import base64
from typing import Any

from .apis import GitLabAPI, GitLabSettings
from .models import GitLabAccessLevel, GitLabVisibility


class GitLabService:
    """A service object for the Gitlab Web Arena evaluations"""

    def __init__(self, apis: GitLabAPI):
        self.apis: GitLabAPI = apis
        self.settings: GitLabSettings = self.apis.settings
        self.validation_data: dict[int, dict[str, Any]] = {}
        self.assertion_msgs: dict[int, list[str]] = {}

    def _update_validation_data(self, *, validation_data: dict[str, Any]):
        """Create a TaskResult and store it in memory. Optionally write all results to disk."""
        if self.settings.task_id is None:
            return
        task_id = self.settings.task_id
        if task_id in self.validation_data:
            self.validation_data[task_id].update(validation_data)
        else:
            self.validation_data[task_id] = validation_data

    def _add_assertion_msg(self, msg: str):
        """Add an assertion message for the current task."""
        if self.settings.task_id is None:
            return
        task_id = self.settings.task_id
        if task_id not in self.assertion_msgs:
            self.assertion_msgs[task_id] = []
        self.assertion_msgs[task_id].append(msg)

    def user_has_project(self, *, project: str | list[str]) -> bool:
        """
        Return True if the user contains the specified project(s).

        - If a string is provided, returns True if that single project exists.
        - If a list of strings is provided, returns True only if all projects exist.
        """
        self._add_assertion_msg(f"Expecting user project {project} to exist")
        projects = self.apis.get_user_projects()
        project_names = {p.name for p in projects} if projects else set()

        self._update_validation_data(
            validation_data={
                "all_user_projects": [
                    p.model_dump() for p in projects
                ]  # Use model_dump for Pydantic objects
            }
        )

        if not projects:
            return False

        if isinstance(project, str):
            return project in project_names

        # Assume a list of project names
        try:
            return all(name in project_names for name in project)
        except TypeError:
            # Provided value isn't iterable like a list of strings
            return False

    def project_has_collaborators(
        self,
        *,
        collaborators: list[str],
        group: str,
        project: str,
        access_level: GitLabAccessLevel = None,
    ) -> bool:
        """
        Return True if the specified collaborators are members of the user's project.
        If access_level is provided, also check that all collaborators have the specified access level.
        """
        self._add_assertion_msg(f"Expecting project {group}/{project} to exist")
        project_obj = self.apis.get_project(group=group, project=project)
        if not project_obj:
            self._add_assertion_msg(
                f"Actual: The project {group}/{project} does not exist"
            )
            return False

        members = self.apis.get_project_members(project_id=project_obj.id)
        member_usernames = (
            {member.username: member for member in members} if members else {}
        )

        self._update_validation_data(
            validation_data={
                "project": project_obj.model_dump(),
                "collaborators": [m.model_dump() for m in members],
            }
        )

        if not members:
            self._add_assertion_msg(
                f"Actual: The project {group}/{project} has no members"
            )
            self._update_validation_data(
                validation_data={
                    "project": project_obj.model_dump(),
                }
            )
            return False

        self._add_assertion_msg(
            f"Expecting all collaborators {collaborators} to be members of project {group}/{project}"
        )
        if access_level is not None:
            self._add_assertion_msg(
                f"Expecting all collaborators to have access level {access_level}"
            )

        actual_members = list(member_usernames.keys())
        self._add_assertion_msg(f"Actual project members: {actual_members}")

        for collaborator_username in collaborators:
            if collaborator_username not in member_usernames:
                self._add_assertion_msg(
                    f"Actual: Collaborator {collaborator_username} is not a member"
                )
                return False  # A collaborator is not a member

            if access_level is not None:
                member = member_usernames[collaborator_username]
                if int(member.access_level) != int(access_level):
                    self._add_assertion_msg(
                        f"Actual: Collaborator {collaborator_username} has access level {member.access_level}, expected {access_level}"
                    )
                    return False  # Access level doesn't match

        return True

    def project_has_visibility_and_members(
        self,
        *,
        group: str,
        project: str,
        visibility: GitLabVisibility,
        members: list[str],
    ) -> bool:
        """
        Return True if the project has the specified visibility and all members are in the project.
        """
        self._add_assertion_msg(f"Expecting project {group}/{project} to exist")
        project_obj = self.apis.get_project(group=group, project=project)
        if not project_obj:
            self._add_assertion_msg(
                f"Actual: The project {group}/{project} does not exist"
            )
            return False

        project_members = self.apis.get_project_members(project_id=project_obj.id)
        member_usernames = (
            {member.username for member in project_members}
            if project_members
            else set()
        )

        self._update_validation_data(
            validation_data={
                "project": project_obj.model_dump(),
                "members": [m.model_dump() for m in project_members],
            }
        )

        self._add_assertion_msg(
            f"Expecting project {group}/{project} to have visibility {visibility}"
        )
        if project_obj.visibility != visibility:
            self._add_assertion_msg(
                f"Actual: The project {group}/{project} has visibility {project_obj.visibility}"
            )
            return False

        self._add_assertion_msg(
            f"Expecting project {group}/{project} to have members: {members}"
        )
        actual_member_usernames = list(member_usernames)
        self._add_assertion_msg(f"Actual project members: {actual_member_usernames}")

        if not project_members:
            self._update_validation_data(
                validation_data={
                    "visibility": project_obj.visibility,
                }
            )
            return not members

        return all(member in member_usernames for member in members)

    def file_has_substring(
        self,
        *,
        group: str,
        project: str,
        file_path: str,
        substring: str | list[str],
        not_substring: str | None = None,
        branch: str = "main",
    ) -> bool:
        """Return True if the specified file in the project contains the substring(s) and not the not_substring.

        - If substring is a string, returns True if that single substring is present.
        - If substring is a list of strings, returns True only if all substrings are present.
        """
        self._add_assertion_msg(f"Expecting project {group}/{project} to exist")
        project_obj = self.apis.get_project(group=group, project=project)
        if not project_obj:
            self._add_assertion_msg(
                f"Actual: The project {group}/{project} does not exist"
            )
            return False

        repo_file = self.apis.get_repository_file(
            project_id=project_obj.id, file_path=file_path, branch=branch
        )
        self._add_assertion_msg(
            f"Expecting file {file_path} to exist in project {group}/{project} on branch {branch}"
        )
        if not repo_file:
            self._add_assertion_msg(
                f"Actual: The repository file {file_path} on branch {branch} does not exist"
            )
            self._update_validation_data(
                validation_data={
                    "project": project_obj.model_dump(),
                }
            )
            return False

        try:
            decoded_content = base64.b64decode(repo_file.content).decode("utf-8")
            self._update_validation_data(
                validation_data={
                    "project": project_obj.model_dump(),
                    "file": repo_file.model_dump(),
                    "content": decoded_content,
                }
            )

            self._add_assertion_msg(
                f"Expecting file {file_path} to contain substring(s): {substring}"
            )
            if not_substring:
                self._add_assertion_msg(
                    f"Expecting file {file_path} to NOT contain substring: '{not_substring}'"
                )

            # Handle both single string and list of strings
            if isinstance(substring, str):
                substring_present = substring in decoded_content
                if not substring_present:
                    self._add_assertion_msg(
                        f"Actual: File {file_path} does not contain required substring: '{substring}'"
                    )
            else:
                # substring is a list of strings - all must be present
                substring_present = all(s in decoded_content for s in substring)
                if not substring_present:
                    missing_substrings = [
                        s for s in substring if s not in decoded_content
                    ]
                    self._add_assertion_msg(
                        f"Actual: File {file_path} does not contain required substring(s): {missing_substrings}"
                    )

            not_substring_absent = (
                not_substring is None or not_substring not in decoded_content
            )

            if not_substring and not not_substring_absent:
                self._add_assertion_msg(
                    f"Actual: File {file_path} contains forbidden substring: '{not_substring}'"
                )

            return substring_present and not_substring_absent
        except Exception as e:
            print(f"Failed to decode file content: {e}")
            return False

    def _get_nested_attr(self, obj: Any, attr_string: str) -> Any:
        """Helper to retrieve nested attributes from an object."""
        attrs = attr_string.split(".")
        value = obj
        for attr in attrs:
            if value is None:
                return None
            if isinstance(value, list):
                # If we encounter a list, check for a match in any item of the list.
                # This is for fields like 'assignees'.
                # The rest of the attr_string is applied to each item.
                remaining_attrs = ".".join(attrs[attrs.index(attr) + 1 :])
                return any(
                    self._get_nested_attr(item, remaining_attrs) for item in value
                )
            try:
                value = getattr(value, attr)
            except AttributeError:
                return None
        return value

    def project_has_issue_with_fields(
        self,
        *,
        group: str,
        project: str,
        author_username: str | None = None,
        fields: dict[str, Any],
        title_field_is_substring: bool = False,
    ) -> bool:
        """Return True if a project has at least one issue matching all specified fields."""
        project_path = f"{group}/{project}"
        issues = self.apis.get_project_issues(
            project_id=project_path, author_username=author_username
        )

        issues_as_dicts = [issue.model_dump() for issue in issues] if issues else []

        self._update_validation_data(validation_data={"issues": issues_as_dicts})

        self._add_assertion_msg(
            f"Expecting project {group}/{project} to have issues for author {author_username}"
        )
        if not issues:
            self._add_assertion_msg(
                f"Actual: The project {group}/{project} has no issues for author {author_username}"
            )
            return False

        for issue in issues:
            is_match = True
            for key, expected_value in fields.items():
                actual_value = self._get_nested_attr(issue, key)

                if (
                    key == "title"
                    and title_field_is_substring
                    and isinstance(actual_value, str)
                    and isinstance(expected_value, str)
                ):
                    if expected_value not in actual_value:
                        is_match = False
                        break
                elif actual_value != expected_value:
                    is_match = False
                    break
            if is_match:
                return True

        return False

    def project_with_visibility_has_initial_commit_message(
        self,
        *,
        group: str,
        project: str,
        visibility: GitLabVisibility,
        initial_commit_message_substrings: list[str],
        require_all_substrings: bool = False,
    ) -> bool:
        """Return True if the project has the specified visibility and the initial commit message matches."""
        self._add_assertion_msg(f"Expecting project {group}/{project} to exist")
        project_obj = self.apis.get_project(group=group, project=project)
        if not project_obj:
            self._add_assertion_msg(
                f"Actual: The project {group}/{project} does not exist"
            )
            return False

        self._add_assertion_msg(
            f"Expecting project {group}/{project} to have visibility {visibility}"
        )
        if project_obj.visibility != visibility:
            self._add_assertion_msg(
                f"Actual: The project {group}/{project} has visibility {project_obj.visibility}"
            )
            return False

        commits = self.apis.get_project_commits(project_id=project_obj.id)

        self._add_assertion_msg(f"Expecting project {group}/{project} to have commits")
        if not commits:
            self._add_assertion_msg(
                f"Actual: The project {group}/{project} has no commits"
            )
            self._update_validation_data(
                validation_data={
                    "visibility": project_obj.visibility,
                }
            )
            return False

        # Commits are returned in reverse chronological order, so the initial commit is the last one.
        initial_commit = commits[-1]

        self._update_validation_data(
            validation_data={
                "project": project_obj.model_dump(),
                "initial_commit": initial_commit.model_dump(),
            }
        )

        # Add detailed assertion messages for commit message checking
        if require_all_substrings:
            self._add_assertion_msg(
                f"Expecting initial commit message to contain ALL substrings: {initial_commit_message_substrings}"
            )
        else:
            self._add_assertion_msg(
                f"Expecting initial commit message to contain ANY of substrings: {initial_commit_message_substrings}"
            )

        self._add_assertion_msg(
            f"Actual initial commit message: '{initial_commit.message}'"
        )

        if require_all_substrings:
            missing_substrings = [
                s
                for s in initial_commit_message_substrings
                if s not in initial_commit.message
            ]
            if missing_substrings:
                self._add_assertion_msg(
                    f"Actual: Missing required substrings: {missing_substrings}"
                )
            return all(
                substring in initial_commit.message
                for substring in initial_commit_message_substrings
            )
        else:
            found_substrings = [
                s
                for s in initial_commit_message_substrings
                if s in initial_commit.message
            ]
            if not found_substrings:
                self._add_assertion_msg(
                    "Actual: None of the required substrings found in commit message"
                )
            else:
                self._add_assertion_msg(f"Actual: Found substrings: {found_substrings}")
            return any(
                substring in initial_commit.message
                for substring in initial_commit_message_substrings
            )

    def project_with_visibility_has_collaborators_and_initial_commit_message(
        self,
        *,
        group: str,
        project: str,
        visibility: GitLabVisibility,
        initial_commit_message_substrings: list[str],
        require_all_substrings: bool = False,
        collaborators: list[str],
        access_level: GitLabAccessLevel = None,
    ) -> bool:
        """
        Return True if the project has the specified visibility, initial commit message, and all collaborators are members.
        """
        self._add_assertion_msg(
            f"Expecting project {group}/{project} to have visibility {visibility}, commit message substrings {initial_commit_message_substrings}, and collaborators {collaborators}"
        )
        if access_level:
            self._add_assertion_msg(
                f"Expecting collaborators to have access level {access_level}"
            )

        has_visibility_and_commit = (
            self.project_with_visibility_has_initial_commit_message(
                group=group,
                project=project,
                visibility=visibility,
                initial_commit_message_substrings=initial_commit_message_substrings,
                require_all_substrings=require_all_substrings,
            )
        )
        has_collaborators = self.project_has_collaborators(
            group=group,
            project=project,
            collaborators=collaborators,
            access_level=access_level,
        )

        # Add detailed results
        if not has_visibility_and_commit:
            self._add_assertion_msg(
                "Actual: Project failed visibility and/or commit message checks"
            )
        if not has_collaborators:
            self._add_assertion_msg("Actual: Project failed collaborators check")
        if has_visibility_and_commit and has_collaborators:
            self._add_assertion_msg(
                "Actual: Project passed all checks (visibility, commit message, and collaborators)"
            )

        self._update_validation_data(
            validation_data={
                "has_visibility_and_commit": has_visibility_and_commit,
                "has_collaborators": has_collaborators,
            }
        )

        return has_visibility_and_commit and has_collaborators

    def project_has_fields(
        self,
        *,
        group: str,
        project: str,
        fields: dict[str, Any],
        description_is_substring: bool = False,
    ) -> bool:
        """Return True if a project has all specified fields matching the given values."""
        project_obj = self.apis.get_project(group=group, project=project)

        self._add_assertion_msg(f"Expecting project {group}/{project} to exist")
        if not project_obj:
            self._add_assertion_msg(
                f"Actual: The project {group}/{project} does not exist"
            )
            return False

        self._update_validation_data(
            validation_data={
                "project": project_obj.model_dump() if project_obj else None
            }
        )

        self._add_assertion_msg(
            f"Expecting project {group}/{project} to have fields: {fields}"
        )

        for key, expected_value in fields.items():
            actual_value = self._get_nested_attr(project_obj, key)

            if (
                key == "description"
                and description_is_substring
                and isinstance(actual_value, str)
                and isinstance(expected_value, str)
            ):
                self._add_assertion_msg(
                    f"Expecting field '{key}' to contain substring: '{expected_value}'"
                )
                self._add_assertion_msg(f"Actual field '{key}' value: '{actual_value}'")
                if expected_value not in actual_value:
                    self._add_assertion_msg(
                        f"Actual: Field '{key}' does not contain expected substring '{expected_value}'"
                    )
                    return False
            else:
                self._add_assertion_msg(
                    f"Expecting field '{key}' to equal: {expected_value}"
                )
                self._add_assertion_msg(f"Actual field '{key}' value: {actual_value}")
                if actual_value != expected_value:
                    self._add_assertion_msg(
                        f"Actual: Field '{key}' mismatch - expected {expected_value}, got {actual_value}"
                    )
                    return False

        self._add_assertion_msg(
            f"Actual: All fields match expected values for project {group}/{project}"
        )
        return True

    def project_has_fields_and_collaborators(
        self,
        *,
        group: str,
        project: str,
        fields: dict[str, Any],
        collaborators: list[str],
        access_level: GitLabAccessLevel = None,
        description_is_substring: bool = False,
    ) -> bool:
        """
        Return True if a project has matching fields and all collaborators are members.
        """
        self._add_assertion_msg(
            f"Expecting project {group}/{project} to have fields {fields} and collaborators {collaborators}"
        )
        if access_level:
            self._add_assertion_msg(
                f"Expecting collaborators to have access level {access_level}"
            )

        has_fields = self.project_has_fields(
            group=group,
            project=project,
            fields=fields,
            description_is_substring=description_is_substring,
        )
        has_collaborators = self.project_has_collaborators(
            collaborators=collaborators,
            group=group,
            project=project,
            access_level=access_level,
        )

        # Add detailed results
        if not has_fields:
            self._add_assertion_msg("Actual: Project failed fields check")
        if not has_collaborators:
            self._add_assertion_msg("Actual: Project failed collaborators check")
        if has_fields and has_collaborators:
            self._add_assertion_msg(
                "Actual: Project passed both fields and collaborators checks"
            )

        self._update_validation_data(
            validation_data={
                "has_fields": has_fields,
                "has_collaborators": has_collaborators,
            }
        )

        return has_fields and has_collaborators

    def does_milestone_exist_with_fields(
        self,
        *,
        group: str,
        project: str,
        values: dict[str, Any],
        title_field_is_substring: bool = False,
    ) -> bool:
        """Return True if a project has at least one milestone matching all specified values. If title_field_is_substring is True and 'title' is provided, the expected title is treated as a substring."""
        milestones = self.apis.get_project_milestones(group=group, project=project)

        milestones_as_dicts = [m.model_dump() for m in milestones] if milestones else []

        self._update_validation_data(
            validation_data={
                "milestones": milestones_as_dicts,
            }
        )

        self._add_assertion_msg(
            f"Expecting project {group}/{project} to have at least one milestone with fields: {values}"
        )

        if not milestones:
            self._add_assertion_msg(
                f"Actual: The project {group}/{project} has no milestones"
            )
            return False

        self._add_assertion_msg(
            f"Actual: Found {len(milestones)} milestone(s) in project {group}/{project}"
        )

        for i, milestone in enumerate(milestones):
            milestone_title = getattr(milestone, "title", "Unknown")
            self._add_assertion_msg(f"Checking milestone {i + 1}: '{milestone_title}'")

            is_match = True
            for key, expected_value in values.items():
                actual_value = self._get_nested_attr(milestone, key)

                if (
                    key == "title"
                    and title_field_is_substring
                    and isinstance(actual_value, str)
                    and isinstance(expected_value, str)
                ):
                    if expected_value not in actual_value:
                        self._add_assertion_msg(
                            f"  Field '{key}': Expected substring '{expected_value}' not found in '{actual_value}'"
                        )
                        is_match = False
                        break
                    else:
                        self._add_assertion_msg(
                            f"  Field '{key}': Found substring '{expected_value}' in '{actual_value}'"
                        )
                elif actual_value != expected_value:
                    self._add_assertion_msg(
                        f"  Field '{key}': Expected {expected_value}, got {actual_value}"
                    )
                    is_match = False
                    break
                else:
                    self._add_assertion_msg(
                        f"  Field '{key}': Matches expected value {expected_value}"
                    )

            if is_match:
                self._add_assertion_msg(
                    f"Actual: Found matching milestone '{milestone_title}' with all required fields"
                )
                return True

        self._add_assertion_msg(
            "Actual: No milestone found matching all required field values"
        )
        return False

    def is_user_following(
        self,
        *,
        username: str,
        following_usernames: list[str],
    ) -> bool:
        """Return True if the user is following all specified usernames."""
        self._add_assertion_msg(f"Expecting user {username} to exist")
        user = self.apis.get_user_by_username(username=username)
        if not user:
            self._add_assertion_msg(f"Actual: The user {username} does not exist")
            return False

        following = self.apis.get_user_following(user_id=user.id)
        following_list = [f.username for f in following] if following else []

        self._update_validation_data(
            validation_data={
                "user": user.model_dump(),
                "following": [f.model_dump() for f in following],
            }
        )

        return all(u in following_list for u in following_usernames)

    def issue_assignee_is(
        self,
        *,
        usernames: list[str],
        title: str | None = None,
        group: str | None = None,
        project: str | None = None,
        issue_iid: int | None = None,
    ) -> bool:
        """Return True if any of the specified usernames is an assignee of the matching issue.

        Two modes:
        - Provide group, project, and issue_iid to check a specific project issue.
        - Or provide title to search issues by exact title.
        If both are provided, the project issue lookup is used.
        """
        username_set = set(usernames)
        issues_as_dicts = []
        actual_issues = []

        # Add detailed expectation messages
        if group and project and issue_iid is not None:
            self._add_assertion_msg(
                f"Expecting issue #{issue_iid} in project {group}/{project} to be assigned to one of: {usernames}"
            )
        elif title:
            self._add_assertion_msg(
                f"Expecting issue with title '{title}' to be assigned to one of: {usernames}"
            )
        else:
            self._add_assertion_msg(
                f"Expecting to find an issue assigned to one of: {usernames}"
            )

        # Prefer explicit project-scoped issue lookup when identifiers are provided
        if group and project and issue_iid is not None:
            issue = self.apis.get_issue_from_project(
                group=group, project=project, issue_iid=issue_iid
            )
            if issue:
                actual_issues.append(issue)
                self._add_assertion_msg(
                    f"Actual: Found issue #{issue_iid} '{issue.title}' in project {group}/{project}"
                )
            else:
                self._add_assertion_msg(
                    f"Actual: Issue #{issue_iid} not found in project {group}/{project}"
                )
        # Fallback to title-based search
        elif title:
            issues = self.apis.get_issue(search=title, in_param="title")
            if issues and isinstance(issues, list):
                actual_issues.extend(issues)
                self._add_assertion_msg(
                    f"Actual: Found {len(issues)} issue(s) matching title search '{title}'"
                )
            else:
                self._add_assertion_msg(
                    f"Actual: No issues found matching title '{title}'"
                )

        if actual_issues:
            issues_as_dicts = [i.model_dump() for i in actual_issues]

        self._update_validation_data(validation_data={"issues": issues_as_dicts})

        if not actual_issues:
            self._add_assertion_msg("Actual: No issues found matching the criteria")
            self._update_validation_data(
                validation_data={
                    "issues": [],
                }
            )
            return False

        for issue in actual_issues:
            if title and issue.title != title:
                self._add_assertion_msg(
                    f"Skipping issue '{issue.title}' - title doesn't match exactly"
                )
                continue

            # Check single assignee
            if issue.assignee and issue.assignee.username in username_set:
                self._add_assertion_msg(
                    f"Actual: Issue '{issue.title}' is assigned to {issue.assignee.username} (matches expected)"
                )
                return True

            # Check multiple assignees
            if issue.assignees:
                matching_assignees = [
                    a.username for a in issue.assignees if a.username in username_set
                ]
                if matching_assignees:
                    self._add_assertion_msg(
                        f"Actual: Issue '{issue.title}' has matching assignees: {matching_assignees}"
                    )
                    return True
                else:
                    all_assignees = [a.username for a in issue.assignees]
                    self._add_assertion_msg(
                        f"Actual: Issue '{issue.title}' has assignees {all_assignees}, none match expected {usernames}"
                    )
            else:
                self._add_assertion_msg(
                    f"Actual: Issue '{issue.title}' has no assignees"
                )

        self._add_assertion_msg(
            "Actual: No matching issues found with any of the expected assignees"
        )
        return False

    def check_user_status_message(
        self, *, user_id: int | str, compare_str: str
    ) -> bool:
        """Return True if the user's status message matches the compare string."""
        user_status = self.apis.get_user_status(user_id=user_id)
        self._update_validation_data(
            validation_data={
                "user_status": user_status.model_dump() if user_status else None
            }
        )
        self._add_assertion_msg(
            f"Expecting user {user_id} to have status message: '{compare_str}'"
        )
        if not user_status:
            self._add_assertion_msg(f"Actual: The user with id {user_id} has no status")
            # If the user has no status, the message is an empty string
            return compare_str == ""

        return user_status.message == compare_str

    def user_has_starred_project(
        self, *, user_id: int | str, project_names: list[str]
    ) -> bool:
        """
        Return True if the user has starred all the specified projects.
        """
        starred_projects = self.apis.get_user_starred_projects(user_id=user_id)
        starred_project_names = (
            {p.name for p in starred_projects} if starred_projects else set()
        )

        self._update_validation_data(
            validation_data={
                "starred_projects": [p.model_dump() for p in starred_projects]
            }
        )

        self._add_assertion_msg(
            f"Expecting user {user_id} to have starred projects: {project_names}"
        )
        if not starred_projects:
            self._add_assertion_msg(
                f"Actual: The user with id {user_id} has no starred projects"
            )
            return False

        return all(name in starred_project_names for name in project_names)

    def does_merge_request_have_comment(
        self,
        *,
        group: str,
        project: str,
        merge_request_iid: int,
        comment: str,
    ) -> bool:
        """Return True if the specified merge request has a comment with the specified text."""
        notes = self.apis.get_merge_request_notes(
            group=group, project=project, merge_request_iid=merge_request_iid
        )

        self._update_validation_data(
            validation_data={"notes": [n.model_dump() for n in notes]}
        )

        self._add_assertion_msg(
            f"Expecting merge request {merge_request_iid} in project {group}/{project} to have comment: '{comment}'"
        )
        if not notes:
            self._add_assertion_msg(
                f"Actual: The merge request {merge_request_iid} in project {group}/{project} has no comments"
            )
            self._update_validation_data(
                validation_data={
                    "notes": [],
                }
            )
            return False

        for note in notes:
            if note.body == comment:
                return True

        return False

    def is_user_website_url(self, *, username: str, website_url: str) -> bool:
        """Return True if the user's website URL matches the specified URL."""
        user = self.apis.get_user_by_username(username=username)
        if not user:
            return False

        user_profile = self.apis.get_user_profile(user_id=user.id)
        user_website_url = user_profile.website_url or "" if user_profile else ""

        self._update_validation_data(
            validation_data={
                "user": user.model_dump(),
                "user_profile": user_profile.model_dump() if user_profile else None,
            }
        )

        self._add_assertion_msg(
            f"Expecting user {username} to have website URL: '{website_url}'"
        )
        if not user_profile:
            self._add_assertion_msg(f"Actual: The user {username} has no profile")
            return False

        return user_website_url == website_url

    def check_merge_request_has_values(
        self,
        *,
        group: str,
        project: str,
        source_branch: str,
        target_branch: str,
        reviewer: str,
    ) -> bool:
        """Return True if a merge request exists with the specified values."""
        merge_requests = self.apis.get_merge_requests(
            group=group, project=project, reviewer_username=reviewer
        )

        mr_as_dicts = (
            [mr.model_dump() for mr in merge_requests] if merge_requests else []
        )

        self._update_validation_data(
            validation_data={
                "merge_requests": mr_as_dicts,
            }
        )

        self._add_assertion_msg(
            f"Expecting merge request in project {group}/{project} with source branch '{source_branch}', target branch '{target_branch}', and reviewer '{reviewer}'"
        )

        if not merge_requests:
            self._add_assertion_msg(
                f"Actual: No merge requests found for reviewer {reviewer} in project {group}/{project}"
            )
            self._update_validation_data(
                validation_data={
                    "merge_requests": [],
                }
            )
            return False

        self._add_assertion_msg(
            f"Actual: Found {len(merge_requests)} merge request(s) for reviewer {reviewer}"
        )

        for i, mr in enumerate(merge_requests):
            mr_title = getattr(mr, "title", "Unknown")
            self._add_assertion_msg(
                f"Checking merge request {i + 1}: '{mr_title}' (source: {mr.source_branch}, target: {mr.target_branch})"
            )

            if mr.source_branch == source_branch and mr.target_branch == target_branch:
                self._add_assertion_msg(
                    f"  Branch match found - source: '{source_branch}', target: '{target_branch}'"
                )

                # Check if the reviewer is in the list of reviewers
                reviewer_usernames = (
                    [r.username for r in mr.reviewers] if mr.reviewers else []
                )
                self._add_assertion_msg(
                    f"  Merge request reviewers: {reviewer_usernames}"
                )

                for r in mr.reviewers:
                    if r.username == reviewer:
                        self._add_assertion_msg(
                            f"Actual: Found matching merge request '{mr_title}' with correct branches and reviewer '{reviewer}'"
                        )
                        return True

                self._add_assertion_msg(
                    f"  Reviewer '{reviewer}' not found in reviewers list"
                )
            else:
                self._add_assertion_msg(
                    f"  Branch mismatch - expected source: '{source_branch}', target: '{target_branch}'; got source: '{mr.source_branch}', target: '{mr.target_branch}'"
                )

        self._add_assertion_msg(
            "Actual: No merge request found matching all criteria (source branch, target branch, and reviewer)"
        )
        return False

    def does_group_exist_with_members(
        self, *, group_name: str, member_usernames: list[str]
    ) -> bool:
        """Return True if a group exists with the specified members."""
        self._add_assertion_msg("Expecting groups to exist")
        groups = self.apis.get_groups()
        if not groups:
            self._add_assertion_msg("Actual: No groups found")
            return False

        target_group = None
        for group in groups:
            if group.name == group_name:
                target_group = group
                break

        self._add_assertion_msg(f"Expecting group {group_name} to exist")
        if not target_group:
            self._add_assertion_msg(f"Actual: The group {group_name} does not exist")
            self._update_validation_data(
                validation_data={
                    "groups": [g.model_dump() for g in groups],
                }
            )
            return False

        members = self.apis.get_group_members(group_id=target_group.id)
        existing_member_usernames = (
            {member.username for member in members} if members else set()
        )

        self._update_validation_data(
            validation_data={
                "group": target_group.model_dump(),
                "members": [m.model_dump() for m in members],
            }
        )

        self._add_assertion_msg(
            f"Expecting group {group_name} to have members: {member_usernames}"
        )
        actual_members = list(existing_member_usernames)
        self._add_assertion_msg(f"Actual group members: {actual_members}")

        if not members:
            self._update_validation_data(
                validation_data={
                    "group": target_group.model_dump(),
                }
            )
            return not member_usernames

        return all(
            username in existing_member_usernames for username in member_usernames
        )
