import logging
from urllib.parse import urlparse

from ..models import AllocationResource

from ..types import (
    EvalFunc,
    WebArenaTask,
    WebarenaTaskEvalResult,
    WebArenaTaskResponse,
)

from .apis import GitLabAPI, GitLabSettings
from .services import GitLabService


class GitLabValidator:
    """
    GitLab validator with custom evaluation functions for WebArena Verified tasks.

    Contains validation methods that are called by the custom evaluator system.
    Uses GitLabService for all data access.
    """

    def __init__(self) -> None:
        self.logger = logging.getLogger(__name__)

    def _get_service(
        self, resource: AllocationResource, task_id: int, recreate_private_token=False
    ) -> GitLabService:
        parsed_url = urlparse(resource.base_url)
        settings = GitLabSettings(
            gitlab_host=parsed_url.hostname,
            gitlab_port=parsed_url.port,
            gitlab_username=resource.username,
            task_id=task_id,
        )
        api = GitLabAPI(settings=settings)

        # gitlab-small comes with known private token but recreate_private_token can be used for other gitlab images
        if recreate_private_token:
            settings.private_token = api._admin_create_personal_access_token()
        return GitLabService(apis=api)

    def user_has_project(
        self,
        task: WebArenaTask,
        task_result: WebArenaTaskResponse,
        resource: AllocationResource,
        eval_func: EvalFunc,
        **kwargs,
    ) -> WebarenaTaskEvalResult:
        service = self._get_service(resource, task.task_id)
        eval_params = eval_func.eval_params
        self.logger.info(f"Calling user_has_project with params: {eval_params}")
        is_success = service.user_has_project(**eval_params)
        task_id = task.task_id
        validation_data = service.validation_data.get(task_id)
        assertion_msgs = service.assertion_msgs.get(task_id, [])

        if is_success:
            return WebarenaTaskEvalResult.create_success(
                assertion_msgs=assertion_msgs,
                site=resource.website_type,
                validation_data=validation_data,
                task_id=task_id,
            )
        else:
            return WebarenaTaskEvalResult.create_failed(
                assertion_msgs=assertion_msgs,
                site=resource.website_type,
                task_id=task_id,
                validation_data=validation_data,
            )

    def project_has_collaborators(
        self,
        task: WebArenaTask,
        task_result: WebArenaTaskResponse,
        resource: AllocationResource,
        eval_func: EvalFunc,
        **kwargs,
    ) -> WebarenaTaskEvalResult:
        service = self._get_service(resource, task.task_id)
        eval_params = eval_func.eval_params
        self.logger.info(
            f"Calling project_has_collaborators with params: {eval_params}"
        )
        is_success = service.project_has_collaborators(**eval_params)
        task_id = task.task_id
        validation_data = service.validation_data.get(task_id)
        assertion_msgs = service.assertion_msgs.get(task_id, [])

        if is_success:
            return WebarenaTaskEvalResult.create_success(
                assertion_msgs=assertion_msgs,
                site=resource.website_type,
                validation_data=validation_data,
                task_id=task_id,
            )
        else:
            return WebarenaTaskEvalResult.create_failed(
                assertion_msgs=assertion_msgs,
                site=resource.website_type,
                task_id=task_id,
                validation_data=validation_data,
            )

    def project_has_visibility_and_members(
        self,
        task: WebArenaTask,
        task_result: WebArenaTaskResponse,
        resource: AllocationResource,
        eval_func: EvalFunc,
        **kwargs,
    ) -> WebarenaTaskEvalResult:
        service = self._get_service(resource, task.task_id)
        eval_params = eval_func.eval_params
        self.logger.info(
            f"Calling project_has_visibility_and_members with params: {eval_params}"
        )
        is_success = service.project_has_visibility_and_members(**eval_params)
        task_id = task.task_id
        validation_data = service.validation_data.get(task_id)
        assertion_msgs = service.assertion_msgs.get(task_id, [])

        if is_success:
            return WebarenaTaskEvalResult.create_success(
                assertion_msgs=assertion_msgs,
                site=resource.website_type,
                validation_data=validation_data,
                task_id=task_id,
            )
        else:
            return WebarenaTaskEvalResult.create_failed(
                assertion_msgs=assertion_msgs,
                site=resource.website_type,
                task_id=task_id,
                validation_data=validation_data,
            )

    def file_has_substring(
        self,
        task: WebArenaTask,
        task_result: WebArenaTaskResponse,
        resource: AllocationResource,
        eval_func: EvalFunc,
        **kwargs,
    ) -> WebarenaTaskEvalResult:
        service = self._get_service(resource, task.task_id)
        eval_params = eval_func.eval_params
        self.logger.info(f"Calling file_has_substring with params: {eval_params}")
        is_success = service.file_has_substring(**eval_params)
        task_id = task.task_id
        validation_data = service.validation_data.get(task_id)
        assertion_msgs = service.assertion_msgs.get(task_id, [])

        if is_success:
            return WebarenaTaskEvalResult.create_success(
                assertion_msgs=assertion_msgs,
                site=resource.website_type,
                validation_data=validation_data,
                task_id=task_id,
            )
        else:
            return WebarenaTaskEvalResult.create_failed(
                assertion_msgs=assertion_msgs,
                site=resource.website_type,
                task_id=task_id,
                validation_data=validation_data,
            )

    def project_has_issue_with_fields(
        self,
        task: WebArenaTask,
        task_result: WebArenaTaskResponse,
        resource: AllocationResource,
        eval_func: EvalFunc,
        **kwargs,
    ) -> WebarenaTaskEvalResult:
        service = self._get_service(resource, task.task_id)
        eval_params = eval_func.eval_params
        self.logger.info(
            f"Calling project_has_issue_with_fields with params: {eval_params}"
        )
        is_success = service.project_has_issue_with_fields(**eval_params)
        task_id = task.task_id
        validation_data = service.validation_data.get(task_id)
        assertion_msgs = service.assertion_msgs.get(task_id, [])

        if is_success:
            return WebarenaTaskEvalResult.create_success(
                assertion_msgs=assertion_msgs,
                site=resource.website_type,
                validation_data=validation_data,
                task_id=task_id,
            )
        else:
            return WebarenaTaskEvalResult.create_failed(
                assertion_msgs=assertion_msgs,
                site=resource.website_type,
                task_id=task_id,
                validation_data=validation_data,
            )

    def project_with_visibility_has_initial_commit_message(
        self,
        task: WebArenaTask,
        task_result: WebArenaTaskResponse,
        resource: AllocationResource,
        eval_func: EvalFunc,
        **kwargs,
    ) -> WebarenaTaskEvalResult:
        service = self._get_service(resource, task.task_id)
        eval_params = eval_func.eval_params
        self.logger.info(
            f"Calling project_with_visibility_has_initial_commit_message with params: {eval_params}"
        )
        is_success = service.project_with_visibility_has_initial_commit_message(
            **eval_params
        )
        task_id = task.task_id
        validation_data = service.validation_data.get(task_id)
        assertion_msgs = service.assertion_msgs.get(task_id, [])

        if is_success:
            return WebarenaTaskEvalResult.create_success(
                assertion_msgs=assertion_msgs,
                site=resource.website_type,
                validation_data=validation_data,
                task_id=task_id,
            )
        else:
            return WebarenaTaskEvalResult.create_failed(
                assertion_msgs=assertion_msgs,
                site=resource.website_type,
                task_id=task_id,
                validation_data=validation_data,
            )

    def project_with_visibility_has_collaborators_and_initial_commit_message(
        self,
        task: WebArenaTask,
        task_result: WebArenaTaskResponse,
        resource: AllocationResource,
        eval_func: EvalFunc,
        **kwargs,
    ) -> WebarenaTaskEvalResult:
        service = self._get_service(resource, task.task_id)
        eval_params = eval_func.eval_params
        self.logger.info(
            f"Calling project_with_visibility_has_collaborators_and_initial_commit_message with params: {eval_params}"
        )
        is_success = service.project_with_visibility_has_collaborators_and_initial_commit_message(
            **eval_params
        )
        task_id = task.task_id
        validation_data = service.validation_data.get(task_id)
        assertion_msgs = service.assertion_msgs.get(task_id, [])

        if is_success:
            return WebarenaTaskEvalResult.create_success(
                assertion_msgs=assertion_msgs,
                site=resource.website_type,
                validation_data=validation_data,
                task_id=task_id,
            )
        else:
            return WebarenaTaskEvalResult.create_failed(
                assertion_msgs=assertion_msgs,
                site=resource.website_type,
                task_id=task_id,
                validation_data=validation_data,
            )

    def project_has_fields(
        self,
        task: WebArenaTask,
        task_result: WebArenaTaskResponse,
        resource: AllocationResource,
        eval_func: EvalFunc,
        **kwargs,
    ) -> WebarenaTaskEvalResult:
        service = self._get_service(resource, task.task_id)
        eval_params = eval_func.eval_params
        self.logger.info(f"Calling project_has_fields with params: {eval_params}")
        is_success = service.project_has_fields(**eval_params)
        task_id = task.task_id
        validation_data = service.validation_data.get(task_id)
        assertion_msgs = service.assertion_msgs.get(task_id, [])

        if is_success:
            return WebarenaTaskEvalResult.create_success(
                assertion_msgs=assertion_msgs,
                site=resource.website_type,
                validation_data=validation_data,
                task_id=task_id,
            )
        else:
            return WebarenaTaskEvalResult.create_failed(
                assertion_msgs=assertion_msgs,
                site=resource.website_type,
                task_id=task_id,
                validation_data=validation_data,
            )

    def project_has_fields_and_collaborators(
        self,
        task: WebArenaTask,
        task_result: WebArenaTaskResponse,
        resource: AllocationResource,
        eval_func: EvalFunc,
        **kwargs,
    ) -> WebarenaTaskEvalResult:
        service = self._get_service(resource, task.task_id)
        eval_params = eval_func.eval_params
        self.logger.info(
            f"Calling project_has_fields_and_collaborators with params: {eval_params}"
        )
        is_success = service.project_has_fields_and_collaborators(**eval_params)
        task_id = task.task_id
        validation_data = service.validation_data.get(task_id)
        assertion_msgs = service.assertion_msgs.get(task_id, [])

        if is_success:
            return WebarenaTaskEvalResult.create_success(
                assertion_msgs=assertion_msgs,
                site=resource.website_type,
                validation_data=validation_data,
                task_id=task_id,
            )
        else:
            return WebarenaTaskEvalResult.create_failed(
                assertion_msgs=assertion_msgs,
                site=resource.website_type,
                task_id=task_id,
                validation_data=validation_data,
            )

    def does_milestone_exist_with_fields(
        self,
        task: WebArenaTask,
        task_result: WebArenaTaskResponse,
        resource: AllocationResource,
        eval_func: EvalFunc,
        **kwargs,
    ) -> WebarenaTaskEvalResult:
        service = self._get_service(resource, task.task_id)
        eval_params = eval_func.eval_params
        self.logger.info(
            f"Calling does_milestone_exist_with_fields with params: {eval_params}"
        )
        is_success = service.does_milestone_exist_with_fields(**eval_params)
        task_id = task.task_id
        validation_data = service.validation_data.get(task_id)
        assertion_msgs = service.assertion_msgs.get(task_id, [])

        if is_success:
            return WebarenaTaskEvalResult.create_success(
                assertion_msgs=assertion_msgs,
                site=resource.website_type,
                validation_data=validation_data,
                task_id=task_id,
            )
        else:
            return WebarenaTaskEvalResult.create_failed(
                assertion_msgs=assertion_msgs,
                site=resource.website_type,
                task_id=task_id,
                validation_data=validation_data,
            )

    def is_user_following(
        self,
        task: WebArenaTask,
        task_result: WebArenaTaskResponse,
        resource: AllocationResource,
        eval_func: EvalFunc,
        **kwargs,
    ) -> WebarenaTaskEvalResult:
        service = self._get_service(resource, task.task_id)
        eval_params = eval_func.eval_params
        self.logger.info(f"Calling is_user_following with params: {eval_params}")
        is_success = service.is_user_following(**eval_params)
        task_id = task.task_id
        validation_data = service.validation_data.get(task_id)
        assertion_msgs = service.assertion_msgs.get(task_id, [])

        if is_success:
            return WebarenaTaskEvalResult.create_success(
                assertion_msgs=assertion_msgs,
                site=resource.website_type,
                validation_data=validation_data,
                task_id=task_id,
            )
        else:
            return WebarenaTaskEvalResult.create_failed(
                assertion_msgs=assertion_msgs,
                site=resource.website_type,
                task_id=task_id,
                validation_data=validation_data,
            )

    def issue_assignee_is(
        self,
        task: WebArenaTask,
        task_result: WebArenaTaskResponse,
        resource: AllocationResource,
        eval_func: EvalFunc,
        **kwargs,
    ) -> WebarenaTaskEvalResult:
        service = self._get_service(resource, task.task_id)
        eval_params = eval_func.eval_params
        self.logger.info(f"Calling issue_assignee_is with params: {eval_params}")
        is_success = service.issue_assignee_is(**eval_params)
        task_id = task.task_id
        validation_data = service.validation_data.get(task_id)
        assertion_msgs = service.assertion_msgs.get(task_id, [])

        if is_success:
            return WebarenaTaskEvalResult.create_success(
                assertion_msgs=assertion_msgs,
                site=resource.website_type,
                validation_data=validation_data,
                task_id=task_id,
            )
        else:
            return WebarenaTaskEvalResult.create_failed(
                assertion_msgs=assertion_msgs,
                site=resource.website_type,
                task_id=task_id,
                validation_data=validation_data,
            )

    def check_user_status_message(
        self,
        task: WebArenaTask,
        task_result: WebArenaTaskResponse,
        resource: AllocationResource,
        eval_func: EvalFunc,
        **kwargs,
    ) -> WebarenaTaskEvalResult:
        service = self._get_service(resource, task.task_id)
        eval_params = eval_func.eval_params
        self.logger.info(
            f"Calling check_user_status_message with params: {eval_params}"
        )
        is_success = service.check_user_status_message(**eval_params)
        task_id = task.task_id
        validation_data = service.validation_data.get(task_id)
        assertion_msgs = service.assertion_msgs.get(task_id, [])

        if is_success:
            return WebarenaTaskEvalResult.create_success(
                assertion_msgs=assertion_msgs,
                site=resource.website_type,
                validation_data=validation_data,
                task_id=task_id,
            )
        else:
            return WebarenaTaskEvalResult.create_failed(
                assertion_msgs=assertion_msgs,
                site=resource.website_type,
                task_id=task_id,
                validation_data=validation_data,
            )

    def user_has_starred_project(
        self,
        task: WebArenaTask,
        task_result: WebArenaTaskResponse,
        resource: AllocationResource,
        eval_func: EvalFunc,
        **kwargs,
    ) -> WebarenaTaskEvalResult:
        service = self._get_service(resource, task.task_id)
        eval_params = eval_func.eval_params
        self.logger.info(f"Calling user_has_starred_project with params: {eval_params}")
        is_success = service.user_has_starred_project(**eval_params)
        task_id = task.task_id
        validation_data = service.validation_data.get(task_id)
        assertion_msgs = service.assertion_msgs.get(task_id, [])

        if is_success:
            return WebarenaTaskEvalResult.create_success(
                assertion_msgs=assertion_msgs,
                site=resource.website_type,
                validation_data=validation_data,
                task_id=task_id,
            )
        else:
            return WebarenaTaskEvalResult.create_failed(
                assertion_msgs=assertion_msgs,
                site=resource.website_type,
                task_id=task_id,
                validation_data=validation_data,
            )

    def does_merge_request_have_comment(
        self,
        task: WebArenaTask,
        task_result: WebArenaTaskResponse,
        resource: AllocationResource,
        eval_func: EvalFunc,
        **kwargs,
    ) -> WebarenaTaskEvalResult:
        service = self._get_service(resource, task.task_id)
        eval_params = eval_func.eval_params
        self.logger.info(
            f"Calling does_merge_request_have_comment with params: {eval_params}"
        )
        is_success = service.does_merge_request_have_comment(**eval_params)
        task_id = task.task_id
        validation_data = service.validation_data.get(task_id)
        assertion_msgs = service.assertion_msgs.get(task_id, [])

        if is_success:
            return WebarenaTaskEvalResult.create_success(
                assertion_msgs=assertion_msgs,
                site=resource.website_type,
                validation_data=validation_data,
                task_id=task_id,
            )
        else:
            return WebarenaTaskEvalResult.create_failed(
                assertion_msgs=assertion_msgs,
                site=resource.website_type,
                task_id=task_id,
                validation_data=validation_data,
            )

    def is_user_website_url(
        self,
        task: WebArenaTask,
        task_result: WebArenaTaskResponse,
        resource: AllocationResource,
        eval_func: EvalFunc,
        **kwargs,
    ) -> WebarenaTaskEvalResult:
        service = self._get_service(resource, task.task_id)
        eval_params = eval_func.eval_params
        self.logger.info(f"Calling is_user_website_url with params: {eval_params}")
        is_success = service.is_user_website_url(**eval_params)
        task_id = task.task_id
        validation_data = service.validation_data.get(task_id)

        if is_success:
            return WebarenaTaskEvalResult.create_success(
                assertion_msgs=["Successfully verified the user's website URL."],
                site=resource.website_type,
                validation_data=validation_data,
                task_id=task_id,
            )
        else:
            return WebarenaTaskEvalResult.create_failed(
                assertion_msgs=["Failed to verify the user's website URL."],
                site=resource.website_type,
                task_id=task_id,
                validation_data=validation_data,
            )

    def check_merge_request_has_values(
        self,
        task: WebArenaTask,
        task_result: WebArenaTaskResponse,
        resource: AllocationResource,
        eval_func: EvalFunc,
        **kwargs,
    ) -> WebarenaTaskEvalResult:
        service = self._get_service(resource, task.task_id)
        eval_params = eval_func.eval_params
        self.logger.info(
            f"Calling check_merge_request_has_values with params: {eval_params}"
        )
        is_success = service.check_merge_request_has_values(**eval_params)
        task_id = task.task_id
        validation_data = service.validation_data.get(task_id)
        assertion_msgs = service.assertion_msgs.get(task_id, [])

        if is_success:
            return WebarenaTaskEvalResult.create_success(
                assertion_msgs=assertion_msgs,
                site=resource.website_type,
                validation_data=validation_data,
                task_id=task_id,
            )
        else:
            return WebarenaTaskEvalResult.create_failed(
                assertion_msgs=assertion_msgs,
                site=resource.website_type,
                task_id=task_id,
                validation_data=validation_data,
            )

    def does_group_exist_with_members(
        self,
        task: WebArenaTask,
        task_result: WebArenaTaskResponse,
        resource: AllocationResource,
        eval_func: EvalFunc,
        **kwargs,
    ) -> WebarenaTaskEvalResult:
        service = self._get_service(resource, task.task_id)
        eval_params = eval_func.eval_params
        self.logger.info(
            f"Calling does_group_exist_with_members with params: {eval_params}"
        )
        is_success = service.does_group_exist_with_members(**eval_params)
        task_id = task.task_id
        validation_data = service.validation_data.get(task_id)
        assertion_msgs = service.assertion_msgs.get(task_id, [])

        if is_success:
            return WebarenaTaskEvalResult.create_success(
                assertion_msgs=assertion_msgs,
                site=resource.website_type,
                validation_data=validation_data,
                task_id=task_id,
            )
        else:
            return WebarenaTaskEvalResult.create_failed(
                assertion_msgs=assertion_msgs,
                site=resource.website_type,
                task_id=task_id,
                validation_data=validation_data,
            )
