"""
Magento Data Helper for retrieving data as Pydantic v2 models.

This module provides data retrieval capabilities for Magento e-commerce
operations, returning structured Pydantic models via API calls where possible.
"""

import logging
import requests
import subprocess
from typing import List, Optional, Dict
from urllib.parse import quote_plus
import json

from .models import (
    Address,
    Cart,
    CartItem,
    CustomerInfo,
    MagentoConfig,
    Order,
    Product,
    ProductInventory,
    WishlistItem,
    NewsletterSubscription,
    ProductReview,
    SalesRule,
    CmsPage,
)


class MagentoDataHelper:
    """
    Helper class for retrieving Magento data as Pydantic models.

    This class handles API calls and database queries as fallback, returning
    structured Pydantic v2 models instead of raw data.
    """

    def __init__(self, config: MagentoConfig):
        self.config = config
        self.logger = logging.getLogger(__name__)
        self.base_url = f"{config.root_url}/rest/all/V1"
        self.admin_token = None
        self.customer_token = None

    def _get_admin_token(self) -> Optional[str]:
        """Get admin authentication token."""
        if self.admin_token:
            return self.admin_token

        try:
            url = f"{self.base_url}/integration/admin/token"
            payload = {
                "username": self.config.admin_username,
                "password": self.config.admin_password,
            }

            response = requests.post(url, json=payload, timeout=30)
            if response.status_code == 200:
                self.admin_token = response.json().strip('"')
                return self.admin_token
            else:
                self.logger.error(f"Failed to get admin token: {response.status_code}")
                return None

        except Exception as e:
            self.logger.error(f"Error getting admin token: {str(e)}")
            return None

    def _get_customer_token(self, email: str, password: str) -> Optional[str]:
        """Get customer authentication token."""
        try:
            url = f"{self.base_url}/integration/customer/token"
            payload = {"username": email, "password": password}

            response = requests.post(url, json=payload, timeout=30)
            if response.status_code == 200:
                return response.json().strip('"')
            else:
                self.logger.error(
                    f"Failed to get customer token: {response.status_code}"
                )
                return None

        except Exception as e:
            self.logger.error(f"Error getting customer token: {str(e)}")
            return None

    def _make_api_request(
        self, endpoint: str, token: str, method: str = "GET", data: dict = None
    ) -> Optional[dict]:
        """Make authenticated API request."""
        try:
            url = f"{self.base_url}/{endpoint.lstrip('/')}"
            headers = {
                "Authorization": f"Bearer {token}",
                "Content-Type": "application/json",
            }

            if method.upper() == "GET":
                response = requests.get(url, headers=headers, timeout=30)
            elif method.upper() == "POST":
                response = requests.post(url, headers=headers, json=data, timeout=30)
            elif method.upper() == "PUT":
                response = requests.put(url, headers=headers, json=data, timeout=30)
            else:
                self.logger.error(f"Unsupported HTTP method: {method}")
                return None

            if response.status_code in [200, 201]:
                return response.json()
            else:
                self.logger.error(
                    f"API request failed: {response.status_code} - {response.text}"
                )
                return None

        except Exception as e:
            self.logger.error(f"Error making API request to {endpoint}: {str(e)}")
            return None

    def get_customer_data_by_email(self, email: str) -> Optional[CustomerInfo]:
        """Get customer information by email using API."""
        admin_token = self._get_admin_token()
        if not admin_token:
            self.logger.warning("No admin token available, falling back to database")
            return self.get_customer_by_email(email)

        try:
            # Search for customer by email
            endpoint = f"customers/search?searchCriteria[filterGroups][0][filters][0][field]=email&searchCriteria[filterGroups][0][filters][0][value]={email}&searchCriteria[filterGroups][0][filters][0][conditionType]=eq"

            result = self._make_api_request(endpoint, admin_token)
            if result and result.get("items"):
                customer_data = result["items"][0]
                return CustomerInfo.model_validate(
                    {
                        "customer_id": customer_data["id"],
                        "firstname": customer_data.get("firstname", ""),
                        "lastname": customer_data.get("lastname", ""),
                        "email": customer_data["email"],
                        "created_at": customer_data.get("created_at"),
                    }
                )
            else:
                self.logger.info(f"No customer found with email: {email}")
                return None

        except Exception as e:
            self.logger.error(f"Error getting customer data via API: {str(e)}")
            # Fallback to database query
            return self.get_customer_by_email(email)

    def get_customer_by_email(self, email: str) -> Optional[CustomerInfo]:
        """Get customer information by email via database (fallback)."""
        query = f"""
        SELECT entity_id as customer_id, firstname, lastname, email, created_at
        FROM customer_entity 
        WHERE email = '{email}'
        """

        result = self._execute_database_query(query)
        if result["success"] and result["rows"]:
            row = result["rows"][0]
            return CustomerInfo.model_validate(
                {
                    "customer_id": int(row["customer_id"]),
                    "firstname": row["firstname"],
                    "lastname": row["lastname"],
                    "email": row["email"],
                    "created_at": row.get("created_at"),
                }
            )
        return None

    def get_addresses_by_email(self, email: str) -> List[Address]:
        """Get all addresses for a customer by email using API."""
        # First get customer data to get customer ID
        customer = self.get_customer_data_by_email(email)
        if not customer:
            return []

        admin_token = self._get_admin_token()
        if not admin_token:
            self.logger.warning("No admin token available, falling back to database")
            return self._get_addresses_by_email_db(email)

        try:
            addresses = []

            # Get billing address
            billing_endpoint = f"customers/{customer.customer_id}/billingAddress"
            billing_result = self._make_api_request(billing_endpoint, admin_token)

            if billing_result:
                billing_address = Address.model_validate(
                    {
                        "address_id": str(billing_result.get("id", "")),
                        "customer_id": customer.customer_id,
                        "customer_name": f"{customer.firstname} {customer.lastname}",
                        "email": customer.email,
                        "is_default_billing": True,
                        "is_default_shipping": False,
                        "firstname": billing_result.get("firstname", ""),
                        "lastname": billing_result.get("lastname", ""),
                        "company": billing_result.get("company", ""),
                        "street": "\n".join(billing_result.get("street", [])),
                        "city": billing_result.get("city", ""),
                        "region": billing_result.get("region", {}).get("region", ""),
                        "region_id": str(
                            billing_result.get("region", {}).get("region_id", "")
                        ),
                        "postcode": billing_result.get("postcode", ""),
                        "country_id": billing_result.get("country_id", ""),
                        "telephone": billing_result.get("telephone", ""),
                        "fax": billing_result.get("fax", ""),
                        "prefix": billing_result.get("prefix", ""),
                        "middlename": billing_result.get("middlename", ""),
                        "suffix": billing_result.get("suffix", ""),
                        "vat_id": billing_result.get("vat_id", ""),
                        "created_at": None,
                        "updated_at": None,
                        "is_active": True,
                    }
                )
                addresses.append(billing_address)

            # Get shipping address
            shipping_endpoint = f"customers/{customer.customer_id}/shippingAddress"
            shipping_result = self._make_api_request(shipping_endpoint, admin_token)

            if shipping_result and shipping_result.get("id") != billing_result.get(
                "id"
            ):
                shipping_address = Address.model_validate(
                    {
                        "address_id": str(shipping_result.get("id", "")),
                        "customer_id": customer.customer_id,
                        "customer_name": f"{customer.firstname} {customer.lastname}",
                        "email": customer.email,
                        "is_default_billing": False,
                        "is_default_shipping": True,
                        "firstname": shipping_result.get("firstname", ""),
                        "lastname": shipping_result.get("lastname", ""),
                        "company": shipping_result.get("company", ""),
                        "street": "\n".join(shipping_result.get("street", [])),
                        "city": shipping_result.get("city", ""),
                        "region": shipping_result.get("region", {}).get("region", ""),
                        "region_id": str(
                            shipping_result.get("region", {}).get("region_id", "")
                        ),
                        "postcode": shipping_result.get("postcode", ""),
                        "country_id": shipping_result.get("country_id", ""),
                        "telephone": shipping_result.get("telephone", ""),
                        "fax": shipping_result.get("fax", ""),
                        "prefix": shipping_result.get("prefix", ""),
                        "middlename": shipping_result.get("middlename", ""),
                        "suffix": shipping_result.get("suffix", ""),
                        "vat_id": shipping_result.get("vat_id", ""),
                        "created_at": None,
                        "updated_at": None,
                        "is_active": True,
                    }
                )
                addresses.append(shipping_address)

            return addresses

        except Exception as e:
            self.logger.error(f"Error getting addresses via API: {str(e)}")
            # Fallback to database query
            return self._get_addresses_by_email_db(email)

    def _get_addresses_by_email_db(self, email: str) -> List[Address]:
        """Get all addresses for a customer by email via database (fallback)."""
        query = f"""
        SELECT 
            ce.entity_id as customer_id,
            CONCAT(ce.firstname, ' ', ce.lastname) as customer_name,
            ce.email,
            ce.default_billing,
            ce.default_shipping,
            cae.entity_id as address_id,
            cae.firstname,
            cae.lastname,
            cae.company,
            cae.street,
            cae.city,
            cae.region,
            cae.region_id,
            cae.postcode,
            cae.country_id,
            cae.telephone,
            cae.fax,
            cae.prefix,
            cae.middlename,
            cae.suffix,
            cae.vat_id,
            cae.created_at,
            cae.updated_at,
            cae.is_active
        FROM customer_entity ce
        JOIN customer_address_entity cae ON ce.entity_id = cae.parent_id
        WHERE ce.email = '{email}' 
        AND cae.is_active = 1
        ORDER BY cae.entity_id DESC
        """

        result = self._execute_database_query(query)
        addresses = []

        if result["success"] and result["rows"]:
            for row in result["rows"]:
                address = Address.model_validate(
                    {
                        "address_id": row["address_id"],
                        "customer_id": int(row["customer_id"]),
                        "customer_name": row["customer_name"],
                        "email": row["email"],
                        "is_default_billing": row["default_billing"]
                        == row["address_id"],
                        "is_default_shipping": row["default_shipping"]
                        == row["address_id"],
                        "firstname": row["firstname"],
                        "lastname": row["lastname"],
                        "company": row["company"],
                        "street": row["street"],
                        "city": row["city"],
                        "region": row["region"],
                        "region_id": row["region_id"],
                        "postcode": row["postcode"],
                        "country_id": row["country_id"],
                        "telephone": row["telephone"],
                        "fax": row["fax"],
                        "prefix": row["prefix"],
                        "middlename": row["middlename"],
                        "suffix": row["suffix"],
                        "vat_id": row["vat_id"],
                        "created_at": row["created_at"],
                        "updated_at": row["updated_at"],
                        "is_active": row["is_active"] == "1",
                    }
                )
                addresses.append(address)

        return addresses

    def get_orders_by_email(self, email: str) -> List[Order]:
        """Get orders for a customer by email using API."""
        admin_token = self._get_admin_token()
        if not admin_token:
            self.logger.warning("No admin token available, falling back to database")
            return self._get_orders_by_email_db(email)

        try:
            # Search for orders by customer email
            endpoint = f"orders?searchCriteria[filterGroups][0][filters][0][field]=customer_email&searchCriteria[filterGroups][0][filters][0][value]={email}&searchCriteria[filterGroups][0][filters][0][conditionType]=eq"

            result = self._make_api_request(endpoint, admin_token)
            orders = []

            if result and result.get("items"):
                for order_data in result["items"]:
                    order = Order.model_validate(order_data)

                    orders.append(order)

            return orders

        except Exception as e:
            self.logger.error(f"Error getting orders via API: {str(e)}")
            # Fallback to database query
            return self._get_orders_by_email_db(email)

    def _get_orders_by_email_db(self, email: str) -> List[Order]:
        """Get orders for a customer by email via database (fallback)."""
        query = f"""
        SELECT 
            so.entity_id as entity_id,
            so.status,
            so.state,
            so.grand_total,
            so.subtotal,
            so.tax_amount,
            so.shipping_amount,
            so.created_at,
            so.updated_at,
            CONCAT(ce.firstname, ' ', ce.lastname) as customer_name,
            ce.email as customer_email,
            COUNT(soi.item_id) as item_count
        FROM sales_order so
        LEFT JOIN customer_entity ce ON so.customer_id = ce.entity_id
        LEFT JOIN sales_order_item soi ON so.entity_id = soi.order_id
        WHERE so.customer_email = '{email}' OR ce.email = '{email}'
        GROUP BY so.entity_id
        ORDER BY so.created_at DESC
        """

        result = self._execute_database_query(query)
        orders = []

        if result["success"] and result["rows"]:
            for row in result["rows"]:
                order = Order.model_validate(
                    {
                        "entity_id": row["entity_id"],
                        "status": row["status"],
                        "state": row["state"],
                        "grand_total": float(row["grand_total"] or 0),
                        "subtotal": float(row["subtotal"] or 0),
                        "tax_amount": float(row["tax_amount"] or 0),
                        "shipping_amount": float(row["shipping_amount"] or 0),
                        "customer_name": row["customer_name"],
                        "customer_email": row["customer_email"],
                        "item_count": int(row["item_count"] or 0),
                        "created_at": row["created_at"],
                        "updated_at": row["updated_at"],
                    }
                )
                orders.append(order)

        return orders

    def get_product_review_by_sku(
        self, product_sku: str, nickname: str = None, rating: int = None
    ) -> List[ProductReview]:
        """Get product reviews by SKU using API."""
        admin_token = self._get_admin_token()

        try:
            # First get product ID from SKU
            product_endpoint = f"products/{product_sku}"
            product_result = self._make_api_request(product_endpoint, admin_token)

            if not product_result:
                self.logger.warning(f"Product not found for SKU: {product_sku}")
                return []

            product_id = product_result.get("id")
            if not product_id:
                return []

            # Search for reviews - note: this endpoint may not exist in all Magento versions
            # If it fails, we'll fall back to database
            reviews_endpoint = f"reviews?searchCriteria[filterGroups][0][filters][0][field]=entity_pk_value&searchCriteria[filterGroups][0][filters][0][value]={product_id}&searchCriteria[filterGroups][0][filters][0][conditionType]=eq"

            result = self._make_api_request(reviews_endpoint, admin_token)
            reviews = []

            if result and result.get("items"):
                for review_data in result["items"]:
                    review = ProductReview.model_validate(review_data)
                    reviews.append(review)

            return reviews

        except Exception as e:
            self.logger.error(f"Error getting product reviews via API: {str(e)}")
            return []

    def _execute_database_query(
        self, query: str, params: Optional[tuple] = None
    ) -> dict:
        """Execute database query via Docker container."""
        try:
            # Format query with parameters if provided
            if params:
                query = query % params

            cmd = [
                "docker",
                "exec",
                self.config.container_name,
                "mysql",
                "-u",
                self.config.db_user,
                f"-p{self.config.db_password}",
                self.config.db_name,
                "-e",
                query,
            ]

            result = subprocess.run(cmd, capture_output=True, text=True, check=True)

            # Parse MySQL output
            lines = result.stdout.strip().split("\n")
            if len(lines) > 1:  # Has header + data
                headers = lines[0].split("\t")
                data = []
                for line in lines[1:]:
                    values = line.split("\t")
                    data.append(dict(zip(headers, values)))

                return {"success": True, "rows": data, "count": len(data)}
            else:
                return {"success": True, "rows": [], "count": 0}

        except subprocess.CalledProcessError as e:
            return {"success": False, "error": e.stderr, "rows": [], "count": 0}
        except Exception as e:
            return {"success": False, "error": str(e), "rows": [], "count": 0}

    def get_carts_by_email(self, email: str) -> List[Cart]:
        """Get cart information by customer email."""
        query = f"""
        SELECT 
            q.entity_id as cart_id,
            q.created_at,
            q.updated_at,
            q.items_count,
            q.items_qty,
            q.grand_total,
            q.is_active,
            CONCAT(ce.firstname, ' ', ce.lastname) as customer_name,
            ce.email as customer_email
        FROM quote q
        LEFT JOIN customer_entity ce ON q.customer_id = ce.entity_id
        WHERE ce.email = '{email}'
        AND q.is_active = 1
        ORDER BY q.updated_at DESC
        """

        result = self._execute_database_query(query)
        carts = []

        if result["success"] and result["rows"]:
            for row in result["rows"]:
                cart = Cart.model_validate(
                    {
                        "cart_id": str(row["cart_id"]),
                        "customer_name": row["customer_name"],
                        "customer_email": row["customer_email"],
                        "items_count": int(float(row["items_count"] or 0)),
                        "items_qty": int(float(row["items_qty"] or 0)),
                        "grand_total": float(row["grand_total"] or 0),
                        "created_at": row["created_at"],
                        "updated_at": row["updated_at"],
                        "is_active": row["is_active"] == "1",
                    }
                )
                carts.append(cart)

        return carts

    def get_cart_items_by_email(self, email: str) -> List[CartItem]:
        """
        Get cart items for a customer by email.

        Gets the first active cart for the customer and returns its items.

        Args:
            email: Customer email address

        Returns:
            List of CartItem objects from the customer's first active cart
        """
        try:
            # Get all carts for the customer by email
            carts = self.get_carts_by_email(email)

            if not carts:
                self.logger.info(f"No active carts found for email: {email}")
                return []

            # Get the first (most recent) cart
            first_cart = carts[0]
            cart_id = first_cart.cart_id

            # Get items from the cart
            cart_items = self.get_cart_items(cart_id)

            self.logger.info(
                f"Retrieved {len(cart_items)} items from cart {cart_id} for email {email}"
            )
            return cart_items

        except Exception as e:
            self.logger.error(f"Error getting cart items by email {email}: {str(e)}")
            return []

    def get_wishlist_items_by_email(self, email: str) -> List[WishlistItem]:
        """Get wishlist items for a customer by email."""
        query = f"""
        SELECT 
            w.wishlist_id,
            wi.wishlist_item_id,
            wi.product_id,
            wi.added_at,
            cpe.sku,
            cpev.value as product_name,
            wi.qty,
            wi.description
        FROM wishlist w 
        JOIN customer_entity ce ON w.customer_id = ce.entity_id
        JOIN wishlist_item wi ON w.wishlist_id = wi.wishlist_id 
        JOIN catalog_product_entity cpe ON wi.product_id = cpe.entity_id
        LEFT JOIN catalog_product_entity_varchar cpev ON cpe.entity_id = cpev.entity_id 
            AND cpev.attribute_id = (
                SELECT attribute_id FROM eav_attribute 
                WHERE attribute_code = 'name' AND entity_type_id = 4
            )
        WHERE ce.email = '{email}'
        ORDER BY wi.added_at DESC
        """

        result = self._execute_database_query(query)
        items = []

        if result["success"] and result["rows"]:
            for row in result["rows"]:
                item = WishlistItem.model_validate(row)
                items.append(item)

        return items

    def get_newsletter_subscription(
        self, email: str
    ) -> Optional[NewsletterSubscription]:
        """Get newsletter subscription by email."""
        query = f"""
        SELECT subscriber_email, subscriber_status, change_status_at 
        FROM newsletter_subscriber 
        WHERE subscriber_email = '{email}'
        """

        result = self._execute_database_query(query)

        if result["success"] and result["rows"]:
            row = result["rows"][0]
            return NewsletterSubscription.model_validate(
                {
                    "subscriber_email": row["subscriber_email"],
                    "subscriber_status": row["subscriber_status"],
                    "change_status_at": row["change_status_at"],
                }
            )

        return None

    def get_all_product_skus(self, output_file: Optional[str] = None) -> List[str]:
        """
        Get all product SKUs from the database.

        Args:
            output_file: Optional file path to save SKUs to

        Returns:
            List of all product SKUs
        """
        try:
            # Enhanced query to get SKU, name, and description
            query = """
            SELECT 
                cpe.sku,
                name_attr.value as name,
                desc_attr.value as description,
                cpe.type_id,
                cpe.created_at,
                cpe.updated_at
            FROM catalog_product_entity cpe
            LEFT JOIN catalog_product_entity_varchar name_attr 
                ON cpe.entity_id = name_attr.entity_id 
                AND name_attr.attribute_id = (
                    SELECT attribute_id FROM eav_attribute 
                    WHERE attribute_code = 'name' 
                    AND entity_type_id = (
                        SELECT entity_type_id FROM eav_entity_type 
                        WHERE entity_type_code = 'catalog_product'
                    )
                )
            LEFT JOIN catalog_product_entity_text desc_attr 
                ON cpe.entity_id = desc_attr.entity_id 
                AND desc_attr.attribute_id = (
                    SELECT attribute_id FROM eav_attribute 
                    WHERE attribute_code = 'description' 
                    AND entity_type_id = (
                        SELECT entity_type_id FROM eav_entity_type 
                        WHERE entity_type_code = 'catalog_product'
                    )
                )
            ORDER BY cpe.sku
            """

            result = self._execute_database_query(query)

            if not result.get("success", False):
                error_msg = result.get("error", "Unknown database error")
                self.logger.error(f"Failed to retrieve product SKUs: {error_msg}")
                return []

            if not result["rows"]:
                self.logger.warning("No product SKUs found in database")
                return []

            rows = result.get("rows", [])
            skus = []

            for row in rows:
                sku = row.get("sku", "")
                if sku:
                    skus.append(sku)

            self.logger.info(f"Retrieved {len(skus)} product SKUs")

            # Save to file if requested
            if output_file:
                import json
                from pathlib import Path

                output_path = Path(output_file)
                output_path.parent.mkdir(parents=True, exist_ok=True)

                # Create detailed product info with name and description
                product_info = []
                for row in rows:
                    product_info.append(
                        {
                            "sku": row.get("sku", ""),
                            "name": row.get("name", ""),
                            "description": row.get("description", ""),
                            "type_id": row.get("type_id", ""),
                            "created_at": row.get("created_at", ""),
                            "updated_at": row.get("updated_at", ""),
                        }
                    )

                with open(output_path, "w", encoding="utf-8") as f:
                    json.dump(
                        {
                            "total_products": len(skus),
                            "skus": skus,
                            "products": product_info,
                        },
                        f,
                        indent=2,
                        ensure_ascii=False,
                    )

                self.logger.info(f"Product SKUs saved to {output_file}")

            return skus

        except Exception as e:
            self.logger.error(f"Error retrieving product SKUs: {str(e)}")
            return []

    def get_cart_items(self, cart_id: str) -> List[CartItem]:
        """Get all items in a cart using API only."""
        if not cart_id:
            raise ValueError("cart_id is required and cannot be None or empty")

        admin_token = self._get_admin_token()
        if not admin_token:
            raise RuntimeError("No admin token available - cannot retrieve cart items")

        # Use regular cart API with admin token
        endpoint = f"carts/{cart_id}/items"
        result = self._make_api_request(endpoint, admin_token)

        if result is None:
            raise RuntimeError(
                f"Failed to retrieve cart items for cart {cart_id} via API"
            )

        items = []
        for item_data in result:
            # Create CartItem using model_validate for proper type conversion
            item = CartItem.model_validate(
                {
                    "item_id": str(item_data.get("item_id"))
                    if item_data.get("item_id") is not None
                    else None,
                    "sku": item_data.get("sku", ""),
                    "name": item_data.get("name"),
                    "qty": int(item_data.get("qty", 0)),
                    "price": float(item_data.get("price", 0.0))
                    if item_data.get("price")
                    else None,
                    "row_total": float(item_data.get("row_total", 0.0))
                    if item_data.get("row_total")
                    else None,
                }
            )
            items.append(item)
        return items

    def get_product_by_id(self, product_id: int) -> Optional[Product]:
        """
        Get product information by product ID using Magento API.

        Since Magento API uses SKU-based endpoints, this method first searches
        for the product by ID to get its SKU, then retrieves the full product data.

        Args:
            product_id: The product ID to retrieve

        Returns:
            Product model if found, None otherwise
        """
        try:
            admin_token = self._get_admin_token()
            if not admin_token:
                self.logger.error("Failed to get admin token for product retrieval")
                return None

            # First, search for the product by ID to get its SKU
            url = f"{self.base_url}/products"
            params = {
                "searchCriteria[filter_groups][0][filters][0][field]": "entity_id",
                "searchCriteria[filter_groups][0][filters][0][value]": product_id,
                "searchCriteria[filter_groups][0][filters][0][condition_type]": "eq",
                "searchCriteria[page_size]": 1,
            }
            headers = {"Authorization": f"Bearer {admin_token}"}

            self.logger.info(f"Searching for product with ID {product_id}")
            response = requests.get(url, headers=headers, params=params, timeout=30)

            if response.status_code != 200:
                self.logger.error(
                    f"API request failed with status {response.status_code}: {response.text}"
                )
                return None

            search_data = response.json()
            items = search_data.get("items", [])

            if not items:
                self.logger.warning(f"Product {product_id} not found")
                return None

            # Get the SKU from the search result
            product_sku = items[0].get("sku")
            if not product_sku:
                self.logger.error(f"No SKU found for product ID {product_id}")
                return None

            # Now get the full product data using the SKU
            return self.get_product_by_sku(product_sku)

        except requests.exceptions.RequestException as e:
            self.logger.error(f"Network error retrieving product {product_id}: {e}")
            return None
        except Exception as e:
            self.logger.error(f"Error retrieving product {product_id}: {e}")
            return None

    def get_product_by_name(self, product_name: str) -> Optional[Product]:
        """
        Get a product by its name.

        Args:
            product_name: The product name to search for

        Returns:
            Product object if found, None otherwise
        """
        try:
            # Get admin token
            token = self._get_admin_token()
            if not token:
                self.logger.error("Failed to get admin token")
                return None

            # Build search criteria URL parameters
            search_params = (
                "searchCriteria[filterGroups][0][filters][0][field]=name&"
                f"searchCriteria[filterGroups][0][filters][0][value]={quote_plus(product_name)}&"
                "searchCriteria[filterGroups][0][filters][0][conditionType]=eq"
            )

            endpoint = f"products?{search_params}"
            response = self._make_api_request(endpoint, token)

            if response and "items" in response and len(response["items"]) > 0:
                product_data = response["items"][0]  # Get first match
                return Product(**product_data)

            self.logger.warning(f"Product not found with name: {product_name}")
            return None

        except Exception as e:
            self.logger.error(f"Error retrieving product by name '{product_name}': {e}")
            return None

    def get_product_by_sku(self, sku: str) -> Optional[Product]:
        """
        Get product information by SKU using Magento API.

        Args:
            sku: The product SKU to retrieve

        Returns:
            Product model if found, None otherwise
        """
        try:
            admin_token = self._get_admin_token()
            if not admin_token:
                self.logger.error("Failed to get admin token for product retrieval")
                return None

            url = f"{self.base_url}/products/{sku}"
            headers = {"Authorization": f"Bearer {admin_token}"}

            self.logger.info(f"Retrieving product {sku} from API: {url}")
            response = requests.get(url, headers=headers, timeout=30)

            if response.status_code == 404:
                self.logger.warning(f"Product {sku} not found")
                return None
            elif response.status_code != 200:
                self.logger.error(
                    f"API request failed with status {response.status_code}: {response.text}"
                )
                return None

            product_data = response.json()

            # Create Product using **product_data to capture all available fields
            # Map entity_id to id for backward compatibility
            if "entity_id" in product_data and "id" not in product_data:
                product_data["id"] = product_data["entity_id"]

            product = Product.model_validate(product_data)

            self.logger.info(
                f"Successfully retrieved product: {product.name} (SKU: {product.sku})"
            )
            return product

        except requests.exceptions.RequestException as e:
            self.logger.error(f"Network error retrieving product {sku}: {e}")
            return None
        except Exception as e:
            self.logger.error(f"Error retrieving product {sku}: {e}")
            return None

    def get_review_by_id(self, review_id: str) -> ProductReview | None:
        """Get a review by its ID."""
        self.logger.info(f"Getting review by ID: {review_id}")

        # Use database query directly since API approach isn't working
        query = """
        SELECT 
            r.review_id,
            r.entity_id,
            r.entity_pk_value,
            r.status_id as review_status,
            r.created_at,
            rd.title,
            rd.detail,
            rd.nickname,
            rd.customer_id,
            GROUP_CONCAT(rs.store_id) as stores,
            COALESCE(
                JSON_ARRAYAGG(
                    JSON_OBJECT(
                        'rating_id', rv.rating_id,
                        'option_id', rv.option_id,
                        'value', rv.value
                    )
                ), 
                JSON_ARRAY()
            ) as ratings
        FROM review r
        LEFT JOIN review_detail rd ON r.review_id = rd.review_id
        LEFT JOIN review_store rs ON r.review_id = rs.review_id
        LEFT JOIN rating_option_vote rv ON r.review_id = rv.review_id
        WHERE r.review_id = %s
        GROUP BY r.review_id
        """

        self.logger.info(f"Executing database query for review_id: {review_id}")
        results = self._execute_database_query(query, (review_id,))

        self.logger.info(
            f"Database query returned {len(results) if results else 0} results"
        )

        if not results:
            self.logger.error(f"Review with ID {review_id} not found in database")
            raise ValueError(f"Review with ID {review_id} not found")

        if len(results["rows"]) == 0:
            return None

        result = results["rows"][0]
        review_data = dict(result)
        self.logger.info(f"Raw database result: {review_data}")

        # Parse JSON fields
        if review_data.get("ratings"):
            try:
                review_data["ratings"] = json.loads(review_data["ratings"])
                self.logger.info(f"Parsed ratings: {review_data['ratings']}")
            except (json.JSONDecodeError, TypeError) as e:
                self.logger.warning(f"Failed to parse ratings JSON: {e}")
                review_data["ratings"] = []
        else:
            review_data["ratings"] = []

        # Parse stores
        if review_data.get("stores"):
            review_data["stores"] = [
                int(s) for s in review_data["stores"].split(",") if s
            ]
            self.logger.info(f"Parsed stores: {review_data['stores']}")
        else:
            review_data["stores"] = []

        # Add backward compatibility for rating field
        if review_data["ratings"] and len(review_data["ratings"]) > 0:
            # Use the first rating value as the main rating
            review_data["rating"] = review_data["ratings"][0].get("value", 0)
        else:
            review_data["rating"] = 0

        # Add default values for fields that might be missing but are expected by the Pydantic model
        review_data.setdefault("review_type", 1)  # Default to customer review
        review_data.setdefault("store_id", 1)  # Default store ID

        self.logger.info(f"Final processed review data: {review_data}")

        return ProductReview(**review_data)

    def get_order_by_id(self, order_id: int) -> Optional[Order]:
        """
        Get order information by order ID using Magento API.

        Args:
            order_id: The order ID to retrieve

        Returns:
            Order model if found, None otherwise
        """
        try:
            admin_token = self._get_admin_token()
            if not admin_token:
                self.logger.error("Failed to get admin token for order retrieval")
                return None

            url = f"{self.base_url}/orders/{order_id}"
            headers = {"Authorization": f"Bearer {admin_token}"}

            self.logger.info(f"Retrieving order {order_id} from API: {url}")
            response = requests.get(url, headers=headers, timeout=30)

            if response.status_code == 404:
                self.logger.warning(f"Order {order_id} not found")
                return None
            elif response.status_code != 200:
                self.logger.error(
                    f"API request failed with status {response.status_code}: {response.text}"
                )
                return None

            order_data = response.json()
            # Create Order using **order_data to capture all available fields
            # Map entity_id to order_id for backward compatibility
            if "entity_id" in order_data and "order_id" not in order_data:
                order_data["order_id"] = str(order_data["entity_id"])

            # Ensure customer_name is set for backward compatibility
            if "customer_name" not in order_data and (
                "customer_firstname" in order_data or "customer_lastname" in order_data
            ):
                firstname = order_data.get("customer_firstname", "")
                lastname = order_data.get("customer_lastname", "")
                order_data["customer_name"] = f"{firstname} {lastname}".strip()

            # Map total_item_count to item_count for backward compatibility
            if "total_item_count" in order_data and "item_count" not in order_data:
                order_data["item_count"] = order_data["total_item_count"]

            order = Order.model_validate(order_data)

            self.logger.info(
                f"Successfully retrieved order: {order.entity_id} for {order.customer_email}"
            )
            return order

        except requests.exceptions.RequestException as e:
            self.logger.error(f"Network error retrieving order {order_id}: {e}")
            return None
        except Exception as e:
            self.logger.error(f"Error retrieving order {order_id}: {e}")
            return None

    def get_products(self, limit: int = 10) -> List[Product]:
        """
        Get a list of products using Magento API.

        Args:
            limit: Maximum number of products to retrieve (default: 10)

        Returns:
            List of Product models
        """
        try:
            admin_token = self._get_admin_token()
            if not admin_token:
                self.logger.error("Failed to get admin token for products retrieval")
                return []

            # Use searchCriteria to limit results
            url = f"{self.base_url}/products"
            params = {
                "searchCriteria[page_size]": limit,
                "searchCriteria[current_page]": 1,
            }
            headers = {"Authorization": f"Bearer {admin_token}"}

            self.logger.info(f"Retrieving first {limit} products from API: {url}")
            response = requests.get(url, headers=headers, params=params, timeout=30)

            if response.status_code != 200:
                self.logger.error(
                    f"API request failed with status {response.status_code}: {response.text}"
                )
                return []

            data = response.json()
            products_data = data.get("items", [])

            products = []
            for product_data in products_data:
                # Create Product using **product_data to capture all available fields
                # Map entity_id to id for backward compatibility
                if "entity_id" in product_data and "id" not in product_data:
                    product_data["id"] = product_data["entity_id"]

                # Ensure price is set for backward compatibility
                if "price" not in product_data and "price_range" in product_data:
                    price_range = product_data["price_range"]
                    if (
                        "minimum_price" in price_range
                        and "final_price" in price_range["minimum_price"]
                    ):
                        product_data["price"] = price_range["minimum_price"][
                            "final_price"
                        ]["value"]

                product = Product.model_validate(product_data)
                products.append(product)

            self.logger.info(f"Successfully retrieved {len(products)} products")
            return products

        except requests.exceptions.RequestException as e:
            self.logger.error(f"Network error retrieving products: {e}")
            return []
        except Exception as e:
            self.logger.error(f"Error retrieving products: {e}")
            return []

    def get_variant_names_for_product(self, product: Product) -> List[str]:
        """
        Get variant names for a configurable product.

        Args:
            product: The configurable product

        Returns:
            List of variant names
        """
        variant_names = []

        try:
            # Check if product has configurable_product_links in extension_attributes
            if (
                product.extension_attributes
                and isinstance(product.extension_attributes, dict)
                and "configurable_product_links" in product.extension_attributes
            ):
                variant_ids = product.extension_attributes["configurable_product_links"]

                # Get each variant product by ID
                for variant_id in variant_ids:
                    variant_product = self.get_product_by_id(variant_id)
                    if variant_product:
                        variant_names.append(variant_product.name)

            self.logger.info(
                f"Found {len(variant_names)} variants for product {product.sku}"
            )
            return variant_names

        except Exception as e:
            self.logger.error(
                f"Error retrieving variants for product {product.sku}: {e}"
            )
            return []

    def get_order_comments(self, order_id: int) -> List[str]:
        """
        Get comments for an order by order ID.

        Args:
            order_id: The order ID to get comments for

        Returns:
            List of comment strings
        """
        try:
            # Get admin token
            token = self._get_admin_token()
            if not token:
                self.logger.error("Failed to get admin token")
                return []

            # Get order comments using Magento API
            endpoint = f"orders/{order_id}/comments"
            response = self._make_api_request(endpoint, token)

            comments = []
            if response and "items" in response:
                for comment_data in response["items"]:
                    if "comment" in comment_data and comment_data["comment"]:
                        comments.append(comment_data["comment"])

            self.logger.info(f"Found {len(comments)} comments for order {order_id}")
            return comments

        except Exception as e:
            self.logger.error(f"Error retrieving comments for order {order_id}: {e}")
            return []

    def get_order_shipments(self, order_id: int) -> List[dict]:
        """
        Get shipment details for an order by order ID.

        Args:
            order_id: The order ID to get shipments for

        Returns:
            List of shipment dictionaries with carrier and tracking info
        """
        try:
            # Get admin token
            token = self._get_admin_token()
            if not token:
                self.logger.error("Failed to get admin token")
                return []

            # Get order shipments using Magento API with search criteria
            # Use /V1/shipments endpoint with order_id filter
            endpoint = f"shipments?searchCriteria[filterGroups][0][filters][0][field]=order_id&searchCriteria[filterGroups][0][filters][0][value]={order_id}&searchCriteria[filterGroups][0][filters][0][conditionType]=eq"
            response = self._make_api_request(endpoint, token)

            shipments = []
            if response and "items" in response:
                for shipment_data in response["items"]:
                    shipment_info = {}

                    # Extract shipping carrier and tracking from tracks
                    if "tracks" in shipment_data and shipment_data["tracks"]:
                        for track in shipment_data["tracks"]:
                            if "title" in track:
                                shipment_info["shipping_carrier"] = track["title"]
                            if "track_number" in track:
                                shipment_info["tracking_number"] = track["track_number"]
                            break  # Use first track for now

                    # Add shipment ID for reference
                    if "entity_id" in shipment_data:
                        shipment_info["shipment_id"] = shipment_data["entity_id"]

                    if shipment_info:
                        shipments.append(shipment_info)

            self.logger.info(f"Found {len(shipments)} shipments for order {order_id}")
            return shipments

        except Exception as e:
            self.logger.error(f"Error retrieving shipments for order {order_id}: {e}")
            return []

    def get_order_addresses(self, order_id: int) -> Dict[str, Dict]:
        """
        Get shipping and billing addresses for an order by order ID.

        Args:
            order_id: The order ID to get addresses for

        Returns:
            Dictionary with 'shipping' and 'billing' address data
        """
        try:
            # Get the order first to extract address information
            order = self.get_order_by_id(order_id)
            if not order:
                self.logger.error(f"Order {order_id} not found")
                return {}

            addresses = {}

            # Extract billing address directly from order (now a dict)
            if order.billing_address:
                addr = order.billing_address
                street = addr.get("street", [])
                addresses["billing"] = {
                    "address": street[0] if street and len(street) > 0 else "",
                    "address2": street[1] if street and len(street) > 1 else "",
                    "city": addr.get("city", ""),
                    "state": addr.get("region", ""),
                    "zip_code": addr.get("postcode", ""),
                    "country": addr.get("country_id", ""),
                }

            # Check if order has extension_attributes with shipping assignments (now a dict)
            if order.extension_attributes:
                ext_attrs = order.extension_attributes
                shipping_assignments = ext_attrs.get("shipping_assignments", [])
                if shipping_assignments:
                    shipping_assignment = shipping_assignments[0]
                    shipping_info = shipping_assignment.get("shipping", {})
                    shipping_address = shipping_info.get("address", {})
                    if shipping_address:
                        street = shipping_address.get("street", [])
                        addresses["shipping"] = {
                            "address": street[0] if street and len(street) > 0 else "",
                            "address2": street[1] if street and len(street) > 1 else "",
                            "city": shipping_address.get("city", ""),
                            "state": shipping_address.get("region", ""),
                            "zip_code": shipping_address.get("postcode", ""),
                            "country": shipping_address.get("country_id", ""),
                        }

            # If no separate shipping address found, use billing as fallback
            if "shipping" not in addresses and "billing" in addresses:
                self.logger.info(
                    f"No separate shipping address found for order {order_id}, using billing address"
                )
                addresses["shipping"] = addresses["billing"].copy()

            self.logger.info(
                f"Retrieved addresses for order {order_id}: {list(addresses.keys())}"
            )
            return addresses

        except Exception as e:
            self.logger.error(f"Error retrieving addresses for order {order_id}: {e}")
            return {}

    def get_sale_by_topic(self, topic: str) -> Optional[SalesRule]:
        """
        Get sales rule information by topic/name using Magento API.

        Args:
            topic: The sales rule name/topic to search for

        Returns:
            SalesRule model if found, None otherwise
        """
        try:
            self.logger.info(f"Searching for sales rule with topic: {topic}")

            admin_token = self._get_admin_token()

            if not admin_token:
                self.logger.error("Failed to get admin token for sales rule retrieval")
                return None

            encoded_topic = quote_plus(topic)

            endpoint = f"salesRules/search?searchCriteria[filterGroups][0][filters][0][field]=name&searchCriteria[filterGroups][0][filters][0][value]={encoded_topic}&searchCriteria[filterGroups][0][filters][0][conditionType]=eq"

            self.logger.debug(f"Calling sales rule API: {endpoint}")
            response = self._make_api_request(endpoint, admin_token)

            if not response:
                self.logger.warning(
                    f"No response from sales rule API for topic: {topic}"
                )
                return None

            if "items" not in response or len(response["items"]) == 0:
                self.logger.warning(f"No sales rule found with topic: {topic}")
                return None

            # Get first matching rule
            rule_data = response["items"][0]

            # Map API response fields to our model
            # Handle field name differences between API and database
            if "rule_id" not in rule_data and "id" in rule_data:
                rule_data["rule_id"] = rule_data["id"]

            sale_rule = SalesRule.model_validate(rule_data)

            self.logger.info(
                f"Found sales rule: {sale_rule.name} (ID: {sale_rule.rule_id})"
            )
            return sale_rule

        except Exception as e:
            self.logger.error(f"Error retrieving sales rule by topic '{topic}': {e}")
            return None

    def get_cms_page_by_id(self, page_id: int) -> Optional[CmsPage]:
        """
        Get CMS page information by page ID using Magento API.

        Args:
            page_id: The page ID to retrieve

        Returns:
            CmsPage model if found, None otherwise
        """
        try:
            admin_token = self._get_admin_token()
            if not admin_token:
                self.logger.error("Failed to get admin token for CMS page retrieval")
                return None

            url = f"{self.base_url}/cmsPage/{page_id}"
            headers = {"Authorization": f"Bearer {admin_token}"}

            response = requests.get(url, headers=headers, timeout=30)

            if response.status_code == 404:
                self.logger.warning(f"CMS page {page_id} not found")
                return None
            elif response.status_code != 200:
                self.logger.error(
                    f"API request failed with status {response.status_code}: {response.text}"
                )
                return None

            page_data = response.json()

            # Create CmsPage using **page_data to capture all available fields
            page = CmsPage.model_validate(page_data)

            self.logger.info(
                f"Successfully retrieved CMS page: {page.title} (ID: {page.page_id})"
            )
            return page

        except requests.exceptions.RequestException as e:
            self.logger.error(f"Network error retrieving CMS page {page_id}: {e}")
            return None
        except Exception as e:
            self.logger.error(f"Error retrieving CMS page {page_id}: {e}")
            return None

    def get_product_inventory_by_sku(self, sku: str) -> Optional[ProductInventory]:
        """
        Get product inventory information by SKU using Magento API.

        Args:
            sku: The product SKU to retrieve inventory for

        Returns:
            ProductInventory model if found, None otherwise
        """
        try:
            admin_token = self._get_admin_token()
            if not admin_token:
                self.logger.error(
                    "Failed to get admin token for product inventory retrieval"
                )
                return None

            url = f"{self.base_url}/stockItems/{sku}"
            headers = {"Authorization": f"Bearer {admin_token}"}

            self.logger.info(
                f"Retrieving product inventory for SKU {sku} from API: {url}"
            )
            response = requests.get(url, headers=headers, timeout=30)

            if response.status_code == 404:
                self.logger.warning(f"Product inventory for SKU {sku} not found")
                return None
            elif response.status_code != 200:
                self.logger.error(
                    f"API request failed with status {response.status_code}: {response.text}"
                )
                return None

            inventory_data = response.json()

            # Create ProductInventory using **inventory_data to capture all available fields
            inventory = ProductInventory.model_validate(inventory_data)

            self.logger.info(
                f"Successfully retrieved product inventory for SKU {sku}: {inventory.qty}"
            )
            return inventory

        except requests.exceptions.RequestException as e:
            self.logger.error(
                f"Network error retrieving product inventory for SKU {sku}: {e}"
            )
            return None
        except Exception as e:
            self.logger.error(f"Error retrieving product inventory for SKU {sku}: {e}")
            return None
