"""
Pydantic v2 models for Magento validation responses.
"""

from datetime import datetime
from typing import List, Optional
from pydantic import BaseModel, Field, ConfigDict, field_validator
from urllib.parse import urlparse


class MagentoConfig(BaseModel):
    """Configuration model for Magento connections."""

    model_config = ConfigDict(from_attributes=True)

    base_url: str
    container_name: str
    admin_username: str = "admin"
    admin_password: str = "admin1234"
    db_user: str = "magentouser"
    db_password: str = "MyPassword"
    db_name: str = "magentodb"

    @property
    def root_url(self) -> str:
        """Return the base URL without any path parameters."""
        parsed = urlparse(self.base_url)
        return f"{parsed.scheme}://{parsed.netloc}"


class CustomerInfo(BaseModel):
    """Customer information model."""

    model_config = ConfigDict(from_attributes=True)

    customer_id: int
    firstname: str
    lastname: str
    email: str
    created_at: Optional[datetime] = None


class CartItem(BaseModel):
    """Cart item model."""

    model_config = ConfigDict(from_attributes=True)

    item_id: Optional[str] = None
    sku: str
    name: Optional[str] = None
    qty: int
    price: Optional[float] = None
    row_total: Optional[float] = None


class Cart(BaseModel):
    """Shopping cart model."""

    model_config = ConfigDict(from_attributes=True)

    cart_id: str
    customer_id: Optional[int] = None
    customer_name: Optional[str] = None
    customer_email: Optional[str] = None
    items_count: int = 0
    items_qty: int = 0
    grand_total: float = 0.0
    created_at: Optional[datetime] = None
    updated_at: Optional[datetime] = None
    is_active: bool = True
    items: List[CartItem] = Field(default_factory=list)


class Address(BaseModel):
    """Customer address model."""

    model_config = ConfigDict(from_attributes=True)

    address_id: str
    customer_id: int
    customer_name: str
    email: str
    is_default_billing: bool = False
    is_default_shipping: bool = False
    firstname: Optional[str] = None
    lastname: Optional[str] = None
    company: Optional[str] = None
    street: Optional[str] = None
    city: Optional[str] = None
    region: Optional[str] = None
    region_id: Optional[str] = None
    postcode: Optional[str] = None
    country_id: Optional[str] = None
    telephone: Optional[str] = None
    fax: Optional[str] = None
    prefix: Optional[str] = None
    middlename: Optional[str] = None
    suffix: Optional[str] = None
    vat_id: Optional[str] = None
    created_at: Optional[datetime] = None
    updated_at: Optional[datetime] = None
    is_active: bool = True


class WishlistItem(BaseModel):
    """Wishlist item model."""

    model_config = ConfigDict(from_attributes=True)

    wishlist_item_id: Optional[int] = None
    product_id: int
    sku: str
    product_name: Optional[str] = None
    qty: int = 1
    description: Optional[str] = None
    added_at: Optional[str] = None


class Order(BaseModel):
    """Order model."""

    model_config = ConfigDict(from_attributes=True)

    # Core order identification
    entity_id: Optional[int] = None  # Order ID from Magento API
    increment_id: Optional[str] = None  # Human-readable order number

    # Order status and state
    status: str
    state: str

    # Financial information - base amounts
    base_grand_total: Optional[float] = None
    base_discount_amount: Optional[float] = None
    base_discount_tax_compensation_amount: Optional[float] = None
    base_shipping_amount: Optional[float] = None
    base_shipping_discount_amount: Optional[float] = None
    base_shipping_discount_tax_compensation_amnt: Optional[float] = None
    base_shipping_incl_tax: Optional[float] = None
    base_shipping_tax_amount: Optional[float] = None
    base_subtotal: Optional[float] = None
    base_subtotal_incl_tax: Optional[float] = None
    base_tax_amount: Optional[float] = None
    base_total_due: Optional[float] = None
    base_to_global_rate: Optional[float] = None
    base_to_order_rate: Optional[float] = None

    # Financial information - order amounts
    grand_total: float
    discount_amount: Optional[float] = None
    discount_tax_compensation_amount: Optional[float] = None
    shipping_amount: float = 0.0
    shipping_discount_amount: Optional[float] = None
    shipping_discount_tax_compensation_amount: Optional[float] = None
    shipping_incl_tax: Optional[float] = None
    shipping_tax_amount: Optional[float] = None
    subtotal: float = 0.0
    subtotal_incl_tax: Optional[float] = None
    tax_amount: float = 0.0
    total_due: Optional[float] = None
    total_paid: Optional[float] = None  # Keep for backward compatibility
    total_refunded: Optional[float] = None  # Keep for backward compatibility

    # Currency information
    base_currency_code: Optional[str] = None
    global_currency_code: Optional[str] = None
    order_currency_code: Optional[str] = None
    store_currency_code: Optional[str] = None
    store_to_base_rate: Optional[float] = None
    store_to_order_rate: Optional[float] = None

    # Customer information
    customer_id: Optional[int] = None
    customer_email: Optional[str] = None
    customer_firstname: Optional[str] = None
    customer_lastname: Optional[str] = None
    customer_name: Optional[str] = None  # Keep for backward compatibility
    customer_is_guest: Optional[int] = None
    customer_group_id: Optional[int] = None
    customer_note_notify: Optional[int] = None

    # Order items and quantities
    total_item_count: Optional[int] = None
    item_count: int = 0  # Keep for backward compatibility
    total_qty_ordered: Optional[float] = None

    # Shipping information
    shipping_description: Optional[str] = None
    is_virtual: Optional[int] = None

    # Store information
    store_id: Optional[int] = None
    store_name: Optional[str] = None

    # Address information
    billing_address_id: Optional[int] = None

    # Quote and protection
    quote_id: Optional[int] = None
    protect_code: Optional[str] = None
    remote_ip: Optional[str] = None

    # Email and notification
    email_sent: Optional[int] = None

    # Weight
    weight: Optional[float] = None

    # Timestamps
    created_at: Optional[str] = None  # Keep as string to match API
    updated_at: Optional[str] = None  # Keep as string to match API

    # Coupon and discount information (keep for backward compatibility)
    coupon_code: Optional[str] = None
    discount_description: Optional[str] = None

    # Complex nested objects (as dicts to preserve full API data)
    billing_address: Optional[dict] = None
    items: Optional[List[dict]] = Field(default_factory=list)
    payment: Optional[dict] = None
    status_histories: Optional[List[dict]] = Field(default_factory=list)
    extension_attributes: Optional[dict] = None


class NewsletterSubscription(BaseModel):
    """Newsletter subscription model."""

    model_config = ConfigDict(from_attributes=True)

    subscriber_email: str
    subscriber_status: str
    change_status_at: Optional[datetime] = None
    is_subscribed: bool = Field(default=False)

    def __init__(self, **data):
        super().__init__(**data)
        # Set is_subscribed based on subscriber_status
        self.is_subscribed = self.subscriber_status == "1"


class ProductReview(BaseModel):
    """Product review model."""

    model_config = ConfigDict(from_attributes=True)

    # Core review fields
    id: Optional[int] = None  # Review ID from API
    review_id: Optional[str] = None
    title: Optional[str] = None
    detail: Optional[str] = None
    nickname: str

    # Review status and type
    review_status: Optional[int] = None  # 1=Approved, 2=Pending, 3=Not Approved
    review_type: Optional[int] = None  # 1=Customer, 2=Guest, 3=Administrator
    review_entity: Optional[str] = None

    # Customer and entity information
    customer_id: Optional[str] = None
    entity_pk_value: Optional[int] = None  # Product ID

    # Store information
    store_id: Optional[int] = None
    stores: Optional[List[int]] = Field(default_factory=list)

    # Rating information
    ratings: Optional[List[dict]] = Field(default_factory=list)

    # Timestamps
    created_at: Optional[str] = None  # Keep as string to match API

    @field_validator("customer_id", mode="before")
    def convert_customer_id_to_str(cls, value):
        """Convert customer_id from int to str if it is passed in."""
        if isinstance(value, int):
            return str(value)
        return value

    @property
    def approved(self) -> bool:
        """Check if review is approved (review_status == 1)."""
        return self.review_status == 1


class Product(BaseModel):
    """Product model."""

    model_config = ConfigDict(from_attributes=True)

    id: int
    sku: str
    name: str
    price: Optional[float] = None
    status: Optional[int] = None
    visibility: Optional[int] = None
    type_id: Optional[str] = None
    created_at: Optional[str] = None
    updated_at: Optional[str] = None
    weight: Optional[float] = None
    extension_attributes: Optional[dict] = None
    product_links: Optional[List[dict]] = Field(default_factory=list)
    options: Optional[List[dict]] = Field(default_factory=list)
    media_gallery_entries: Optional[List[dict]] = Field(default_factory=list)
    tier_prices: Optional[List[dict]] = Field(default_factory=list)
    custom_attributes: Optional[List[dict]] = Field(default_factory=list)


class ProductInventory(BaseModel):
    """Product inventory/stock item model."""

    model_config = ConfigDict(from_attributes=True)

    item_id: Optional[int] = None
    product_id: Optional[int] = None
    stock_id: Optional[int] = None
    qty: Optional[float] = None
    is_in_stock: Optional[bool] = None
    is_qty_decimal: Optional[bool] = None
    show_default_notification_message: Optional[bool] = None
    use_config_min_qty: Optional[bool] = None
    min_qty: Optional[float] = None
    use_config_min_sale_qty: Optional[int] = None
    min_sale_qty: Optional[float] = None
    use_config_max_sale_qty: Optional[bool] = None
    max_sale_qty: Optional[float] = None
    use_config_backorders: Optional[bool] = None
    backorders: Optional[int] = None
    use_config_notify_stock_qty: Optional[bool] = None
    notify_stock_qty: Optional[float] = None
    use_config_qty_increments: Optional[bool] = None
    qty_increments: Optional[float] = None
    use_config_enable_qty_inc: Optional[bool] = None
    enable_qty_increments: Optional[bool] = None
    use_config_manage_stock: Optional[bool] = None
    manage_stock: Optional[bool] = None
    low_stock_date: Optional[str] = None
    is_decimal_divided: Optional[bool] = None
    stock_status_changed_auto: Optional[int] = None


class SalesRule(BaseModel):
    """Sales rule model for Magento discount/promotion rules."""

    model_config = ConfigDict(from_attributes=True)

    # Core rule identification
    rule_id: Optional[int] = None
    name: str
    description: Optional[str] = None
    store_labels: List[str] = Field(default_factory=list)

    # Rule status and configuration
    is_active: bool = True
    website_ids: List[int] = Field(default_factory=list)
    customer_group_ids: List[int] = Field(default_factory=list)

    # Usage limits
    uses_per_customer: int = 0
    uses_per_coupon: int = 0
    times_used: int = 0

    # Rule conditions (complex nested objects)
    condition: Optional[dict] = None
    action_condition: Optional[dict] = None

    # Rule behavior
    stop_rules_processing: bool = False
    is_advanced: bool = False
    sort_order: int = 0

    # Discount configuration
    simple_action: Optional[str] = (
        None  # e.g., "buy_x_get_y", "by_percent", "cart_fixed"
    )
    discount_amount: float = 0.0
    discount_step: Optional[int] = None
    apply_to_shipping: bool = False

    # Additional features
    is_rss: bool = False
    coupon_type: Optional[str] = None  # e.g., "NO_COUPON", "SPECIFIC_COUPON"
    use_auto_generation: bool = False
    simple_free_shipping: Optional[str] = None  # "0", "1", etc.

    # Date range (keeping as optional strings to match API)
    from_date: Optional[str] = None
    to_date: Optional[str] = None

    @property
    def discount_amount_type(self) -> str:
        """Get discount amount type from simple_action field."""
        return self.simple_action or ""


class CmsPage(BaseModel):
    """CMS page model for Magento content pages."""

    model_config = ConfigDict(from_attributes=True)

    # Core page identification
    id: Optional[int] = None
    page_id: Optional[int] = None  # Alternative ID field
    identifier: Optional[str] = None
    title: str

    # Page content
    content: Optional[str] = None
    content_heading: Optional[str] = None
    meta_title: Optional[str] = None
    meta_description: Optional[str] = None
    meta_keywords: Optional[str] = None

    # Page configuration
    is_active: bool = True
    page_layout: Optional[str] = None
    layout_update_xml: Optional[str] = None
    custom_theme: Optional[str] = None
    custom_root_template: Optional[str] = None
    custom_layout_update_xml: Optional[str] = None

    # SEO and routing
    sort_order: int = 0
    store_id: List[int] = Field(default_factory=list)

    # Timestamps
    creation_time: Optional[str] = None
    update_time: Optional[str] = None


# Response models
class CartResponse(BaseModel):
    """Response model for cart operations."""

    model_config = ConfigDict(from_attributes=True)

    success: bool
    message: str
    cart_ids: List[str] = Field(default_factory=list)
    carts: List[Cart] = Field(default_factory=list)
    customer_name: Optional[str] = None


class CartItemsResponse(BaseModel):
    """Response model for cart items."""

    model_config = ConfigDict(from_attributes=True)

    success: bool
    message: str
    cart_id: str
    items: List[CartItem] = Field(default_factory=list)
    item_count: int = 0


class AddressResponse(BaseModel):
    """Response model for address operations."""

    model_config = ConfigDict(from_attributes=True)

    success: bool
    message: str
    email: str
    addresses: List[Address] = Field(default_factory=list)
    address_count: int = 0


class WishlistResponse(BaseModel):
    """Response model for wishlist operations."""

    model_config = ConfigDict(from_attributes=True)

    success: bool
    message: str
    customer_id: Optional[int] = None
    customer_info: Optional[CustomerInfo] = None
    items: List[WishlistItem] = Field(default_factory=list)
    item_count: int = 0


class OrderResponse(BaseModel):
    """Response model for order operations."""

    model_config = ConfigDict(from_attributes=True)

    success: bool
    message: str
    email: str
    orders: List[Order] = Field(default_factory=list)
    order_count: int = 0


class NewsletterResponse(BaseModel):
    """Response model for newsletter operations."""

    model_config = ConfigDict(from_attributes=True)

    success: bool
    message: str
    email: str
    subscription: Optional[NewsletterSubscription] = None


class ReviewResponse(BaseModel):
    """Response model for review operations."""

    model_config = ConfigDict(from_attributes=True)

    success: bool
    message: str
    product_sku: str
    review: Optional[ProductReview] = None


class ProductResponse(BaseModel):
    """Response model for product operations."""

    model_config = ConfigDict(from_attributes=True)

    success: bool
    message: str
    product_id: int
    product: Optional[Product] = None


# Eval Params Models
class ProductDetailsEvalParams(BaseModel):
    """Eval params for verify_product_details function."""

    model_config = ConfigDict(from_attributes=True)

    product_id: Optional[str] = None
    name: Optional[str] = None


class OrderDetailsEvalParams(BaseModel):
    """Eval params for verify_order_details function."""

    model_config = ConfigDict(from_attributes=True)

    order_id: str
    address_type: Optional[str] = "shipping"  # "shipping" or "billing"


class ReviewDetailsEvalParams(BaseModel):
    """Eval params for verify_review_details function."""

    model_config = ConfigDict(from_attributes=True)

    review_id: str


class SaleDetailsEvalParams(BaseModel):
    """Eval params for verify_sale_details function."""

    model_config = ConfigDict(from_attributes=True)

    topic: str


class SitePageDetailsEvalParams(BaseModel):
    """Eval params for verify_site_page_details function."""

    model_config = ConfigDict(from_attributes=True)

    page_id: str


# Expected Data Models
class SkuExpectedData(BaseModel):
    """Expected data for SKU-based validations (cart, orders, wishlist)."""

    model_config = ConfigDict(from_attributes=True)

    sku: Optional[str] = None
    skus: Optional[List[str]] = None


class ReviewExpectedData(BaseModel):
    """Expected data for review validation."""

    model_config = ConfigDict(from_attributes=True)

    sku: str
    num_star: int
    nickname: Optional[str] = None


class AddressExpectedData(BaseModel):
    """Expected data for address validation."""

    model_config = ConfigDict(from_attributes=True)

    address: str
    address2: Optional[str] = None
    city: str
    state: str
    zip_code: str
    country: Optional[str] = None


class ProductDetailsExpectedData(BaseModel):
    """Expected data for product details validation."""

    model_config = ConfigDict(from_attributes=True)

    in_stock: Optional[bool] = None
    name: Optional[str] = None
    price: Optional[float] = None
    stock_qty: Optional[int] = None
    color: Optional[str] = None
    size: Optional[str] = None
    description: Optional[List[str]] = None
    is_disabled: Optional[bool] = None
    on_sale: Optional[bool] = None
    variants: Optional[List[str]] = None


class OrderDetailsExpectedData(BaseModel):
    """Expected data for order details validation."""

    model_config = ConfigDict(from_attributes=True)

    status: Optional[str] = None
    comment: Optional[str] = None
    shipping_carrier: Optional[str] = None
    tracking_number: Optional[str] = None
    # Address fields
    address: Optional[str] = None
    address2: Optional[str] = None
    city: Optional[str] = None
    state: Optional[str] = None
    zip_code: Optional[str] = None
    country: Optional[str] = None


class ReviewDetailsExpectedData(BaseModel):
    """Expected data for review details validation."""

    model_config = ConfigDict(from_attributes=True)

    approved: Optional[bool] = None
    exists: Optional[bool] = None


class SaleDetailsExpectedData(BaseModel):
    """Expected data for sale details validation."""

    model_config = ConfigDict(from_attributes=True)

    name: Optional[str] = None
    website_ids: Optional[List[int]] = None
    customer_group_ids: Optional[List[int]] = None
    discount_amount_type: Optional[str] = None
    discount_amount: Optional[float] = None


class SitePageDetailsExpectedData(BaseModel):
    """Expected data for site page details validation."""

    model_config = ConfigDict(from_attributes=True)

    title: Optional[str] = None
    identifier: Optional[str] = None
    is_active: Optional[bool] = None
    content: Optional[str] = None


class ReportGenerationFormEvalParams(BaseModel):
    """Expected data for sales report generation form validation."""

    model_config = ConfigDict(from_attributes=True)

    from_date: str | None = None
    to_date: str | None = None
    order_statuses: List[str] | None = None
    period_type: str | None = None
    report_type: str | None = None
    show_empty_rows: str | None = None
    show_order_statuses: str | None = None
    store_ids: str | None = None
    url: str
