"""
Magento Custom Validation Functions for WebArena Verified Tasks.

This module contains the custom validation functions required by the
custom evaluator system for Magento-based tasks.
"""

import logging
from ..playwright_utils import get_form_fields
import json

from .models import (
    MagentoConfig,
    # Eval Params Models
    ProductDetailsEvalParams,
    OrderDetailsEvalParams,
    ReviewDetailsEvalParams,
    SaleDetailsEvalParams,
    SitePageDetailsEvalParams,
    # Expected Data Models
    SkuExpectedData,
    ReviewExpectedData,
    AddressExpectedData,
    ProductDetailsExpectedData,
    OrderDetailsExpectedData,
    ReviewDetailsExpectedData,
    SaleDetailsExpectedData,
    SitePageDetailsExpectedData,
    ReportGenerationFormEvalParams,
)
from ..datatype_evaluators import DateMatchEvaluator
from .data_helper import MagentoDataHelper
from ..types import (
    WebArenaTask,
    WebArenaTaskResponse,
    WebarenaTaskEvalResult,
    EvalFunc,
)
from ..models import AllocationResource


class MagentoValidator:
    """
    Magento validator with custom evaluation functions for WebArena Verified tasks.

    Contains validation methods that are called by the custom evaluator system.
    Uses MagentoDataHelper for all data access.
    """

    def __init__(self):
        self.logger = logging.getLogger(__name__)

    def validate_in_shopping_cart(
        self,
        task: WebArenaTask,
        task_result: WebArenaTaskResponse,
        resource: AllocationResource,
        eval_func: EvalFunc,
        **kwargs,
    ) -> WebarenaTaskEvalResult:
        """
        Validate that a product SKU is present in the shopping cart.

        Expected eval_func format:
        {
            "name": "validate_in_shopping_cart",
            "expected_data": {"sku": "B00CPTR7WS"}
        }
        Email comes from resource.username
        """
        try:
            # Extract expected_data from eval_func and validate with Pydantic
            if not eval_func or not hasattr(eval_func, "expected_data"):
                return WebarenaTaskEvalResult.create_failed(
                    assertion_msgs=["No expected_data found in eval_func"],
                    site=resource.website_type,
                )

            try:
                expected_data = SkuExpectedData.model_validate(eval_func.expected_data)
            except Exception as e:
                return WebarenaTaskEvalResult.create_failed(
                    assertion_msgs=[f"Invalid expected_data format: {str(e)}"],
                    site=resource.website_type,
                )

            # Create MagentoConfig from AllocationResource
            config = MagentoConfig(
                base_url=resource.base_url, container_name=resource.container_name
            )

            # Create data helper with resource-specific config
            data_helper = MagentoDataHelper(config)

            # Get email from resource username (assuming it's an email)
            email = resource.username

            # Use data helper to get cart items by email
            cart_items = data_helper.get_cart_items_by_email(email)

            if expected_data.sku is not None:
                # Check if any cart item has the specified SKU
                matching_items = [
                    item for item in cart_items if item.sku == expected_data.sku
                ]
            else:
                # Check if any cart item has the specified SKUs
                matching_items = [
                    item for item in cart_items if item.sku in expected_data.skus
                ]

            matching_sku_str = ", ".join([item.sku for item in matching_items])

            if matching_items:
                return WebarenaTaskEvalResult.create_success(
                    assertion_msgs=[
                        f"Product with SKU(s) {matching_sku_str} found in shopping cart for {email}"
                    ],
                    site=resource.website_type,
                    validation_data={
                        "sku": expected_data.sku,
                        "email": email,
                        "cart_items": [item.model_dump() for item in cart_items],
                    },
                    task_id=task.task_id,
                )
            else:
                if expected_data.sku is not None:
                    assert_msg = f"Product with SKU {expected_data.sku} not found in shopping cart for {email}"
                else:
                    assert_msg = f"No product with any of the expected SKUs found in shopping cart for {email}"
                return WebarenaTaskEvalResult.create_failed(
                    assertion_msgs=[assert_msg],
                    site=resource.website_type,
                    task_id=task.task_id,
                    validation_data={
                        "sku": expected_data.sku,
                        "email": email,
                        "cart_items": [item.model_dump() for item in cart_items],
                    },
                )

        except Exception as e:
            return WebarenaTaskEvalResult.create_failed(
                assertion_msgs=[f"Cart validation failed: {str(e)}"],
                site=resource.website_type,
                task_id=task.task_id,
            )

    def validate_ordered(
        self,
        task: WebArenaTask,
        task_result: WebArenaTaskResponse,
        resource: AllocationResource,
        eval_func: EvalFunc,
        **kwargs,
    ) -> WebarenaTaskEvalResult:
        """
        Validate that a product SKU has been ordered.

        Expected data format: {"sku": "B01JMA0YX6"}
        """
        try:
            # Extract expected_data from eval_func and validate with Pydantic
            if not eval_func or not hasattr(eval_func, "expected_data"):
                return WebarenaTaskEvalResult.create_failed(
                    assertion_msgs=["No expected_data found in eval_func"],
                    site=resource.website_type,
                    task_id=task.task_id,
                )

            try:
                expected_data = SkuExpectedData.model_validate(eval_func.expected_data)
            except Exception as e:
                return WebarenaTaskEvalResult.create_failed(
                    assertion_msgs=[f"Invalid expected_data format: {str(e)}"],
                    site=resource.website_type,
                    task_id=task.task_id,
                )

            # Create MagentoConfig from AllocationResource
            config = MagentoConfig(
                base_url=resource.base_url, container_name=resource.container_name
            )

            # Create data helper with resource-specific config
            data_helper = MagentoDataHelper(config)

            # Get email from resource username
            email = resource.username

            # Use data helper to get all orders by email
            orders = data_helper.get_orders_by_email(email)

            # Loop through orders looking for the SKU
            matching_orders = []
            assert_msg = []
            for order in orders:
                # Get order items for each order to check for SKU
                try:
                    order_items = order.items
                    for item in order_items:
                        if item["sku"] == expected_data.sku:
                            if order.status == "canceled":
                                assert_msg.append(
                                    f"Product with SKU {expected_data.sku} found in canceled order {order.entity_id}. Ignoring."
                                )
                            else:
                                matching_orders.append(order)
                                break
                except Exception as e:
                    self.logger.warning(
                        f"Could not get items for order {order.entity_id}: {e}"
                    )
                    continue

            if matching_orders:
                return WebarenaTaskEvalResult.create_success(
                    site=resource.website_type,
                    assertion_msgs=[
                        f"Product with SKU {expected_data.sku} found in order for {email}",
                        *assert_msg,
                    ],
                    validation_data={
                        "sku": expected_data.sku,
                        "email": email,
                        "matching_orders": matching_orders,
                        "all_orders": [order.model_dump() for order in orders],
                    },
                    task_id=task.task_id,
                )
            else:
                return WebarenaTaskEvalResult.create_failed(
                    assertion_msgs=[
                        f"Product with SKU {expected_data.sku} not found in any orders for {email}",
                        *assert_msg,
                    ],
                    validation_data={
                        "sku": expected_data.sku,
                        "email": email,
                        "matching_orders": [],
                        "all_orders": [order.model_dump() for order in orders],
                    },
                    site=resource.website_type,
                    task_id=task.task_id,
                )

        except Exception as e:
            return WebarenaTaskEvalResult.create_failed(
                assertion_msgs=[f"Order validation failed: {str(e)}"],
                site=resource.website_type,
                task_id=task.task_id,
            )

    def validate_in_wishlist(
        self,
        task: WebArenaTask,
        task_result: WebArenaTaskResponse,
        resource: AllocationResource,
        eval_func: EvalFunc,
        **kwargs,
    ) -> WebarenaTaskEvalResult:
        """
        Validate that a product SKU is present in a customer's wishlist.

        Expected data format: {"sku": "B074QVN413"}
        """
        try:
            # Extract SKU from task's expected_data
            expected_data = eval_func.expected_data
            if hasattr(expected_data, "sku"):
                sku = expected_data.sku
            elif isinstance(expected_data, dict) and "sku" in expected_data:
                sku = expected_data["sku"]
            else:
                return WebarenaTaskEvalResult.create_failed(
                    assertion_msgs=[
                        "No SKU found in expected_data for wishlist validation"
                    ],
                    site=resource.website_type,
                )

            # Create MagentoConfig from AllocationResource
            config = MagentoConfig(
                base_url=resource.base_url, container_name=resource.container_name
            )

            # Create data helper with resource-specific config
            data_helper = MagentoDataHelper(config)

            # Get email from resource username
            email = resource.username

            # Use data helper to get wishlist items by email
            wishlist_items = data_helper.get_wishlist_items_by_email(email)

            # Loop through wishlist items looking for the SKU
            matching_items = [item for item in wishlist_items if item.sku == sku]

            validation_data = {
                "sku": sku,
                "email": email,
                "wishlist_items": [item.model_dump() for item in wishlist_items],
            }
            if matching_items:
                return WebarenaTaskEvalResult.create_success(
                    assertion_msgs=[
                        f"Product with SKU {sku} found in wishlist for {email}"
                    ],
                    site=resource.website_type,
                    validation_data=validation_data,
                    task_id=task.task_id,
                )
            else:
                return WebarenaTaskEvalResult.create_failed(
                    assertion_msgs=[
                        f"Product with SKU {sku} not found in wishlist for {email}"
                    ],
                    validation_data=validation_data,
                    site=resource.website_type,
                    task_id=task.task_id,
                )

        except Exception as e:
            return WebarenaTaskEvalResult.create_failed(
                assertion_msgs=[f"Wishlist validation failed: {str(e)}"],
                site=resource.website_type,
                task_id=task.task_id,
            )

    def validate_newsletter(
        self,
        task: WebArenaTask,
        task_result: WebArenaTaskResponse,
        resource: AllocationResource,
        **kwargs,
    ) -> WebarenaTaskEvalResult:
        """
        Validate newsletter subscription status.

        This function checks if newsletter subscription was successful.
        Expected data: null (no specific data expected)
        """
        try:
            # Create MagentoConfig from AllocationResource
            config = MagentoConfig(
                base_url=resource.base_url, container_name=resource.container_name
            )

            # Create data helper with resource-specific config
            data_helper = MagentoDataHelper(config)

            # Get email from resource username
            email = resource.username

            # Use data helper to get newsletter subscription by email
            subscription = data_helper.get_newsletter_subscription(email)

            if subscription and subscription.is_subscribed:
                return WebarenaTaskEvalResult.create_success(
                    site=resource.website_type,
                    validation_data={
                        "email": email,
                        "subscription": subscription.model_dump(),
                    },
                    task_id=task.task_id,
                )
            else:
                return WebarenaTaskEvalResult.create_failed(
                    assertion_msgs=[
                        f"No active newsletter subscription found for {email}"
                    ],
                    site=resource.website_type,
                    task_id=task.task_id,
                )

        except Exception as e:
            return WebarenaTaskEvalResult.create_failed(
                assertion_msgs=[f"Newsletter validation failed: {str(e)}"],
                site=resource.website_type,
                task_id=task.task_id,
            )

    def validate_address(
        self,
        task: WebArenaTask,
        task_result: WebArenaTaskResponse,
        resource: AllocationResource,
        eval_func: EvalFunc,
        **kwargs,
    ) -> WebarenaTaskEvalResult:
        """
        Validate that a specific address exists in customer addresses.

        Expected data format: {
            "address": "231 Willow Way",
            "address2": "Suite 100",
            "city": "Chicago",
            "state": "IL",
            "zip_code": "60601",
            "country": "US"
        }
        """
        try:
            # Extract and validate expected_data from eval_func using Pydantic
            if not eval_func or not hasattr(eval_func, "expected_data"):
                return WebarenaTaskEvalResult.create_failed(
                    assertion_msgs=["No expected_data found in eval_func"],
                    site=resource.website_type,
                    task_id=task.task_id,
                )

            try:
                expected_data = AddressExpectedData.model_validate(
                    eval_func.expected_data
                )
            except Exception as e:
                return WebarenaTaskEvalResult.create_failed(
                    assertion_msgs=[
                        f"Invalid expected_data format for address validation: {str(e)}"
                    ],
                    site=resource.website_type,
                    task_id=task.task_id,
                )

            # Create MagentoConfig from AllocationResource
            config = MagentoConfig(
                base_url=resource.base_url, container_name=resource.container_name
            )

            # Create data helper with resource-specific config
            data_helper = MagentoDataHelper(config)

            # Get email from resource username
            email = resource.username

            # Use data helper to get addresses by email
            addresses = data_helper.get_addresses_by_email(email)

            # Find matching addresses by comparing individual fields
            matching_addresses = []
            assert_msg = []
            for address in addresses:
                # Check each field for matches (case-insensitive where appropriate)
                field_matches = []
                field_messages = []

                # Address match
                if expected_data.address:
                    address_match = (
                        expected_data.address.lower() == (address.street or "").lower()
                    )
                    field_matches.append(address_match)
                    if address_match:
                        field_messages.append(
                            f"Address match: expected '{expected_data.address}' found in '{address.street}'"
                        )
                    else:
                        field_messages.append(
                            f"Address mismatch: expected '{expected_data.address}' not found in '{address.street}'"
                        )
                else:
                    address_match = True

                # Address2 match
                if expected_data.address2:
                    address2_match = (
                        expected_data.address2.lower() == (address.street or "").lower()
                    )
                    field_matches.append(address2_match)
                    if address2_match:
                        field_messages.append(
                            f"Address2 match: expected '{expected_data.address2}' == actual '{address.street}'"
                        )
                    else:
                        field_messages.append(
                            f"Address2 mismatch: expected '{expected_data.address2}' != actual '{address.street}'"
                        )
                else:
                    address2_match = True

                # City match
                if expected_data.city:
                    city_match = (
                        expected_data.city.lower() == (address.city or "").lower()
                    )
                    field_matches.append(city_match)
                    if city_match:
                        field_messages.append(
                            f"City match: expected '{expected_data.city}' == actual '{address.city}'"
                        )
                    else:
                        field_messages.append(
                            f"City mismatch: expected '{expected_data.city}' != actual '{address.city}'"
                        )
                else:
                    city_match = True

                # State match
                if expected_data.state:
                    state_match = (
                        expected_data.state.lower() == (address.region or "").lower()
                    )
                    field_matches.append(state_match)
                    if state_match:
                        field_messages.append(
                            f"State match: expected '{expected_data.state}' == actual '{address.region}'"
                        )
                    else:
                        field_messages.append(
                            f"State mismatch: expected '{expected_data.state}' != actual '{address.region}'"
                        )
                else:
                    state_match = True

                # Zip code match
                if expected_data.zip_code:
                    zip_match = expected_data.zip_code == (address.postcode or "")
                    field_matches.append(zip_match)
                    if zip_match:
                        field_messages.append(
                            f"Zip match: expected '{expected_data.zip_code}' == actual '{address.postcode}'"
                        )
                    else:
                        field_messages.append(
                            f"Zip mismatch: expected '{expected_data.zip_code}' != actual '{address.postcode}'"
                        )
                else:
                    zip_match = True

                # Country match
                if expected_data.country:
                    country_match = (
                        expected_data.country.upper()
                        == (address.country_id or "").upper()
                    )
                    field_matches.append(country_match)
                    if country_match:
                        field_messages.append(
                            f"Country match: expected '{expected_data.country}' == actual '{address.country_id}'"
                        )
                    else:
                        field_messages.append(
                            f"Country mismatch: expected '{expected_data.country}' != actual '{address.country_id}'"
                        )
                else:
                    country_match = True

                # Add field messages to assert_msg for this address
                assert_msg.extend(field_messages)

                # Check if all configured fields match
                if (
                    address_match
                    and address2_match
                    and city_match
                    and state_match
                    and zip_match
                    and country_match
                ):
                    matching_addresses.append(address)

            validation_data = {
                "expected_address": expected_data.model_dump(),
                "email": email,
                "addresses": [addr.model_dump() for addr in addresses],
            }
            if matching_addresses:
                return WebarenaTaskEvalResult.create_success(
                    assertion_msgs=[f"Address found for {email}", *assert_msg],
                    site=resource.website_type,
                    validation_data=validation_data,
                    task_id=task.task_id,
                )
            else:
                return WebarenaTaskEvalResult.create_failed(
                    assertion_msgs=[
                        f"Address not found for {email}. Expected: {expected_data.address}, {expected_data.city}, {expected_data.state}, {expected_data.zip_code}",
                        *assert_msg,
                    ],
                    validation_data=validation_data,
                    site=resource.website_type,
                    task_id=task.task_id,
                )

        except Exception as e:
            return WebarenaTaskEvalResult.create_failed(
                assertion_msgs=[f"Address validation failed: {str(e)}"],
                site=resource.website_type,
                task_id=task.task_id,
            )

    def validate_review(
        self,
        task: WebArenaTask,
        task_result: WebArenaTaskResponse,
        resource: AllocationResource,
        eval_func: EvalFunc,
        **kwargs,
    ) -> WebarenaTaskEvalResult:
        """
        Validate that a product review exists with specified SKU and star rating.

        Expected data format: {"sku": "B00J8RZL7I", "num_star": 5}
        """
        try:
            # Extract expected_data from eval_func and validate with Pydantic
            if not eval_func or not hasattr(eval_func, "expected_data"):
                return WebarenaTaskEvalResult.create_failed(
                    assertion_msgs=["No expected_data found in eval_func"],
                    site=resource.website_type,
                    task_id=task.task_id,
                )

            try:
                expected_data = ReviewExpectedData.model_validate(
                    eval_func.expected_data
                )
            except Exception as e:
                return WebarenaTaskEvalResult.create_failed(
                    assertion_msgs=[f"Invalid expected_data format: {str(e)}"],
                    site=resource.website_type,
                    task_id=task.task_id,
                )

            # Create MagentoConfig from AllocationResource
            config = MagentoConfig(
                base_url=resource.base_url, container_name=resource.container_name
            )

            # Create data helper with resource-specific config
            data_helper = MagentoDataHelper(config)

            # Get email from resource username
            email = resource.username

            # Use data helper's method to get product reviews
            reviews = data_helper.get_product_review_by_sku(expected_data.sku)

            # Filter reviews by rating
            matching_reviews = [
                review
                for review in reviews
                if any(
                    rating["value"] == expected_data.num_star
                    for rating in review.ratings
                )
            ]

            if expected_data.nickname:
                matching_reviews = [
                    review
                    for review in matching_reviews
                    if review.nickname == expected_data.nickname
                ]

            validation_data = {
                "sku": expected_data.sku,
                "num_star": expected_data.num_star,
                "email": email,
                "reviews": [r.model_dump() for r in reviews],
            }
            if matching_reviews:
                return WebarenaTaskEvalResult.create_success(
                    assertion_msgs=[
                        f"Review found for product {expected_data.sku} with {expected_data.num_star} star rating and nickname {expected_data.nickname}"
                    ],
                    site=resource.website_type,
                    validation_data=validation_data,
                    task_id=task.task_id,
                )
            else:
                return WebarenaTaskEvalResult.create_failed(
                    assertion_msgs=[
                        f"No review found for product {expected_data.sku} with {expected_data.num_star} star rating and nickname {expected_data.nickname}"
                    ],
                    site=resource.website_type,
                    task_id=task.task_id,
                    validation_data=validation_data,
                )

        except Exception as e:
            return WebarenaTaskEvalResult.create_failed(
                assertion_msgs=[f"Review validation failed: {str(e)}"],
                site=resource.website_type,
                task_id=task.task_id,
            )

    def verify_product_details(
        self,
        *,
        task: WebArenaTask,
        task_result: WebArenaTaskResponse,
        resource: AllocationResource,
        eval_func: EvalFunc,
        **kwargs,
    ) -> WebarenaTaskEvalResult:
        """
        Verify product details match expected values.

        Expected eval_func format:
        {
            "name": "verify_product_details",
            "eval_params": {"product_id": "350"} or {"name": "Product Name"},
            "expected_data": {"in_stock": false}
        }
        """
        try:
            # Extract and validate eval_params from eval_func
            if not eval_func or not hasattr(eval_func, "eval_params"):
                return WebarenaTaskEvalResult.create_failed(
                    assertion_msgs=["No eval_params found in eval_func"],
                    site=resource.website_type,
                    task_id=task.task_id,
                )

            try:
                eval_params = ProductDetailsEvalParams.model_validate(
                    eval_func.eval_params
                )
            except Exception as e:
                return WebarenaTaskEvalResult.create_failed(
                    assertion_msgs=[f"Invalid eval_params format: {str(e)}"],
                    site=resource.website_type,
                    task_id=task.task_id,
                )

            # Extract and validate expected_data from eval_func
            if not hasattr(eval_func, "expected_data"):
                return WebarenaTaskEvalResult.create_failed(
                    assertion_msgs=["No expected_data found in eval_func"],
                    site=resource.website_type,
                    task_id=task.task_id,
                )

            try:
                expected_data = ProductDetailsExpectedData.model_validate(
                    eval_func.expected_data
                )
            except Exception as e:
                return WebarenaTaskEvalResult.create_failed(
                    assertion_msgs=[f"Invalid expected_data format: {str(e)}"],
                    site=resource.website_type,
                    task_id=task.task_id,
                )

            # Create MagentoConfig from AllocationResource
            config = MagentoConfig(
                base_url=resource.base_url, container_name=resource.container_name
            )

            # Create data helper with resource-specific config
            data_helper = MagentoDataHelper(config)

            # Check if we should lookup by name or ID
            if eval_params.name:
                product = data_helper.get_product_by_name(eval_params.name)
                identifier = f"name '{eval_params.name}'"
            elif eval_params.product_id:
                product_id = int(eval_params.product_id)
                product = data_helper.get_product_by_id(product_id)
                identifier = f"ID {product_id}"
            else:
                return WebarenaTaskEvalResult.create_failed(
                    assertion_msgs=[
                        "Either 'name' or 'product_id' must be provided in eval_params"
                    ],
                    site=resource.website_type,
                    task_id=task.task_id,
                )

            if not product:
                return WebarenaTaskEvalResult.create_failed(
                    assertion_msgs=[f"Product {identifier} not found"],
                    site=resource.website_type,
                    task_id=task.task_id,
                )

            # Validate individual fields if present in expected_data
            assert_msgs = []
            failed_validation = False

            # Validate in_stock if present in expected_data
            if expected_data.in_stock is not None:
                # Check stock using extension_attributes.stock_item.is_in_stock for better accuracy
                actual_in_stock = False
                if (
                    product.extension_attributes
                    and isinstance(product.extension_attributes, dict)
                    and "stock_item" in product.extension_attributes
                ):
                    stock_item = product.extension_attributes["stock_item"]
                    if isinstance(stock_item, dict) and "is_in_stock" in stock_item:
                        actual_in_stock = stock_item["is_in_stock"]
                    else:
                        # Fallback to product status if stock_item structure is unexpected
                        actual_in_stock = product.status == 1
                else:
                    # Fallback to product status if no stock_item in extension_attributes
                    actual_in_stock = product.status == 1

                if actual_in_stock != expected_data.in_stock:
                    assert_msgs.append(
                        f"Mismatch in_stock: expected {expected_data.in_stock}, actual {actual_in_stock}"
                    )
                    failed_validation = True
                else:
                    assert_msgs.append(
                        f"Match in_stock: expected {expected_data.in_stock}, actual {actual_in_stock}"
                    )

            # Check name
            if expected_data.name:
                if product.name != expected_data.name:
                    assert_msgs.append(
                        f"Name mismatch: expected '{expected_data.name}', got '{product.name}'"
                    )
                    failed_validation = True
                else:
                    assert_msgs.append(
                        f"Name match: expected '{expected_data.name}', got '{product.name}'"
                    )

            # Check price
            if expected_data.price is not None:
                expected_price = self._normalize_currency(expected_data.price)
                actual_price = self._normalize_currency(product.price)
                if (
                    abs(actual_price - expected_price) > 0.01
                ):  # Allow small floating point differences
                    assert_msgs.append(
                        f"Price mismatch: expected {expected_price}, got {actual_price}"
                    )
                    failed_validation = True
                else:
                    assert_msgs.append(
                        f"Price match: expected {expected_price}, got {actual_price}"
                    )

            # Check stock_qty
            inventory_data = None
            if expected_data.stock_qty is not None:
                # Get stock quantity from extension_attributes first
                actual_stock_qty = None
                if (
                    product.extension_attributes
                    and isinstance(product.extension_attributes, dict)
                    and "stock_item" in product.extension_attributes
                ):
                    stock_item = product.extension_attributes["stock_item"]
                    if isinstance(stock_item, dict) and "qty" in stock_item:
                        actual_stock_qty = int(stock_item["qty"])

                # If stock quantity not available in product data, get it from inventory API
                if actual_stock_qty is None:
                    self.logger.debug(
                        f"Stock quantity not available in product data for {product.sku}, fetching from inventory API"
                    )
                    inventory_data = data_helper.get_product_inventory_by_sku(
                        product.sku
                    )
                    if inventory_data and inventory_data.qty is not None:
                        actual_stock_qty = int(inventory_data.qty)

                if actual_stock_qty is None:
                    assert_msgs.append(
                        "Stock quantity not available in product data or inventory API"
                    )
                    failed_validation = True
                elif actual_stock_qty != expected_data.stock_qty:
                    assert_msgs.append(
                        f"Stock quantity mismatch: expected {expected_data.stock_qty}, got {actual_stock_qty}"
                    )
                    failed_validation = True
                else:
                    assert_msgs.append(
                        f"Stock quantity match: expected {expected_data.stock_qty}, got {actual_stock_qty}"
                    )

            # Check color (custom attribute)
            if expected_data.color:
                actual_color = None
                if product.custom_attributes:
                    for attr in product.custom_attributes:
                        if attr["attribute_code"] == "color":
                            actual_color = str(attr["value"])
                            break

                if actual_color is None:
                    assert_msgs.append("Color attribute not found in product")
                    failed_validation = True
                elif actual_color != expected_data.color:
                    assert_msgs.append(
                        f"Color mismatch: expected '{expected_data.color}', got '{actual_color}'"
                    )
                    failed_validation = True
                else:
                    assert_msgs.append(
                        f"Color match: expected '{expected_data.color}', got '{actual_color}'"
                    )

            # Check size (custom attribute)
            if expected_data.size:
                actual_size = None
                if product.custom_attributes:
                    for attr in product.custom_attributes:
                        if attr["attribute_code"] == "size":
                            actual_size = str(attr["value"])
                            break

                if actual_size is None:
                    assert_msgs.append("Size attribute not found in product")
                    failed_validation = True
                elif actual_size != expected_data.size:
                    assert_msgs.append(
                        f"Size mismatch: expected '{expected_data.size}', got '{actual_size}'"
                    )
                    failed_validation = True
                else:
                    assert_msgs.append(
                        f"Size match: expected '{expected_data.size}', got '{actual_size}'"
                    )

            # Check description (substring matching)
            if expected_data.description:
                actual_description = None
                # Find description in custom_attributes
                if product.custom_attributes:
                    for attr in product.custom_attributes:
                        if attr["attribute_code"] == "description":
                            actual_description = (
                                str(attr["value"]) if attr["value"] else ""
                            )
                            break

                if actual_description is None:
                    assert_msgs.append("Description attribute not found in product")
                    failed_validation = True
                else:
                    missing_substrings = []
                    for expected_substring in expected_data.description:
                        if expected_substring not in actual_description:
                            missing_substrings.append(expected_substring)

                    if missing_substrings:
                        assert_msgs.append(
                            f"Description missing expected substrings: {missing_substrings}. "
                            f"Actual description: '{actual_description}'"
                        )
                        failed_validation = True
                    else:
                        assert_msgs.append(
                            f"Description match: expected '{expected_data.description}', got '{actual_description}'"
                        )

            # Check is_disabled (product status)
            if expected_data.is_disabled is not None:
                # Product is disabled if status != 1 (enabled)
                actual_disabled = product.status != 1

                if actual_disabled != expected_data.is_disabled:
                    assert_msgs.append(
                        f"Product disabled status mismatch: expected {expected_data.is_disabled}, got {actual_disabled}. "
                        f"Product status: {product.status}"
                    )
                    failed_validation = True
                else:
                    assert_msgs.append(
                        f"Product disabled status match: expected {expected_data.is_disabled}, got {actual_disabled}. "
                        f"Product status: {product.status}"
                    )

            # Check on_sale (custom attribute 'sale')
            if expected_data.on_sale is not None:
                # Product is on sale if custom attribute 'sale' is 1
                actual_on_sale = None
                if product.custom_attributes:
                    for attr in product.custom_attributes:
                        if attr["attribute_code"] == "sale":
                            actual_on_sale = int(attr["value"]) == 1
                            break

                if actual_on_sale is None:
                    assert_msgs.append("Sale attribute not found in product")
                    failed_validation = True
                elif actual_on_sale != expected_data.on_sale:
                    assert_msgs.append(
                        f"Product on sale status mismatch: expected {expected_data.on_sale}, got {actual_on_sale}"
                    )
                    failed_validation = True
                else:
                    assert_msgs.append(
                        f"Product on sale status match: expected {expected_data.on_sale}, got {actual_on_sale}"
                    )

            actual_variants = None
            # Validate variants if present in expected_data
            if expected_data.variants:
                # Get actual variant names using helper method
                actual_variants = data_helper.get_variant_names_for_product(product)

                # Check if all expected variants are present
                missing_variants = []
                for expected_variant in expected_data.variants:
                    if expected_variant not in actual_variants:
                        missing_variants.append(expected_variant)

                if missing_variants:
                    assert_msgs.append(
                        f"Product {identifier} missing expected variants: {missing_variants}. "
                        f"Actual variants: {actual_variants}"
                    )
                    failed_validation = True
                else:
                    assert_msgs.append(
                        f"Product {identifier} variants match: expected {expected_data.variants}, got {actual_variants}"
                    )

            validation_data = {
                "product": product.model_dump(),
            }

            if actual_variants:
                validation_data["variants"] = actual_variants

            if inventory_data:
                validation_data["inventory"] = inventory_data.model_dump()

            # If there are validation errors, return failure
            if failed_validation:
                return WebarenaTaskEvalResult.create_failed(
                    assertion_msgs=[
                        f"Product {identifier} validation failed:",
                        *assert_msgs,
                    ],
                    site=resource.website_type,
                    validation_data=validation_data,
                    task_id=task.task_id,
                )

            # Validation passed
            return WebarenaTaskEvalResult.create_success(
                assertion_msgs=[
                    f"Success Check for Product: {product.name} (SKU: {product.sku})",
                    *assert_msgs,
                ],
                site=resource.website_type,
                validation_data=validation_data,
                task_id=task.task_id,
            )

        except ValueError as e:
            return WebarenaTaskEvalResult.create_failed(
                assertion_msgs=[f"Invalid product_id format: {str(e)}"],
                site=resource.website_type,
                task_id=task.task_id,
            )
        except Exception as e:
            return WebarenaTaskEvalResult.create_failed(
                assertion_msgs=[f"Product details validation failed: {str(e)}"],
                site=resource.website_type,
                task_id=task.task_id,
            )

    def verify_order_details(
        self,
        *,
        task: WebArenaTask,
        task_result: WebArenaTaskResponse,
        resource: AllocationResource,
        eval_func: EvalFunc,
        **kwargs,
    ) -> WebarenaTaskEvalResult:
        """
        Verify order details match expected values.

        Expected eval_func format:
        {
            "name": "verify_order_details",
            "eval_params": {"order_id": "302"},
            "expected_data": {"status": "Canceled"}
        }
        """
        try:
            # Extract and validate eval_params from eval_func
            if not eval_func or not hasattr(eval_func, "eval_params"):
                return WebarenaTaskEvalResult.create_failed(
                    assertion_msgs=["No eval_params found in eval_func"],
                    site=resource.website_type,
                    task_id=task.task_id,
                )

            try:
                eval_params = OrderDetailsEvalParams.model_validate(
                    eval_func.eval_params
                )
            except Exception as e:
                return WebarenaTaskEvalResult.create_failed(
                    assertion_msgs=[f"Invalid eval_params format: {str(e)}"],
                    site=resource.website_type,
                    task_id=task.task_id,
                )

            # Extract and validate expected_data from eval_func
            if not hasattr(eval_func, "expected_data"):
                return WebarenaTaskEvalResult.create_failed(
                    assertion_msgs=["No expected_data found in eval_func"],
                    site=resource.website_type,
                    task_id=task.task_id,
                )

            try:
                expected_data = OrderDetailsExpectedData.model_validate(
                    eval_func.expected_data
                )
            except Exception as e:
                return WebarenaTaskEvalResult.create_failed(
                    assertion_msgs=[f"Invalid expected_data format: {str(e)}"],
                    site=resource.website_type,
                    task_id=task.task_id,
                )

            # Create MagentoConfig from AllocationResource
            config = MagentoConfig(
                base_url=resource.base_url, container_name=resource.container_name
            )

            # Create data helper with resource-specific config
            data_helper = MagentoDataHelper(config)

            # Get order_id from eval_params
            order_id = int(eval_params.order_id)
            order = data_helper.get_order_by_id(order_id)
            identifier = f"ID {order_id}"

            if not order:
                return WebarenaTaskEvalResult.create_failed(
                    assertion_msgs=[f"Order not found with {identifier}"],
                    site=resource.website_type,
                    task_id=task.task_id,
                )

            # Validate individual fields if present in expected_data
            assert_msgs = []
            failed_validation = False

            # Check status
            if expected_data.status:
                expected_status = expected_data.status.lower()
                actual_status = order.status.lower()

                if actual_status != expected_status:
                    assert_msgs.append(
                        f"Status mismatch: expected '{expected_status}', got '{actual_status}'"
                    )
                    failed_validation = True
                else:
                    assert_msgs.append(
                        f"Status match: expected '{expected_status}', got '{actual_status}'"
                    )

            # Validate order comments
            order_comments = None
            if expected_data.comment:
                order_comments = data_helper.get_order_comments(order_id)

                # Check if expected comment string exists in any of the order comments
                comment_found = False
                actual_comment = None
                if order_comments:
                    for comment in order_comments:
                        if expected_data.comment.lower() in comment.lower():
                            comment_found = True
                            actual_comment = comment
                            break

                if not comment_found:
                    assert_msgs.append(
                        f"Expected comment '{expected_data.comment}' not found in order comments. "
                        f"Available comments: {[c.comment for c in order_comments if hasattr(c, 'comment') and c.comment]}"
                    )
                    failed_validation = True
                else:
                    assert_msgs.append(
                        f"Comment match: expected '{expected_data.comment}', got '{actual_comment}'"
                    )

            # Validate order shipments (shipping carrier and tracking number)
            order_shipments = None
            if expected_data.shipping_carrier or expected_data.tracking_number:
                order_shipments = data_helper.get_order_shipments(order_id)

                if expected_data.shipping_carrier:
                    carrier_found = False
                    actual_carrier = None
                    if order_shipments:
                        for shipment in order_shipments:
                            if (
                                "shipping_carrier" in shipment
                                and shipment["shipping_carrier"]
                            ):
                                if (
                                    expected_data.shipping_carrier.lower()
                                    in shipment["shipping_carrier"].lower()
                                ):
                                    carrier_found = True
                                    actual_carrier = shipment["shipping_carrier"]
                                    break

                    if not carrier_found:
                        available_carriers = (
                            [s.get("shipping_carrier", "N/A") for s in order_shipments]
                            if order_shipments
                            else []
                        )
                        assert_msgs.append(
                            f"Expected shipping carrier '{expected_data.shipping_carrier}' not found. "
                            f"Available carriers: {available_carriers}"
                        )
                        failed_validation = True
                    else:
                        assert_msgs.append(
                            f"Shipping carrier match: expected '{expected_data.shipping_carrier}', got '{actual_carrier}'"
                        )

            if expected_data.tracking_number:
                tracking_found = False
                actual_tracking = None
                if order_shipments:
                    for shipment in order_shipments:
                        if (
                            "tracking_number" in shipment
                            and shipment["tracking_number"]
                        ):
                            if (
                                expected_data.tracking_number
                                == shipment["tracking_number"]
                            ):
                                tracking_found = True
                                actual_tracking = shipment["tracking_number"]
                                break

                if not tracking_found:
                    available_tracking = (
                        [s.get("tracking_number", "N/A") for s in order_shipments]
                        if order_shipments
                        else []
                    )
                    assert_msgs.append(
                        f"Expected tracking number '{expected_data.tracking_number}' not found. "
                        f"Available tracking numbers: {available_tracking}"
                    )
                    failed_validation = True
                else:
                    assert_msgs.append(
                        f"Tracking number match: expected '{expected_data.tracking_number}', got '{actual_tracking}'"
                    )

            # Validate order addresses (shipping/billing)
            order_addresses = None
            address_fields = [
                expected_data.address,
                expected_data.address2,
                expected_data.city,
                expected_data.state,
                expected_data.zip_code,
                expected_data.country,
            ]
            if any(field for field in address_fields):
                order_addresses = data_helper.get_order_addresses(order_id)

                # Determine address type (default to shipping)
                address_type = eval_params.address_type

                if address_type not in order_addresses:
                    assert_msgs.append(
                        f"No {address_type} address found for order {order_id}. "
                        f"Available address types: {list(order_addresses.keys())}"
                    )
                    failed_validation = True
                else:
                    actual_address = order_addresses[address_type]

                    # Validate each address field individually
                    if expected_data.address:
                        actual_addr = actual_address.get("address", "")
                        if expected_data.address.lower() not in actual_addr.lower():
                            assert_msgs.append(
                                f"Address mismatch: expected '{expected_data.address}', got '{actual_addr}'"
                            )
                            failed_validation = True
                        else:
                            assert_msgs.append(
                                f"Address match: expected '{expected_data.address}', got '{actual_addr}'"
                            )

                    if expected_data.address2:
                        actual_addr2 = actual_address.get("address2", "")
                        if expected_data.address2.lower() not in actual_addr2.lower():
                            assert_msgs.append(
                                f"Address2 mismatch: expected '{expected_data.address2}', got '{actual_addr2}'"
                            )
                            failed_validation = True
                        else:
                            assert_msgs.append(
                                f"Address2 match: expected '{expected_data.address2}', got '{actual_addr2}'"
                            )

                    if expected_data.city:
                        actual_city = actual_address.get("city", "")
                        if expected_data.city.lower() != actual_city.lower():
                            assert_msgs.append(
                                f"City mismatch: expected '{expected_data.city}', got '{actual_city}'"
                            )
                            failed_validation = True
                        else:
                            assert_msgs.append(
                                f"City match: expected '{expected_data.city}', got '{actual_city}'"
                            )

                    if expected_data.state:
                        actual_state = actual_address.get("state", "")
                        if expected_data.state.lower() != actual_state.lower():
                            assert_msgs.append(
                                f"State mismatch: expected '{expected_data.state}', got '{actual_state}'"
                            )
                            failed_validation = True
                        else:
                            assert_msgs.append(
                                f"State match: expected '{expected_data.state}', got '{actual_state}'"
                            )

                    if expected_data.zip_code:
                        actual_zip = actual_address.get("zip_code", "")
                        if expected_data.zip_code != actual_zip:
                            assert_msgs.append(
                                f"Zip code mismatch: expected '{expected_data.zip_code}', got '{actual_zip}'"
                            )
                            failed_validation = True
                        else:
                            assert_msgs.append(
                                f"Zip code match: expected '{expected_data.zip_code}', got '{actual_zip}'"
                            )

                    if expected_data.country:
                        actual_country = actual_address.get("country", "")
                        if expected_data.country.lower() != actual_country.lower():
                            assert_msgs.append(
                                f"Country mismatch: expected '{expected_data.country}', got '{actual_country}'"
                            )
                            failed_validation = True
                        else:
                            assert_msgs.append(
                                f"Country match: expected '{expected_data.country}', got '{actual_country}'"
                            )

            # Validation passed
            validation_data = order.model_dump()
            # Include comments in validation data if they were fetched
            if order_comments is not None:
                validation_data["comments"] = order_comments
            # Include shipments in validation data if they were fetched
            if order_shipments is not None:
                validation_data["shipments"] = order_shipments
            # Include addresses in validation data if they were fetched
            if order_addresses is not None:
                validation_data["addresses"] = order_addresses

            # If there are validation errors, return failure
            if failed_validation:
                return WebarenaTaskEvalResult.create_failed(
                    assertion_msgs=[
                        f"Order {identifier} validation failed:",
                        *assert_msgs,
                    ],
                    site=resource.website_type,
                    validation_data=validation_data,
                    task_id=task.task_id,
                )

            return WebarenaTaskEvalResult.create_success(
                assertion_msgs=[
                    f"Order {identifier} details validation passed. ",
                    *assert_msgs,
                ],
                site=resource.website_type,
                validation_data=validation_data,
                task_id=task.task_id,
            )

        except ValueError as e:
            return WebarenaTaskEvalResult.create_failed(
                assertion_msgs=[f"Invalid order_id format: {str(e)}"],
                site=resource.website_type,
                task_id=task.task_id,
            )
        except Exception as e:
            return WebarenaTaskEvalResult.create_failed(
                assertion_msgs=[f"Order details validation failed: {str(e)}"],
                site=resource.website_type,
                task_id=task.task_id,
            )

    def verify_review_details(
        self,
        *,
        task: WebArenaTask,
        task_result: WebArenaTaskResponse,
        resource: AllocationResource,
        eval_func: EvalFunc,
        **kwargs,
    ) -> WebarenaTaskEvalResult:
        """
        Verify review details match expected values.

        Expected eval_func format:
        {
            "name": "verify_review_details",
            "eval_params": {"review_id": "352"},
            "expected_data": {"approved": true}
        }
        """
        try:
            # Extract and validate eval_params from eval_func
            if not eval_func or not hasattr(eval_func, "eval_params"):
                return WebarenaTaskEvalResult.create_failed(
                    assertion_msgs=["No eval_params found in eval_func"],
                    site=resource.website_type,
                    task_id=task.task_id,
                )

            try:
                eval_params = ReviewDetailsEvalParams.model_validate(
                    eval_func.eval_params
                )
            except Exception as e:
                return WebarenaTaskEvalResult.create_failed(
                    assertion_msgs=[f"Invalid eval_params format: {str(e)}"],
                    site=resource.website_type,
                    task_id=task.task_id,
                )

            # Extract and validate expected_data from eval_func
            if not hasattr(eval_func, "expected_data"):
                return WebarenaTaskEvalResult.create_failed(
                    assertion_msgs=["No expected_data found in eval_func"],
                    site=resource.website_type,
                    task_id=task.task_id,
                )

            try:
                expected_data = ReviewDetailsExpectedData.model_validate(
                    eval_func.expected_data
                )
            except Exception as e:
                return WebarenaTaskEvalResult.create_failed(
                    assertion_msgs=[f"Invalid expected_data format: {str(e)}"],
                    site=resource.website_type,
                    task_id=task.task_id,
                )

            # Create MagentoConfig from AllocationResource
            config = MagentoConfig(
                base_url=resource.base_url, container_name=resource.container_name
            )

            # Create data helper with resource-specific config
            data_helper = MagentoDataHelper(config)

            # Get review_id from eval_params
            review_id = int(eval_params.review_id)
            review = data_helper.get_review_by_id(review_id)
            identifier = f"ID {review_id}"

            # Validate individual fields if present in expected_data
            assert_msgs = []
            failed_validation = False
            if expected_data.exists is not None:
                if expected_data.exists != bool(review):
                    assert_msgs.append(
                        f"Review {identifier} exists mismatch: expected '{expected_data.exists}', got '{bool(review)}'"
                    )
                    failed_validation = True
                else:
                    assert_msgs.append(
                        f"Review {identifier} exists match: expected '{expected_data.exists}', got '{bool(review)}'"
                    )

            elif not review:
                return WebarenaTaskEvalResult.create_failed(
                    assertion_msgs=[f"Review not found with {identifier}"],
                    site=resource.website_type,
                    task_id=task.task_id,
                )

            if review and expected_data.approved is not None:
                # Check approved status
                expected_text = "approved" if expected_data.approved else "not approved"
                status_text = "approved" if review.approved else "not approved"
                if review.approved != expected_data.approved:
                    assert_msgs.append(
                        f"Approval status mismatch: expected '{expected_text}', got '{status_text}' (review_status: {review.review_status})"
                    )
                    failed_validation = True
                else:
                    assert_msgs.append(
                        f"Approval status match: expected '{expected_text}', got '{status_text}' (review_status: {review.review_status})"
                    )

            validation_data = {}
            if review:
                validation_data = review.model_dump()

            # If validation passed, return success
            if not failed_validation:
                return WebarenaTaskEvalResult.create_success(
                    assertion_msgs=[
                        f"Review {identifier} validation passed: ",
                        *assert_msgs,
                    ],
                    validation_data=validation_data,
                    site=resource.website_type,
                    task_id=task.task_id,
                )

            # Return validation errors
            return WebarenaTaskEvalResult.create_failed(
                assertion_msgs=assert_msgs,
                site=resource.website_type,
                task_id=task.task_id,
                validation_data=validation_data,
            )

        except Exception as e:
            self.logger.error(f"Error validating review details: {e}")
            return WebarenaTaskEvalResult.create_failed(
                assertion_msgs=[f"Error validating review details: {str(e)}"],
                site=resource.website_type,
                task_id=task.task_id,
            )

    def verify_sale_details(
        self,
        *,
        task: WebArenaTask,
        task_result: WebArenaTaskResponse,
        resource: AllocationResource,
        eval_func: EvalFunc,
        **kwargs,
    ) -> WebarenaTaskEvalResult:
        """
        Verify sales rule details match expected values.

        Expected eval_func format:
        {
            "name": "verify_sale_details",
            "eval_params": {"topic": "spring sale"},
            "expected_data": {
                "name": "spring sale",
                "website_ids": 0,
                "customer_group_ids": 1,
                "discount_amount_type": "by_percent",
                "discount_amount": 20
            }
        }
        """
        try:
            # Extract and validate eval_params from eval_func
            if not eval_func or not hasattr(eval_func, "eval_params"):
                return WebarenaTaskEvalResult.create_failed(
                    assertion_msgs=["No eval_params found in eval_func"],
                    site=resource.website_type,
                )

            try:
                eval_params = SaleDetailsEvalParams.model_validate(
                    eval_func.eval_params
                )
            except Exception as e:
                return WebarenaTaskEvalResult.create_failed(
                    assertion_msgs=[f"Invalid eval_params format: {str(e)}"],
                    site=resource.website_type,
                    task_id=task.task_id,
                )

            # Extract and validate expected_data from eval_func
            if not hasattr(eval_func, "expected_data"):
                return WebarenaTaskEvalResult.create_failed(
                    assertion_msgs=["No expected_data found in eval_func"],
                    site=resource.website_type,
                    task_id=task.task_id,
                )

            try:
                expected_data = SaleDetailsExpectedData.model_validate(
                    eval_func.expected_data
                )
            except Exception as e:
                return WebarenaTaskEvalResult.create_failed(
                    assertion_msgs=[f"Invalid expected_data format: {str(e)}"],
                    site=resource.website_type,
                    task_id=task.task_id,
                )

            # Create MagentoConfig from AllocationResource
            config = MagentoConfig(
                base_url=resource.base_url, container_name=resource.container_name
            )

            # Create data helper with resource-specific config
            data_helper = MagentoDataHelper(config)

            # Get sales rule by topic
            sale_rule = data_helper.get_sale_by_topic(eval_params.topic)

            if not sale_rule:
                return WebarenaTaskEvalResult.create_failed(
                    assertion_msgs=[
                        f"Sales rule with topic '{eval_params.topic}' not found"
                    ],
                    site=resource.website_type,
                    task_id=task.task_id,
                )

            # Validate expected fields
            assert_msgs = []
            validation_failed = False

            # Check name
            if expected_data.name:
                if sale_rule.name != expected_data.name:
                    assert_msgs.append(
                        f"Name mismatch: expected '{expected_data.name}', got '{sale_rule.name}'"
                    )
                    validation_failed = True
                else:
                    assert_msgs.append(
                        f"Name match: expected '{expected_data.name}', got '{sale_rule.name}'"
                    )

            # Check website_ids
            if expected_data.website_ids is not None:
                # Handle list comparison for website_ids
                if isinstance(expected_data.website_ids, list) and isinstance(
                    sale_rule.website_ids, list
                ):
                    if set(sale_rule.website_ids) != set(expected_data.website_ids):
                        assert_msgs.append(
                            f"Website IDs mismatch: expected {expected_data.website_ids}, got {sale_rule.website_ids}"
                        )
                        validation_failed = True
                    else:
                        assert_msgs.append(
                            f"Website IDs match: expected {expected_data.website_ids}, got {sale_rule.website_ids}"
                        )
                else:
                    # Fallback to string comparison for backward compatibility
                    if str(sale_rule.website_ids) != str(expected_data.website_ids):
                        assert_msgs.append(
                            f"Website IDs mismatch: expected '{expected_data.website_ids}', got '{sale_rule.website_ids}'"
                        )
                        validation_failed = True
                    else:
                        assert_msgs.append(
                            f"Website IDs match: expected '{expected_data.website_ids}', got '{sale_rule.website_ids}'"
                        )

            # Check customer_group_ids
            if expected_data.customer_group_ids is not None:
                # Handle list comparison for customer_group_ids
                if isinstance(expected_data.customer_group_ids, list) and isinstance(
                    sale_rule.customer_group_ids, list
                ):
                    if set(sale_rule.customer_group_ids) != set(
                        expected_data.customer_group_ids
                    ):
                        assert_msgs.append(
                            f"Customer group IDs mismatch: expected {expected_data.customer_group_ids}, got {sale_rule.customer_group_ids}"
                        )
                        validation_failed = True
                    else:
                        assert_msgs.append(
                            f"Customer group IDs match: expected {expected_data.customer_group_ids}, got {sale_rule.customer_group_ids}"
                        )
                else:
                    # Fallback to string comparison for backward compatibility
                    if str(sale_rule.customer_group_ids) != str(
                        expected_data.customer_group_ids
                    ):
                        assert_msgs.append(
                            f"Customer group IDs mismatch: expected '{expected_data.customer_group_ids}', got '{sale_rule.customer_group_ids}'"
                        )
                        validation_failed = True
                    else:
                        assert_msgs.append(
                            f"Customer group IDs match: expected '{expected_data.customer_group_ids}', got '{sale_rule.customer_group_ids}'"
                        )

            # Check discount_amount_type
            if expected_data.discount_amount_type:
                if sale_rule.discount_amount_type != expected_data.discount_amount_type:
                    assert_msgs.append(
                        f"Discount amount type mismatch: expected '{expected_data.discount_amount_type}', got '{sale_rule.discount_amount_type}'"
                    )
                    validation_failed = True
                else:
                    assert_msgs.append(
                        f"Discount amount type match: expected '{expected_data.discount_amount_type}', got '{sale_rule.discount_amount_type}'"
                    )

            # Check discount_amount
            if expected_data.discount_amount is not None:
                expected_discount_amount = float(expected_data.discount_amount)
                if abs(sale_rule.discount_amount - expected_discount_amount) > 0.01:
                    assert_msgs.append(
                        f"Discount amount mismatch: expected {expected_discount_amount}, got {sale_rule.discount_amount}"
                    )
                    validation_failed = True
                else:
                    assert_msgs.append(
                        f"Discount amount match: expected {expected_discount_amount}, got {sale_rule.discount_amount}"
                    )

            # Return success with validation data
            validation_data = {
                "topic": eval_params.topic,
                "sale_data": sale_rule.model_dump(),
                "validated_fields": {
                    field: getattr(expected_data, field)
                    for field in [
                        "name",
                        "website_ids",
                        "customer_group_ids",
                        "discount_amount_type",
                        "discount_amount",
                    ]
                    if getattr(expected_data, field) is not None
                },
            }

            # If validation errors exist, return failure
            if validation_failed:
                return WebarenaTaskEvalResult.create_failed(
                    assertion_msgs=assert_msgs,
                    site=resource.website_type,
                    task_id=task.task_id,
                    validation_data=validation_data,
                )

            return WebarenaTaskEvalResult.create_success(
                validation_data=validation_data,
                site=resource.website_type,
                task_id=task.task_id,
                assertion_msgs=assert_msgs,
            )

        except Exception as e:
            self.logger.error(f"Error validating sale details: {e}")
            return WebarenaTaskEvalResult.create_failed(
                assertion_msgs=[f"Error validating sale details: {str(e)}"],
                site=resource.website_type,
                task_id=task.task_id,
            )

    def verify_site_page_details(
        self,
        task: WebArenaTask,
        task_result: WebArenaTaskResponse,
        resource: AllocationResource,
        eval_func: EvalFunc,
        **kwargs,
    ) -> WebarenaTaskEvalResult:
        """
        Verify CMS page details using Magento API.

        Expected eval_func format:
        {
            "name": "verify_site_page_details",
            "eval_params": {"page_id": "1"},
            "expected_data": {"title": "Bruh bro you clicked the wrong page"}
        }
        """
        try:
            # Extract and validate eval_params from eval_func
            if not eval_func or not hasattr(eval_func, "eval_params"):
                return WebarenaTaskEvalResult.create_failed(
                    assertion_msgs=["No eval_params found in eval_func"],
                    site=resource.website_type,
                    task_id=task.task_id,
                )

            try:
                eval_params = SitePageDetailsEvalParams.model_validate(
                    eval_func.eval_params
                )
            except Exception as e:
                return WebarenaTaskEvalResult.create_failed(
                    assertion_msgs=[f"Invalid eval_params format: {str(e)}"],
                    site=resource.website_type,
                    task_id=task.task_id,
                )

            # Extract and validate expected_data from eval_func
            if not hasattr(eval_func, "expected_data"):
                return WebarenaTaskEvalResult.create_failed(
                    assertion_msgs=["No expected_data found in eval_func"],
                    site=resource.website_type,
                    task_id=task.task_id,
                )

            try:
                expected_data = SitePageDetailsExpectedData.model_validate(
                    eval_func.expected_data
                )
            except Exception as e:
                return WebarenaTaskEvalResult.create_failed(
                    assertion_msgs=[f"Invalid expected_data format: {str(e)}"],
                    site=resource.website_type,
                    task_id=task.task_id,
                )

            # Convert page_id to int
            try:
                page_id = int(eval_params.page_id)
            except ValueError:
                return WebarenaTaskEvalResult.create_failed(
                    assertion_msgs=[f"Invalid page_id format: {eval_params.page_id}"],
                    site=resource.website_type,
                    task_id=task.task_id,
                )

            # Create config and data helper
            config = MagentoConfig(
                base_url=resource.base_url, container_name=resource.container_name
            )
            data_helper = MagentoDataHelper(config)

            # Get CMS page by ID
            cms_page = data_helper.get_cms_page_by_id(page_id)

            if not cms_page:
                return WebarenaTaskEvalResult.create_failed(
                    assertion_msgs=[f"CMS page with ID '{page_id}' not found"],
                    site=resource.website_type,
                    task_id=task.task_id,
                )

            # Validate expected fields
            assert_msgs = []
            validation_failed = False

            # Check title
            if expected_data.title:
                if cms_page.title != expected_data.title:
                    assert_msgs.append(
                        f"Title mismatch: expected '{expected_data.title}', got '{cms_page.title}'"
                    )
                    validation_failed = True
                else:
                    assert_msgs.append(
                        f"Title match: expected '{expected_data.title}', got '{cms_page.title}'"
                    )

            # Check identifier
            if expected_data.identifier:
                if cms_page.identifier != expected_data.identifier:
                    assert_msgs.append(
                        f"Identifier mismatch: expected '{expected_data.identifier}', got '{cms_page.identifier}'"
                    )
                    validation_failed = True
                else:
                    assert_msgs.append(
                        f"Identifier match: expected '{expected_data.identifier}', got '{cms_page.identifier}'"
                    )

            # Check is_active status
            if expected_data.is_active is not None:
                if cms_page.is_active != expected_data.is_active:
                    assert_msgs.append(
                        f"Active status mismatch: expected {expected_data.is_active}, got {cms_page.is_active}"
                    )
                    validation_failed = True
                else:
                    assert_msgs.append(
                        f"Active status match: expected {expected_data.is_active}, got {cms_page.is_active}"
                    )

            # Check content (partial match for long content)
            if expected_data.content:
                actual_content = cms_page.content or ""
                if expected_data.content not in actual_content:
                    assert_msgs.append(
                        f"Content does not contain expected text: '{expected_data.content}'"
                    )
                    validation_failed = True
                else:
                    assert_msgs.append(
                        f"Content match: expected '{expected_data.content}', got '{actual_content}'"
                    )

            # Return success with validation data
            validation_data = {
                "page_id": page_id,
                "page_data": cms_page.model_dump(),
                "validated_fields": {
                    field: getattr(expected_data, field)
                    for field in ["title", "identifier", "is_active", "content"]
                    if getattr(expected_data, field) is not None
                },
            }

            # If validation errors exist, return failure
            if validation_failed:
                return WebarenaTaskEvalResult.create_failed(
                    assertion_msgs=assert_msgs,
                    site=resource.website_type,
                    validation_data=validation_data,
                    task_id=task.task_id,
                )

            return WebarenaTaskEvalResult.create_success(
                assertion_msgs=[
                    f"CMS page validation successful for page ID {page_id}",
                    *assert_msgs,
                ],
                site=resource.website_type,
                validation_data=validation_data,
                task_id=task.task_id,
            )

        except Exception as e:
            self.logger.error(f"Error in verify_site_page_details: {str(e)}")
            return WebarenaTaskEvalResult.create_failed(
                assertion_msgs=[f"Error validating CMS page: {str(e)}"],
                site=resource.website_type,
                task_id=task.task_id,
            )

    async def verify_report_generation_form(
        self,
        *,
        task: WebArenaTask,
        task_result: WebArenaTaskResponse,
        resource: AllocationResource,
        eval_func: EvalFunc,
        **kwargs,
    ) -> WebarenaTaskEvalResult:
        """UI based verification of sales report generation form"""

        if not eval_func or eval_func.eval_params is None:
            return WebarenaTaskEvalResult.create_failed(
                assertion_msgs=["No eval_params found in eval_func"],
                site=resource.website_type,
                task_id=task.task_id,
            )

        try:
            eval_params = ReportGenerationFormEvalParams.model_validate(
                eval_func.eval_params
            )
        except Exception as e:
            return WebarenaTaskEvalResult.create_failed(
                assertion_msgs=[f"Invalid eval_params format: {str(e)}"],
                site=resource.website_type,
                task_id=task.task_id,
            )

        assert_msgs = []
        validation_failed = False
        url = self._expand_url(resource=resource, url=eval_params.url)
        if url is not None:
            found = False
            for last_url in task_result.last_urls:
                if last_url.startswith(url):
                    found = True
                    break
            if not found:
                assert_msgs.append(
                    f"URL not found in last_urls. URL: '{url}' last_urls: {json.dumps(task_result.last_urls)}"
                )
                validation_failed = True
            else:
                assert_msgs.append(
                    f"URL found in last_urls. URL: '{url}' last_urls: {json.dumps(task_result.last_urls)}"
                )

        try:
            form_fields = await get_form_fields(
                cdp_url=resource.cdp_url,
                locator="form input, form select, form textarea",
            )
        except Exception as e:
            return WebarenaTaskEvalResult.create_failed(
                assertion_msgs=[f"Error accessing form fields: {str(e)}"],
                site=resource.website_type,
                task_id=task.task_id,
                validation_data={
                    "form_fields": None,
                    "urls": task_result.last_urls,
                },
            )
        date_evaluator = DateMatchEvaluator()
        for field_name, value in eval_params.model_dump().items():
            if field_name == "url":
                continue

            if value is not None:
                if field_name == "from_date":
                    field_name = "from"
                if field_name == "to_date":
                    field_name = "to"

                if field_name not in form_fields:
                    assert_msgs.append(f"Field '{field_name}' not found in form fields")
                    validation_failed = True

                if field_name in form_fields:
                    if field_name == "from" or field_name == "to":
                        expected_date = date_evaluator._normalize_date(value)
                        actual_date = date_evaluator._normalize_date(
                            form_fields[field_name]
                        )

                        if expected_date != actual_date:
                            assert_msgs.append(
                                f"Field '{field_name}' value mismatch: expected '{value}', got '{form_fields[field_name]}'"
                            )
                            validation_failed = True
                        else:
                            assert_msgs.append(
                                f"Field '{field_name}' value match: expected '{value}', got '{form_fields[field_name]}'"
                            )
                    elif form_fields[field_name] != value:
                        assert_msgs.append(
                            f"Field '{field_name}' value mismatch: expected '{value}', got '{form_fields[field_name]}'"
                        )
                        validation_failed = True
                    else:
                        assert_msgs.append(
                            f"Field '{field_name}' value match: expected '{value}', got '{form_fields[field_name]}'"
                        )

        validation_data = {
            "form_fields": form_fields,
            "urls": task_result.last_urls,
        }
        if validation_failed:
            return WebarenaTaskEvalResult.create_failed(
                assertion_msgs=assert_msgs,
                site=resource.website_type,
                task_id=task.task_id,
                validation_data=validation_data,
            )
        else:
            return WebarenaTaskEvalResult.create_success(
                assertion_msgs=assert_msgs,
                site=resource.website_type,
                task_id=task.task_id,
                validation_data=validation_data,
            )

    def _normalize_currency(self, value) -> float:
        """
        Normalize a value to a currency amount for comparison.
        Handles various formats like "$3", "3", "$3.00", "3.00", "€3", etc.
        Note: This is format normalization only, no currency exchange.

        Args:
            value: The value to normalize (can be str, int, float with currency symbols)

        Returns:
            float: The normalized currency amount

        Raises:
            ValueError: If the value cannot be converted to a valid currency amount
        """
        try:
            # Handle None
            if value is None:
                raise ValueError("Cannot convert None to currency")

            # Convert to string and normalize
            str_value = str(value).strip()

            # Handle empty strings
            if not str_value:
                raise ValueError("Cannot convert empty string to currency")

            import re

            # Remove common currency symbols, spaces, and commas
            # Supports: $, €, £, ¥, ₹, ₽, ¢, etc.
            cleaned = re.sub(r"[\$€£¥₹₽¢\s,]", "", str_value)

            # Handle empty result after cleaning
            if not cleaned:
                raise ValueError(
                    "No numeric value found after removing currency symbols"
                )

            return float(cleaned)

        except (ValueError, TypeError) as e:
            raise ValueError(f"Cannot convert '{value}' to currency: {str(e)}")

    def _expand_url(self, resource: AllocationResource, url: str) -> str:
        return url.replace(f"__{resource.website_type.upper()}__", resource.base_url)
