import copy
import uuid
from datetime import UTC, datetime
from enum import StrEnum
from typing import Any, Literal, Optional, Self
from urllib.parse import urlparse

from pydantic import BaseModel, Field


class SessionRequest(BaseModel, extra="forbid", frozen=True):
    session_id: str = Field(default_factory=lambda: uuid.uuid4().hex)
    metadata: dict[str, Any] = Field(default_factory=dict)


class SessionState(BaseModel, extra="forbid", frozen=True):
    session_id: str
    created_at: datetime
    task_count: int = 0
    metadata: dict[str, Any] = Field(default_factory=dict)

    @classmethod
    def create(cls, *, session_id: str, metadata: dict[str, Any] | None = None) -> Self:
        return cls(
            session_id=session_id,
            created_at=datetime.now(UTC),
            metadata=metadata or {},
        )

    def increment_task_count(self) -> Self:
        return self.model_copy(update={"task_count": self.task_count + 1})


class TaskRequest(BaseModel, extra="forbid", frozen=True):
    request_id: str = Field(default_factory=lambda: uuid.uuid4().hex)
    session_id: str | None = None
    trace_id: str | None = None
    data: dict[str, Any]


class TaskResultRequest(BaseModel, extra="forbid", frozen=True):
    result: dict[str, Any]
    llm_calls_log: list[dict]


class TaskStatus(StrEnum):
    PENDING = "pending"
    RUNNING = "running"
    DONE = "done"


class TaskState(BaseModel, extra="forbid", frozen=True):
    request_id: str
    session_id: str | None = None
    status: TaskStatus
    result: dict[str, Any] | None = None
    llm_calls_log: list[dict] | None = None
    started_at: datetime | None = None
    completed_at: datetime | None = None

    @classmethod
    def pending(cls, *, request_id: str, session_id: str | None = None) -> Self:
        return cls(
            request_id=request_id, session_id=session_id, status=TaskStatus.PENDING
        )

    def start(self) -> Self:
        return self.model_copy(
            update={
                "status": TaskStatus.RUNNING,
                "started_at": datetime.now(UTC),
            }
        )

    def complete(self, *, result: dict[str, Any], llm_calls_log: list[dict]) -> Self:
        return self.model_copy(
            update={
                "status": TaskStatus.DONE,
                "completed_at": datetime.now(UTC),
                "result": copy.deepcopy(result),
                "llm_calls_log": copy.deepcopy(llm_calls_log),
            }
        )


class BenchmarkTask(BaseModel, frozen=True, extra="forbid"):
    input: Any
    expected_output: Any
    metadata: dict[str, Any] = Field(default_factory=dict)


# Public API request models
class AllocationRequest(BaseModel, extra="forbid", frozen=True):
    """Request to allocate a website resource.

    Fields:
        website_type: The type/category of website requested (e.g., "gitlab").
        readonly: If true, allocate a readonly user; otherwise exclusive/admin.
    """

    website_type: str
    readonly: bool = False


# Shared requirement model used by pytest plugin hooks and benchmark conftests
class WebsiteRequirement(BaseModel, extra="forbid", frozen=True):
    website_type: str
    readonly: bool = False


# Public API response models (shared by dispatcher and clients)
class AllocationResource(BaseModel, extra="forbid", frozen=True):
    allocation_id: str
    site_id: str
    container_name: str | None = None
    website_type: str
    base_url: str
    cdp_url: str
    vnc_url: str
    readonly: bool
    username: str
    password: str
    role: Literal["admin", "readonly"]

    @property
    def base_host(self) -> str:
        return urlparse(self.base_url).hostname

    @property
    def base_port(self) -> int:
        return urlparse(self.base_url).port

    @property
    def root_url(self) -> str:
        parsed_url = urlparse(self.base_url)
        return f"{parsed_url.scheme}://{parsed_url.netloc}"


class ReleaseResponse(BaseModel, extra="forbid", frozen=True):
    allocation_id: str
    site_id: str


# Internal admin API request models (shared with control client)
class HealthCheckSpec(BaseModel, extra="forbid", frozen=True):
    url: str
    expected_status: int = 200
    expected_substring: str | None = None
    timeout: float = 5.0


class WebsiteUserRegistration(BaseModel, extra="forbid", frozen=True):
    username: str
    password: str
    email: Optional[str] = ""
    role: Literal["admin", "readonly"] = "readonly"


class WebsiteRegistrationRequest(BaseModel, extra="forbid", frozen=True):
    website_type: str
    base_url: str
    cdp_url: str
    vnc_url: str
    container_name: str | None = None
    users: list[WebsiteUserRegistration]
    health_check: HealthCheckSpec
