from playwright.async_api import (  # noqa: F401
    Page,
    async_playwright,
)


async def get_element_value(
    cdp_url: str, locator: str
) -> tuple[str | None, str | None]:
    async with async_playwright() as playwright:
        browser = await playwright.chromium.connect_over_cdp(cdp_url)
        context = browser.contexts[0]
        page = context.pages[0]
        return await get_value_from_page(page, locator)


async def get_element_values(cdp_url: str, locator: str) -> tuple[list[str], list[str]]:
    async with async_playwright() as playwright:
        browser = await playwright.chromium.connect_over_cdp(cdp_url)
        if len(browser.contexts) == 0:
            raise ValueError("No browser context available")
        context = browser.contexts[0]
        if len(context.pages) == 0:
            raise ValueError("No browser page available")
        page = context.pages[0]
        return await get_values_from_page(page, locator)


async def get_values_from_page(page: Page, locator: str) -> tuple[list[str], list[str]]:
    elements = await page.query_selector_all(locator)
    return [await element.text_content() for element in elements], [
        await element.evaluate("el => el.outerHTML") for element in elements
    ]


async def get_value_from_page(
    page: Page, locator: str
) -> tuple[str | None, str | None]:
    element = await page.query_selector(locator)
    if element:
        text_content = await element.text_content()
        html = await element.evaluate("el => el.outerHTML")
        return (text_content, html)
    return (None, None)


async def get_form_fields(cdp_url: str, locator: str | None = None) -> dict[str, any]:
    async with async_playwright() as playwright:
        browser = await playwright.chromium.connect_over_cdp(cdp_url)
        context = browser.contexts[0]
        page = context.pages[0]
        return await pull_form_fields(page, locator)


async def pull_form_fields(
    page: Page, locator: str = "form input, form select, form textarea"
) -> dict[str, any]:
    form_fields = {}
    form_elements = await page.query_selector_all(locator)
    for element in form_elements:
        name = await element.get_attribute("name")
        if name:
            is_array = False
            if name.endswith("[]"):
                name = name[:-2]
                is_array = True
                form_fields[name] = []

            tag_name = await element.evaluate("el => el.tagName.toLowerCase()")

            if tag_name == "select":
                # Handle select elements - get the selected option value
                selected_value = await element.evaluate("""
                    select => {
                        const selectedOption = select.options[select.selectedIndex];
                        return selectedOption ? selectedOption.value : '';
                    }
                """)
                if is_array and selected_value:
                    form_fields[name].append(selected_value)
                elif selected_value:
                    form_fields[name] = selected_value
            elif await element.get_attribute("type") == "checkbox":
                form_fields[name] = await element.is_checked()
            elif await element.get_attribute("type") == "radio":
                # For radio inputs, capture the value of the checked option (grouped by name)
                if await element.is_checked():
                    radio_value = await element.get_attribute("value")
                    if is_array:
                        if radio_value:
                            form_fields[name].append(radio_value)
                    else:
                        if radio_value:
                            form_fields[name] = radio_value
                        else:
                            # Fallback: some radios may not have explicit value; try input_value()
                            input_value = await element.input_value()
                            if input_value:
                                form_fields[name] = input_value
            else:
                value = await element.get_attribute("value")
                input_value = await element.input_value()

                if is_array and value:
                    form_fields[name].append(value)
                elif is_array and input_value:
                    form_fields[name].append(input_value)
                elif value:
                    form_fields[name] = value
                elif input_value:
                    form_fields[name] = input_value
    return form_fields
