"""
Reddit Data Helper for retrieving data as Pydantic v2 models.

This module provides Reddit data retrieval capabilities using PostgreSQL database
via Docker exec, returning structured Pydantic models for evaluation.
"""

import json
import logging
import subprocess
from typing import Any, Dict, List, Optional

from .models import (
    CommentInfo,
    PostInfo,
    RedditConfig,
    SubredditInfo,
    SubscriptionInfo,
    UserInfo,
)


class RedditDataHelper:
    """
    Helper class for retrieving Reddit data as Pydantic models.

    This class handles PostgreSQL database queries via Docker exec, returning
    structured Pydantic v2 models instead of raw data.
    """

    def __init__(self, config: Optional[RedditConfig] = None):
        self.config = config or RedditConfig()
        self.logger = logging.getLogger(__name__)
        self.container_name = self.config.container_name
        self.database = self.config.database
        self.username = self.config.username

    def _execute_query_json(self, query: str) -> List[Dict[str, Any]]:
        """Execute a SQL query and return results as JSON."""
        try:
            # Wrap the query to return JSON
            json_query = f"""
            SELECT array_to_json(array_agg(row_to_json(t))) as result
            FROM ({query}) t;
            """

            psql_cmd = [
                "docker",
                "exec",
                "-i",
                self.container_name,
                "psql",
                "-U",
                self.username,
                "-d",
                self.database,
                "-t",  # tuples only
                "-A",  # unaligned output
                "--quiet",
                "-c",
                json_query,
            ]

            self.logger.debug(f"Executing query: {query}")

            result = subprocess.run(
                psql_cmd, capture_output=True, text=True, timeout=30
            )

            if result.returncode != 0:
                error_msg = (
                    f"JSON query failed with code {result.returncode}: {result.stderr}"
                )
                self.logger.error(error_msg)
                raise Exception(error_msg)

            output = result.stdout.strip()
            if not output or output == "null":
                return []

            # Parse JSON result
            try:
                json_result = json.loads(output)
                return json_result if json_result else []
            except json.JSONDecodeError as e:
                self.logger.error(f"Failed to parse JSON result: {e}")
                self.logger.debug(f"Raw output: {output}")
                return []

        except Exception as e:
            error_msg = f"JSON query execution failed: {e}"
            self.logger.error(error_msg)
            raise Exception(error_msg)

    def _execute_query(self, query: str) -> None:
        """Execute a SQL query without returning results (for INSERT, UPDATE, DELETE)."""
        try:
            psql_cmd = [
                "docker",
                "exec",
                "-i",
                self.container_name,
                "psql",
                "-U",
                self.username,
                "-d",
                self.database,
                "--quiet",
                "-c",
                query,
            ]

            self.logger.debug(f"Executing query: {query}")

            result = subprocess.run(
                psql_cmd, capture_output=True, text=True, timeout=30
            )

            if result.returncode != 0:
                error_msg = (
                    f"Query failed with code {result.returncode}: {result.stderr}"
                )
                self.logger.error(error_msg)
                raise Exception(error_msg)

        except Exception as e:
            error_msg = f"Query execution failed: {e}"
            self.logger.error(error_msg)
            raise Exception(error_msg)

    def get_user_by_username(self, username: str) -> Optional[UserInfo]:
        """Get user information by username."""
        try:
            # Handle PostgreSQL parameter substitution manually for now
            safe_username = username.replace(
                "'", "''"
            )  # Basic SQL injection protection
            query = f"""
                SELECT username, email, biography, created
                FROM users 
                WHERE username = '{safe_username}' OR username ILIKE '{safe_username}'
                LIMIT 1
            """

            results = self._execute_query_json(query)

            if not results:
                self.logger.warning(f"User {username} not found")
                return None

            user_data = results[0]
            self.logger.debug(f"Retrieved user info for {username}")
            self.logger.debug(f"Raw user data: {user_data}")

            user_info = UserInfo.model_validate(
                {
                    "username": user_data["username"],
                    "email": user_data.get("email"),
                    "bio": user_data.get("biography", ""),
                    "created_utc": user_data.get("created"),
                }
            )
            self.logger.debug(f"Mapped user info: {user_info}")
            return user_info

        except Exception as e:
            self.logger.error(f"Failed to get user info for {username}: {e}")
            return None

    def get_user_profile_bio(self, username: str) -> Optional[str]:
        """Get user's profile bio."""
        user = self.get_user_by_username(username)
        if user:
            self.logger.debug(f"User {username} bio: '{user.bio}'")
            return user.bio
        else:
            self.logger.warning(f"User {username} not found")
            return None

    def user_has_bio_content(self, username: str, content: str) -> bool:
        """Check if user's bio contains specific content."""
        bio = self.get_user_profile_bio(username)
        if not bio:
            return False
        return content.lower() in bio.lower()

    def get_subreddit_by_name(self, subreddit: str) -> Optional[SubredditInfo]:
        """Get subreddit/forum information."""
        try:
            safe_subreddit = subreddit.replace("'", "''")
            # Get forum info with subscriber count and sidebar
            query = f"""
                SELECT f.name, f.title, f.description, f.sidebar, f.created,
                       COUNT(fs.user_id) as subscriber_count
                FROM forums f
                LEFT JOIN forum_subscriptions fs ON f.id = fs.forum_id
                WHERE f.name = '{safe_subreddit}' OR f.name ILIKE '{safe_subreddit}'
                GROUP BY f.id, f.name, f.title, f.description, f.sidebar, f.created
                LIMIT 1
            """

            results = self._execute_query_json(query)

            if not results:
                self.logger.warning(f"Forum {subreddit} not found")
                return None

            forum_data = results[0]
            self.logger.debug(f"Retrieved forum info for {subreddit}")

            return SubredditInfo.model_validate(
                {
                    "display_name": forum_data["name"],
                    "title": forum_data["title"],
                    "description": forum_data.get("description", ""),
                    "sidebar": forum_data.get("sidebar", ""),
                    "subscribers": forum_data.get("subscriber_count", 0),
                    "created_utc": forum_data.get("created"),
                }
            )

        except Exception as e:
            self.logger.error(f"Failed to get forum info for {subreddit}: {e}")
            return None

    def subreddit_exists(self, subreddit: str) -> bool:
        """Check if a subreddit exists."""
        sub = self.get_subreddit_by_name(subreddit)
        return sub is not None

    def get_subreddit_posts(
        self, subreddit: str, limit: int = 25, sort_by: str = "new"
    ) -> List[PostInfo]:
        """Get posts from a subreddit/forum with different sorting options.

        Args:
            subreddit: Subreddit/forum name
            limit: Maximum number of posts to return
            sort_by: Sorting method - "new" (timestamp DESC), "top" (net_score DESC), "hot" (combined score)
        """
        try:
            safe_subreddit = subreddit.replace("'", "''")

            # Determine sort order based on sort_by parameter
            if sort_by == "top":
                order_clause = "ORDER BY s.net_score DESC, s.timestamp DESC"
            elif sort_by == "hot":
                # Hot could be a combination of score and recency (simplified here)
                order_clause = "ORDER BY (s.net_score * 1.0 / (EXTRACT(EPOCH FROM NOW() - s.timestamp) / 3600 + 1)) DESC"
            else:  # default to "new"
                order_clause = "ORDER BY s.timestamp DESC"

            query = f"""
                SELECT s.id, s.title, s.body, s.timestamp, s.net_score, s.comment_count,
                       u.username as author, f.name as forum_name
                FROM submissions s
                JOIN users u ON s.user_id = u.id
                JOIN forums f ON s.forum_id = f.id
                WHERE f.name = '{safe_subreddit}' OR f.name ILIKE '{safe_subreddit}'
                {order_clause}
                LIMIT {limit}
            """

            results = self._execute_query_json(query)
            self.logger.debug(
                f"Retrieved {len(results)} posts from {subreddit} (sorted by {sort_by})"
            )

            posts = []
            for post_data in results:
                posts.append(
                    PostInfo.model_validate(
                        {
                            "id": str(post_data["id"]),
                            "title": post_data["title"],
                            "selftext": post_data.get("body", ""),
                            "author": post_data["author"],
                            "subreddit": post_data["forum_name"],
                            "created_utc": post_data.get("timestamp"),
                            "score": post_data.get("net_score", 0),
                            "num_comments": post_data.get("comment_count", 0),
                            "url": f"/f/{post_data['forum_name']}/comments/{post_data['id']}/",
                        }
                    )
                )

            return posts

        except Exception as e:
            self.logger.error(f"Failed to get posts from {subreddit}: {e}")
            return []

    def get_post_by_id(self, post_id: str, subreddit: str = None) -> Optional[PostInfo]:
        """Get post information by ID."""
        try:
            base_query = f"""
                SELECT s.id, s.title, s.body, s.timestamp, s.net_score, s.comment_count,
                       u.username as author, f.name as forum_name
                FROM submissions s
                JOIN users u ON s.user_id = u.id
                JOIN forums f ON s.forum_id = f.id
                WHERE s.id = {post_id}
            """

            if subreddit:
                safe_subreddit = subreddit.replace("'", "''")
                base_query += f" AND (f.name = '{safe_subreddit}' OR f.name ILIKE '{safe_subreddit}')"

            base_query += " LIMIT 1"

            results = self._execute_query_json(base_query)

            if not results:
                self.logger.warning(f"Post {post_id} not found")
                return None

            post_data = results[0]
            self.logger.debug(f"Retrieved post {post_id}")

            return PostInfo.model_validate(
                {
                    "id": str(post_data["id"]),
                    "title": post_data["title"],
                    "selftext": post_data.get("body", ""),
                    "author": post_data["author"],
                    "subreddit": post_data["forum_name"],
                    "created_utc": post_data.get("timestamp"),
                    "score": post_data.get("net_score", 0),
                    "num_comments": post_data.get("comment_count", 0),
                    "url": f"/f/{post_data['forum_name']}/comments/{post_data['id']}/",
                }
            )

        except Exception as e:
            self.logger.error(f"Failed to get post {post_id}: {e}")
            return None

    def check_post_exists(self, post_id: str, subreddit: str = None) -> bool:
        """Check if a post exists."""
        post = self.get_post_by_id(post_id, subreddit)
        return post is not None

    def post_exists_in_subreddit(
        self,
        subreddit: str,
        post_id: str = None,
        author: str = None,
        title_contains: str | list[str] = None,
        content_contains: str | list[str] = None,
    ) -> bool:
        """Check if a post exists in subreddit with specific criteria."""
        try:
            safe_subreddit = subreddit.replace("'", "''")

            query = f"""
                SELECT COUNT(*) as count
                FROM submissions s
                JOIN users u ON s.user_id = u.id
                JOIN forums f ON s.forum_id = f.id
                WHERE (f.name = '{safe_subreddit}' OR f.name ILIKE '{safe_subreddit}')
            """

            conditions = []
            if post_id:
                conditions.append(f"s.id = {post_id}")
            if author:
                safe_author = author.replace("'", "''")
                conditions.append(f"u.username = '{safe_author}'")

            # Handle title_contains - string or list of strings
            if title_contains:
                if isinstance(title_contains, str):
                    safe_title = title_contains.replace("'", "''")
                    conditions.append(f"s.title ILIKE '%{safe_title}%'")
                elif isinstance(title_contains, list):
                    # ALL elements in the list must be present in the title
                    title_conditions = []
                    for title in title_contains:
                        safe_title = title.replace("'", "''")
                        title_conditions.append(f"s.title ILIKE '%{safe_title}%'")
                    if title_conditions:
                        conditions.append(f"({' AND '.join(title_conditions)})")

            # Handle content_contains - string or list of strings
            if content_contains:
                if isinstance(content_contains, str):
                    safe_content = content_contains.replace("'", "''")
                    conditions.append(f"s.body ILIKE '%{safe_content}%'")
                elif isinstance(content_contains, list):
                    # ALL elements in the list must be present in the content
                    content_conditions = []
                    for content in content_contains:
                        safe_content = content.replace("'", "''")
                        content_conditions.append(f"s.body ILIKE '%{safe_content}%'")
                    if content_conditions:
                        conditions.append(f"({' AND '.join(content_conditions)})")

            if conditions:
                query += " AND " + " AND ".join(conditions)

            results = self._execute_query_json(query)

            if results and len(results) > 0:
                count = results[0].get("count", 0)
                return int(count) > 0

            return False

        except Exception as e:
            self.logger.error(f"Failed to check post existence in {subreddit}: {e}")
            return False

        except Exception as e:
            self.logger.error(f"Failed to check post existence in {subreddit}: {e}")
            return False

    def get_post_comments(self, post_id: str, limit: int = 100) -> List[CommentInfo]:
        """Get comments for a specific post."""
        try:
            query = f"""
                SELECT c.id, c.body, c.timestamp, c.net_score,
                       u.username as author, s.id as submission_id
                FROM comments c
                JOIN users u ON c.user_id = u.id
                JOIN submissions s ON c.submission_id = s.id
                WHERE s.id = {post_id}
                ORDER BY c.timestamp DESC
                LIMIT {limit}
            """

            results = self._execute_query_json(query)
            self.logger.debug(f"Retrieved {len(results)} comments for post {post_id}")

            comments = []
            for comment_data in results:
                comments.append(
                    CommentInfo.model_validate(
                        {
                            "id": str(comment_data["id"]),
                            "body": comment_data["body"],
                            "author": comment_data["author"],
                            "created_utc": comment_data.get("timestamp"),
                            "score": comment_data.get("net_score", 0),
                            "submission_id": str(comment_data["submission_id"]),
                        }
                    )
                )

            return comments

        except Exception as e:
            self.logger.error(f"Failed to get comments for post {post_id}: {e}")
            return []

    def comment_exists_on_post(
        self,
        post_id: str,
        author: str = None,
        content_contains: str = None,
    ) -> bool:
        """Check if a comment exists on a post with specific criteria."""
        try:
            query = f"""
                SELECT COUNT(*) as count
                FROM comments c
                JOIN users u ON c.user_id = u.id
                WHERE c.submission_id = {post_id}
            """

            conditions = []
            if author:
                safe_author = author.replace("'", "''")
                conditions.append(f"u.username = '{safe_author}'")
            if content_contains:
                safe_content = content_contains.replace("'", "''")
                conditions.append(f"c.body ILIKE '%{safe_content}%'")

            if conditions:
                query += " AND " + " AND ".join(conditions)

            results = self._execute_query_json(query)

            if results and len(results) > 0:
                count = results[0].get("count", 0)
                return int(count) > 0

            return False

        except Exception as e:
            self.logger.error(
                f"Failed to check comment existence on post {post_id}: {e}"
            )
            return False

    def reply_exists_for_comment(
        self,
        comment_id: str,
        author: str = None,
        content_contains: str = None,
    ) -> bool:
        """Check if a reply exists for a specific comment with given criteria."""
        try:
            query = f"""
                SELECT COUNT(*) as count
                FROM comments c
                JOIN users u ON c.user_id = u.id
                WHERE c.parent_id = {comment_id}
            """

            conditions = []
            if author:
                safe_author = author.replace("'", "''")
                conditions.append(f"u.username = '{safe_author}'")
            if content_contains:
                safe_content = content_contains.replace("'", "''")
                conditions.append(f"c.body ILIKE '%{safe_content}%'")

            if conditions:
                query += " AND " + " AND ".join(conditions)

            results = self._execute_query_json(query)

            if results and len(results) > 0:
                count = results[0].get("count", 0)
                return int(count) > 0

            return False

        except Exception as e:
            self.logger.error(
                f"Failed to check reply existence for comment {comment_id}: {e}"
            )
            return False

    def get_user_posts(
        self, username: str, subreddit: str = None, limit: int = 25
    ) -> List[PostInfo]:
        """Get posts by a specific user."""
        try:
            safe_username = username.replace("'", "''")

            query = f"""
                SELECT s.id, s.title, s.body, s.timestamp, s.net_score, s.comment_count,
                       u.username as author, f.name as forum_name
                FROM submissions s
                JOIN users u ON s.user_id = u.id
                JOIN forums f ON s.forum_id = f.id
                WHERE u.username = '{safe_username}'
            """

            if subreddit:
                safe_subreddit = subreddit.replace("'", "''")
                query += f" AND (f.name = '{safe_subreddit}' OR f.name ILIKE '{safe_subreddit}')"

            query += f" ORDER BY s.timestamp DESC LIMIT {limit}"

            results = self._execute_query_json(query)
            self.logger.debug(f"Retrieved {len(results)} posts by {username}")

            posts = []
            for post_data in results:
                posts.append(
                    PostInfo.model_validate(
                        {
                            "id": str(post_data["id"]),
                            "title": post_data["title"],
                            "selftext": post_data.get("body", ""),
                            "author": post_data["author"],
                            "subreddit": post_data["forum_name"],
                            "created_utc": post_data.get("timestamp"),
                            "score": post_data.get("net_score", 0),
                            "num_comments": post_data.get("comment_count", 0),
                            "url": f"/f/{post_data['forum_name']}/comments/{post_data['id']}/",
                        }
                    )
                )

            return posts

        except Exception as e:
            self.logger.error(f"Failed to get posts by {username}: {e}")
            return []

    def check_user_subscribed_to_subreddit(self, username: str, subreddit: str) -> bool:
        """Check if a user is subscribed to a subreddit."""
        try:
            safe_username = username.replace("'", "''")
            safe_subreddit = subreddit.replace("'", "''")

            query = f"""
                SELECT COUNT(*) as count
                FROM forum_subscriptions fs
                JOIN users u ON fs.user_id = u.id
                JOIN forums f ON fs.forum_id = f.id
                WHERE u.username = '{safe_username}' 
                AND (f.name = '{safe_subreddit}' OR f.name ILIKE '{safe_subreddit}')
            """

            results = self._execute_query_json(query)

            if results and len(results) > 0:
                count = results[0].get("count", 0)
                return int(count) > 0

            return False

        except Exception as e:
            self.logger.error(
                f"Failed to check subscription for {username} to {subreddit}: {e}"
            )
            return False

    def get_user_subscriptions(self, username: str) -> List[SubscriptionInfo]:
        """Get all subreddit subscriptions for a user."""
        try:
            safe_username = username.replace("'", "''")

            query = f"""
                SELECT f.name as subreddit_name, f.title, fs.created_at
                FROM forum_subscriptions fs
                JOIN users u ON fs.user_id = u.id
                JOIN forums f ON fs.forum_id = f.id
                WHERE u.username = '{safe_username}'
                ORDER BY fs.created_at DESC
            """

            results = self._execute_query_json(query)
            self.logger.debug(f"Retrieved {len(results)} subscriptions for {username}")

            subscriptions = []
            for sub_data in results:
                subscriptions.append(
                    SubscriptionInfo.model_validate(
                        {
                            "subreddit": sub_data["subreddit_name"],
                            "title": sub_data.get("title", ""),
                            "subscribed_at": sub_data.get("created_at"),
                        }
                    )
                )

            return subscriptions

        except Exception as e:
            self.logger.error(f"Failed to get subscriptions for {username}: {e}")
            return []

    def close(self):
        """Close any connections (placeholder for compatibility)."""
        # PostgreSQL connections via docker exec don't need explicit closing
        pass

    def create_user(
        self,
        username: str,
        email: Optional[str] = None,
        admin: bool = False,
        password_hash: Optional[str] = None,
    ) -> bool:
        """
        Create a new user in the Postmill database.

        Args:
            username: Unique username for the new user
            email: Optional email address
            admin: Whether the user should have admin privileges (default: False)
            password_hash: Pre-hashed password (if None, uses default test password)

        Returns:
            True if user was created successfully, False otherwise
        """
        from datetime import datetime

        # Use default password hash if none provided (common test password)
        if password_hash is None:
            password_hash = (
                "$2y$13$1I/hMfQdvPmKTYkrLk4NqOb5jC9Mf1l1nbnkTcRJm/ik8nLPdl5.i"
            )

        # Normalize username (lowercase)
        normalized_username = username.lower()

        # Current timestamp
        created_time = datetime.now().strftime("%Y-%m-%d %H:%M:%S+00")

        # Check if user already exists
        if self.user_exists(username):
            self.logger.warning(f"User {username} already exists")
            return False

        # Build SQL based on whether email is provided
        if email:
            sql = f"""
            INSERT INTO users (
                id, username, password, normalized_username, created, admin, locale, night_mode,
                front_page, open_external_links_in_new_tab, auto_fetch_submission_titles,
                enable_post_previews, show_thumbnails, notify_on_reply, notify_on_mentions,
                front_page_sort_mode, timezone, allow_private_messages, poppers_enabled,
                full_width_display_enabled, submission_link_destination, trusted,
                show_custom_stylesheets, email, normalized_email
            ) VALUES (
                nextval('users_id_seq'), '{username}', '{password_hash}', '{normalized_username}', '{created_time}',
                {str(admin).lower()}, 'en', 'auto', 'subscribed', false, true, true, true,
                true, true, 'hot', 'America/New_York', true, true, false, 'url', false,
                true, '{email}', '{email.lower()}'
            );
            """
        else:
            sql = f"""
            INSERT INTO users (
                id, username, password, normalized_username, created, admin, locale, night_mode,
                front_page, open_external_links_in_new_tab, auto_fetch_submission_titles,
                enable_post_previews, show_thumbnails, notify_on_reply, notify_on_mentions,
                front_page_sort_mode, timezone, allow_private_messages, poppers_enabled,
                full_width_display_enabled, submission_link_destination, trusted,
                show_custom_stylesheets
            ) VALUES (
                nextval('users_id_seq'), '{username}', '{password_hash}', '{normalized_username}', '{created_time}',
                {str(admin).lower()}, 'en', 'auto', 'subscribed', false, true, true, true,
                true, true, 'hot', 'America/New_York', true, true, false, 'url', false, true
            );
            """

        try:
            self._execute_query(sql)
            self.logger.debug(f"Successfully created user: {username}")
            return True
        except Exception as e:
            self.logger.error(f"Failed to create user {username}: {e}")
            return False

    def user_exists(self, username: str) -> bool:
        """Check if a user exists in the database."""
        try:
            query = f"SELECT COUNT(*) as count FROM users WHERE username = '{username}'"
            result = self._execute_query_json(query)
            if result and len(result) > 0:
                return result[0].get("count", 0) > 0
            return False
        except Exception as e:
            self.logger.error(f"Error checking if user exists: {e}")
            return False

    def check_user_upvoted_post(self, username: str, post_id: str) -> bool:
        """
        Check if a user has upvoted a specific post.

        Args:
            username: Username to check
            post_id: Post ID to check for upvote

        Returns:
            True if user upvoted the post, False otherwise
        """
        try:
            safe_username = username.replace("'", "''")
            safe_post_id = post_id.replace("'", "''")

            query = f"""
            SELECT COUNT(*) as count
            FROM submission_votes sv
            JOIN users u ON sv.user_id = u.id
            JOIN submissions s ON sv.submission_id = s.id
            WHERE u.username = '{safe_username}'
            AND s.id = '{safe_post_id}'
            AND sv.upvote = true
            """

            result = self._execute_query_json(query)
            if result and len(result) > 0:
                vote_count = result[0].get("count", 0)
                self.logger.debug(
                    f"User {username} upvote status for post {post_id}: {vote_count > 0}"
                )
                return vote_count > 0
            return False

        except Exception as e:
            self.logger.error(
                f"Error checking upvote status for user {username} on post {post_id}: {e}"
            )
            return False

    def check_user_downvoted_post(self, username: str, post_id: str) -> bool:
        """
        Check if a user has downvoted a specific post.

        Args:
            username: Username to check
            post_id: Post ID to check for downvote

        Returns:
            True if user downvoted the post, False otherwise
        """
        try:
            safe_username = username.replace("'", "''")
            safe_post_id = post_id.replace("'", "''")

            query = f"""
            SELECT COUNT(*) as count
            FROM submission_votes sv
            JOIN users u ON sv.user_id = u.id
            JOIN submissions s ON sv.submission_id = s.id
            WHERE u.username = '{safe_username}'
            AND s.id = '{safe_post_id}'
            AND sv.upvote = false
            """

            result = self._execute_query_json(query)
            if result and len(result) > 0:
                vote_count = result[0].get("count", 0)
                self.logger.debug(
                    f"User {username} downvote status for post {post_id}: {vote_count > 0}"
                )
                return vote_count > 0
            return False

        except Exception as e:
            self.logger.error(
                f"Error checking downvote status for user {username} on post {post_id}: {e}"
            )
            return False

    def check_user_vote_on_post(self, username: str, post_id: str) -> Optional[bool]:
        """
        Check a user's vote status on a specific post.

        Args:
            username: Username to check
            post_id: Post ID to check for vote

        Returns:
            True if user upvoted, False if user downvoted, None if no vote
        """
        try:
            safe_username = username.replace("'", "''")
            safe_post_id = post_id.replace("'", "''")

            query = f"""
            SELECT sv.upvote
            FROM submission_votes sv
            JOIN users u ON sv.user_id = u.id
            JOIN submissions s ON sv.submission_id = s.id
            WHERE u.username = '{safe_username}'
            AND s.id = '{safe_post_id}'
            """

            result = self._execute_query_json(query)
            if result and len(result) > 0:
                upvote = result[0].get("upvote")
                self.logger.debug(
                    f"User {username} vote status for post {post_id}: {'upvote' if upvote else 'downvote'}"
                )
                return upvote
            else:
                self.logger.debug(f"User {username} has not voted on post {post_id}")
                return None

        except Exception as e:
            self.logger.error(
                f"Error checking vote status for user {username} on post {post_id}: {e}"
            )
            return None

    def check_user_voted_on_posts(
        self, username: str, post_ids: List[str], upvote: bool = True
    ) -> Dict[str, bool]:
        """
        Check voting status for multiple posts at once (more efficient than individual calls).

        Args:
            username: Username to check
            post_ids: List of post IDs to check
            upvote: True to check upvotes, False to check downvotes

        Returns:
            Dictionary mapping post_id -> vote_status
        """
        try:
            safe_username = username.replace("'", "''")
            safe_post_ids = [pid.replace("'", "''") for pid in post_ids]
            post_ids_str = "','".join(safe_post_ids)

            query = f"""
            SELECT s.id as post_id, COUNT(sv.id) > 0 as voted
            FROM submissions s
            LEFT JOIN submission_votes sv ON s.id = sv.submission_id 
                AND sv.user_id = (SELECT u.id FROM users u WHERE u.username = '{safe_username}')
                AND sv.upvote = {str(upvote).lower()}
            WHERE s.id IN ('{post_ids_str}')
            GROUP BY s.id
            """

            result = self._execute_query_json(query)
            vote_status = {}

            for row in result:
                post_id = str(row.get("post_id", ""))
                voted = row.get("voted", False)
                vote_status[post_id] = voted

            # Ensure all requested post_ids are in the result
            for post_id in post_ids:
                if post_id not in vote_status:
                    vote_status[post_id] = False

            vote_type = "upvotes" if upvote else "downvotes"
            self.logger.debug(
                f"User {username} {vote_type} status for {len(post_ids)} posts: {sum(vote_status.values())} voted"
            )

            return vote_status

        except Exception as e:
            self.logger.error(
                f"Error checking bulk vote status for user {username}: {e}"
            )
            return {post_id: False for post_id in post_ids}

    def post_exists_with_image_file_name(
        self,
        subreddit: str,
        file_name: str,
        author: str = None,
        title_contains: str = None,
    ) -> bool:
        """Check if a post exists in subreddit with a specific image file name and optionally title content."""
        try:
            safe_subreddit = subreddit.replace("'", "''")
            safe_file_name = file_name.replace("'", "''")

            query = f"""
                SELECT COUNT(*) as count
                FROM submissions s
                JOIN users u ON s.user_id = u.id
                JOIN forums f ON s.forum_id = f.id
                JOIN images i ON s.image_id = i.id
                WHERE (f.name = '{safe_subreddit}' OR f.name ILIKE '{safe_subreddit}')
                AND i.file_name = '{safe_file_name}'
                AND s.media_type = 'image'
            """

            if author:
                safe_author = author.replace("'", "''")
                query += f" AND u.username = '{safe_author}'"
            
            if title_contains:
                safe_title = title_contains.replace("'", "''")
                query += f" AND s.title ILIKE '%{safe_title}%'"

            results = self._execute_query_json(query)

            if results and len(results) > 0:
                count = results[0].get("count", 0)
                self.logger.debug(f"Found {count} posts with image file name '{file_name}' in {subreddit}")
                return count > 0

            return False

        except Exception as e:
            self.logger.error(f"Failed to check post existence with image file name in {subreddit}: {e}")
            return False
