"""
Pydantic v2 models for Reddit validation responses.
"""

from datetime import datetime
from typing import Any, Dict, Optional

from pydantic import BaseModel, ConfigDict


class RedditConfig(BaseModel):
    """Configuration model for Reddit connections."""

    model_config = ConfigDict(from_attributes=True)

    container_name: str = "forum"
    database: str = "postmill"
    username: str = "postmill"


class UserInfo(BaseModel):
    """Reddit user information model."""

    model_config = ConfigDict(from_attributes=True)

    username: str
    bio: Optional[str] = None  # Changed from biography to bio for consistency
    email: Optional[str] = None
    created_utc: Optional[datetime] = None  # Changed from created to created_utc


class SubredditInfo(BaseModel):
    """Subreddit/forum information model."""

    model_config = ConfigDict(from_attributes=True)

    display_name: str  # Changed from name to display_name to match database
    title: Optional[str] = None
    description: Optional[str] = None
    sidebar: Optional[str] = None  # Raw sidebar content (markdown)
    subscribers: Optional[int] = 0  # Changed from subscriber_count to subscribers
    created_utc: Optional[datetime] = None  # Changed from created to created_utc


class PostInfo(BaseModel):
    """Post/submission information model."""

    model_config = ConfigDict(from_attributes=True)

    id: str
    title: str
    selftext: Optional[str] = (
        None  # Changed from body to selftext for Reddit compatibility
    )
    author: str
    subreddit: str  # Changed from forum_name to subreddit
    created_utc: Optional[datetime] = None  # Changed from timestamp to created_utc
    score: Optional[int] = 0  # Changed from net_score to score
    num_comments: Optional[int] = 0  # Changed from comment_count to num_comments
    url: Optional[str] = None


class CommentInfo(BaseModel):
    """Comment information model."""

    model_config = ConfigDict(from_attributes=True)

    id: str
    body: str
    author: str
    submission_id: str
    parent_id: Optional[str] = None
    timestamp: Optional[datetime] = None
    net_score: Optional[int] = 0


class ValidationResponse(BaseModel):
    """Standard validation response model."""

    model_config = ConfigDict(from_attributes=True)

    success: bool
    message: str
    data: Optional[Dict[str, Any]] = None


class VoteInfo(BaseModel):
    """Vote information model."""

    model_config = ConfigDict(from_attributes=True)

    user_id: int
    submission_id: int
    choice: int  # 1 for upvote, -1 for downvote
    username: str

    @property
    def is_upvote(self) -> bool:
        return self.choice == 1


class SubscriptionInfo(BaseModel):
    """Subscription information model."""

    model_config = ConfigDict(from_attributes=True)

    user_id: int
    forum_id: int
    username: str
    forum_name: str
    is_subscribed: bool = True


# Expected Data Models for Validation Functions
class BioContentExpectedData(BaseModel):
    """Expected data for user bio content validation."""

    content: str


class SubredditSubscriptionExpectedData(BaseModel):
    """Expected data for subreddit subscription validation."""

    subreddit: str


class PostExistsExpectedData(BaseModel):
    """Expected data for post existence validation."""

    subreddit: str
    title: Optional[str | list[str]] = None
    content: Optional[str | list[str]] = None
    author: Optional[str] = None
    post_id: Optional[int] = None


class CommentExistsExpectedData(BaseModel):
    """Expected data for comment existence validation."""

    post_id: str
    comment_content: Optional[str] = None


class SubredditDescriptionExpectedData(BaseModel):
    """Expected data for subreddit description validation."""

    subreddit: str
    description: Optional[str] = None
    sidebar: Optional[list[str]] = None


class UserPostsVoteExpectedData(BaseModel):
    """Expected data for user posts vote validation."""

    target_user: str
    subreddit: str


class PostIdsUpvoteExpectedData(BaseModel):
    """Expected data for post IDs upvote validation."""

    post_ids: list[int]


class PostIdsDownvoteExpectedData(BaseModel):
    """Expected data for post IDs downvote validation."""

    post_ids: list[int]


class ReplyExistsExpectedData(BaseModel):
    """Expected data for reply existence validation."""

    comment_id: str
    reply_content: Optional[str] = None


class PostImageFileNameExpectedData(BaseModel):
    """Expected data for post image file name validation."""

    subreddit: str
    file_name: str
    author: Optional[str] = None
    title: Optional[str] = None
