"""
Reddit Custom Validation Functions for WebArena Verified Tasks.

This module contains the custom validation functions required by the
custom evaluator system for Reddit-based tasks.
"""

import logging

from ..models import AllocationResource

from ..types import (
    EvalFunc,
    WebArenaTask,
    WebarenaTaskEvalResult,
    WebArenaTaskResponse,
)

from .data_helper import RedditDataHelper
from .models import (
    # Expected Data Models
    BioContentExpectedData,
    CommentExistsExpectedData,
    PostExistsExpectedData,
    PostIdsDownvoteExpectedData,
    PostIdsUpvoteExpectedData,
    PostImageFileNameExpectedData,
    RedditConfig,
    ReplyExistsExpectedData,
    SubredditDescriptionExpectedData,
    SubredditSubscriptionExpectedData,
)


class RedditValidator:
    """
    Reddit validator with custom evaluation functions for WebArena Verified tasks.

    Contains validation methods that are called by the custom evaluator system.
    Uses RedditDataHelper for all data access via PostgreSQL database.
    """

    def __init__(self):
        self.logger = logging.getLogger(__name__)

    def validate_user_bio_content(
        self,
        task: WebArenaTask,
        task_result: WebArenaTaskResponse,
        resource: AllocationResource,
        eval_func: EvalFunc,
        **kwargs,
    ) -> WebarenaTaskEvalResult:
        """
        Validate that a user's bio contains specific content.

        Expected eval_func format:
        {
            "name": "validate_user_bio_content",
            "expected_data": {"content": "I am a robot"}
        }
        Username is obtained from resource.username
        """
        try:
            # Extract expected_data from eval_func and validate with Pydantic
            if not eval_func or not hasattr(eval_func, "expected_data"):
                return WebarenaTaskEvalResult.create_failed(
                    assertion_msgs=["No expected_data found in eval_func"],
                    site=resource.website_type,
                    task_id=task.task_id,
                )

            try:
                expected_data = BioContentExpectedData.model_validate(
                    eval_func.expected_data
                )
            except Exception as e:
                return WebarenaTaskEvalResult.create_failed(
                    assertion_msgs=[f"Invalid expected_data format: {str(e)}"],
                    site=resource.website_type,
                    task_id=task.task_id,
                )

            # Extract username from resource
            username = resource.username

            # Create RedditConfig from AllocationResource
            config = RedditConfig(container_name=resource.container_name)

            # Create service with resource-specific config
            service = RedditDataHelper(config)

            try:
                # Check if user's bio contains the expected content
                has_content = service.user_has_bio_content(
                    username, expected_data.content
                )

                if has_content:
                    # Get the actual bio for verification
                    actual_bio = service.get_user_profile_bio(username)
                    return WebarenaTaskEvalResult.create_success(
                        assertion_msgs=[
                            f"User {username} bio contains expected content: '{expected_data.content}'"
                        ],
                        site=resource.website_type,
                        task_id=task.task_id,
                        validation_data={
                            "username": username,
                            "expected_content": expected_data.content,
                            "actual_bio": actual_bio,
                        },
                    )
                else:
                    actual_bio = service.get_user_profile_bio(username)
                    return WebarenaTaskEvalResult.create_failed(
                        assertion_msgs=[
                            f"User {username} bio does not contain expected content: '{expected_data.content}'"
                        ],
                        site=resource.website_type,
                        task_id=task.task_id,
                        validation_data={
                            "username": username,
                            "expected_content": expected_data.content,
                            "actual_bio": actual_bio,
                        },
                    )

            finally:
                service.close()

        except Exception as e:
            return WebarenaTaskEvalResult.create_failed(
                assertion_msgs=[f"Bio validation failed: {str(e)}"],
                site=resource.website_type,
                task_id=task.task_id,
            )

    def validate_user_subscribed_to_subreddit(
        self,
        task: WebArenaTask,
        task_result: WebArenaTaskResponse,
        resource: AllocationResource,
        eval_func: EvalFunc,
        **kwargs,
    ) -> WebarenaTaskEvalResult:
        """
        Validate that a user is subscribed to a specific subreddit.

        Expected eval_func format:
        {
            "name": "validate_user_subscribed_to_subreddit",
            "expected_data": {"subreddit": "books"}
        }
        Username is obtained from resource.username
        """
        try:
            # Extract expected_data from eval_func and validate with Pydantic
            if not eval_func or not hasattr(eval_func, "expected_data"):
                return WebarenaTaskEvalResult.create_failed(
                    assertion_msgs=["No expected_data found in eval_func"],
                    site=resource.website_type,
                    task_id=task.task_id,
                )

            try:
                expected_data = SubredditSubscriptionExpectedData.model_validate(
                    eval_func.expected_data
                )
            except Exception as e:
                return WebarenaTaskEvalResult.create_failed(
                    assertion_msgs=[f"Invalid expected_data format: {str(e)}"],
                    site=resource.website_type,
                    task_id=task.task_id,
                )

            # Extract username from resource
            username = resource.username

            # Create RedditConfig from AllocationResource
            config = RedditConfig(container_name=resource.container_name)

            # Create service with resource-specific config
            service = RedditDataHelper(config)

            try:
                # Check if user is subscribed to subreddit
                is_subscribed = service.check_user_subscribed_to_subreddit(
                    username, expected_data.subreddit
                )

                if is_subscribed:
                    return WebarenaTaskEvalResult.create_success(
                        assertion_msgs=[
                            f"User {username} is successfully subscribed to subreddit {expected_data.subreddit}"
                        ],
                        site=resource.website_type,
                        task_id=task.task_id,
                        validation_data={
                            "username": username,
                            "subreddit": expected_data.subreddit,
                            "subscribed": True,
                        },
                    )
                else:
                    return WebarenaTaskEvalResult.create_failed(
                        assertion_msgs=[
                            f"User {username} is not subscribed to subreddit {expected_data.subreddit}"
                        ],
                        site=resource.website_type,
                        task_id=task.task_id,
                        validation_data={
                            "username": username,
                            "subreddit": expected_data.subreddit,
                            "subscribed": False,
                        },
                    )

            finally:
                service.close()

        except Exception as e:
            return WebarenaTaskEvalResult.create_failed(
                assertion_msgs=[f"Subscription validation failed: {str(e)}"],
                site=resource.website_type,
                task_id=task.task_id,
            )

    def validate_post_exists_in_subreddit(
        self,
        task: WebArenaTask,
        task_result: WebArenaTaskResponse,
        resource: AllocationResource,
        eval_func: EvalFunc,
        **kwargs,
    ) -> WebarenaTaskEvalResult:
        """
        Validate that a post exists in a specific subreddit.

        Expected eval_func format:
        {
            "name": "validate_post_exists_in_subreddit",
            "expected_data": {
                "subreddit": "books",
                "title": "Harry Potter",           # Optional: check if title contains this text
                "content": "Edit: This is new",    # Optional: check if body contains this text
                "post_id": 12345                   # Optional: check specific post ID
            }
        }
        Author is obtained from resource.username

        Logic:
        - If only title provided: check title contains the text
        - If only content provided: check body contains the text
        - If both title and content: check BOTH conditions
        - If neither: use post_id and/or author only
        """
        try:
            # Extract expected_data from eval_func and validate with Pydantic
            if not eval_func or not hasattr(eval_func, "expected_data"):
                return WebarenaTaskEvalResult.create_failed(
                    assertion_msgs=["No expected_data found in eval_func"],
                    site=resource.website_type,
                    task_id=task.task_id,
                )

            try:
                expected_data = PostExistsExpectedData.model_validate(
                    eval_func.expected_data
                )
            except Exception as e:
                return WebarenaTaskEvalResult.create_failed(
                    assertion_msgs=[f"Invalid expected_data format: {str(e)}"],
                    site=resource.website_type,
                    task_id=task.task_id,
                )

            # Extract username from resource
            username = resource.username

            # Create RedditConfig from AllocationResource
            config = RedditConfig(container_name=resource.container_name)

            # Create service with resource-specific config
            service = RedditDataHelper(config)

            try:
                # Check if post exists based on provided criteria
                post_exists = service.post_exists_in_subreddit(
                    subreddit=expected_data.subreddit,
                    post_id=expected_data.post_id,
                    author=username,
                    title_contains=expected_data.title,
                    content_contains=expected_data.content,
                )

                if post_exists:
                    return WebarenaTaskEvalResult.create_success(
                        assertion_msgs=[
                            f"Post found in subreddit {expected_data.subreddit} matching criteria"
                        ],
                        site=resource.website_type,
                        task_id=task.task_id,
                        validation_data={
                            "subreddit": expected_data.subreddit,
                            "title": expected_data.title,
                            "content": expected_data.content,
                            "author": username,
                            "post_id": expected_data.post_id,
                            "exists": True,
                        },
                    )
                else:
                    return WebarenaTaskEvalResult.create_failed(
                        assertion_msgs=[
                            f"Post not found in subreddit {expected_data.subreddit}"
                        ],
                        site=resource.website_type,
                        task_id=task.task_id,
                        validation_data={
                            "subreddit": expected_data.subreddit,
                            "title": expected_data.title,
                            "content": expected_data.content,
                            "author": username,
                            "post_id": expected_data.post_id,
                            "exists": False,
                        },
                    )

            finally:
                service.close()

        except Exception as e:
            return WebarenaTaskEvalResult.create_failed(
                assertion_msgs=[f"Post validation failed: {str(e)}"],
                site=resource.website_type,
                task_id=task.task_id,
            )

    def validate_comment_exists_in_post(
        self,
        task: WebArenaTask,
        task_result: WebArenaTaskResponse,
        resource: AllocationResource,
        eval_func: EvalFunc,
        **kwargs,
    ) -> WebarenaTaskEvalResult:
        """
        Validate that a comment exists in a specific post.

        Expected eval_func format:
        {
            "name": "validate_comment_exists_in_post",
            "expected_data": {"post_id": "59421", "comment_content": "I am a big fan of the bookorg"}
        }
        The comment author is taken from resource.username.
        """
        try:
            # Extract expected_data from eval_func and validate with Pydantic
            if not eval_func or not hasattr(eval_func, "expected_data"):
                return WebarenaTaskEvalResult.create_failed(
                    assertion_msgs=["No expected_data found in eval_func"],
                    site=resource.website_type,
                    task_id=task.task_id,
                )

            try:
                expected_data = CommentExistsExpectedData.model_validate(
                    eval_func.expected_data
                )
            except Exception as e:
                return WebarenaTaskEvalResult.create_failed(
                    assertion_msgs=[f"Invalid expected_data format: {str(e)}"],
                    site=resource.website_type,
                    task_id=task.task_id,
                )

            # Get the username from the allocation resource
            username = resource.username

            # Create RedditConfig from AllocationResource
            config = RedditConfig(container_name=resource.container_name)

            # Create service with resource-specific config
            service = RedditDataHelper(config)

            try:
                # Check if comment exists in the post by the specified user
                comment_exists = service.comment_exists_on_post(
                    post_id=expected_data.post_id,
                    author=username,
                    content_contains=expected_data.comment_content,
                )

                if comment_exists:
                    return WebarenaTaskEvalResult.create_success(
                        assertion_msgs=[
                            f"Comment by {username} found in post {expected_data.post_id} with expected content"
                        ],
                        site=resource.website_type,
                        task_id=task.task_id,
                        validation_data={
                            "post_id": expected_data.post_id,
                            "comment_content": expected_data.comment_content,
                            "author": username,
                            "exists": True,
                        },
                    )
                else:
                    return WebarenaTaskEvalResult.create_failed(
                        assertion_msgs=[
                            f"Comment by {username} not found in post {expected_data.post_id}"
                        ],
                        site=resource.website_type,
                        task_id=task.task_id,
                        validation_data={
                            "post_id": expected_data.post_id,
                            "comment_content": expected_data.comment_content,
                            "author": username,
                            "exists": False,
                        },
                    )

            finally:
                service.close()

        except Exception as e:
            return WebarenaTaskEvalResult.create_failed(
                assertion_msgs=[f"Comment validation failed: {str(e)}"],
                site=resource.website_type,
                task_id=task.task_id,
            )

    def validate_subreddit_has_description(
        self,
        task: WebArenaTask,
        task_result: WebArenaTaskResponse,
        resource: AllocationResource,
        eval_func: EvalFunc,
        **kwargs,
    ) -> WebarenaTaskEvalResult:
        """
        Validate that a subreddit/forum has the right description and sidebar tags.

        Expected data format: {
            "subreddit": "sci_fi",
            "description": "Science Fiction Discussion",
            "sidebar": ["New", "Classic", "Movies"]
        }
        Description must match exactly (case-insensitive).
        All sidebar tags must be found in the actual sidebar.
        """
        try:
            # Extract expected_data from eval_func and validate with Pydantic
            if not eval_func or not hasattr(eval_func, "expected_data"):
                return WebarenaTaskEvalResult.create_failed(
                    assertion_msgs=["No expected_data found in eval_func"],
                    site=resource.website_type,
                    task_id=task.task_id,
                )

            try:
                expected_data = SubredditDescriptionExpectedData.model_validate(
                    eval_func.expected_data
                )
            except Exception as e:
                return WebarenaTaskEvalResult.create_failed(
                    assertion_msgs=[f"Invalid expected_data format: {str(e)}"],
                    site=resource.website_type,
                    task_id=task.task_id,
                )

            # Create RedditConfig from AllocationResource
            config = RedditConfig(container_name=resource.container_name)

            # Create service with resource-specific config
            service = RedditDataHelper(config)

            try:
                # Check if subreddit exists
                subreddit_info = service.get_subreddit_by_name(expected_data.subreddit)

                if subreddit_info:
                    # Validate description if provided
                    description_match = True
                    description_msg = None
                    if expected_data.description:
                        actual_description = subreddit_info.description or ""
                        if (
                            actual_description.lower().strip()
                            != expected_data.description.lower().strip()
                        ):
                            description_match = False
                            description_msg = f"Description mismatch. Expected: '{expected_data.description}', Actual: '{actual_description}'"

                    # Validate sidebar tags if provided
                    sidebar_match = True
                    missing_sidebar_tags = []
                    if expected_data.sidebar:
                        actual_sidebar = subreddit_info.sidebar or ""
                        actual_sidebar_lower = actual_sidebar.lower()

                        # Check that all expected sidebar tags are found as substrings
                        for expected_tag in expected_data.sidebar:
                            if expected_tag.lower() not in actual_sidebar_lower:
                                sidebar_match = False
                                missing_sidebar_tags.append(expected_tag)

                    # Overall validation result
                    overall_match = description_match and sidebar_match

                    validation_data = {
                        "subreddit": expected_data.subreddit,
                        "expected_description": expected_data.description,
                        "actual_description": subreddit_info.description,
                        "expected_sidebar": expected_data.sidebar,
                        "actual_sidebar": subreddit_info.sidebar,
                        "description_match": description_match,
                        "sidebar_match": sidebar_match,
                        "overall_match": overall_match,
                    }

                    if overall_match:
                        return WebarenaTaskEvalResult.create_success(
                            assertion_msgs=[
                                f"Subreddit {expected_data.subreddit} has correct description and sidebar tags"
                            ],
                            site=resource.website_type,
                            task_id=task.task_id,
                            validation_data=validation_data,
                        )
                    else:
                        error_msgs = []
                        if not description_match:
                            error_msgs.append(description_msg)
                        if not sidebar_match:
                            error_msgs.append(
                                f"Missing sidebar tags: {missing_sidebar_tags}"
                            )

                        return WebarenaTaskEvalResult.create_failed(
                            assertion_msgs=[
                                f"Subreddit {expected_data.subreddit} validation failed: {'; '.join(error_msgs)}"
                            ],
                            site=resource.website_type,
                            task_id=task.task_id,
                            validation_data={
                                **validation_data,
                                "missing_sidebar_tags": missing_sidebar_tags,
                                "description_error": description_msg,
                            },
                        )
                else:
                    return WebarenaTaskEvalResult.create_failed(
                        assertion_msgs=[
                            f"Subreddit {expected_data.subreddit} does not exist"
                        ],
                        site=resource.website_type,
                        task_id=task.task_id,
                        validation_data={
                            "subreddit": expected_data.subreddit,
                            "expected_description": expected_data.description,
                            "expected_sidebar": expected_data.sidebar,
                            "exists": False,
                        },
                    )

            finally:
                service.close()

        except Exception as e:
            return WebarenaTaskEvalResult.create_failed(
                assertion_msgs=[f"Subreddit validation failed: {str(e)}"],
                site=resource.website_type,
                task_id=task.task_id,
            )

    def validate_user_upvoted_posts(
        self,
        task: WebArenaTask,
        task_result: WebArenaTaskResponse,
        resource: AllocationResource,
        eval_func: EvalFunc,
        **kwargs,
    ) -> WebarenaTaskEvalResult:
        """
        Validate that a user has upvoted specific posts by their IDs.

        This is a consolidated validation function that replaces the need for
        validate_user_upvoted_newest_post and validate_user_upvoted_all_user_posts
        by accepting a list of post IDs to check.

        Expected data format: {"post_ids": [123, 456, 789]}
        Username is obtained from resource.username
        """
        try:
            # Extract expected_data from eval_func and validate with Pydantic
            if not eval_func or not hasattr(eval_func, "expected_data"):
                return WebarenaTaskEvalResult.create_failed(
                    assertion_msgs=["No expected_data found in eval_func"],
                    site=resource.website_type,
                    task_id=task.task_id,
                )

            try:
                expected_data = PostIdsUpvoteExpectedData.model_validate(
                    eval_func.expected_data
                )
            except Exception as e:
                return WebarenaTaskEvalResult.create_failed(
                    assertion_msgs=[f"Invalid expected_data format: {str(e)}"],
                    site=resource.website_type,
                    task_id=task.task_id,
                )

            # Extract username from resource
            username = resource.username

            # Validate post_ids is not empty
            if not expected_data.post_ids:
                return WebarenaTaskEvalResult.create_failed(
                    assertion_msgs=["No post IDs provided for upvote validation"],
                    site=resource.website_type,
                    task_id=task.task_id,
                )

            # Create RedditConfig from AllocationResource
            config = RedditConfig(container_name=resource.container_name)

            # Create service with resource-specific config
            service = RedditDataHelper(config)

            try:
                # Check if user upvoted all specified posts
                # Convert post_ids to strings as the service expects List[str]
                post_ids_str = [str(pid) for pid in expected_data.post_ids]
                upvote_status = service.check_user_voted_on_posts(
                    username, post_ids_str, upvote=True
                )

                # Count how many were upvoted
                upvoted_count = sum(upvote_status.values())
                all_upvoted = upvoted_count == len(expected_data.post_ids)

                # Get post details for validation data
                post_details = []
                for post_id in expected_data.post_ids:
                    try:
                        # Try to get post info for better reporting
                        post_info = service.get_post_by_id(str(post_id))
                        post_details.append(
                            {
                                "id": post_id,
                                "title": post_info.title
                                if post_info
                                else f"Post {post_id}",
                                "upvoted": upvote_status.get(str(post_id), False),
                            }
                        )
                    except Exception:
                        # If we can't get post info, just include basic data
                        post_details.append(
                            {
                                "id": post_id,
                                "title": f"Post {post_id}",
                                "upvoted": upvote_status.get(str(post_id), False),
                            }
                        )

                validation_data = {
                    "username": username,
                    "post_ids": expected_data.post_ids,
                    "posts_checked": len(expected_data.post_ids),
                    "upvoted_count": upvoted_count,
                    "all_upvoted": all_upvoted,
                    "post_details": post_details,
                }

                if all_upvoted:
                    return WebarenaTaskEvalResult.create_success(
                        assertion_msgs=[
                            f"User {username} has successfully upvoted all {len(expected_data.post_ids)} specified posts"
                        ],
                        site=resource.website_type,
                        validation_data=validation_data,
                        task_id=task.task_id,
                    )
                else:
                    not_upvoted = [
                        post_id
                        for post_id in expected_data.post_ids
                        if not upvote_status.get(str(post_id), False)
                    ]
                    return WebarenaTaskEvalResult.create_failed(
                        assertion_msgs=[
                            f"User {username} has only upvoted {upvoted_count} of {len(expected_data.post_ids)} specified posts. "
                            f"Missing upvotes on posts: {not_upvoted}"
                        ],
                        site=resource.website_type,
                        validation_data={
                            **validation_data,
                            "not_upvoted": not_upvoted,
                        },
                        task_id=task.task_id,
                    )

            finally:
                service.close()

        except Exception as e:
            return WebarenaTaskEvalResult.create_failed(
                assertion_msgs=[f"Post upvote validation failed: {str(e)}"],
                site=resource.website_type,
                task_id=task.task_id,
            )

    def validate_user_downvoted_posts(
        self,
        task: WebArenaTask,
        task_result: WebArenaTaskResponse,
        resource: AllocationResource,
        eval_func: EvalFunc,
        **kwargs,
    ) -> WebarenaTaskEvalResult:
        """
        Validate that a user has downvoted specific posts by their IDs.

        This is a consolidated validation function that replaces the need for
        validate_user_downvoted_top_posts and validate_user_downvoted_all_user_posts
        by accepting a list of post IDs to check.

        Expected data format: {"post_ids": [123, 456, 789]}
        Username is obtained from resource.username
        """
        try:
            # Extract expected_data from eval_func and validate with Pydantic
            if not eval_func or not hasattr(eval_func, "expected_data"):
                return WebarenaTaskEvalResult.create_failed(
                    assertion_msgs=["No expected_data found in eval_func"],
                    site=resource.website_type,
                    task_id=task.task_id,
                )

            try:
                expected_data = PostIdsDownvoteExpectedData.model_validate(
                    eval_func.expected_data
                )
            except Exception as e:
                return WebarenaTaskEvalResult.create_failed(
                    assertion_msgs=[f"Invalid expected_data format: {str(e)}"],
                    site=resource.website_type,
                    task_id=task.task_id,
                )

            # Extract username from resource
            username = resource.username

            # Validate post_ids is not empty
            if not expected_data.post_ids:
                return WebarenaTaskEvalResult.create_failed(
                    assertion_msgs=["No post IDs provided for downvote validation"],
                    site=resource.website_type,
                    task_id=task.task_id,
                )

            # Create RedditConfig from AllocationResource
            config = RedditConfig(container_name=resource.container_name)

            # Create service with resource-specific config
            service = RedditDataHelper(config)

            try:
                # Check if user downvoted all specified posts
                # Convert post_ids to strings as the service expects List[str]
                post_ids_str = [str(pid) for pid in expected_data.post_ids]
                downvote_status = service.check_user_voted_on_posts(
                    username, post_ids_str, upvote=False
                )

                # Count how many were downvoted
                downvoted_count = sum(downvote_status.values())
                all_downvoted = downvoted_count == len(expected_data.post_ids)

                # Get post details for validation data
                post_details = []
                for post_id in expected_data.post_ids:
                    try:
                        # Try to get post info for better reporting
                        post_info = service.get_post_by_id(str(post_id))
                        post_details.append(
                            {
                                "id": post_id,
                                "title": post_info.title
                                if post_info
                                else f"Post {post_id}",
                                "downvoted": downvote_status.get(str(post_id), False),
                            }
                        )
                    except Exception:
                        # If we can't get post info, just include basic data
                        post_details.append(
                            {
                                "id": post_id,
                                "title": f"Post {post_id}",
                                "downvoted": downvote_status.get(str(post_id), False),
                            }
                        )

                validation_data = {
                    "username": username,
                    "post_ids": expected_data.post_ids,
                    "posts_checked": len(expected_data.post_ids),
                    "downvoted_count": downvoted_count,
                    "all_downvoted": all_downvoted,
                    "post_details": post_details,
                }

                if all_downvoted:
                    return WebarenaTaskEvalResult.create_success(
                        assertion_msgs=[
                            f"User {username} has successfully downvoted all {len(expected_data.post_ids)} specified posts"
                        ],
                        site=resource.website_type,
                        validation_data=validation_data,
                        task_id=task.task_id,
                    )
                else:
                    not_downvoted = [
                        post_id
                        for post_id in expected_data.post_ids
                        if not downvote_status.get(str(post_id), False)
                    ]
                    return WebarenaTaskEvalResult.create_failed(
                        assertion_msgs=[
                            f"User {username} has only downvoted {downvoted_count} of {len(expected_data.post_ids)} specified posts. "
                            f"Missing downvotes on posts: {not_downvoted}"
                        ],
                        site=resource.website_type,
                        validation_data={
                            **validation_data,
                            "not_downvoted": not_downvoted,
                        },
                        task_id=task.task_id,
                    )

            finally:
                service.close()

        except Exception as e:
            return WebarenaTaskEvalResult.create_failed(
                assertion_msgs=[f"Post downvote validation failed: {str(e)}"],
                site=resource.website_type,
                task_id=task.task_id,
            )

    def validate_reply_exists_for_comment(
        self,
        task: WebArenaTask,
        task_result: WebArenaTaskResponse,
        resource: AllocationResource,
        eval_func: EvalFunc,
        **kwargs,
    ) -> WebarenaTaskEvalResult:
        """
        Validate that a reply exists for a specific comment.

        Expected eval_func format:
        {
            "name": "validate_reply_exists_for_comment",
            "expected_data": {
                "comment_id": "1235250",
                "reply_content": "thanks! I am a big fan of your website."  # Optional: check if reply contains this text
            }
        }
        The reply author is taken from resource.username.
        """
        try:
            # Extract expected_data from eval_func and validate with Pydantic
            if not eval_func or not hasattr(eval_func, "expected_data"):
                return WebarenaTaskEvalResult.create_failed(
                    assertion_msgs=["No expected_data found in eval_func"],
                    site=resource.website_type,
                    task_id=task.task_id,
                )

            try:
                expected_data = ReplyExistsExpectedData.model_validate(
                    eval_func.expected_data
                )
            except Exception as e:
                return WebarenaTaskEvalResult.create_failed(
                    assertion_msgs=[f"Invalid expected_data format: {str(e)}"],
                    site=resource.website_type,
                    task_id=task.task_id,
                )

            # Get the username from the allocation resource
            username = resource.username

            # Create RedditConfig from AllocationResource
            config = RedditConfig(container_name=resource.container_name)

            # Create service with resource-specific config
            service = RedditDataHelper(config)

            try:
                # Check if reply exists for the comment by the specified user
                reply_exists = service.reply_exists_for_comment(
                    comment_id=expected_data.comment_id,
                    author=username,
                    content_contains=expected_data.reply_content,
                )

                if reply_exists:
                    return WebarenaTaskEvalResult.create_success(
                        assertion_msgs=[
                            f"Reply by {username} found for comment {expected_data.comment_id}"
                            + (
                                " with expected content"
                                if expected_data.reply_content
                                else ""
                            )
                        ],
                        site=resource.website_type,
                        task_id=task.task_id,
                        validation_data={
                            "comment_id": expected_data.comment_id,
                            "reply_content": expected_data.reply_content,
                            "author": username,
                            "exists": True,
                        },
                    )
                else:
                    return WebarenaTaskEvalResult.create_failed(
                        assertion_msgs=[
                            f"Reply by {username} not found for comment {expected_data.comment_id}"
                            + (
                                " with expected content"
                                if expected_data.reply_content
                                else ""
                            )
                        ],
                        site=resource.website_type,
                        task_id=task.task_id,
                        validation_data={
                            "comment_id": expected_data.comment_id,
                            "reply_content": expected_data.reply_content,
                            "author": username,
                            "exists": False,
                        },
                    )

            finally:
                service.close()

        except Exception as e:
            return WebarenaTaskEvalResult.create_failed(
                assertion_msgs=[f"Reply validation failed: {str(e)}"],
                site=resource.website_type,
                task_id=task.task_id,
            )

    def validate_post_image_file_name(
        self,
        task: WebArenaTask,
        task_result: WebArenaTaskResponse,
        resource: AllocationResource,
        eval_func: EvalFunc,
        **kwargs,
    ) -> WebarenaTaskEvalResult:
        """
        Validate that a post exists in a subreddit with a specific image file name.

        Expected eval_func format:
        {
            "name": "validate_post_image_file_name",
            "expected_data": {
                "subreddit": "pics",
                "file_name": "b02113033af32feae9ff147dbbe3764039368d67d193885bd04e65c2e6beea9c.jpg",
                "author": "username",  # Optional: specific author
                "title": "My Image"     # Optional: title must contain this text
            }
        }
        If author is not provided, it uses resource.username
        """
        try:
            # Extract expected_data from eval_func and validate with Pydantic
            if not eval_func or not hasattr(eval_func, "expected_data"):
                return WebarenaTaskEvalResult.create_failed(
                    assertion_msgs=["No expected_data found in eval_func"],
                    site=resource.website_type,
                    task_id=task.task_id,
                )

            try:
                expected_data = PostImageFileNameExpectedData.model_validate(
                    eval_func.expected_data
                )
            except Exception as e:
                return WebarenaTaskEvalResult.create_failed(
                    assertion_msgs=[f"Invalid expected_data format: {str(e)}"],
                    site=resource.website_type,
                    task_id=task.task_id,
                )

            # Extract username from resource if not provided in expected_data
            author = expected_data.author or resource.username

            # Create RedditConfig from AllocationResource
            config = RedditConfig(container_name=resource.container_name)

            # Create service with resource-specific config
            service = RedditDataHelper(config)

            try:
                # Check if post exists with the specified image file name
                post_exists = service.post_exists_with_image_file_name(
                    subreddit=expected_data.subreddit,
                    file_name=expected_data.file_name,
                    author=author,
                    title_contains=expected_data.title,
                )

                if post_exists:
                    return WebarenaTaskEvalResult.create_success(
                        assertion_msgs=[
                            f"Post found in subreddit {expected_data.subreddit} with image file name '{expected_data.file_name}'"
                        ],
                        site=resource.website_type,
                        task_id=task.task_id,
                        validation_data={
                            "subreddit": expected_data.subreddit,
                            "file_name": expected_data.file_name,
                            "author": author,
                            "title": expected_data.title,
                            "exists": True,
                        },
                    )
                else:
                    return WebarenaTaskEvalResult.create_failed(
                        assertion_msgs=[
                            f"No post found in subreddit {expected_data.subreddit} with image file name '{expected_data.file_name}'"
                        ],
                        site=resource.website_type,
                        task_id=task.task_id,
                        validation_data={
                            "subreddit": expected_data.subreddit,
                            "file_name": expected_data.file_name,
                            "author": author,
                            "title": expected_data.title,
                            "exists": False,
                        },
                    )

            finally:
                service.close()

        except Exception as e:
            return WebarenaTaskEvalResult.create_failed(
                assertion_msgs=[f"Image file name validation failed: {str(e)}"],
                site=resource.website_type,
                task_id=task.task_id,
            )
