#!/usr/bin/env python3
import json
import os
from os import path
import time
import signal
import shutil
from subprocess import check_call, Popen
import argparse
import requests
import yaml
import sys

client_init_num = 0
CLIENT_DIR = './weights/running'
CLIENT_FILENAME = 'client_init_num.txt'


def reload_client_init_num():
    global client_init_num
    if not os.path.exists(CLIENT_DIR):
        return

    pathname = os.path.join(CLIENT_DIR, CLIENT_FILENAME)
    if os.path.exists(pathname):
        with open(pathname, 'r') as f:
            client_init_num = int(f.read()) + 1


def save_client_init_num():
    if not os.path.exists(CLIENT_DIR):
        os.makedirs(CLIENT_DIR)

    pathname = os.path.join(CLIENT_DIR, CLIENT_FILENAME)
    with open(pathname, 'w') as f:
        f.write(str(client_init_num))


def run_offline_media_servers(yaml_settings):
    run_server_html_cmd = 'python3.7 ./src/portal/manage.py runserver 0:8080'
    p1 = Popen(run_server_html_cmd,
               shell=True, preexec_fn=os.setsid)
    time.sleep(3)
    run_servers_cmd = './src/media-server/run_servers {} {}'.format(
        yaml_settings, client_init_num)
    p2 = Popen(run_servers_cmd, shell=True, preexec_fn=os.setsid)
    time.sleep(8)
    return p1, p2


def start_mahimahi_clients(num_clients, trace_dir, epochs, yaml_settings):
    global client_init_num
    # create a temporary directory to store Chrome sessions
    chrome_sessions = "chrome-sessions"
    if not path.exists(chrome_sessions):
        os.makedirs(chrome_sessions)

    plist = []
    try:
        files = sorted(os.listdir(trace_dir))
        print('Running {} epochs on {} files'.format(
            epochs, len(files)), flush=True)
        for epoch_idx in range(epochs):
            for file_idx in range(0, len(files), num_clients):
                base_port = 9361
                remote_base_port = 10000

                save_client_init_num()
                print('loaded client_init_num', client_init_num)

                plist = list(run_offline_media_servers(yaml_settings))

                for i in range(1, num_clients + 1):
                    port = base_port + i
                    remote_debugging_port = remote_base_port + i

                    client_file_idx = file_idx + i - 1
                    if client_file_idx > len(files) - 1:
                        break
                    filename = files[client_file_idx]

                    print('file {}/{}, epoch {}/{}'.format(client_file_idx,
                                                           len(files), epoch_idx, epochs), end='\t', flush=True)

                    # downlink: trace_dir/TRACE (to transmit video)
                    # uplink: 12 Mbps (fast enough for ack messages)
                    mahimahi_chrome_cmd = "mm-delay 40 mm-link ./src/media-server/12mbps {}/{} -- sh -c 'chromium-browser --headless --disable-gpu --remote-debugging-port={} http://$MAHIMAHI_BASE:8080/player/?wsport={} --user-data-dir={}/{}.profile'".format(
                        trace_dir, filename, remote_debugging_port, port, chrome_sessions, port)
                    print(mahimahi_chrome_cmd, flush=True)

                    p = Popen(mahimahi_chrome_cmd.encode('utf-8'),
                              shell=True, preexec_fn=os.setsid)
                    plist.append(p)
                    time.sleep(3)  # don't know why but seems necessary

                time.sleep(60 * 8)

                client_init_num += 2

                for p in plist:
                    os.killpg(os.getpgid(p.pid), signal.SIGTERM)
                time.sleep(20)

                shutil.rmtree(chrome_sessions, ignore_errors=True)
    except Exception as e:
        print("exception: " + str(e))
        pass
    finally:
        for p in plist:
            os.killpg(os.getpgid(p.pid), signal.SIGTERM)
            shutil.rmtree(chrome_sessions, ignore_errors=True)
        time.sleep(5)


def main():
    parser = argparse.ArgumentParser(description="Tester")
    parser.add_argument(
        "--yaml-settings",
        default='./src/settings.yml'
    )
    parser.add_argument(
        "--trace-dir",
        default='/home/ubuntu/traces/mahimahi/'
    )
    parser.add_argument(
        "-t",
        "--test",
        default=False,
        action='store_true'
    )
    parser.add_argument(
        "--epochs",
        type=int,
        default=1000
    ),
    parser.add_argument(
        "-f",
        "--force",
        default=False,
        action='store_true'
    )
    parser.add_argument(
        "-c",
        "--clients",
        type=int,
        default=0,
    )
    args = parser.parse_args()

    with open(args.yaml_settings, 'r') as fh:
        yaml_settings = yaml.safe_load(fh)

    clients = yaml_settings["experiments"][0]['num_servers'] if args.clients == 0 else args.clients
    training_mode = not args.test
    trace_dir = args.trace_dir + ('train' if training_mode else 'test')
    epochs = 1 if args.test else args.epochs

    if not os.path.isdir('./logs'):
        os.mkdir('./logs')
    sys.stdout = open('./logs/offline_test.txt', 'w')

    print('running on traces dir={}, training mode={}'.format(
          trace_dir, training_mode), flush=True)
    print('running experiment {}'.format(
        yaml_settings["experiments"]), flush=True)

    check_call('sudo sysctl -w net.ipv4.ip_forward=1', shell=True)

    if not args.force:
        reload_client_init_num()
    start_mahimahi_clients(clients, trace_dir, epochs, args.yaml_settings)


if __name__ == '__main__':
    main()
