#include <algorithm>
#include <assert.h>
#include <cmath>
#include <iostream>
#include <fstream>

#include "filesystem.hh"
#include "kmeans.hh"
#include "json.hpp"

using json = nlohmann::json;

Kmeans::Kmeans(std::string kmeans_path)
{
    load_clusters(fs::path(kmeans_path) / "clusters.json");
    mean_ = read_file(fs::path(kmeans_path) / "mean.txt");
    std_ = read_file(fs::path(kmeans_path) / "std.txt");
}

std::vector<double> Kmeans::read_file(fs::path filename)
{
    std::ifstream file(filename);
    std::vector<double> result;
    double a;

    while (file >> a)
    {
        result.push_back(a);
    }

    return result;
}

void Kmeans::normalize_inplace(std::vector<double> &input)
{
    assert(input.size() == mean_.size());
    assert(input.size() == std_.size());

    for (std::size_t i = 0; i < input.size(); i++)
    {
        input[i] = (input[i] - mean_[i]) / std_[i];
    }
}

std::size_t Kmeans::find_optimal_cluster(std::vector<double> input)
{
    normalize_inplace(input);

    // find min cluster
    const auto &min_cluster = *std::min_element(
        clusters_.begin(),
        clusters_.end(),
        [&](std::pair<int, std::vector<double>> elem1, std::pair<int, std::vector<double>> elem2)
        {
            return dist(input, elem1.second) < dist(input, elem2.second);
        });

    return min_cluster.first;
}

double Kmeans::dist(std::vector<double> v1, std::vector<double> v2)
{
    assert(v1.size() == v2.size());
    double distance = 0;

    for (std::size_t i = 0; i < v1.size(); i++)
    {
        // if (std::pow(v1[i] - v2[i], 2) > 1e4) {
        //   distance += 1e10;
        //   std::cout << "input:" << v1[i] << "cluster: " << v2[i] << std::endl;
        //   continue;
        // }

        distance += std::pow(v1[i] - v2[i], 2);
    }

    return std::sqrt(distance);
}

void Kmeans::load_clusters(fs::path clusters_path)
{
    std::ifstream ifs(clusters_path);
    json json_data = json::parse(ifs);

    for (auto &el : json_data.items())
    {
        auto cluster = el.value().get<std::vector<double>>();
        clusters_[stoi(el.key())] = cluster;
    }
}

std::size_t Kmeans::calc_inner_index(std::size_t cluster, 
                                         std::size_t dis_buffer)
{
  return cluster + dis_buffer; // buf_size_
}