import os
import shutil
import numpy as np
import argparse
import matplotlib.pyplot as plt


FILE_SIZE = 2000
BYTES_PER_PKT = 1500.0
MILLISEC_IN_SEC = 1000.0
EXP_LEN = 5000.0  # millisecond
MAX_TRACE_TIME = (10*60*1000)


def validate_dir(dir):
    if os.path.exists(dir):
        shutil.rmtree(dir)

    subdirs = dir.split('/')
    dir = '.'
    for subdir in subdirs:
        dir += '/' + subdir
        if os.path.exists(dir):
            continue

        os.mkdir(dir)


def convert_file(trace_file, output_file):
    with open(trace_file, 'r') as f, open(output_file, 'w') as mf:
        millisec_time = 0
        mf.write(str(millisec_time) + '\n')
        for line in f:
            throughput = float(line.split()[0])
            pkt_per_millisec = throughput / BYTES_PER_PKT / MILLISEC_IN_SEC

            millisec_count = 0
            pkt_count = 0
            while True:
                millisec_count += 1
                millisec_time += 1
                to_send = (millisec_count * pkt_per_millisec) - pkt_count
                to_send = np.floor(to_send)

                for i in range(int(to_send)):
                    if millisec_time > MAX_TRACE_TIME:
                        return

                    mf.write(str(millisec_time) + '\n')

                pkt_count += to_send

                if millisec_count >= EXP_LEN:
                    break


def convert_to_mb(args, sub_dir, files):
    mahimahi_dir = args.cooked + sub_dir
    mahimahi_mb_dir = 'mahimahi_mb/' + sub_dir
    validate_dir(mahimahi_mb_dir)

    for trace_file in files:
        with open(mahimahi_dir + trace_file, 'r') as f, open(mahimahi_mb_dir + trace_file, 'w') as o:
            for line in f:
                t = float(line)
                o.write(str(t/(1000000.0 / 8.0)) + '\n')


def plot_variance(args, files):
    vars = []
    for trace_file in files:
        file_var = []
        with open(args.output + trace_file, 'r') as f:
            v = None
            for line in f:
                if v is None:
                    v = float(line)
                    continue

                file_var.append(abs(float(line) - v))
                v = float(line)
        a = np.array(file_var)
        vars.append(a.mean())

    a = np.array(vars)
    hist, bins = np.histogram(vars, 100)

    fig, ax = plt.subplots()
    ax.plot(bins[1:], hist)
    plt.savefig('./files_variances.png', bbox_inches='tight')


def convert_pensieve_traces(args, files):
    for trace_file in files:
        with open(args.cooked + trace_file, 'r') as f, open(args.output + trace_file, 'w') as o:
            for line in f:
                t = float(line.split('\t')[-1])
                o.write(str(t) + '\n')


def parse_dir(input_dir, output_dir):
    files = os.listdir(input_dir)

    i = 0
    for trace_file in files:
        # if os.stat(input_dir + trace_file).st_size >= FILE_SIZE:
        input = input_dir + trace_file
        output = output_dir + trace_file
        convert_file(input, output)

        i += 1
        print(str(i) + '/' + str(len(files)))


def main():
    parser = argparse.ArgumentParser(description="Traces converter")
    parser.add_argument(
        "--cooked",
        default='./traces/',
        help='cooked traces dir'
    )
    parser.add_argument(
        "--output",
        default='./mahimahi/',
        help='mahimahi trace dir'
    )

    args = parser.parse_args()

    validate_dir(args.output + 'test/')
    validate_dir(args.output + 'train/')
    parse_dir(args.cooked + 'test/', args.output + 'test/')
    parse_dir(args.cooked + 'train/', args.output + 'train/')


if __name__ == '__main__':
    main()
