import numpy as np
import numpy.typing as npt

N_COVERAGE_SAMPLES = 100
N_FPR_SAMPLES = 100
N_MU_SAMPLES = 360
N_RECALL_SAMPLES = 100
ALPHA_PARAMETERIZATION = True


class Metrics:
    """Metrics to evaluate OOD detector given a single uncertainty scores.
    """

    def __init__(self,
                 y_true: npt.NDArray[int],
                 y_pred: npt.NDArray[int],
                 score: npt.NDArray[float]):
        """
        Initializes the object and computes all achievable values of:
            1) selective risk, 
            2) TPR (a.k.a. coverage or recall),
            3) FPR, 
            4) Precision, 
            5) CCR (correct-classification-rate, a.k.a. 1-selective risk for the 0/1 loss), 
            6) areas under curves defined by the above, e.g., AUROC, AUPR, OSCR ...

        Args:
            y_true (npt.NDArray[np.int]): Array of groundtruth labels. For ID samples, the class labels are [0, 1, 2, ...]. The value of -1 corresponds to OOD samples.
            y_pred (npt.NDArray[np.int]): Array of predicted class labels.
            score (npt.NDArray[np.float]): Array of scores obtained by some OOD detection method.
        """
        y_pred[y_true == -1] = -1

        # 0/1 loss
        loss = np.array(y_true != y_pred, dtype=np.float16)

        # sort scores and losses according to score
        indices = np.argsort(score)
        sorted_loss = loss[indices]
        sorted_y_true = y_true[indices]

        # number of samples
        n = len(y_true)
        n_in = np.sum(y_true != -1)
        n_out = n-n_in

        # initialize empty arrays to hold the metric values at different thresholds
        self.sel_risk = np.zeros(n+1)
        self.coverage = np.zeros(n+1)
        self.fpr = np.zeros(n+1)
        self.prec = np.zeros(n+1)
        self.score = score[indices]
        cur_sel_risk = 0
        cur_coverage = 0
        cur_fpr = 0
        cur_tp = 0

        self.prec[0] = np.NaN
        self.ccr = np.zeros(n+1)
        self.ccr[0] = np.NaN
        cur_ccr = 0

        # loop over all thresholds (defined by sample scores) and fill in the metric arrays
        for i in range(n):
            if sorted_y_true[i] == -1:
                cur_fpr += 1
            else:
                cur_tp += 1
                cur_coverage += 1
                cur_sel_risk += sorted_loss[i]
                cur_ccr += 1-sorted_loss[i]

            self.coverage[i+1] = cur_coverage / n_in
            self.sel_risk[i+1] = cur_sel_risk / \
                cur_coverage if cur_coverage > 0 else 0.0
            self.ccr[i+1] = cur_ccr / cur_coverage if cur_coverage > 0 else 1.0
            self.fpr[i+1] = cur_fpr / n_out if n_out > 0 else 0.0
            self.prec[i+1] = cur_tp / (i+1)

        self.y_true = sorted_y_true
        self.loss = sorted_loss
        self.AUROC = np.trapz(self.coverage[1:], self.fpr[1:])
        self.AUPR = np.trapz(self.prec[1:], self.coverage[1:])
        self.AURF = np.trapz(self.sel_risk[1:], self.fpr[1:])
        self.AURC = np.trapz(self.sel_risk[1:], self.coverage[1:])
        self.tpr = self.coverage
        self.recall = self.coverage
        self.OSCR = np.trapz(self.ccr[1:], self.fpr[1:])

    def risk_coverage_curve_at_fpr(self,
                                   fpr: float,
                                   n_coverage_samples: int = N_COVERAGE_SAMPLES) -> [npt.NDArray[float], npt.NDArray[float]]:
        """
        Computes the Risk-Coverage curve, constrained by maximal false-positive-rate.
        I.e., returns only the section of the Risk-Coverage curve, where the constraint is satisfied.

        Args:
            fpr (float): Maximal admissible false-positive-rate.
            n_coverage_samples (int, optional): Sampling parameter for the X-axis. Defaults to N_COVERAGE_SAMPLES.

        Returns:
            [npt.NDArray[np.float], npt.NDArray[np.float]]: Risk and Coverage arrays, i.e., Y-axis values, X-axis values for the Risk-Coverage curve.
        """

        coverage = np.linspace(0, 1, n_coverage_samples)
        sel_risk = np.ones_like(coverage)*np.Inf
        for i, cov in enumerate(coverage):
            ind = np.argwhere((self.fpr <= fpr) & (self.coverage >= cov))
            sel_risk[i] = min((sel_risk[i], np.min(self.sel_risk[ind]))) if len(
                ind) > 0 else sel_risk[i]

        ind = np.isfinite(sel_risk)
        sel_risk = sel_risk[ind]
        coverage = coverage[ind]
        return sel_risk, coverage

    def risk_coverage_curve_at_prec(self,
                                    prec: float,
                                    n_coverage_samples: int = N_COVERAGE_SAMPLES) -> [npt.NDArray[float], npt.NDArray[float]]:
        """
        Computes the Risk-Coverage curve, constrained by minimal precision.
        I.e., returns only the section of the Risk-Coverage curve, where the constraint is satisfied.

        Args:
            prec (float): Minimal admissible precision.
            n_coverage_samples (int, optional): Sampling parameter for the X-axis. Defaults to N_COVERAGE_SAMPLES.

        Returns:
            [npt.NDArray[np.float], npt.NDArray[np.float]]: Risk and Coverage arrays, i.e., Y-axis values, X-axis values for the Risk-Coverage curve.
        """

        coverage = np.linspace(0, 1, n_coverage_samples)
        sel_risk = np.ones_like(coverage)*np.Inf
        for i, cov in enumerate(coverage):
            ind = np.argwhere((self.prec >= prec) & (self.coverage >= cov))
            sel_risk[i] = min((sel_risk[i], np.min(self.sel_risk[ind]))) if len(
                ind) > 0 else sel_risk[i]

        ind = np.isfinite(sel_risk)
        sel_risk = sel_risk[ind]
        coverage = coverage[ind]
        return sel_risk, coverage


class MetricsDouble:
    """
    Class to compute Metrics to evaluate OOD detector given by two uncertainty scores.
    """

    def __init__(self,
                 y_true: npt.NDArray[int],
                 y_pred: npt.NDArray[int],
                 scoreA: npt.NDArray[float],
                 scoreB: npt.NDArray[float]):
        """
        Initializes the object, saving the inputs but performing no additonal computation.

        Args:
            y_true (npt.NDArray[np.int]): Array of groundtruth labels. For ID samples, the class labels are [0, 1, 2, ...]. The value of -1 corresponds to OOD samples.
            y_pred (npt.NDArray[np.int]): Array of predicted class labels.
            scoreA (npt.NDArray[np.float]): Array of scores obtained by some OOD detection method.
            scoreB (npt.NDArray[np.float]): Array of scores obtained by some OOD detection method.
        """
        self.y_true = y_true
        self.y_pred = y_pred
        self.scoreA = scoreA
        self.scoreB = scoreB

        self.y_pred[y_true == -1] = -1

    def risk_coverage_curve_at_prec(self,
                                    prec: float,
                                    n_coverage_samples: int = N_COVERAGE_SAMPLES,
                                    n_mu_samples: int = N_MU_SAMPLES,
                                    use_alpha_parameterization: bool = ALPHA_PARAMETERIZATION) -> [npt.NDArray[float], npt.NDArray[float]]:
        """
        Computes the Risk-Coverage curve, constrained by minimal precision. 
        For each possible value of coverage, the optimal (minimizing selective risk) selective classifier is found among all selective classifiers parameterized by the sampled mu parameter.

        Args:
            prec (float): Minimal admissible precision.
            n_coverage_samples (int, optional): Sampling parameter for the X-axis. Defaults to N_COVERAGE_SAMPLES.
            n_mu_samples (int, optional): Sampling parameter for optimal selective function search. Defaults to N_MU_SAMPLES.
            use_alpha_parameterization (bool, optional): If True, the selective function is parameterized by the angle of the separating hyperplane. Defaults to ALPHA_PARAMETERIZATION.

        Returns:
            [npt.NDArray[np.float], npt.NDArray[np.float]]: Risk and Coverage arrays, i.e., Y-axis values, X-axis values for the Risk-Coverage curve.
        """
        coverage = np.linspace(0, 1, n_coverage_samples)
        sel_risk = np.ones_like(coverage)*np.Inf

        mu_samples = np.linspace(0, 1, n_mu_samples)
        if use_alpha_parameterization:
            mu_samples = [i for i in np.linspace(
                0, 2*np.pi, n_mu_samples)] + [np.pi/2., np.pi, np.pi*3./2.]

        for mu in mu_samples:
            score = (1-mu)*self.scoreA+mu*self.scoreB
            if use_alpha_parameterization:
                score = np.cos(mu)*self.scoreA + np.sin(mu)*self.scoreB

            metric = Metrics(self.y_true, self.y_pred, score)
            for i, cov in enumerate(coverage):
                ind = np.argwhere((metric.prec >= prec) &
                                  (metric.coverage >= cov))
                sel_risk[i] = min((sel_risk[i], np.min(metric.sel_risk[ind]))) if len(
                    ind) > 0 else sel_risk[i]

        ind = np.isfinite(sel_risk)
        sel_risk = sel_risk[ind]
        coverage = coverage[ind]
        return sel_risk, coverage

    def risk_coverage_curve_at_fpr(self,
                                   fpr: float,
                                   n_coverage_samples: int = N_COVERAGE_SAMPLES,
                                   n_mu_samples: int = N_MU_SAMPLES,
                                   use_alpha_parameterization: bool = ALPHA_PARAMETERIZATION) -> [npt.NDArray[float], npt.NDArray[float]]:
        """
        Computes the Risk-Coverage curve, constrained by maximal false-positive-rate. 
        For each possible value of coverage, the optimal (minimizing selective risk) selective classifier is found among all selective classifiers parameterized by the sampled mu parameter.

        Args:
            fpr (float): Maximal admissible false-positive-rate.
            n_coverage_samples (int, optional): Sampling parameter for the X-axis. Defaults to N_COVERAGE_SAMPLES.
            n_mu_samples (int, optional): Sampling parameter for optimal selective function search. Defaults to N_MU_SAMPLES.
            use_alpha_parameterization (bool, optional): If True, the selective function is parameterized by the angle of the separating hyperplane. Defaults to ALPHA_PARAMETERIZATION.

        Returns:
            [npt.NDArray[np.float], npt.NDArray[np.float]]: Risk and Coverage arrays, i.e., Y-axis values, X-axis values for the Risk-Coverage curve.
        """
        coverage = np.linspace(0, 1, n_coverage_samples)
        sel_risk = np.ones_like(coverage)*np.Inf

        mu_samples = np.linspace(0, 1, n_mu_samples)
        if use_alpha_parameterization:
            mu_samples = [i for i in np.linspace(
                0, 2*np.pi, n_mu_samples)] + [np.pi/2., np.pi, np.pi*3./2.]

        for mu in mu_samples:
            score = (1-mu)*self.scoreA+mu*self.scoreB
            if use_alpha_parameterization:
                score = np.cos(mu)*self.scoreA + np.sin(mu)*self.scoreB

            metric = Metrics(self.y_true, self.y_pred, score)
            for i, cov in enumerate(coverage):
                ind = np.argwhere((metric.fpr <= fpr) &
                                  (metric.coverage >= cov))
                sel_risk[i] = min((sel_risk[i], np.min(metric.sel_risk[ind]))) if len(
                    ind) > 0 else sel_risk[i]

        ind = np.isfinite(sel_risk)
        sel_risk = sel_risk[ind]
        coverage = coverage[ind]
        return sel_risk, coverage

    def risk_fpr_curve_at_coverage(self,
                                   coverage: float,
                                   n_fpr_samples: int = N_FPR_SAMPLES,
                                   n_mu_samples: int = N_MU_SAMPLES,
                                   use_alpha_parameterization: bool = ALPHA_PARAMETERIZATION) -> [npt.NDArray[float], npt.NDArray[float]]:
        """
        Computes the Risk-FPR curve, constrained by minimal coverage (TPR). 
        For each possible value of FPR, the optimal (minimizing selective risk) selective classifier is found among all selective classifiers parameterized by the sampled mu parameter.

        Args:
            coverage (float): Minimal admissible true-positive-rate.
            n_fpr_samples (int, optional): Sampling parameter for the X-axis. Defaults to N_COVERAGE_SAMPLES.
            n_mu_samples (int, optional): Sampling parameter for optimal selective function search. Defaults to N_MU_SAMPLES.
            use_alpha_parameterization (bool, optional): If True, the selective function is parameterized by the angle of the separating hyperplane. Defaults to ALPHA_PARAMETERIZATION.

        Returns:
            [npt.NDArray[np.float], npt.NDArray[np.float]]: Risk and FPR arrays, i.e., Y-axis values, X-axis values for the Risk-FPR curve.
        """

        fpr = np.linspace(0, 1, n_fpr_samples)
        sel_risk = np.ones_like(fpr)*np.Inf

        mu_samples = np.linspace(0, 1, n_mu_samples)
        if use_alpha_parameterization:
            mu_samples = [i for i in np.linspace(
                0, 2*np.pi, n_mu_samples)] + [np.pi/2., np.pi, np.pi*3./2.]

        for mu in mu_samples:
            score = (1-mu)*self.scoreA+mu*self.scoreB
            if use_alpha_parameterization:
                score = np.cos(mu)*self.scoreA + np.sin(mu)*self.scoreB

            metric = Metrics(self.y_true, self.y_pred, score)
            for i, f in enumerate(fpr):
                ind = np.argwhere((metric.fpr <= f) & (
                    metric.coverage >= coverage))
                sel_risk[i] = min((sel_risk[i], np.min(metric.sel_risk[ind]))) if len(
                    ind) > 0 else sel_risk[i]

        return sel_risk, fpr

    def tpr_vs_fpr(self,
                   n_fpr_samples: int = N_FPR_SAMPLES,
                   n_mu_samples: int = N_MU_SAMPLES,
                   use_alpha_parameterization: bool = ALPHA_PARAMETERIZATION) -> [npt.NDArray[float], npt.NDArray[float], float]:
        """
        Computes the TPR-FPR, a.k.a. ROC curve. 
        For each possible value of FPR, the optimal (maximizing TPR) selective function is found among all selective classifiers parameterized by the sampled mu parameter.

        Args:
            n_fpr_samples (int, optional): Sampling parameter for the X-axis. Defaults to N_COVERAGE_SAMPLES.
            n_mu_samples (int, optional): Sampling parameter for optimal selective function search. Defaults to N_MU_SAMPLES.
            use_alpha_parameterization (bool, optional): If True, the selective function is parameterized by the angle of the separating hyperplane. Defaults to ALPHA_PARAMETERIZATION.

        Returns:
            [npt.NDArray[np.float], npt.NDArray[np.float], float]: TPR and FPR arrays and area-under-the-curve, i.e., Y-axis values, X-axis values for the ROC curve and AUROC.
        """
        fpr = np.linspace(0, 1, n_fpr_samples)
        tpr = np.zeros_like(fpr)

        mu_samples = np.linspace(0, 1, n_mu_samples)
        if use_alpha_parameterization:
            mu_samples = [i for i in np.linspace(
                0, 2*np.pi, n_mu_samples)] + [np.pi/2., np.pi, np.pi*3./2.]

        for mu in mu_samples:
            score = (1-mu)*self.scoreA+mu*self.scoreB
            if use_alpha_parameterization:
                score = np.cos(mu)*self.scoreA + np.sin(mu)*self.scoreB

            metric = Metrics(self.y_true, self.y_pred, score)
            for i, f in enumerate(fpr):
                ind = np.argwhere((metric.fpr <= f))
                tpr[i] = max(tpr[i], np.max(metric.coverage[ind]))

        auc = np.trapz(tpr, fpr)
        return tpr, fpr, auc

    def ccr_vs_fpr(self,
                   n_fpr_samples: int = N_FPR_SAMPLES,
                   n_mu_samples: int = N_MU_SAMPLES,
                   use_alpha_parameterization: bool = ALPHA_PARAMETERIZATION) -> [npt.NDArray[float], npt.NDArray[float], float]:
        """
        Computes the CCR-FPR curve. 
        For each possible value of FPR, the optimal (maximizing CCR) selective function is found among all selective classifiers parameterized by the sampled mu parameter.

        Args:
            n_fpr_samples (int, optional): Sampling parameter for the X-axis. Defaults to N_COVERAGE_SAMPLES.
            n_mu_samples (int, optional): Sampling parameter for optimal selective function search. Defaults to N_MU_SAMPLES.
            use_alpha_parameterization (bool, optional): If True, the selective function is parameterized by the angle of the separating hyperplane. Defaults to ALPHA_PARAMETERIZATION.

        Returns:
            [npt.NDArray[np.float], npt.NDArray[np.float], float]: CCR and FPR arrays and area-under-the-curve, i.e., Y-axis values, X-axis values for the CCR-FPR curve and corresponding OSCR.
        """
        fpr = None
        sel_risk = None
        best_oscr = 0

        mu_samples = np.linspace(0, 1, n_mu_samples)
        if use_alpha_parameterization:
            mu_samples = [i for i in np.linspace(
                0, 2*np.pi, n_mu_samples)] + [np.pi/2., np.pi, np.pi*3./2.]

        for mu in mu_samples:
            score = (1-mu)*self.scoreA+mu*self.scoreB
            if use_alpha_parameterization:
                score = np.cos(mu)*self.scoreA + np.sin(mu)*self.scoreB

            metric = Metrics(self.y_true, self.y_pred, score)
            if metric.OSCR >= best_oscr:
                best_oscr = metric.OSCR
                fpr = metric.fpr
                sel_risk = metric.sel_risk

        auc = np.trapz(1-sel_risk, fpr)
        return 1-sel_risk, fpr, auc

    def prec_vs_recall(self,
                       n_recall_samples: int = N_RECALL_SAMPLES,
                       n_mu_samples: int = N_MU_SAMPLES,
                       use_alpha_parameterization: bool = ALPHA_PARAMETERIZATION) -> [npt.NDArray[float], npt.NDArray[float], float]:
        """
        Computes the Precision-Recall curve. 
        For each possible value X of TPR (Recall), a (maximizing Precision) selective function is found among all selective classifiers parameterized by the sampled mu parameter,
        such that the TPR of the selective function is higher or equal to X.

        Args:
            n_recall_samples (int, optional): Sampling parameter for the X-axis. Defaults to N_COVERAGE_SAMPLES.
            n_mu_samples (int, optional): Sampling parameter for optimal selective function search. Defaults to N_MU_SAMPLES.
            use_alpha_parameterization (bool, optional): If True, the selective function is parameterized by the angle of the separating hyperplane. Defaults to ALPHA_PARAMETERIZATION.

        Returns:
            [npt.NDArray[np.float], npt.NDArray[np.float], float]: Precision and Recall arrays and area-under-the-curve, i.e., Y-axis values, X-axis values for the PR curve and AUPR.
        """
        recall = np.linspace(0.001, 1, n_recall_samples)
        prec = np.zeros_like(recall)

        mu_samples = np.linspace(0, 1, n_mu_samples)
        if use_alpha_parameterization:
            mu_samples = [i for i in np.linspace(
                0, 2*np.pi, n_mu_samples)] + [np.pi/2., np.pi, np.pi*3./2.]

        for mu in mu_samples:
            score = (1-mu)*self.scoreA+mu*self.scoreB
            if use_alpha_parameterization:
                score = np.cos(mu)*self.scoreA + np.sin(mu)*self.scoreB

            metric = Metrics(self.y_true, self.y_pred, score)
            for i, r in enumerate(recall):
                ind = np.argwhere((metric.recall >= r))
                prec[i] = max(prec[i], np.max(metric.prec[ind]))

        auc = np.trapz(prec, recall)

        return prec, recall, auc

    def risk_vs_coverage(self,
                         n_coverage_samples: int = N_COVERAGE_SAMPLES,
                         n_mu_samples: int = N_MU_SAMPLES,
                         use_alpha_parameterization: bool = ALPHA_PARAMETERIZATION) -> [npt.NDArray[float], npt.NDArray[float], float]:
        """
        Computes the Risk-Coverage curve. 
        For each possible value of coverage, the optimal (minimizing selective risk) selective classifier is found among all selective classifiers parameterized by the sampled mu parameter.

        Args:
            n_coverage_samples (int, optional): Sampling parameter for the X-axis. Defaults to N_COVERAGE_SAMPLES.
            n_mu_samples (int, optional): Sampling parameter for optimal selective function search. Defaults to N_MU_SAMPLES.
            use_alpha_parameterization (bool, optional): If True, the selective function is parameterized by the angle of the separating hyperplane. Defaults to ALPHA_PARAMETERIZATION.

        Returns:
            [npt.NDArray[np.float], npt.NDArray[np.float], float]: Risk and Coverage arrays and are under the curve, i.e., Y-axis values, X-axis values for the Risk-Coverage curve and area under the curve.
        """

        coverage = np.linspace(0, 1, n_coverage_samples)
        risk = np.ones_like(coverage)*np.Inf

        mu_samples = np.linspace(0, 1, n_mu_samples)
        if use_alpha_parameterization:
            mu_samples = [i for i in np.linspace(
                0, 2*np.pi, n_mu_samples)] + [np.pi/2., np.pi, np.pi*3./2.]

        for mu in mu_samples:
            score = (1-mu)*self.scoreA+mu*self.scoreB
            if use_alpha_parameterization:
                score = np.cos(mu)*self.scoreA + np.sin(mu)*self.scoreB

            metric = Metrics(self.y_true, self.y_pred, score)
            for i, c in enumerate(coverage):
                ind = np.argwhere((metric.coverage >= c))
                risk[i] = min(risk[i], np.min(metric.sel_risk[ind]))

        auc = np.trapz(risk, coverage)

        return risk, coverage, auc

    def get_mu(self,
               max_fpr: float,
               min_coverage: float,
               n_mu_samples: int = N_MU_SAMPLES) -> [float, float]:
        """
        Finds the optimal selective function (minimizing selective risk), constrained by conditions on minimal TPR and maximal FPR.
        The optimal selective function is found among all selective classifiers parameterized by the sampled mu parameter.

        Args:
            max_fpr (float): Maximal admissible FPR.
            min_coverage (float): Minimal admissible TPR.
            n_mu_samples (int, optional): Sampling parameter for optimal selective function search. Defaults to N_MU_SAMPLES.

        Returns:
            [float, float]: Values of mu and the corresponding threshold, which define a separating hyperplane of the selective function.
        """

        best_risk = np.inf
        final_threshold = None
        final_mu = None
        for mu in np.linspace(0, 1, n_mu_samples):
            score = (1-mu)*self.scoreA+mu*self.scoreB

            metric = Metrics(self.y_true, self.y_pred, score)
            select = (metric.fpr <= max_fpr) & (
                metric.coverage >= min_coverage)
            potential_risks = metric.sel_risk[select]
            potential_thresholds = metric.score[select[1:]]

            if potential_risks.size > 0 and np.min(potential_risks) < best_risk:
                select_best = np.argmin(potential_risks)
                best_risk = potential_risks[select_best]
                final_threshold = potential_thresholds[select_best]
                final_mu = mu

        return final_mu, final_threshold

    def get_alpha(self,
                  max_fpr: float,
                  min_coverage: float,
                  n_mu_samples: int = N_MU_SAMPLES) -> [float, float]:
        """
        Finds the optimal selective function (minimizing selective risk), constrained by conditions on minimal TPR and maximal FPR.
        The optimal selective function is found among all selective classifiers parameterized by the sampled mu parameter.
        Uses alpha parameterization instead of mu parameterization.

        Args:
            max_fpr (float): Maximal admissible FPR.
            min_coverage (float): Minimal admissible TPR.
            n_mu_samples (int, optional): Sampling parameter for optimal selective function search. Defaults to N_MU_SAMPLES.

        Returns:
            [float, float]: Values of alpha and the corresponding threshold, which define a separating hyperplane of the selective function.
        """
        best_risk = np.inf
        final_threshold = None
        final_alpha = None
        possible_alphas = [i for i in np.linspace(
            0, 2*np.pi, n_mu_samples)] + [np.pi/2., np.pi, np.pi*3./2.]

        for alpha in possible_alphas:
            score = np.cos(alpha)*self.scoreA+np.sin(alpha)*self.scoreB
            metric = Metrics(self.y_true, self.y_pred, score)
            select = (metric.fpr <= max_fpr) & (
                metric.coverage >= min_coverage)
            potential_risks = metric.sel_risk[select]
            potential_thresholds = metric.score[select[:-1]]

            if potential_risks.size > 0 and np.min(potential_risks) < best_risk:
                select_best = np.argmin(potential_risks)
                best_risk = potential_risks[select_best]
                final_threshold = potential_thresholds[select_best]
                final_alpha = alpha

        return final_alpha, final_threshold

    def get_oscr(self):
        max_oscr = 0
        for mu in np.linspace(0,1,100):
            score = (1-mu)*self.scoreA+mu*self.scoreB 
            metric = Metrics(self.y_true,self.y_pred,score)
            if max_oscr < metric.OSCR:
                max_oscr = metric.OSCR
                fpr, ccr = metric.fpr, metric.ccr

        return fpr, ccr, max_oscr
