import numpy as np
import numpy.typing as npt


def load(path: str) -> [npt.NDArray[int], npt.NDArray[int], npt.NDArray[float]]:
    """
    Loads groundtruth labels, predictions and scores of a selective classifier.

    Args:
        path (str): Path to npz file with 'label', 'pred' and 'conf' keys which define the label, prediction and 1-score arrays.

    Returns:
        [npt.NDArray[np.int], npt.NDArray[np.int], npt.NDArray[np.float]]: Numpy arrays of labels, predictions and corresponding OOD scores.
    """
    data = np.load(path)
    return data['label'], data['pred'], 1-data['conf']


def load2(path1: str, path2: str) -> [npt.NDArray[int], npt.NDArray[int], npt.NDArray[float]]:
    """
    Wrapper around load(...). Loads two files and concatenates their contents.
    The function is used to obtain labels, predictions and scores in the OOD setups.

    E.g., path1 defines file where predictions of the model on ID data are saved.
          path2 defines file where predictions of the model on OOD data are saved.

    The returned arrays simulate the situation where the model is evaluated on a mixture of the ID and OOD data.
    """
    data1 = np.load(path1)
    data2 = np.load(path2)
    return np.concatenate([data1['label'], data2['label']]), \
        np.concatenate([data1['pred'], data2['pred']]), \
        np.concatenate([1-data1['conf'], 1-data2['conf']])
