
## Generic module imports ##


import numpy as np
from copy import deepcopy

# For doing compact nested for loops
import itertools

# For garbage collection, seemed to be an issue with earlier
# issues
import gc


# For parallel processing
import concurrent.futures as cf

# Pandas is a very useful module for saving and analyzing data
# Dataframe and Dataseries in particular
import pandas as pd

# Pickle module allows us to serialize Python-based classes
# (e.g., pandas objects and so on)
import pickle as pkl


# Let's log what goes on in the simulation
import logging

# A handy container class
from collections import namedtuple

# For running shell scripts within Python
from subprocess import call
import os

#################################################################
## Customized module imports ##

from MountainCar import MountainCar

# Various basic information (name of columns, data directories)
from base_info import dfcols, inner_data_dir, data_dir, log_dir

from base_info import make_inner_pklname as make_inner_pklname

##################################################################

def run_single_trial(cls, k):
    """
    Takes MountainCar cls objects,
    integer k (current trial number), and runs a trial.
    RETURNS: vals = [ n, mean_val, var_val, Bell_err, trialnum, k ]
             where mean_val and var_val are mean and variance of
             30 independent trials; Bell_err is measured by the
             Frobenius norm of the difference of linear coefficients
             of Q-function, divided by sqrt(d).
    This is called within a for loop over trials by run_outer_trial
    """

    #print("run_single_trial")

    # Generate fresh data
    cls.simulate_data()

    # Compute FQI estimate
    weight_Q, Bell_err = cls.conduct_FQI()

    # Evaluate error
    value_list = np.zeros(30)
    for i in range(30):
        Traj, value = cls.evaluate_value(weight_Q)
        print(i, "value", value)
        value_list[i] = value
    mean_val = np.nanmean(value_list)
    var_val = np.nanvar(value_list)

    # Return values as list in column order for the Pandas dataframe
    vals = [cls.n, mean_val, var_val, np.nanmean(Bell_err), k]
    print(vals)
    logging.info("%d %.12f %.3e %.3e %d" %(cls.n, mean_val, var_val, np.nanmean(Bell_err), k))
    return vals

###################################################################
def get_trial_info(k, numTrials, cls):
    """
    Helper routine to get information about a trial.
    Used in logging.
    """

    #print("get_trial_info")

    info_string = "%d/%d of (%d)" \
                  %(k, numTrials, cls.n)
    if k >= 0:
        return "Finished " + info_string
    elif k==-1:
        return "Starting " + info_string

#############################################################

def stitch_results(df_data,
                   n=-100,
                   pklfile="hack.pkl",
                   numTrials=37):
    """
    Helper routine to stitch in a new dataframe to an existing one

    INPUTS: df_data == current data frame, we will append to this
            (n, mean_val, var_val, Bell_err,
             pklfile,
             k):
            information on the new data that we want to append
    OUTPUTS: new data frame with appended data.
    """

    #print("stitch_results")

    # Make a dummy class for calling make_inner_pklname
    dummy_cls = namedtuple("dummy_cls", "n")
    myclass = dummy_cls(n=n)
    #print(pklfile)
    # Now make the requisite name, load the data, append it
    inner_pklname = make_inner_pklname(myclass, numTrials, pklfile)
    #print(inner_pklname)
    inner_df = pkl.load(open(os.path.join(inner_data_dir, inner_pklname), "rb"))
    # Here is the join
    return df_data.append(inner_df, ignore_index=True)


##################################################################

def construct_full_pkl(n_list,
                       numTrials,
                       pklfile="hack2.pkl"):
    """
    Routine to construct the full pickle file based on
    a set of input parameters.
    INPUTS:  n_list == list of sample sizes
             numTrials == int, num trials in the sym

    OUTPUTS:  None, save the pickle file directory as a dump.
    """

    #print("contruct_full_pkl")

    df_data = pd.DataFrame(columns = dfcols)

    for n in n_list:
        print("Adding n=%d" %(int(n)))
        df_data = stitch_results(df_data, n=n, pklfile=pklfile, numTrials=numTrials)

    pkl.dump(df_data, open(os.path.join(data_dir, pklfile), "wb"))

####################################################################
def run_full_outer(n, cls, numTrials, pklfile):
    """
    A full set of trials for a given problem instance.
    Routine called by each core in parallel processing.

    INPUTS:  n == int sample size
             cls == a mountain car class
             numTrials == number of trials
             pkfile == name of pickle file in which to save results
    """

    cls.update_params(n=n)

    #cls_td.recount = True
    inner_df = pd.DataFrame(columns = dfcols)
    inner_pklname = make_inner_pklname(cls, numTrials, pklfile)

    if os.path.exists(os.path.join(inner_data_dir, inner_pklname)):
        logging.info(get_trial_info(numTrials, numTrials, cls))
        return None

    logging.info(get_trial_info(-1, numTrials, cls))
    # Now loop over the trials
    for k in np.arange(0, numTrials):
        #cls_td.recount = True
        # Run a single trial and get a sequence of values
        vals = run_single_trial(cls, k)
        # cls_td.recount = False
        # Zip the column labels and values into a dictionary
        tmp_dict = dict(zip(dfcols, vals))
        # Append these entries as a row to our dataframe
        inner_df = inner_df.append(tmp_dict, ignore_index=True)
        #if np.mod(k+1, 25)==0 and k > 0:
        logging.info(get_trial_info(k+1, numTrials, cls))

    pkl.dump(inner_df,
             open(os.path.join(inner_data_dir, inner_pklname), "wb"))
    logging.info("SAVING TO ----------------------------> \n")
    print("File: %s" %inner_pklname)



############################################################
## MAIN ROUTINE

if __name__=="__main__":


    #####################################

    testing = False

    if testing:
        # My testing version
        n_list = [10]
        numTrials = 4
        pklfile = "steam_test.pkl"
    else:
        # This is where the parameters are loaded
        from paramfiles.hamster_parameters import n_list, \
                                                  pklfile, numTrials, max_workers
    # Now import the eigen information



    # Results will be saved to pklfile
    # Make the logfile match the pickle file
    logfile = pklfile.strip("pkl") + "log"

    # Some formatting stuff for the logger
    log_format = '%(asctime)s %(filename)s: %(message)s'
    full_logfile = os.path.join(log_dir, logfile)
    logging.basicConfig(level=logging.INFO,
                        handlers=[logging.FileHandler(full_logfile),
                                logging.StreamHandler()],
                        format=log_format,
                        datefmt='%Y-%m-%d %H:%M:%S')

#####################################################################


    # Get some generic instances of the parameters for initializing
    # the class
    first_n = n_list[0]

    cls = MountainCar(n=first_n)

    for n in n_list:
        run_full_outer(n, cls, numTrials, pklfile)

    # Now reconstruct the full pickle based on all the pickle files
    # that were saved in parallel jobs
    construct_full_pkl(n_list,
                       numTrials,
                       pklfile=pklfile)
