% *************************************************************************
% *************************************************************************
% *************************************************************************
%
% REFERENCE COMMAND INPUT (RCI) MAIN PROGRAM
%
% [*** ANONYMIZED ***] 
%
% 2023-09-20
%
% Main file for RCI algorithm. Considers infinite-horizon, input-affine
% nonlinear systems.
%
% Associated with work:
%
%   [*** ANONYMIZED ***] "A New, Physics-Based Continuous-Time
%   Reinforcement Learning Algorithm with Performance Guarantees" ICLR
%   2024. In review.
%
%
% ***** PROGRAM EXECUTION FLOW:
%
% * Select system (see variable 'systag') below
% * Select preset groups to execute (see variable 'preset_group_list')
%   below
% * Configure master settings -- master_settings.m
%   * Configure relative paths to programs, data
%   * Configure algorithm settings
%   * Configure master plot formatting
%   * Configure frequency response plot settings
%   * System initialization
%       * Modeling error parameter values \nu -- see variables 'nuvec',
%       'nueval'
%       * Integral augmentation settings
%       * Configure misc system settings -- config_sys.m
%   * Configure relative paths to data for selected system
%   * Configure controllers for selected system
%   * Configure individual preset group settings 
%       -- config_preset_group_cell.m  
%       -- config_preset_group.m 
% * For each preset in each preset group selected:
%   * Configure individual preset's alg hyperparams -- config_preset.m
%   * Run preset's algorithm  -- programs in 'algs' folder
% * Plot/save figures -- plot_main.m
%   * For each preset group executed, generate plots -- plot_preset_group.m
%       * State trajectory, control signal plots -- plot_x_u.m
%       * Algorithm-specific plots -- programs in 'plot' folder
%       * (x_0 sweeps only) Plots for x_0 sweeps, sweep statistical data --
%       plot_x0_sweep.m
%       * (Evaluation groups only) Plots for evaluation of critic NN,
%       policy cost -- plot_val_pol.m
%   * Save alg data to directory "00 figures/...../data/"
%   
% ***** DATA MANAGEMENT:
%
% Figures created by this program are written to the folder "00 figures/"
% in a time-stamped subfolder. Raw program data is written to the same
% location. The program data falls under the following three structures:
% 
%   group_settings_master   (Cell array) Each entry contains an
%                           'group_settings' struct for the respective
%                           preset group (see description below)
%   alg_settings_cell_master  (Cell array) Each entry contains a cell array
%                           of 'alg_settings_cell' objects containing the
%                           individual preset settings for each of the
%                           presets executed for the respective preset
%                           group (see description below)
%   out_data_cell_master    (Cell array) Each entry contains an
%                           'out_data_cell' struct for the respective
%                           preset group (see description below)
%
% These three objects are each cell arrays, with the number of entries
% corresponding to the number of preset groups executed. Each entry of
% these cell arrays contains data for the respective preset group of the
% form.
%
%   group_settings      (Struct) Contains shared settings common to all
%                       presets (e.g., state penalty matrix Q). In the case
%                       of an x_0 sweep, this will contain the ICs chosen.
%                       This struct is mainly initialized in the program
%                       config_preset_group.m
%   alg_settings_cell   ('numpresets' x 1 cell) Cell array containing the
%                       algorithm settings of each of the presets in the
%                       group. This cell array is mainly initialized in the
%                       program config_preset.m. Each entry's parameters
%                       are algorithm-specific. See respective algorithm
%                       .m-file for detailed descriptions of each
%                       algorithm's hyperparameters
%   out_data_cell       ('numpresets' x 1 cell) After each preset is run,
%                       its algorithm output data (e.g., state trajectory
%                       x(t), NN weight responses, etc.) are stored in this
%                       cell array. The data stored is algorithm-specific.
%                       See respective algorithm .m-file for detailed
%                       descriptions of each algorithm's output data
%   eval_data           (Struct) Contains evaluation data of the critic NN
%                       and policy cost. Generated by 'plot_eval_pol' only.
%
% Data from each individual preset group is also saved under a subfolder
% with the preset group's name
%
%
% ***** PLOTTING PREVIOUSLY-GENERATED DATA:
%
% The program does not need to re-run all of the algorithms to generate
% their data and generate plots again. Given data saved in pre-generated
% 'group_settings_master', 'alg_settings_cell_master', and
% 'out_data_cell_master' structs, the function re_plot_script.m may be used
% to plot the pre-existing data. See re_plot_script.m for details.
%
% ***** PLOT FORMATTING:
%
% The formatting used for the plots is set programattically by the program
% plot_format.m. The procedure followed to generate plots usually goes as
% follows:
%   Generate plot (title, axes labels, data, etc.) 
%   Specify the figure count in the 'figcount' variable.
%   Format the plot via the following call:
%           p_sett.figcount = figcount;
%           plot_format(p_sett, group_settings);
%       Note: Default formatting options can be over-written for an
%       individual plot by adding additional specs to the 'p_sett' struct
%       (see description of plot_format.m for how to do this).
%   
%
% ***** GENERAL TIPS:
%
% * NOTE: In order to execute properly, the programs must be kept in the
% same file path locations relative to main.m.
% * Set 'savefigs' = 1 (below) to save figures and algorithm data. Data
% gets saved to the relative path specified by the 'relpath' variable
% * To change preset group settings, go to config_preset_group.m
% * Generally speaking, shared hyperparameter values are written in
% config_preset_group.m (e.g., ICs, reference command settings, etc.).
% Specific algorithm learning hyperparameters are written in
% config_preset.m
%
% *************************************************************************
% *************************************************************************
% *************************************************************************


% *************************************************************************
% *************************************************************************
% *************************************************************************
%
% CONFIG
% 
% *************************************************************************
% *************************************************************************
% *************************************************************************

% ***********************
%
% CLEAR VARIABLES, COMMAND WINDOW, FIGURES
%
clear
clear global
clc
close all


% ***********************
%
% FIGURES
%
savefigs = 0;               % Save figures control
savedata = savefigs;        % Save data control
relpath = '00 figures/';  	% Relative file path for saved figures

% ***********************
%
% SETTINGS -- SYSTEM SELECTION
%

% System names -- DON'T CHANGE
sysnames.pendulum = 'pendulum';
sysnames.businjet = 'businjet';
sysnames.ddmr = 'ddmr';
master_settings.sysnames = sysnames;        % Save list to master settings

% *** System selection
systag = sysnames.pendulum;         % Inverted pendulum
% systag = sysnames.businjet;       % Jet aircraft 
% systag = sysnames.ddmr;           % DDMR (ground bot)

% ***********************
%
% SETTINGS -- SYSTEM/CONTROLLER INIT
%
% Initialize (=1) or load from previous (=0)
%

% Model
% init1_load0_model = 1;
init1_load0_model = 0;

% LQ servo controllers
% init1_load0_lq = 1; 
% init1_load0_lq = 0;
init1_load0_lq = init1_load0_model;




% *************************************************************************
%
% METHOD/SYSTEM/DESIGN PRESET GROUPS
%
% These tags correspond to the group of presets to be executed. Each preset
% within the group contains the specific
%
%   Algorithm/methodology (e.g., RCI, cFVI, rFVI, etc.)
%   System (e.g., pendulum)
%   Design (with numerical design parameters)
%
% Uncomment the preset groups to execute them. The preset groups will be
% executed in the order they apper in 'preset_group_list'.
%
% *************************************************************************


switch systag


    % ***********************
    %
    % PENDULUM
    %    
    
    case sysnames.pendulum

        preset_group = 'main';

        preset_group_list = {                           
                                'ES_eval_val_pol'
%                                 'ES_nom_step_th'
%                                 'ES_error_step_th'
%                                 'ES_error_rci_training_sweep' 
%                                 'ES_nom_rci_training'      
%                                 'ES_error_rci_training'    
                                        };    
 
    % ***********************
    %
    % JET
    %

    case sysnames.businjet

        preset_group = 'main';

        preset_group_list = {                         
                                'ES_eval_val_pol'
%                                 'ES_nom_step_g'
%                                 'ES_error_step_g'
%                                 'ES_error_rci_training_sweep'   
%                                 'ES_nom_rci_training'      
%                                 'ES_error_rci_training'    
                                        };      


    % ***********************
    %
    % DDMR
    %

    case sysnames.ddmr

        preset_group = 'main';

        preset_group_list = {    
%                                 'ES_eval_val_pol'
%                                 'ES_nom_step_w'
%                                 'ES_error_step_w'
%                                 'ES_error_rci_training_sweep'
                                'ES_nom_rci_training' 
%                                 'ES_error_rci_training'       
                                        };


  

end

% Number of preset groups executed
numgroups = size(preset_group_list, 1);



% *************************************************************************
% *************************************************************************
%
% MASTER SETTINGS
%
% *************************************************************************
% *************************************************************************


% ***********************
%
% STORE SETTINGS
%

% Figures
master_settings.savefigs = savefigs;           
master_settings.savedata = savedata;      
master_settings.relpath = relpath;  

% System tag
master_settings.systag = systag;

% Model/controller init
master_settings.init1_load0_model = init1_load0_model;
master_settings.init1_load0_lq = init1_load0_lq;

% Preset group
master_settings.preset_group = preset_group;

% Preset group list
master_settings.preset_group_list = preset_group_list;

% Number of preset groups executed
master_settings.numgroups = numgroups;


% ***********************
%
% INITIALIZE MASTER SETTINGS
%

addpath('config');
[master_settings, group_settings_master, preset_list_cell] = ...
    config_settings(master_settings);

    


%%
% *************************************************************************
% *************************************************************************
% *************************************************************************
%
% MAIN LOOP
%
% For each preset in the preset group:
%
%   Run preset configuration to initialize design parameters
%   Run respective preset algorithm and collect algorithm output data 
% 
% *************************************************************************
% *************************************************************************
% *************************************************************************

% Holds algorithm settings for each preset in each group
alg_settings_cell_master = cell(numgroups, 1);

% Holds algorithm output data for each preset in each group
out_data_cell_master = cell(numgroups, 1);

% Cumulative preset counter
cumpresetcnt = 1;

% Total number of presets
numpresets_tot = master_settings.numpresets_tot;


for i = 1:numgroups


    % ***********************
    %
    % PULL CURRENT PRESET GROUP
    %  
    
    % Preset list
    preset_listi = preset_list_cell{i};
    
    % Group settings
    group_settingsi = group_settings_master{i};
    
    % Current preset group name
    preset_group_namei = preset_group_list{i};


    % ***********************
    %
    % INITIALIZATION
    %  
    
    % Number of presets to execute in the group
    numpresetsi = size(preset_listi, 1);

    % Extract sweep settings
    sweepsettsi = group_settingsi.sweepsetts;

    % Is a sweep preset group (=1) or not (=0)
    issweep = sweepsettsi.issweep;

    % Get number of sweep dimensions, sweep size in each dimension
    if issweep
        sweepdimi = sweepsettsi.sweepdim;
        sweepsizeveci = sweepsettsi.sweepsizevec;
    else
        sweepdimi = 1;
        sweepsizeveci = 1;
    end

    % Total number of sweep iterations
    numsweepits = prod(sweepsizeveci);
    
    % ***********************
    %
    % STORAGE
    %  
    
    % Holds algorithm settings for each preset in the group
    if issweep
        alg_settings_celli = cell([sweepsizeveci' numpresetsi]);
    else
        alg_settings_celli = cell(numpresetsi, 1);
    end
    
    % Holds algorithm output data for each preset in the group
    if issweep
        out_data_celli = cell([sweepsizeveci' numpresetsi]);
    else
        out_data_celli = cell(numpresetsi, 1);
    end    

    % Display preset group count
    disp('***************************************************************')
    disp('*')
    disp(['* EXECUTING PRESET GROUP   ' num2str(i)...
            '   OUT OF      ' num2str(numgroups)])
    disp(['* CURRENT PRESET GROUP:   ' preset_group_namei] )
    disp('*')
    disp('***************************************************************')

    % ***********************
    %
    % BEGIN SWEEP LOOP
    %  

    for sweepcnt = 1:numsweepits

    % Current sweep index -- 'sweepdim'-dimensional variable
    switch sweepdimi
        case 1
            sweepindvec = sweepcnt;
        case 2
            [i1, i2] = ind2sub(sweepsizeveci,sweepcnt);
            sweepindvec = [i1; i2];
        case 3
            [i1, i2, i3] = ind2sub(sweepsizeveci,sweepcnt);
            sweepindvec = [i1; i2; i3];
    end

    % ***********************
    %
    % BEGIN PRESET LOOP
    %     
    
    for j = 1:numpresetsi
        
        % Display algorithm count
        disp('************************')
        disp('*')
        disp(['* EXECUTING PRESET    ' num2str(cumpresetcnt)...
                '   OUT OF      ' num2str(numpresets_tot)])
        disp('*')
        disp('************************')
        
        % Store the current sweep index vector in group_settings, in case
        % it is needed in the config
        group_settingsi.sweepindvec = sweepindvec;
        
        % Store the current preset count in group_settings, in case it is
        % needed in the config
        group_settingsi.presetcount = j;
        
        % *********************************************************************
        %
        % SELECT ALGORITHM, SYSTEM, DESIGN PARAMETERS BASED ON PRESET
        % 
        
        alg_settings = config_preset(preset_listi{j},...
            group_settingsi, master_settings);
        
        % *********************************************************************
        %
        % RUN ALGORITHM
        % 
        
        % Get algorithm elapsed time -- start timer
        tic;
     
        if group_settingsi.runpresets

            out_data = eval(['alg_' alg_settings.alg ...
                '(alg_settings, group_settingsi, master_settings)']);

        end

    
        % Get algorithm elapsed time -- stop timer
        out_data.runtime = toc;
    
        % Display run time
        disp(['***** RUN TIME OF PRESET ' num2str(cumpresetcnt) ...
                ':     ' num2str(toc) ' s'])
    
        % *****************************************************************
        %
        % STORAGE -- CURRENT PRESET
        % 
        
        % Store preset settings
        switch sweepdimi
            case 1
                alg_settings_celli{j} = alg_settings;
            case 2
                alg_settings_celli{i1, i2, j} = alg_settings;
            case 3
                alg_settings_celli{i1, i2, i3, j} = alg_settings;
        end
        
        
        % Store algorithm output data
        switch sweepdimi
            case 1
                out_data_celli{j} = out_data;
            case 2
                out_data_celli{i1, i2, j} = out_data;
            case 3
                out_data_celli{i1, i2, i3, j} = out_data;
        end

        % Increment cumulative preset counter
        cumpresetcnt = cumpresetcnt + 1;
        
    end         % END PRESET LOOP

    end         % END SWEEP LOOP



    % *****************************************************************
    %
    % STORAGE -- CURRENT PRESET GROUP
    % 
    
    % Store preset settings
    alg_settings_cell_master{i} = alg_settings_celli;
    
    % Store algorithm output data
    out_data_cell_master{i} = out_data_celli;    

end



% *************************************************************************
% *************************************************************************
% *************************************************************************
%
% PLOT/SAVE FIGURES
% 
% *************************************************************************
% *************************************************************************
% *************************************************************************


% Call plot function
plot_main(alg_settings_cell_master, out_data_cell_master,...
    group_settings_master, master_settings);



% *************************************************************************
% *************************************************************************
% *************************************************************************
%
% END MAIN
% 
% *************************************************************************
% *************************************************************************
% *************************************************************************

% Display complete
disp('*******************************************************************')
disp('*******************************************************************')
disp('*')
disp(['* MAIN PROGRAM COMPLETE'])
disp('*')
disp('*******************************************************************')
disp('*******************************************************************')

