import argparse
import torch
from exp.exp_long_term_forecasting import Exp_Long_Term_Forecast
from utils.print_args import print_args
import random
import numpy as np
from utils.str2bool import str2bool

if __name__ == '__main__':
    parser = argparse.ArgumentParser(description='SSNet')

    # basic config
    parser.add_argument('--task_name', type=str, required=False, default='long_term_forecast',
                        help='task name, options:[long_term_forecast, short_term_forecast, imputation, classification, anomaly_detection]')
    parser.add_argument('--random_seed', type=int, default=2021, help="random seed")
    parser.add_argument('--is_training', type=int, required=True, default=1, help='status')
    parser.add_argument('--model_id', type=str, required=True, default='test', help='model id')
    parser.add_argument('--model', type=str, required=True, default='SSNet',
                        help='model name, options: [SSNet, PatchTST, PDF]')

    # data loader
    parser.add_argument('--data', type=str, required=True, default='ETTm1', help='dataset type')
    parser.add_argument('--root_path', type=str, default='./data/ETT/', help='root path of the data file')
    parser.add_argument('--data_path', type=str, default='ETTh1.csv', help='data file')
    parser.add_argument('--features', type=str, default='M',
                        help='forecasting task, options:[M, S, MS]; M:multivariate predict multivariate, S:univariate predict univariate, MS:multivariate predict univariate')
    parser.add_argument('--target', type=str, default='OT', help='target feature in S or MS task')
    parser.add_argument('--freq', type=str, default='h',
                        help='freq for time features encoding, options:[s:secondly, t:minutely, h:hourly, d:daily, b:business days, w:weekly, m:monthly], you can also use more detailed freq like 15min or 3h')
    parser.add_argument('--checkpoints', type=str, default='./checkpoints/', help='location of model checkpoints')

    # forecasting task
    parser.add_argument('--seq_len', type=int, default=96, help='input sequence length')
    parser.add_argument('--label_len', type=int, default=48, help='start token length')
    parser.add_argument('--pred_len', type=int, default=96, help='prediction sequence length')
    parser.add_argument('--seasonal_patterns', type=str, default='Monthly', help='subset for M4')
    parser.add_argument('--inverse', action='store_true', help='inverse output data', default=False)

    # model define
    parser.add_argument('--top_k', type=int, default=5, help='for SSNet, TimesBlock')
    parser.add_argument('--num_kernels', type=int, default=6, help='for Inception')
    parser.add_argument('--enc_in', type=int, default=7, help='encoder input size')
    parser.add_argument('--dec_in', type=int, default=7, help='decoder input size')
    parser.add_argument('--c_out', type=int, default=7, help='output size')
    parser.add_argument('--d_model', type=int, default=512, help='dimension of model')
    parser.add_argument('--n_heads', type=int, default=8, help='num of heads')
    parser.add_argument('--e_layers', type=int, default=2, help='num of encoder layers')
    parser.add_argument('--d_layers', type=int, default=1, help='num of decoder layers')
    parser.add_argument('--d_ff', type=int, default=2048, help='dimension of fcn')
    parser.add_argument('--moving_avg', type=int, default=25, help='window size of moving average')
    parser.add_argument('--factor', type=int, default=1, help='attn factor')
    parser.add_argument('--distil', action='store_false',
                        help='whether to use distilling in encoder, using this argument means not using distilling',
                        default=True)
    parser.add_argument('--dropout', type=float, default=0.05, help='dropout')
    parser.add_argument('--embed', type=str, default='timeF',
                        help='time features encoding, options:[timeF, fixed, learned]')
    parser.add_argument('--activation', type=str, default='gelu', help='activation')
    parser.add_argument('--output_attention', action='store_true', help='whether to output attention in ecoder')
    parser.add_argument('--channel_independence', type=int, default=0,
                        help='0: channel dependence 1: channel independence for FreTS model')
    parser.add_argument('--decomp_method', type=str, default='moving_avg',
                        help='method of series decompsition, only support moving_avg or dft_decomp')
    parser.add_argument('--use_norm', type=int, default=1, help='whether to use normalize; True 1 False 0')
    parser.add_argument('--down_sampling_layers', type=int, default=3, help='num of down sampling layers')
    parser.add_argument('--down_sampling_window', type=int, default=2, help='down sampling window size')
    parser.add_argument('--down_sampling_method', type=str, default="avg",
                        help='down sampling method, only support avg, max, conv')
    parser.add_argument('--seg_len', type=int, default=48,
                        help='the length of segmen-wise iteration of SegRNN')
    parser.add_argument('--fc_dropout', type=float, default=0.05, help='attention dropout')
    parser.add_argument('--head_dropout', type=float, default=0.0, help='head dropout')
    parser.add_argument('--pct_start', type=float, default=0.3, help='pct_start')

    # model define for PDF
    parser.add_argument('--period', type=int, nargs='+', default=[24, 12], help='period list, for PDF')
    parser.add_argument('--kernel_list', type=int, nargs='+', default=[3, 7, 9], help='kernel size list, for PDF')
    parser.add_argument('--patch_len_list', type=int, nargs='+', default=[16], help='patch high, for PDF')
    parser.add_argument('--stride_list', type=int, nargs='+', default=None, help='stride, for PDF')
    parser.add_argument('--add', action='store_true', default=False, help='add')
    parser.add_argument('--wo_conv', action='store_true', default=False, help='without convolution')
    parser.add_argument('--serial_conv', action='store_true', default=False, help='serial convolution')

    # model define for PatchTST
    parser.add_argument('--patch_len', type=int, default=16, help='patch length')
    parser.add_argument('--stride', type=int, default=8, help='stride')
    parser.add_argument('--padding_patch', default='end', help='None: None; end: padding on the end')
    parser.add_argument('--revin', type=int, default=1, help='RevIN; True 1 False 0')
    parser.add_argument('--affine', type=int, default=0, help='RevIN-affine; True 1 False 0')
    parser.add_argument('--subtract_last', type=int, default=0, help='0: subtract mean; 1: subtract last')
    parser.add_argument('--decomposition', type=int, default=0, help='decomposition; True 1 False 0')
    parser.add_argument('--kernel_size', type=int, default=25, help='decomposition-kernel')
    parser.add_argument('--individual', type=int, default=0, help='individual head; True 1 False 0')
    parser.add_argument('--embed_type', type=int, default=0,
                        help='0: default 1: value embedding + temporal embedding + positional embedding 2: value embedding + temporal embedding 3: value embedding + positional embedding 4: value embedding')
    # parser.add_argument('--do_predict', action='store_true', help='whether to predict unseen future data')

    # model define for SSNet
    parser.add_argument('--mlp_type', type=str, default='Skip', help='mlp type')

    # model define for ModernTCN
    parser.add_argument('--stem_ratio', type=int, default=6, help='stem ratio')
    parser.add_argument('--downsample_ratio', type=int, default=2, help='downsample_ratio')
    parser.add_argument('--ffn_ratio', type=int, default=2, help='ffn_ratio')
    parser.add_argument('--patch_size', type=int, default=16, help='the patch size')
    parser.add_argument('--patch_stride', type=int, default=8, help='the patch stride')

    parser.add_argument('--num_blocks', nargs='+', type=int, default=[1, 1, 1, 1], help='num_blocks in each stage')
    parser.add_argument('--large_size', nargs='+', type=int, default=[31, 29, 27, 13], help='big kernel size')
    parser.add_argument('--small_size', nargs='+', type=int, default=[5, 5, 5, 5],
                        help='small kernel size for structral reparam')
    parser.add_argument('--dims', nargs='+', type=int, default=[256, 256, 256, 256], help='dmodels in each stage')
    parser.add_argument('--dw_dims', nargs='+', type=int, default=[256, 256, 256, 256])

    parser.add_argument('--small_kernel_merged', type=str2bool, default=False,
                        help='small_kernel has already merged or not')
    parser.add_argument('--call_structural_reparam', type=bool, default=False, help='structural_reparam after training')
    parser.add_argument('--use_multi_scale', type=str2bool, default=True, help='use_multi_scale fusion')

    # model define for FITS
    # parser.add_argument('--train_mode', type=int, default=0)
    parser.add_argument('--cut_freq', type=int, default=0)
    parser.add_argument('--base_T', type=int, default=24)
    parser.add_argument('--H_order', type=int, default=2)
    parser.add_argument('--train_mode', type=int, default=0,
                        help='Does not work, defaults to supervision on the forecasting task')

    # model define for Koopa
    parser.add_argument('--dynamic_dim', type=int, default=128, help='latent dimension of koopman embedding')
    parser.add_argument('--hidden_dim', type=int, default=64, help='hidden dimension of en/decoder')
    parser.add_argument('--hidden_layers', type=int, default=2, help='number of hidden layers of en/decoder')
    parser.add_argument('--alpha', type=float, default=0.2, help='spectrum filter ratio')
    parser.add_argument('--multistep', action='store_true', help='whether to use approximation for multistep K',
                        default=False)

    # model define for CrossGNN
    parser.add_argument('--tvechidden', type=int, default=1, help='scale vec dim')
    parser.add_argument('--nvechidden', type=int, default=1, help='variable vec dim')
    parser.add_argument('--use_tgcn', type=int, default=1, help='use cross-scale gnn')
    parser.add_argument('--use_ngcn', type=int, default=1, help='use cross-variable gnn')
    parser.add_argument('--anti_ood', type=int, default=1, help='simple strategy to solve data shift')
    parser.add_argument('--scale_number', type=int, default=4, help='scale number')
    parser.add_argument('--hidden', type=int, default=8, help='channel dim')
    parser.add_argument('--tk', type=int, default=10, help='constant w.r.t corss-scale neighbors')

    # model define for WITRAN
    parser.add_argument('--WITRAN_deal', type=str, default='None',
        help='WITRAN deal data type, options:[None, standard]')
    parser.add_argument('--WITRAN_grid_cols', type=int, default=24,
        help='Numbers of data grid cols for WITRAN')

    # optimization
    parser.add_argument('--num_workers', type=int, default=10, help='data loader num workers')
    parser.add_argument('--itr', type=int, default=2, help='experiments times')
    parser.add_argument('--train_epochs', type=int, default=100, help='train epochs')
    parser.add_argument('--batch_size', type=int, default=128, help='batch size of train input data')
    parser.add_argument('--patience', type=int, default=20, help='early stopping patience')
    parser.add_argument('--learning_rate', type=float, default=0.0001, help='optimizer learning rate')
    parser.add_argument('--des', type=str, default='test', help='exp description')
    parser.add_argument('--loss', type=str, default='MSE', help='loss function')
    parser.add_argument('--lradj', type=str, default='type3', help='adjust learning rate')
    parser.add_argument('--use_amp', action='store_true', help='use automatic mixed precision training', default=False)

    # GPU
    parser.add_argument('--use_gpu', type=bool, default=True, help='use gpu')
    parser.add_argument('--gpu', type=int, default=0, help='gpu')
    parser.add_argument('--use_multi_gpu', action='store_true', help='use multiple gpus', default=False)
    parser.add_argument('--devices', type=str, default='0,1,2,3', help='device ids of multile gpus')

    # de-stationary projector params
    parser.add_argument('--p_hidden_dims', type=int, nargs='+', default=[128, 128],
                        help='hidden layer dimensions of projector (List)')
    parser.add_argument('--p_hidden_layers', type=int, default=2, help='number of hidden layers in projector')

    # metrics (dtw)
    parser.add_argument('--use_dtw', type=bool, default=False,
                        help='the controller of using dtw metric (dtw is time consuming, not suggested unless necessary)')

    # Augmentation
    parser.add_argument('--augmentation_ratio', type=int, default=0, help="How many times to augment")
    parser.add_argument('--seed', type=int, default=2, help="Randomization seed")
    parser.add_argument('--jitter', default=False, action="store_true", help="Jitter preset augmentation")
    parser.add_argument('--scaling', default=False, action="store_true", help="Scaling preset augmentation")
    parser.add_argument('--permutation', default=False, action="store_true",
                        help="Equal Length Permutation preset augmentation")
    parser.add_argument('--randompermutation', default=False, action="store_true",
                        help="Random Length Permutation preset augmentation")
    parser.add_argument('--magwarp', default=False, action="store_true", help="Magnitude warp preset augmentation")
    parser.add_argument('--timewarp', default=False, action="store_true", help="Time warp preset augmentation")
    parser.add_argument('--windowslice', default=False, action="store_true", help="Window slice preset augmentation")
    parser.add_argument('--windowwarp', default=False, action="store_true", help="Window warp preset augmentation")
    parser.add_argument('--rotation', default=False, action="store_true", help="Rotation preset augmentation")
    parser.add_argument('--spawner', default=False, action="store_true", help="SPAWNER preset augmentation")
    parser.add_argument('--dtwwarp', default=False, action="store_true", help="DTW warp preset augmentation")
    parser.add_argument('--shapedtwwarp', default=False, action="store_true", help="Shape DTW warp preset augmentation")
    parser.add_argument('--wdba', default=False, action="store_true", help="Weighted DBA preset augmentation")
    parser.add_argument('--discdtw', default=False, action="store_true",
                        help="Discrimitive DTW warp preset augmentation")
    parser.add_argument('--discsdtw', default=False, action="store_true",
                        help="Discrimitive shapeDTW warp preset augmentation")
    parser.add_argument('--extra_tag', type=str, default="", help="Anything extra")

    args = parser.parse_args()
    # args.use_gpu = True if torch.cuda.is_available() and args.use_gpu else False
    args.use_gpu = True if torch.cuda.is_available() else False

    random.seed(args.random_seed)
    torch.manual_seed(args.random_seed)
    np.random.seed(args.random_seed)

    if args.use_gpu and args.use_multi_gpu:
        args.devices = args.devices.replace(' ', '')
        device_ids = args.devices.split(',')
        args.device_ids = [int(id_) for id_ in device_ids]
        print('device_ids:', args.device_ids)
        args.gpu = args.device_ids[0]

    if args.cut_freq == 0:
        args.cut_freq = int(args.seq_len // args.base_T + 1) * args.H_order + 10

    print('Args in experiment:')
    print_args(args)
    assert args.task_name == 'long_term_forecast'
    Exp = Exp_Long_Term_Forecast

    class Setting:

        def __init__(self, ii):
            self.task_name = args.task_name
            self.model_id = args.model_id
            self.model = args.model
            self.data = args.data
            self.data_path = args.data_path.split('.')[0]
            self.features = args.features
            self.seq_len = args.seq_len
            self.label_len = args.label_len
            self.pred_len = args.pred_len
            self.d_model = args.d_model
            self.n_heads = args.n_heads
            self.e_layers = args.e_layers
            self.d_layers = args.d_layers
            self.d_ff = args.d_ff
            self.factor = args.factor
            self.embed = args.embed
            self.distil = args.distil
            self.des = args.des
            self.itr = ii
            self.period = args.period
            self.dropout = args.dropout
            self.fc_dropout = args.fc_dropout
            self.lradj = args.lradj
            self.learning_rate = args.learning_rate
            self.train_epochs = args.train_epochs
            self.patience = args.patience
            self.batch_size = args.batch_size

        def __str__(self):
            s = \
            f"{self.task_name}_" \
            f"{self.data}_" \
            f"{self.model}_" \
            f"{self.seq_len}_" \
            f"{self.pred_len}_" \
            f"period_{self.period[0]}_" \
            f"d_model_{self.d_model}_"\
            f"dropout_{self.dropout}_" \
            f"batch_size_{self.batch_size}_"\
            f"learning_rate_{self.learning_rate}_"\
            f"train_epochs_{self.train_epochs}_"\
            f"patience_{self.patience}_"\
            f"lradj_{self.lradj}_"\
            f"{self.des}_"
            return s

    if args.is_training:
        for ii in range(args.itr):
            # setting record of experiments
            exp = Exp(args)  # set experiments
            setting = Setting(ii)

            print('>>>>>>>start training : {}>>>>>>>>>>>>>>>>>>>>>>>>>>'.format(setting))
            exp.train(setting)

            print('>>>>>>>testing : {}<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<'.format(setting))
            exp.test(setting, )
            torch.cuda.empty_cache()
    else:
        ii = 0
        setting = Setting(ii)
        exp = Exp(args)  # set experiments
        print('>>>>>>>testing : {}<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<'.format(setting))
        exp.test(setting, test=1)
        torch.cuda.empty_cache()

