import os
import pickle
import torch
import numpy as np
from sklearn.neighbors import NearestNeighbors

from llm_router.router.base_router import BaseRouter, RouterInput, RouterOutput, PREFERENCES
from llm_router.data.utils import get_costs

class KNNPredictorRouter(BaseRouter):
    
    @property
    def name(self):
        return f"knn_predictor_router"
    
    def fit(self, trainset, valset, configs):
        self.benchmark = trainset.get_benchmark()
        
        if os.path.exists(os.path.join(configs.training.output_dir, "knn.pkl")):
            with open(os.path.join(configs.training.output_dir, "knn.pkl"), "rb") as f:
                self.knn = pickle.load(f)
            return
        
        self.knn = NearestNeighbors(
            n_neighbors=self.config.n_neighbors,
            metric="cosine",
            leaf_size=self.config.leaf_size,
            n_jobs=-1,
        ).fit(self.benchmark["embeddings"].cpu().numpy())
        
        os.makedirs(configs.training.output_dir, exist_ok=True)
        with open(os.path.join(configs.training.output_dir, "knn.pkl"), "wb") as f:
            pickle.dump(self.knn, f)
    
    def route(self, router_input: RouterInput):
        _, indices = self.knn.kneighbors(router_input.embedding.unsqueeze(0).cpu().numpy())
        predicted_scores = torch.stack([self.benchmark['scores'][idx] for idx in indices[0]]).mean(dim=0)
        predicted_output_tokens = torch.stack([self.benchmark['output_tokens'][idx] for idx in indices[0]]).mean(dim=0)
        predicted_costs = get_costs(router_input.input_tokens, predicted_output_tokens, router_input.routing_config)
        assert len(predicted_scores) == len(predicted_costs)
        
        outputs = {}
        for preference in PREFERENCES:
            q = [s - preference * c for s, c in zip(predicted_scores, predicted_costs)]
            routing_id = np.argmax(q)
            routing_model = list(router_input.routing_config.keys())[routing_id]
            outputs[preference] = RouterOutput(
                idx=router_input.idx,
                routing_config=router_input.routing_config,
                scores=router_input.scores,
                costs=router_input.costs,
                input_tokens=router_input.input_tokens,
                output_tokens=router_input.output_tokens,
                routing_id=routing_id,
                routing_model=routing_model,
                info={
                    "predicted_scores": predicted_scores,
                    "predicted_output_tokens": predicted_output_tokens,
                    "predicted_costs": predicted_costs,
                },
            )
            
        return outputs