import os
import torch
import numpy as np

from llm_router.router.base_router import BaseRouter, RouterInput, RouterOutput, PREFERENCES
from llm_router.data.utils import get_costs
from llm_router.model.registry import get_model
from llm_router.model.utils import Trainer

class PredRouter(BaseRouter):
    
    @property
    def name(self):
        return f"pred_router"
    
    def fit(self, trainset, valset, configs):        
        # load model
        self.model = get_model(configs.model)
        self.model.to(configs.model.device)
        self.model.initialize(trainset)
        # trainer
        trainer = Trainer(self.model, trainset, valset, configs.training)
        ckpt_dir = os.path.join(configs.training.output_dir, "model", configs.model.ckpt)
        if not os.path.exists(ckpt_dir):
            trainer.train()
        trainer.load(ckpt_dir)
        
    def route(self, router_input: RouterInput):
        predicted_scores, predicted_output_tokens = self.model.predict(router_input)
        predicted_costs = get_costs(router_input.input_tokens, predicted_output_tokens, router_input.routing_config)
        assert len(predicted_scores) == len(predicted_costs)
        
        outputs = {}
        for preference in PREFERENCES:
            q = [s - preference * c for s, c in zip(predicted_scores, predicted_costs)]
            routing_id = np.argmax(q)
            routing_model = list(router_input.routing_config.keys())[routing_id]
            outputs[preference] = RouterOutput(
                idx=router_input.idx,
                routing_config=router_input.routing_config,
                scores=router_input.scores,
                costs=router_input.costs,
                input_tokens=router_input.input_tokens,
                output_tokens=router_input.output_tokens,
                routing_id=routing_id,
                routing_model=routing_model,
                info={
                    "predicted_scores": predicted_scores,
                    "predicted_output_tokens": predicted_output_tokens,
                    "predicted_costs": predicted_costs,
                },
            )
            
        return outputs