import numpy as np
from numpy.linalg import norm, eigh

# Utility functions
def dinfty(uhat, ustar):
    return min(max(np.abs(uhat - ustar)), max(np.abs(uhat + ustar)))

def dtti(Uhat, Ustar):
    r = Ustar.shape[1]
    s = np.array([0]*r)
    for k in range(r):
        s[k] = 1 if max(np.abs(Uhat[:,k] - Ustar[:,k])) <= max(np.abs(Uhat[:,k] + Ustar[:,k])) else -1
    D = Uhat * s - Ustar
    return np.max(norm(D, axis=1))

def pmdiff(uhat, ustar):
  return min(np.abs(uhat-ustar), np.abs(uhat+ustar))

def sample_spherical(npoints, ndim):
    vec = np.random.randn(ndim, npoints)
    vec /= np.linalg.norm(vec, axis=0)
    return vec

def get_ortho(n):
  A = np.random.randn(n * n).reshape(n, n).astype("float32")
  O, _ = np.linalg.qr(A)
  return O.astype("float32")

def simulator(n,r,a,noise_type):

    I = np.identity(n)
    lambdas = [np.sqrt(n*np.log(n)) + 0.5*i*np.sqrt(n*np.log(n)) for i in range(r)]
    lambdas.reverse()

    # Generate Eigenvectors
    xi = sample_spherical(1, n-1)
    xi = xi.reshape(-1)
    b1 = np.sqrt(1-a**2)*xi/norm(xi,2) 
    u_new = np.concatenate(([a],b1)).reshape(-1,1)
    loc = np.random.randint(0, n)
    c = u_new[loc, 0]
    u_new[loc, 0], u_new[0,0] = a, c
    U = u_new
    for _ in range(r-1):
        xi = sample_spherical(1, n-1)
        xi = xi.reshape(-1)
        b1 = np.sqrt(1-a**2)*xi/norm(xi,2)
        u_new = np.concatenate(([a],b1)).reshape(-1,1)
        loc = np.random.randint(0, n)
        c = u_new[loc, 0]
        u_new[loc, 0], u_new[0,0] = a, c
        u_new = (I - U @ U.T) @ u_new
        u_new = u_new / norm(u_new, 2)
        U = np.hstack((U, u_new))

    # Record coherence locations
    coh_loc = {}
    for i in range(r):
        coh_loc[i] = np.where(np.abs(U[:,i]) > np.log(n) / np.sqrt(n))[0].tolist()

    # Create Symmetric Noise Matrix
    if noise_type == 1:
        W = np.random.randn(n * n).reshape(n, n).astype("float32")
    elif noise_type == 2:
        W = np.random.laplace(0, 1/np.sqrt(2), n*n).reshape(n,n).astype("float32")
    else:
        W = (2*np.random.binomial(1, 0.5, size=n*n)-1).reshape(n,n).astype("float32")
    W_diag = np.diag(W)
    W_upper = np.triu(W)
    W_symm = W_upper + W_upper.T
    np.fill_diagonal(W_symm, W_diag)
    M = U @ np.diag(lambdas) @ U.T
    Y = (M + W_symm).astype("float32")
    del M, W, W_symm, W_diag, W_upper
    H = get_ortho(n)
    Yhat0 = H @ Y @ H.T
    eigvals, eigvecs = eigh(Yhat0)
    # Obtain top eigenvectors
    idx = np.argsort(np.abs(eigvals.real))[::-1]  # Sort eigenvalues in descending order
    uhat_specs = H.T @ eigvecs[:, idx[0:r]]
    ev_specs = eigvals[idx[0:r]]
    sighat2 = np.sum(np.triu((Y - ((ev_specs * uhat_specs) @ uhat_specs.T))**2))/(n*(n+1)/2)

    uspec_dinfty = [0]*r
    evs_c = [0]*r

    for k in range(r):
        evs_c[k] = (eigvals[idx[k]] + np.sqrt(eigvals[idx[k]]**2 - 4*n*sighat2))/2
        uspec_dinfty[k] = "{:.4f}".format(dinfty(uhat_specs[:,k], U[:,k]))

    uhat_c_dinfty = [0]*r

    noise_dict = {1:'Gaussian', 2:'Laplacian', 3:'Rademacher'}

    results2 = {'n':[], 'r':[], 'a':[], 'k':[], 'noise':[], 'largest.loc':[], 'U.largest':[], 'Ours.Est':[], 'Spec.Est':[], 'Ours.Error':[],
                'Spec.Error':[]}
    
    results3 = {'n':[n], 'r':[r], 'a':[a], 'noise':[noise_dict[noise_type]], 'Ours.Error':[], 'Spec.Error':[]}
    results3['Spec.Error'].append("{:.4f}".format(dtti(uhat_specs, U)))
    
    Uhat =  np.zeros((n, r), dtype="float32")

    # Consider the k-th largest eigenpair in absolute value.
    for k in range(r):
        qs = np.sign(eigvecs[:,idx[k]])
        alpha0 = np.quantile(np.abs(eigvecs[:,idx[k]]), 0.5)
        # Calculate Yhats
        Yhats = qs.reshape(-1, 1) * Yhat0 * qs
        Ihat = np.where(np.abs(eigvecs[:,idx[k]]) > alpha0)[0]
        uhat_c = np.zeros(n)
        # Calculate u.sum.root
        u_sum_root_c = np.sqrt(np.sum(Yhats[Ihat[:, np.newaxis], Ihat]) / evs_c[k])
        # Calculate each element of uhat
        for i in range(n):
            uhat_c[i] = np.sum(Yhats[i, Ihat]) / (evs_c[k] * u_sum_root_c)

        # Apply Qs to uhat
        uhat_c = qs * uhat_c
        # Apply Qhat to uhat
        uhat_c = H.T @ uhat_c
        for i in range(n):
            if np.abs(uhat_specs[i,k]) <= np.abs(np.log(n)/evs_c[k]):
                uhat_c[i] = uhat_specs[i,k]
        Uhat[:,k] = uhat_c
        uhat_c_dinfty[k] = "{:.4f}".format(dinfty(uhat_c, U[:,k]))

        for loc in coh_loc[k]:
            results2['n'].append(n)
            results2['r'].append(r)
            results2['a'].append(a)
            results2['noise'].append(noise_dict[noise_type])
            results2['k'].append(k)
            results2['largest.loc'].append(loc)
            results2['U.largest'].append("{:.4f}".format(U[loc, k]))
            results2['Ours.Est'].append("{:.4f}".format(uhat_c[loc]))
            uhatc_diff = pmdiff(uhat_c[loc], U[loc,k])
            results2['Ours.Error'].append("{:.4f}".format(uhatc_diff))
            results2['Spec.Est'].append("{:.4f}".format(uhat_specs[loc,k]))
            uspec_diff = pmdiff(uhat_specs[loc,k],U[loc,k])
            results2['Spec.Error'].append("{:.4f}".format(uspec_diff))
    
    results1 = {"n": n,
                "r": r,
                "a": a,
                "k": list(range(r)),
                "noise": noise_dict[noise_type],
                "Ours.Error":uhat_c_dinfty,
                "Spec.Error":uspec_dinfty}
    results3['Ours.Error'].append("{:.4f}".format(dtti(Uhat, U)))
    results = {"results1":results1, "results2":results2, "results3":results3}
    return results
