from jax import numpy as jnp
from flax import linen as nn
import einops

def rotate_every_two(x):
    """x: jax.Array(B, nh, T, hs)"""
    x1 = x[:, :, :, ::2]
    x2 = x[:, :, :, 1::2]
    x = jnp.stack((-x2, x1), axis=-1)
    return einops.rearrange(x, '... d j -> ... (d j)') # x.flatten(-2)  # in einsum notation: rearrange(x, '... d j -> ... (d j)')\

def duplicate_interleave(m):
    """
    A simple version of `torch.repeat_interleave` for duplicating a matrix while interleaving the copy.
    """
    dim0 = m.shape[0]
    m = m.reshape(-1, 1)  # flatten the matrix
    m = jnp.tile(m, (1, 2))  # repeat all elements into the 2nd dimension
    m = m.reshape(dim0, -1)  # reshape into a matrix, interleaving the copy
    return m

def apply_rotary_pos_emb(x, sin, cos, scale=1):
    """x: jax.Array(B, nh, T, hs)"""
    sin, cos = map(lambda t: duplicate_interleave(t * scale), (sin, cos))
    # einsum notation for lambda t: repeat(t[offset:x.shape[1]+offset,:], "n d -> () n () (d j)", j=2)
    return (x * cos) + (rotate_every_two(x) * sin)


def fixed_pos_embedding(x):
    seq_len, dim = x.shape
    inv_freq = 1.0 / (10000 ** (jnp.arange(0, dim) / dim))
    sinusoid_inp = (
        jnp.einsum("i , j -> i j", jnp.arange(0, seq_len), inv_freq).astype(x.dtype)
    )
    return jnp.sin(sinusoid_inp), jnp.cos(sinusoid_inp)


class XPos(nn.Module):
    head_dim: int
    scale_base: int = 512
    dtype: jnp.dtype = jnp.float32
    
    def setup(self):
        self.scale =  (jnp.arange(0, self.head_dim, 2) + 0.4 * self.head_dim) / (1.4 * self.head_dim)

    def __call__(self, x, offset=0, downscale=False):
        """x: jax.Array(B, nh, T, hs)"""
        # scale = self.scale ** jnp.divide((jnp.arange(0, len, 1) - len // 2), self.scale_base)[:, None]
        # sin, cos = fixed_pos_embedding(scale)
        # return (sin, cos, scale)
        length = x.shape[2]
        min_pos = -(length + offset) // 2
        max_pos = length + offset + min_pos
        scale = self.scale ** jnp.divide(jnp.arange(min_pos, max_pos, 1), self.scale_base)[:, None]
        scale = scale.astype(self.dtype)
        sin, cos = fixed_pos_embedding(scale)

        if scale.shape[0] > length:
            scale = scale[-length:]
            sin = sin[-length:]
            cos = cos[-length:]
        
        if downscale:
            scale = 1 / scale

        x = apply_rotary_pos_emb(x, sin, cos, scale)
        return x.astype(x.dtype)