import os, sys
import numpy as np
import pickle as pkl
import json, ast, re
import tempfile, uuid, copy
import pathlib, importlib
import asyncio, subprocess
import PyPDF2

from os.path import isfile, join
from typing import Annotated, Literal, Dict, List, Tuple, Optional
from typing_extensions import TypedDict
from pydantic import BaseModel, Field

from langchain_core.tools import tool
from langchain_core.prompts import ChatPromptTemplate, MessagesPlaceholder
from langchain_core.prompt_values import ChatPromptValue
from langchain_core.messages import AnyMessage, SystemMessage, HumanMessage, AIMessage, ToolMessage
from langchain_core.output_parsers import StrOutputParser, JsonOutputParser
from langchain_core.runnables import RunnableLambda
from langchain.tools.render import render_text_description_and_args

from langgraph.checkpoint.memory import MemorySaver
from langgraph.graph import END, START, StateGraph, MessagesState
from langgraph.graph.message import add_messages
from langgraph.prebuilt import ToolNode

from utils.agent_templates import *
from utils.tool_definitions import tool_dict, custom_stop, final_answer, set_conversation_buffer, json_output_parser, verification_function, set_sop_file, set_smv_files
from utils.tool_definitions import read_docs, read_nusmvs, remove_thinking_tokens
from utils.kripke_templates import KRIPKE_TASK_PROMPT, KRIPKE_PLAN_SYSTEM, KRIPKE_PLAN_PROMPT, KRIPKE_CORRECTION_TEMPLATE, KRIPKE_EXECUTION_TASK
from utils.extraction_templates import *

BASE_LLM = None
GLOBAL_LLM = None
CODING_LLM = None
REASONING_LLM = None
LLM_SERVER_OWNER = None
SOLVER_STRATEGY: Literal["ReWOO", "plan-and-execute"] = "ReWOO"

def set_node_llm_state(base_llm, global_llm, coding_llm, reasoning_llm = None):
    global BASE_LLM
    global GLOBAL_LLM
    global CODING_LLM
    global REASONING_LLM
    BASE_LLM = base_llm
    GLOBAL_LLM = global_llm
    CODING_LLM = coding_llm
    
    if reasoning_llm is not None:
        REASONING_LLM = ( reasoning_llm | RunnableLambda(remove_thinking_tokens))
    return GLOBAL_LLM, CODING_LLM

def set_solver_strategy(strategy):
    global SOLVER_STRATEGY
    SOLVER_STRATEGY = strategy.lower().replace(' ', '-')
    return SOLVER_STRATEGY

def set_llm_owner(model_dict, model_index):
    global LLM_SERVER_OWNER
    try:
        LLM_SERVER_OWNER = model_dict['data'][model_index]['owned_by']
    except Exception:
        LLM_SERVER_OWNER = "default"
    return LLM_SERVER_OWNER

def bind_tools_func(tools):
    global BASE_LLM
    global CODING_LLM
    try:
        if LLM_SERVER_OWNER.lower()=="vllm":
            TOOLS_LLM = CODING_LLM.bind_tools(tools, tool_choice="auto")
        else:
            TOOLS_LLM = CODING_LLM.bind_tools(tools, tool_choice="required")
    except Exception as e:
        print(f"Tool binding failed with error: {e}. Trying BASE_LLM bind.")
        if LLM_SERVER_OWNER.lower()=="vllm":
            TOOLS_LLM = BASE_LLM.bind_tools(tools, tool_choice="auto")
        else:
            TOOLS_LLM = BASE_LLM.bind_tools(tools, tool_choice="required")
        TOOLS_LLM = (TOOLS_LLM | RunnableLambda(remove_thinking_tokens))
    return TOOLS_LLM

class ReWOO(TypedDict):
    task: str = None
    plan_string: str = None
    result: str = None
    results: dict = None
    steps: List = None
    messages: Annotated[list[AnyMessage], add_messages]

class NUSMV_planner:
    def __init__(self, filesys_root, previous_conversation=[], nusmv_model_file=None):
        if nusmv_model_file is not None and os.path.isdir(filesys_root):
            self.model_file = nusmv_model_file
            if '_' in self.model_file:
                model_num = self.model_file.split('_')[1].split('.')[0]
                set_sop_file(join(filesys_root, f'sop_{model_num}.pdf'))
            else:
                set_sop_file(join(filesys_root, 'SOP.pdf'))
            set_smv_files(filesys_root)
        else:
            print(f"{nusmv_model_file} does not exist. Using {filesys_root} as default.")
            self.model_file = "model.smv"
            set_sop_file(join(filesys_root, 'SOP.pdf'))
            
        self.agent_state = ReWOO(messages=[])
        self.react_messages = {"messages": []}
        self.tools = None
        self.tool_name = None
        self.tool_renderer = None
        self.tool_node = None
        self.planning_node1_context = []
        self.filesystem_root = filesys_root

        self.tool_messages_offset = -1
        self.final_answer_tool_flag = False
        self.PREVIOUS_CONVERSATION = previous_conversation
        self.plan_prompt1_flag = False
        self.plan_prompt2_flag = False
    
    @staticmethod
    def plan_parser(text: AIMessage) -> List[str]:
        steps = [v for v in re.split("\n\s*\d+\. ", text.content)[1:]]
        return steps

    def set_node_tools(self, tool_list):
        self.tools = [tool_dict[f] for f in tool_list if f in tool_dict.keys()]
        self.tool_names = [f for f in tool_list if f in tool_dict.keys()]
        self.tools_renderer = render_text_description_and_args(self.tools)
        self.tool_node = ToolNode(self.tools+[final_answer])
        return self.tools, self.tool_names, self.tools_renderer, self.tool_node

    def planning_node1(self, state:ReWOO) -> Dict:
        global GLOBAL_LLM
        for state_key in state.keys():
            self.agent_state[state_key] = state[state_key]
        
        self.set_node_tools(tool_list=['crag_tool', 'direct_response'])
        llm_with_stop = GLOBAL_LLM.bind(stop=custom_stop['plan'])
        plan_prompt1 = ChatPromptTemplate.from_messages([
            ('system', NUSMV_PLAN_SYS),
            MessagesPlaceholder(variable_name="previous_conversation", optional=True),
            ('human', NUSMV_PLAN_PROMPT),
        ])
        self.plan_prompt1_flag = True
        plan_prompt_with_tools = plan_prompt1.partial(tools=self.tools_renderer, tool_names = self.tool_names)
        planning_chain = (plan_prompt_with_tools | llm_with_stop)

        self.agent_state['task'] = NUSMV_TASK_PROMPT_1
        result = planning_chain.invoke({'previous_conversation': self.PREVIOUS_CONVERSATION})
        matches = self.plan_parser(result)
        self.agent_state["steps"] = matches
        self.agent_state["plan_string"] = result.content
        self.agent_state["messages"] = add_messages(state["messages"], [result])
        self.planning_node1_context.append(state["messages"])
        # print(self.agent_state)
        return self.agent_state

    def planning_node2(self, state:ReWOO) -> Dict:
        global GLOBAL_LLM
        for state_key in state.keys():
            self.agent_state[state_key] = state[state_key]
        '''
        state_pth = join(self.filesystem_root, 'states.smv')
        trans_pth = join(self.filesystem_root, 'trans.smv')
        props_pth = join(self.filesystem_root, 'properties.smv')

        states = open(state_pth, "r").read()
        trans = open(trans_pth, "r").read()
        props = open(props_pth, "r").read()
        '''
        # print("KRIPKE AGENT")
        # print(self.agent_state["messages"])
        # set_conversation_buffer(self.agent_state["messages"])
        self.set_node_tools(tool_list=['list_directory', 'save_nusmv', 'crag_tool', 'nusmv_codepad', 'direct_response', 'read_file'])
        llm_with_stop = GLOBAL_LLM.bind(stop=custom_stop['plan'])
        plan_prompt2 = ChatPromptTemplate.from_messages([
            ('system', NUSMV_PLAN_SYS_1),
            MessagesPlaceholder(variable_name="previous_conversation", optional=True),
            ('human', NUSMV_PLAN_PROMPT_1),
        ])
        self.plan_prompt2_flag = True
        plan_prompt_with_tools = plan_prompt2.partial(tools=self.tools_renderer, tool_names = self.tool_names)
        planning_chain = (plan_prompt_with_tools | llm_with_stop)
        self.agent_state['task'] = NUSMV_TASK_PROMPT.format(context = read_nusmvs())
        result = planning_chain.invoke({'previous_conversation':self.agent_state["messages"],
                                        'NUSMV_TASK': self.agent_state['task'],
                                        'model_name': self.model_file,
                                        })
        matches = self.plan_parser(result)
        self.agent_state["steps"] = matches
        self.agent_state["plan_string"] = result.content
        self.agent_state["messages"] = add_messages(state["messages"], [result])
        # print(self.agent_state)
        return self.agent_state

    def replanning_node(self, state:ReWOO) -> Dict:
        global GLOBAL_LLM
        for state_key in state.keys():
            self.agent_state[state_key] = state[state_key]

        self.set_node_tools(tool_list=['list_directory', 'save_nusmv', 'crag_tool', 'nusmv_codepad', 'direct_response', 'read_file'])
        llm_with_stop = GLOBAL_LLM.bind(stop=custom_stop['plan'])   
        plan_prompt = ChatPromptTemplate.from_messages([
            ("system", PROP_PLAN_SYSPROMPT),
            MessagesPlaceholder(variable_name="messages", optional=True)
        ])
        plan_prompt_with_tools = plan_prompt.partial(tools=self.tools_renderer, tool_names=self.tool_names)
        planning_chain = (plan_prompt_with_tools | llm_with_stop)
        last_plan_step_index = self.get_current_step(state)-1
        _results = (state["results"] or {}) if "results" in state else {}
        plan_str = PROP_CORRECTION_PROMPT.format(**{
                "kripke": self.kripke_config,
                'DOC_TXT': self.SOP_txt,
                "plan": "\n".join([f"STEP {i+1}. {step}" for i, step in enumerate(state["steps"])]),
                "future_step": last_plan_step_index+1,
                "current_step": last_plan_step_index,
                "intermediate_steps": '\n\n'.join([f"\nOUTPUT for STEP {i_f+1}: \n{(chr(92)+'n').join([g.content for g in _results[i_f]])}" \
                                                for i_f in range(last_plan_step_index)]),})
        
        result = planning_chain.invoke({"messages": [HumanMessage(plan_str)]})
        matches = self.plan_parser(result)

        self.agent_state["steps"] = matches
        self.agent_state["plan_string"] = result.content
        self.agent_state["messages"] = add_messages(state["messages"], [result])
        return self.agent_state

    def react_agent_node(self, state: ReWOO) -> Dict:
        global GLOBAL_LLM
        global REASONING_LLM
        for state_key in state.keys():
            self.agent_state[state_key] = state[state_key]
        if self.tool_messages_offset<0:
            self.tool_messages_offset = len(self.agent_state["messages"])

        self.set_node_tools(tool_list=['list_directory', 'read_nusmv', 'save_nusmv', 'execute_nusmv', 'debug_nusmv', 'crag_tool', 'direct_response'])
        llm_with_stop = GLOBAL_LLM.bind(stop=custom_stop['observation'])
        # if REASONING_LLM is not None:
        #     llm_with_stop = REASONING_LLM.bind(stop=custom_stop['observation'])
        executor_system_prompt = '\n'.join([EXECUTE_STEP_SYSTEM, FUNCTION_PROMPT_TEMPLATE, RESPONSE_FORMAT_SYSTEM])
        react_prompt = ChatPromptTemplate.from_messages([
            ("system", executor_system_prompt),
            ("human", HUMAN_PROMPT_TEMPLATE),
            MessagesPlaceholder(variable_name="messages", optional=True),
            ])
        react_prompt_with_tools = react_prompt.partial(tools=self.tools_renderer, tool_names=self.tool_names)
        executor_chain = (react_prompt_with_tools | llm_with_stop)
        response = executor_chain.invoke({
            "input": NUSMV_PLANNER_EXECUTION_TASK.format(model_name=self.model_file),
            "messages": self.agent_state["messages"][self.tool_messages_offset:] if len(self.agent_state["messages"])>self.tool_messages_offset else [],
        })
        print("REACT AGENT EXECUTED SUCESSFULLY!!!!!!!!!!!!!!")
        response.content = f"{response.content}\n\nObservation: "
        self.agent_state["messages"] = add_messages(state["messages"], [response])
        return self.agent_state
    
    def tool_execution_node(self, state: ReWOO) -> Dict:
        global CODING_LLM
        for state_key in state.keys():
            self.agent_state[state_key] = state[state_key]
        # set_conversation_buffer(self.agent_state["messages"])
        set_conversation_buffer(self.agent_state["messages"][self.tool_messages_offset:] if len(self.agent_state["messages"])>self.tool_messages_offset else [])
        def check_action_key(action_message):
            for k in custom_stop["action"]:
                if k in action_message:
                    return k
            return False
        
        TOOLS_LLM = bind_tools_func(self.tools+[final_answer])
        tool_message = state["messages"][-1].content
        tool_prompt = ChatPromptTemplate.from_messages([
            ("system", TOOL_PROMPT_SYSTEM),
            ("human", TOOL_PROMPT_TEMPLATE),
        ])
        tool_model = (tool_prompt | TOOLS_LLM)
        
        try:
            if action := check_action_key(tool_message):
                tool_message_str = tool_message[tool_message.find(action)+len(action):]
            else:
                tool_message_str = copy.deepcopy(tool_message)
            parsed_dict = json_output_parser(CODING_LLM, tool_message_str, required_keys=["tool_name", "tool_input"], force_parse=True)
            print(parsed_dict)
        except Exception as e:
            # raise(ValueError(ERROR_PROMPT_TEMPLATE.format(error_description=e)))
            print(f"Parser failed with error: {e}")
            if action := check_action_key(tool_message):
                tool_message_str = tool_message[tool_message.find(action)+len(action):]
            else:
                tool_message_str = json.dumps({
                    "tool_name": "direct_response", 
                    "tool_input": 'Initiate the next "Thought-Action-Observation" step and ensure that your response aligns with the specified $JSON_BLOB format.',
                })
            print(tool_message_str)
        finally:
            if '```json' in tool_message_str:
                tool_message_str = tool_message_str[tool_message_str.find('```json')+len('```json'):tool_message_str.rfind('```')]
            response = asyncio.run(tool_model.ainvoke({
                "tool_call": tool_message_str,
            }))
            try:
                print(response)
                response = agentic_graph.format_aimessage_for_toolcall(response)
                tool_response = asyncio.run(self.tool_node.ainvoke({"messages": [response]}))
            except Exception as e:
                response.tool_calls.append({"name": "invalid"})
                tool_response = {"messages": []}
                print(f"Tool execution failed with error: {e}")
            if len(tool_response["messages"])==0:
                error_msg = ERROR_PROMPT_TEMPLATE.format(error_description=f"The specified action {tool_message_str} could not be parsed.")
                tool_response["messages"].append(AIMessage(
                    content=error_msg)
                )
                print(tool_response)
            
        if response.tool_calls[-1]["name"]=="final_answer":
            self.final_answer_tool_flag = True
        self.agent_state["messages"] = add_messages(state["messages"], tool_response["messages"])
        return self.agent_state
    
    def solver(self, state: ReWOO) -> Literal["__end__", "nusmv_react_agent_node"]:
        response = state["messages"][-1]
        if self.final_answer_tool_flag:
            return "__end__"
        elif response.type=="tool":
            return "nusmv_react_agent_node"
        return "__end__"
class properties_agent:
    def __init__(self, filesys_root, previous_conversation=[], SOP_pth = None):
        if SOP_pth is not None and isfile(SOP_pth):
            text = ""
            with open(SOP_pth, "rb") as file:
                pdf = PyPDF2.PdfReader(file)
                for page_num in range(len(pdf.pages)):
                    page = pdf.pages[page_num]
                    text += page.extract_text()
                self.SOP_txt = text
            print("SOP PDF successfully read.")
        else:
            self.SOP_txt = None
            print("SOP PDF could not be read.")
        set_sop_file(SOP_pth)

        self.agent_state = ReWOO(messages=[])
        self.react_messages = {"messages": []}
        self.tools = None
        self.tool_name = None
        self.tools_renderer = None
        self.tool_node = None

        self.filesystem_root = filesys_root
        #self.properties_pth = join(self.filesystem_root, "properties.json")

        self.tool_messages_offset = -1
        self.final_answer_tool_flag = False
        self.properties_content = None
        self.PREVIOUS_CONVERSATION = previous_conversation
        self.plan_prompt1_flag = False
        self.plan_prompt2_flag = False

    @staticmethod
    def plan_parser(text: AIMessage) -> List[str]:
        steps = [v for v in re.split("\n\s*\d+\. ", text.content)[1:]]
        return steps

    def set_node_tools(self, tool_list):
        self.tools = [tool_dict[f] for f in tool_list if f in tool_dict.keys()]
        self.tool_names = [f for f in tool_list if f in tool_dict.keys()]
        self.tools_renderer = render_text_description_and_args(self.tools)
        self.tool_node = ToolNode(self.tools+[final_answer])
        return self.tools, self.tool_names, self.tools_renderer, self.tool_node
    
    def planning_node1(self, state:ReWOO) -> Dict:
        global GLOBAL_LLM
        for state_key in state.keys():
            self.agent_state[state_key] = state[state_key]
        
        self.set_node_tools(tool_list=['crag_tool', 'direct_response'])
        llm_with_stop = GLOBAL_LLM.bind(stop=custom_stop['plan'])
        plan_prompt1 = ChatPromptTemplate.from_messages([
            ('system', PROP_PLAN_SYSPROMPT_1),
            MessagesPlaceholder(variable_name="previous_conversation", optional=True),
            ('human', PROP_PLAN_PROMPT_1),
        ])
        self.plan_prompt1_flag = True
        plan_prompt_with_tools = plan_prompt1.partial(tools=self.tools_renderer, tool_names = self.tool_names)
        planning_chain = (plan_prompt_with_tools | llm_with_stop)

        self.agent_state['task'] = PROP_TASK_PROMPT_1
        result = planning_chain.invoke({'previous_conversation': self.PREVIOUS_CONVERSATION})
        matches = self.plan_parser(result)
        self.agent_state["steps"] = matches
        self.agent_state["plan_string"] = result.content
        self.agent_state["messages"] = add_messages(state["messages"], [result])
        # print(self.agent_state)
        return self.agent_state
    
    def planning_node2(self, state:ReWOO) -> Dict:
        global GLOBAL_LLM
        for state_key in state.keys():
            self.agent_state[state_key] = state[state_key]
        # print("KRIPKE AGENT")
        # print(self.agent_state["messages"])
        # set_conversation_buffer(self.agent_state["messages"])
        
        self.set_node_tools(tool_list=['list_directory', 'save_nusmv', 'crag_tool', 'generate_spec_properties', 'generate_nl_descriptions', 'categorize_properties', 'direct_response'])
        llm_with_stop = GLOBAL_LLM.bind(stop=custom_stop['plan'])
        plan_prompt2 = ChatPromptTemplate.from_messages([
            ('system', PROP_PLAN_SYSPROMPT),
            MessagesPlaceholder(variable_name="previous_conversation", optional=True),
            ('human', PROP_PLAN_PROMPT),
        ])
        self.plan_prompt2_flag = True
        plan_prompt_with_tools = plan_prompt2.partial(tools=self.tools_renderer, tool_names = self.tool_names)
        planning_chain = (plan_prompt_with_tools | llm_with_stop)

        self.agent_state['task'] = PROP_TASK_PROMPT.format(SOP_TXT = self.SOP_txt)
        result = planning_chain.invoke({'previous_conversation':self.agent_state["messages"],
                                        'PROP_TASK': self.agent_state['task']})
        matches = self.plan_parser(result)
        self.agent_state["steps"] = matches
        self.agent_state["plan_string"] = result.content
        self.agent_state["messages"] = add_messages(state["messages"], [result])
        # print(self.agent_state)
        return self.agent_state
    
    def replanning_node(self, state:ReWOO) -> Dict:
        global GLOBAL_LLM
        for state_key in state.keys():
            self.agent_state[state_key] = state[state_key]

        self.set_node_tools(tool_list=['list_directory', 'save_nusmv', 'crag_tool', 'generate_spec_properties', 'generate_nl_descriptions', 'categorize_properties', 'direct_response'])
        llm_with_stop = GLOBAL_LLM.bind(stop=custom_stop['plan'])   
        plan_prompt = ChatPromptTemplate.from_messages([
            ("system", PROP_PLAN_SYSPROMPT),
            MessagesPlaceholder(variable_name="messages", optional=True)
        ])
        plan_prompt_with_tools = plan_prompt.partial(tools=self.tools_renderer, tool_names=self.tool_names)
        planning_chain = (plan_prompt_with_tools | llm_with_stop)
        last_plan_step_index = self.get_current_step(state)-1
        _results = (state["results"] or {}) if "results" in state else {}
        plan_str = PROP_CORRECTION_PROMPT.format(**{
                "kripke": self.kripke_config,
                'DOC_TXT': self.SOP_txt,
                "plan": "\n".join([f"STEP {i+1}. {step}" for i, step in enumerate(state["steps"])]),
                "future_step": last_plan_step_index+1,
                "current_step": last_plan_step_index,
                "intermediate_steps": '\n\n'.join([f"\nOUTPUT for STEP {i_f+1}: \n{(chr(92)+'n').join([g.content for g in _results[i_f]])}" \
                                                for i_f in range(last_plan_step_index)]),})
        
        result = planning_chain.invoke({"messages": [HumanMessage(plan_str)]})
        matches = self.plan_parser(result)

        self.agent_state["steps"] = matches
        self.agent_state["plan_string"] = result.content
        self.agent_state["messages"] = add_messages(state["messages"], [result])
        return self.agent_state

    def react_agent_node(self, state: ReWOO) -> Dict:
        global GLOBAL_LLM
        for state_key in state.keys():
            self.agent_state[state_key] = state[state_key]
        if self.tool_messages_offset<0:
            self.tool_messages_offset = len(self.agent_state["messages"])

        self.set_node_tools(tool_list=['list_directory', 'read_file', 'direct_response'])    
        llm_with_stop = GLOBAL_LLM.bind(stop=custom_stop['observation'])
        executor_system_prompt = '\n'.join([EXECUTE_STEP_SYSTEM, FUNCTION_PROMPT_TEMPLATE, RESPONSE_FORMAT_SYSTEM])
        react_prompt = ChatPromptTemplate.from_messages([
            ("system", executor_system_prompt),
            ("human", HUMAN_PROMPT_TEMPLATE),
            MessagesPlaceholder(variable_name="messages", optional=True),
            ])
        react_prompt_with_tools = react_prompt.partial(tools=self.tools_renderer, tool_names=self.tool_names)
        executor_chain = (react_prompt_with_tools | llm_with_stop)
        response = executor_chain.invoke({
            "input": PROP_EXECUTION_TASK,
            "messages": self.agent_state["messages"][self.tool_messages_offset:] if len(self.agent_state["messages"])>self.tool_messages_offset else [],
        })
        print("REACT AGENT EXECUTED SUCESSFULLY!!!!!!!!!!!!!!")
        response.content = f"{response.content}\n\nObservation: "
        self.agent_state["messages"] = add_messages(state["messages"], [response])
        return self.agent_state
    
    def tool_execution_node(self, state: ReWOO) -> Dict:
        global CODING_LLM
        for state_key in state.keys():
            self.agent_state[state_key] = state[state_key]
        # set_conversation_buffer(self.agent_state["messages"])
        set_conversation_buffer(self.agent_state["messages"][self.tool_messages_offset:] if len(self.agent_state["messages"])>self.tool_messages_offset else [])
        def check_action_key(action_message):
            for k in custom_stop["action"]:
                if k in action_message:
                    return k
            return False
        
        TOOLS_LLM = bind_tools_func(self.tools+[final_answer])
        tool_message = state["messages"][-1].content
        tool_prompt = ChatPromptTemplate.from_messages([
            ("system", TOOL_PROMPT_SYSTEM),
            ("human", TOOL_PROMPT_TEMPLATE),
        ])
        tool_model = (tool_prompt | TOOLS_LLM)
        
        try:
            if action := check_action_key(tool_message):
                tool_message_str = tool_message[tool_message.find(action)+len(action):]
            else:
                tool_message_str = copy.deepcopy(tool_message)
            parsed_dict = json_output_parser(CODING_LLM, tool_message_str, required_keys=["tool_name", "tool_input"], force_parse=True)
            print(parsed_dict)
        except Exception as e:
            # raise(ValueError(ERROR_PROMPT_TEMPLATE.format(error_description=e)))
            print(f"Parser failed with error: {e}")
            if action := check_action_key(tool_message):
                tool_message_str = tool_message[tool_message.find(action)+len(action):]
            else:
                tool_message_str = json.dumps({
                    "tool_name": "direct_response", 
                    "tool_input": 'Initiate the next "Thought-Action-Observation" step and ensure that your response aligns with the specified $JSON_BLOB format.',
                })
            print(tool_message_str)
        finally:
            if '```json' in tool_message_str:
                tool_message_str = tool_message_str[tool_message_str.find('```json')+len('```json'):tool_message_str.rfind('```')]
            response = asyncio.run(tool_model.ainvoke({
                "tool_call": tool_message_str,
            }))
            try:
                print(response)
                response = agentic_graph.format_aimessage_for_toolcall(response)
                tool_response = asyncio.run(self.tool_node.ainvoke({"messages": [response]}))
            except Exception as e:
                response.tool_calls.append({"name": "invalid"})
                tool_response = {"messages": []}
                print(f"Tool execution failed with error: {e}")
            if len(tool_response["messages"])==0:
                error_msg = ERROR_PROMPT_TEMPLATE.format(error_description=f"The specified action {tool_message_str} could not be parsed.")
                tool_response["messages"].append(AIMessage(
                    content=error_msg)
                )
                print(tool_response)
            
        if response.tool_calls[-1]["name"]=="final_answer":
            self.final_answer_tool_flag = True
        self.agent_state["messages"] = add_messages(state["messages"], tool_response["messages"])
        return self.agent_state
    
    def solver(self, state: ReWOO) -> Literal["__end__", "prop_react_agent_node"]:
        response = state["messages"][-1]
        if self.final_answer_tool_flag:
            return "__end__"
        elif response.type=="tool":
            return "prop_react_agent_node"
        return "__end__"
    
class kripke_agent:
    def __init__(self, filesys_root, SOP_pth=None, previous_conversation=[]):
        if SOP_pth is not None and isfile(SOP_pth):
            text = ""
            with open(SOP_pth, "rb") as file:
                pdf = PyPDF2.PdfReader(file)
                for page_num in range(len(pdf.pages)):
                    page = pdf.pages[page_num]
                    text += page.extract_text()
                self.SOP_txt = text
            print("SOP PDF successfully read.")
        else:
            self.SOP_txt = None
            print("SOP PDF could not be read.")
        set_sop_file(SOP_pth)
        
        self.agent_state = ReWOO(messages=[])
        self.react_messages = {"messages": []}
        self.tools = None
        self.tool_names = None
        self.tools_renderer = None
        self.tool_node = None

        self.filesys_root = filesys_root
        
        self.tool_messages_offset = -1
        self.final_answer_tool_flag = False
        self.kripke_struct = None
        self.PREVIOUS_CONVERSATION = previous_conversation
        self.plan_prompt1_flag = False
        self.plan_prompt2_flag = False
    
    @staticmethod
    def plan_parser(text: AIMessage) -> List[str]:
        steps = [v for v in re.split("\n\s*\d+\. ", text.content)[1:]]
        return steps

    def set_node_tools(self, tool_list):
        self.tools = [tool_dict[f] for f in tool_list if f in tool_dict.keys()]
        self.tool_names = [f for f in tool_list if f in tool_dict.keys()]
        self.tools_renderer = render_text_description_and_args(self.tools)
        self.tool_node = ToolNode(self.tools+[final_answer])
        return self.tools, self.tool_names, self.tools_renderer, self.tool_node
    
    def planning_node1(self, state:ReWOO) -> Dict:
        global GLOBAL_LLM
        for state_key in state.keys():
            self.agent_state[state_key] = state[state_key]
        
        self.set_node_tools(tool_list=['crag_tool', 'direct_response'])
        llm_with_stop = GLOBAL_LLM.bind(stop=custom_stop['plan'])
        plan_prompt1 = ChatPromptTemplate.from_messages([
            ('system', PLAN_SYSPROMPT),
            MessagesPlaceholder(variable_name="previous_conversation", optional=True),
            ('human', PLAN_PROMPT_1),
        ])
        self.plan_prompt1_flag = True
        plan_prompt_with_tools = plan_prompt1.partial(tools=self.tools_renderer, tool_names = self.tool_names)
        planning_chain = (plan_prompt_with_tools | llm_with_stop)

        self.agent_state['task'] = TASK_PROMPT_1
        result = planning_chain.invoke({'previous_conversation':self.PREVIOUS_CONVERSATION})
        matches = self.plan_parser(result)
        self.agent_state["steps"] = matches
        self.agent_state["plan_string"] = result.content
        self.agent_state["messages"] = add_messages(state["messages"], [result])
        # print(self.agent_state)
        return self.agent_state
    
    def planning_node2(self, state:ReWOO) -> Dict:
        global GLOBAL_LLM
        for state_key in state.keys():
            self.agent_state[state_key] = state[state_key]
        # print("KRIPKE AGENT")
        # print(self.agent_state["messages"])
        # set_conversation_buffer(self.agent_state["messages"])
        
        self.set_node_tools(tool_list=['list_directory', 'save_nusmv', 'crag_tool', 'create_states', 'create_transitions', 'direct_response'])
        llm_with_stop = GLOBAL_LLM.bind(stop=custom_stop['plan'])
        plan_prompt2 = ChatPromptTemplate.from_messages([
            ('system', PLAN_SYSPROMPT),
            MessagesPlaceholder(variable_name="previous_conversation", optional=True),
            ('human', PLAN_PROMPT),
        ])
        self.plan_prompt2_flag = True
        plan_prompt_with_tools = plan_prompt2.partial(tools=self.tools_renderer, tool_names = self.tool_names)
        planning_chain = (plan_prompt_with_tools | llm_with_stop)

        self.agent_state['task'] = TASK_PROMPT.format(SOP_TXT = self.SOP_txt)
        result = planning_chain.invoke({'previous_conversation':self.agent_state["messages"],
                                        'SOP_TASK': self.agent_state['task']})
        matches = self.plan_parser(result)
        self.agent_state["steps"] = matches
        self.agent_state["plan_string"] = result.content
        self.agent_state["messages"] = add_messages(state["messages"], [result])
        # print(self.agent_state)
        return self.agent_state
    
    def replanning_node(self, state:ReWOO)-> Dict:
        global GLOBAL_LLM
        for state_key in state.keys():
            self.agent_state[state_key] = state[state_key]
        
        self.set_node_tools(tool_list=['list_directory', 'save_nusmv', 'crag_tool', 'create_states', 'create_transitions', 'direct_response'])
        llm_with_stop = GLOBAL_LLM.bind(stop=custom_stop['plan'])
        plan_prompt = ChatPromptTemplate.from_messages([
            ('system',PLAN_SYSPROMPT),
            MessagesPlaceholder(variable_name="messages", optional=True)
        ])
        plan_prompt_with_tools = plan_prompt.partial(tools= self.tools_renderer, tool_names=self.tool_names)
        planning_chain = (plan_prompt_with_tools | llm_with_stop)

        last_plan_step_index = self.get_current_step(state)-1
        _results = (state["results"] or {}) if "results" in state else {}
        plan_str = CORRECTION_PROMPT.format(**{
            "DOC_TXT": self.SOP_txt,
            "plan": "\n".join([f"STEP {i+1}. {step}" for i, step in enumerate(state["steps"])]),
            "future_step": last_plan_step_index+1,
            "current_step": last_plan_step_index,
            "intermediate_steps": '\n\n'.join([f"\nOUTPUT for STEP {i_f+1}: \n{(chr(92)+'n').join([g.content for g in _results[i_f]])}" \
                                                for i_f in range(last_plan_step_index)]),})
        
        result = planning_chain.invoke({"messages": [HumanMessage(plan_str)]})
        matches = self.plan_parser(result)

        self.agent_state["steps"] = matches
        self.agent_state["plan_string"] = result.content
        self.agent_state["messages"] = add_messages(state["messages"], [result])
        return self.agent_state
    
    def react_agent_node(self, state: ReWOO) -> Dict:
        global GLOBAL_LLM
        for state_key in state.keys():
            self.agent_state[state_key] = state[state_key]
        if self.tool_messages_offset<0:
            self.tool_messages_offset = len(self.agent_state["messages"])
        
        self.set_node_tools(tool_list=['list_directory', 'read_file', 'direct_response'])
        llm_with_stop = GLOBAL_LLM.bind(stop=custom_stop['observation'])
        executor_system_prompt = '\n'.join([EXECUTE_STEP_SYSTEM, FUNCTION_PROMPT_TEMPLATE, RESPONSE_FORMAT_SYSTEM])
        react_prompt = ChatPromptTemplate.from_messages([
            ("system", executor_system_prompt),
            ("human", HUMAN_PROMPT_TEMPLATE),
            MessagesPlaceholder(variable_name="messages", optional=True),
            ])
        react_prompt_with_tools = react_prompt.partial(tools=self.tools_renderer, tool_names=self.tool_names)
        executor_chain = (react_prompt_with_tools | llm_with_stop)
        response = executor_chain.invoke({
            "input": NUSMV_EXECUTION_TASK,
            "messages": self.agent_state["messages"][self.tool_messages_offset:] if len(self.agent_state["messages"])>self.tool_messages_offset else [],
        })
        print("REACT AGENT EXECUTED SUCESSFULLY!!!!!!!!!!!!!!")
        response.content = f"{response.content}\n\nObservation: "
        self.agent_state["messages"] = add_messages(state["messages"], [response])
        return self.agent_state
    
    def tool_execution_node(self, state: ReWOO) -> Dict:
        global CODING_LLM
        for state_key in state.keys():
            self.agent_state[state_key] = state[state_key]
        # set_conversation_buffer(self.agent_state["messages"])
        set_conversation_buffer(self.agent_state["messages"][self.tool_messages_offset:] if len(self.agent_state["messages"])>self.tool_messages_offset else [])
        def check_action_key(action_message):
            for k in custom_stop["action"]:
                if k in action_message:
                    return k
            return False
        
        TOOLS_LLM = bind_tools_func(self.tools+[final_answer])
        tool_message = state["messages"][-1].content
        tool_prompt = ChatPromptTemplate.from_messages([
            ("system", TOOL_PROMPT_SYSTEM),
            ("human", TOOL_PROMPT_TEMPLATE),
        ])
        tool_model = (tool_prompt | TOOLS_LLM)
        
        try:
            if action := check_action_key(tool_message):
                tool_message_str = tool_message[tool_message.find(action)+len(action):]
            else:
                tool_message_str = copy.deepcopy(tool_message)
            parsed_dict = json_output_parser(CODING_LLM, tool_message_str, required_keys=["tool_name", "tool_input"], force_parse=True)
            print(parsed_dict)
        except Exception as e:
            # raise(ValueError(ERROR_PROMPT_TEMPLATE.format(error_description=e)))
            print(f"Parser failed with error: {e}")
            if action := check_action_key(tool_message):
                tool_message_str = tool_message[tool_message.find(action)+len(action):]
            else:
                tool_message_str = json.dumps({
                    "tool_name": "direct_response", 
                    "tool_input": 'Initiate the next "Thought-Action-Observation" step and ensure that your response aligns with the specified $JSON_BLOB format.',
                })
            print(tool_message_str)
        finally:
            if '```json' in tool_message_str:
                tool_message_str = tool_message_str[tool_message_str.find('```json')+len('```json'):tool_message_str.rfind('```')]
            response = asyncio.run(tool_model.ainvoke({
                "tool_call": tool_message_str,
            }))
            try:
                print(response)
                response = agentic_graph.format_aimessage_for_toolcall(response)
                tool_response = asyncio.run(self.tool_node.ainvoke({"messages": [response]}))
            except Exception as e:
                response.tool_calls.append({"name": "invalid"})
                tool_response = {"messages": []}
                print(f"Tool execution failed with error: {e}")
            if len(tool_response["messages"])==0:
                error_msg = ERROR_PROMPT_TEMPLATE.format(error_description=f"The specified action {tool_message_str} could not be parsed.")
                tool_response["messages"].append(AIMessage(
                    content=error_msg)
                )
                print(tool_response)
            
        if response.tool_calls[-1]["name"]=="final_answer":
            self.final_answer_tool_flag = True
        self.agent_state["messages"] = add_messages(state["messages"], tool_response["messages"])
        return self.agent_state
    
    def solver(self, state: ReWOO) -> Literal["__end__", "kripke_react_agent_node"]:
        response = state["messages"][-1]
        if self.final_answer_tool_flag:
            return "__end__"
        elif response.type=="tool":
            return "kripke_react_agent_node"
        return "__end__"

class agentic_graph:
    def __init__(self, filesystem_root, ltool_dict: Dict=None):
        if ltool_dict is None:
            self.tools = [f for f in tool_dict.values()]
            self.tool_names = [f for f in tool_dict.keys()]
        else:
            self.tools = [f for f in ltool_dict.values()]
            self.tool_names = [f for f in ltool_dict.keys()]
        self.tools_renderer = render_text_description_and_args(self.tools)
        self.tool_node = ToolNode(self.tools+[final_answer])
        self.multi_plan_offset = 0
        
        self.agent_state = ReWOO(messages=[])
        self.filesystem_root = filesystem_root
        self.kripke_extractor: Optional[kripke_agent] = None
        self.properties_extractor: Optional[properties_agent] = None
        self.nusmv_planner: Optional[NUSMV_planner] = None
        
        self.PREVIOUS_CONVERSATION = []
        self.GLOBAL_PLAN = []
        self.GLOBAL_ANSWERS = []
        self.FINAL_ANSWER = None
        
    def set_auxilary_planner(self, 
                             kripke_extractor: kripke_agent = None,
                             properties_extractor: properties_agent = None,
                             nusmv_planner: NUSMV_planner = None):
        if kripke_extractor:
            self.kripke_extractor = kripke_extractor
        if properties_extractor:
            self.properties_extractor = properties_extractor
        if nusmv_planner:
            self.nusmv_planner = nusmv_planner
        
        
        return self.kripke_extractor, self.properties_extractor, self.nusmv_planner

    def set_node_tools(self, tool_list):
        if "All" in tool_list:
            tool_list = [f for f in tool_dict.keys()]
        elif "direct_response" not in tool_list:
            tool_list += ["direct_response"]
        
        self.tools = [tool_dict[f] for f in tool_list if f in tool_dict.keys()]
        self.tool_names = [f for f in tool_list if f in tool_dict.keys()]
        self.tools_renderer = render_text_description_and_args(self.tools)
        self.tool_node = ToolNode(self.tools+[final_answer])
        print(self.tools_renderer)
        return self.tools, self.tool_names, self.tools_renderer, self.tool_node

    def set_previous_conversation(self, buffer):
        self.PREVIOUS_CONVERSATION = buffer
        return self.PREVIOUS_CONVERSATION

    @staticmethod
    def format_aimessage_for_toolcall(ai_message: AIMessage) -> AIMessage:
        tool_message = AIMessage(content="")
        print("Formatting TOOL CALLS")
        if hasattr(ai_message, 'invalid_tool_calls'):
            print("Invalid TOOL CALLS:", ai_message.invalid_tool_calls)
        if ai_message.additional_kwargs.get("tool_calls", None):
            try:
                assert len(ai_message.tool_calls)
                tool_message.additional_kwargs = ai_message.additional_kwargs
                tool_message.tool_calls = [{
                    "name": ai_message.tool_calls[f]['name'],
                    "args": ai_message.tool_calls[f]['args'],
                    "id": ai_message.tool_calls[f]['id'],
                    "type": "tool_call",
                } for f in range(len(ai_message.tool_calls))]
                tool_message.id = ai_message.id
                tool_message.response_metadata = ai_message.response_metadata
                return tool_message
            except Exception as e:
                raise ValueError("Tool arguments could not be parsed.")
        
        message_dict = json_output_parser(CODING_LLM, ai_message.content, required_keys=["name", "arguments"])
        if type(message_dict["arguments"])==str:
            argument_dict = json_output_parser(CODING_LLM, message_dict["arguments"], required_keys=None, force_parse=False)
        elif type(message_dict["arguments"])==dict:
            argument_dict = message_dict["arguments"]
        else:
            e = type(message_dict["arguments"])
            raise ValueError(f"argument type is unknown: {e}")
        print(argument_dict)
        
        tool_id = f"call_{uuid.uuid4()}"
        tool_message.additional_kwargs = { "tool_calls":[{
            "id": tool_id,
            "function": {
                "name": message_dict["name"], 
                "arguments": message_dict["arguments"] if type(message_dict["arguments"])==str else json.dumps(message_dict["arguments"])
                },
            "type": "function",
        }]}
        tool_message.tool_calls = [{
            "name": message_dict["name"],
            "args": argument_dict,
            "id": tool_id,
            "type": "tool_call",
        }]
        tool_message.id = ai_message.id
        tool_message.response_metadata = ai_message.response_metadata
        return tool_message

    @staticmethod
    def get_current_step(state: ReWOO):
        if "results" not in state or state["results"] is None:
            return 1
        if len(state["results"]) > len(state["steps"]):
            return None
        else:
            return len(state["results"])
        
    @staticmethod
    def plan_parser(text: AIMessage) -> List[str]:
        steps = [v for v in re.split("\n\s*\d+\. ", text.content)[1:]]
        return steps
    
    def verification_node(self, state: ReWOO) -> Dict:
        for state_key in state.keys():
            self.agent_state[state_key] = state[state_key]
        print("Verifying final answer...")
        set_conversation_buffer(state["messages"][self.multi_plan_offset:])
        response_dict = verification_function(instruction_prompt=state["messages"][self.multi_plan_offset], final_answer=state["messages"][-1])
        response = AIMessage(content="The final answer has a confidence of **'{}'** on a scale of 1 to 10. \n\nExplanation: {}".format(response_dict["score"], response_dict["explanation"]))
        self.GLOBAL_PLAN = state["steps"]
        self.GLOBAL_ANSWERS = state["results"]
        self.FINAL_ANSWER = state["messages"][-1]
        if self.nusmv_planner is not None and self.nusmv_planner.plan_prompt2_flag==False:
            print("Messages RESET")
            self.agent_state["messages"] = [self.FINAL_ANSWER]
            self.agent_state["task"] = None
            self.agent_state["result"] = None
            self.agent_state["results"] = None
            self.multi_plan_offset = len(state["messages"])
        if self.properties_extractor is not None and self.properties_extractor.plan_prompt2_flag==False:
            print("Messages RESET")
            self.agent_state["messages"] = [self.FINAL_ANSWER]
            self.agent_state["task"] = None
            self.agent_state["result"] = None
            self.agent_state["results"] = None
            self.multi_plan_offset = len(state["messages"])
        elif self.kripke_extractor is not None and self.kripke_extractor.plan_prompt2_flag==False:
            print("Messages RESET")
            self.agent_state["messages"] = [self.FINAL_ANSWER]
            self.agent_state["task"] = None
            self.agent_state["result"] = None
            self.agent_state["results"] = None
            self.multi_plan_offset = len(state["messages"])
        else:
            self.agent_state["messages"] = add_messages(state["messages"], [response])
        return self.agent_state

    def tool_execution_node(self, state: ReWOO) -> Dict:
        global CODING_LLM
        for state_key in state.keys():
            self.agent_state[state_key] = state[state_key]
        
        set_conversation_buffer(self.agent_state["messages"][self.multi_plan_offset:])
        def check_action_key(action_message):
            for k in custom_stop["action"]:
                if k in action_message:
                    return k
            return False
        
        TOOLS_LLM = bind_tools_func(self.tools+[final_answer])
        tool_message = state["messages"][-1].content
        tool_prompt = ChatPromptTemplate.from_messages([
            ("system", TOOL_PROMPT_SYSTEM),
            ("human", TOOL_PROMPT_TEMPLATE),
        ])
        tool_model = (tool_prompt | TOOLS_LLM)
        
        try:
            if action := check_action_key(tool_message):
                tool_message_str = tool_message[tool_message.find(action)+len(action):]
            else:
                tool_message_str = copy.deepcopy(tool_message)
            parsed_dict = json_output_parser(CODING_LLM, tool_message_str, required_keys=["tool_name", "tool_input"], force_parse=True)
            print(parsed_dict)
        except Exception as e:
            # raise(ValueError(ERROR_PROMPT_TEMPLATE.format(error_description=e)))
            print(f"Parser failed with error: {e}")
            if action := check_action_key(tool_message):
                tool_message_str = tool_message[tool_message.find(action)+len(action):]
            else:
                tool_message_str = json.dumps({
                    "tool_name": "direct_response", 
                    "tool_input": 'Initiate the next "Thought-Action-Observation" step and ensure that your response aligns with the specified $JSON_BLOB format.',
                })
            print(tool_message_str)
        finally:
            if '```json' in tool_message_str:
                tool_message_str = tool_message_str[tool_message_str.find('```json')+len('```json'):tool_message_str.rfind('```')]
            response = asyncio.run(tool_model.ainvoke({
                "tool_call": tool_message_str,
            }))
            try:
                print(response)
                response = self.format_aimessage_for_toolcall(response)
                tool_response = asyncio.run(self.tool_node.ainvoke({"messages": [response]}))
            except Exception as e:
                response.tool_calls.append({"name": "invalid"})
                tool_response = {"messages": []}
                print(f"Tool execution failed with error: {e}")
            if len(tool_response["messages"])==0:
                error_msg = ERROR_PROMPT_TEMPLATE.format(error_description=f"The specified action {tool_message_str} could not be parsed.")
                tool_response["messages"].append(AIMessage(
                    content=error_msg)
                )
                print(tool_response)
        
        last_plan_step_index = self.get_current_step(state)-1
        _results = (state["results"] or {}) if "results" in state else {}
        _results[last_plan_step_index].append(AIMessage(content=tool_response["messages"][-1].content))
        if response.tool_calls[-1]["name"]=="final_answer":
            if last_plan_step_index+1==len(state["steps"]):
                self.agent_state["result"] = tool_response["messages"][-1].content
                self.agent_state["results"] = _results
                self.agent_state["messages"] = add_messages(state["messages"], tool_response["messages"])
            else:
                print(f"Adding new step {last_plan_step_index+1}.")
                _results[last_plan_step_index+1] = []
                self.agent_state["result"] = None
                self.agent_state["results"] = _results
                self.agent_state["messages"] = add_messages(state["messages"], tool_response["messages"])
            # print(self.agent_state)
            print(f"Step {last_plan_step_index} completed.")
            return self.agent_state
        
        self.agent_state["result"] = None
        self.agent_state["results"] = _results
        self.agent_state["messages"] = add_messages(state["messages"], tool_response["messages"])
        # print(self.agent_state)
        print(f"Processing step {last_plan_step_index}")
        return self.agent_state

    def planning_node(self, state: ReWOO) -> Dict:
        global GLOBAL_LLM
        for state_key in state.keys():
            self.agent_state[state_key] = state[state_key]
        
        llm_with_stop = GLOBAL_LLM.bind(stop=custom_stop['plan'])
        plan_prompt = ChatPromptTemplate.from_messages([
            ('system', PLANNING_STEP_SYSTEM),
            MessagesPlaceholder(variable_name="previous_conversation", optional=True),
            ('human', PLANNING_STEP_TEMPLATE),
        ])
        plan_prompt_with_tools = plan_prompt.partial(tools=self.tools_renderer, tool_names=self.tool_names)
        planning_chain = (plan_prompt_with_tools | llm_with_stop)
        
        self.agent_state["task"] = state["messages"][-1].content
        result = planning_chain.invoke({'previous_conversation': self.PREVIOUS_CONVERSATION,
                                        'input': self.agent_state["task"]})
        matches = self.plan_parser(result)
        
        self.agent_state["steps"] = matches
        self.agent_state["plan_string"] = result.content
        self.agent_state["messages"] = add_messages(state["messages"], [result])
        # print(self.agent_state)
        return self.agent_state

    def replanning_node(self, state: ReWOO) -> Dict:
        global GLOBAL_LLM
        for state_key in state.keys():
            self.agent_state[state_key] = state[state_key]
        
        llm_with_stop = GLOBAL_LLM.bind(stop=custom_stop['plan'])
        plan_prompt = ChatPromptTemplate.from_messages([
            ("system", PLANNING_STEP_SYSTEM),
            MessagesPlaceholder(variable_name="messages", optional=True)
        ])
        plan_prompt_with_tools = plan_prompt.partial(tools=self.tools_renderer, tool_names=self.tool_names)
        planning_chain = (plan_prompt_with_tools | llm_with_stop)
        
        last_plan_step_index = self.get_current_step(state)-1
        _results = (state["results"] or {}) if "results" in state else {}
        plan_str = PLANNING_CORRECTION_TEMPLATE.format(**{
                "input": state["task"],
                "plan": "\n".join([f"STEP {i+1}. {step}" for i, step in enumerate(state["steps"])]),
                "future_step": last_plan_step_index+1,
                "current_step": last_plan_step_index,
                "intermediate_steps": '\n\n'.join([f"\nOUTPUT for STEP {i_f+1}: \n{(chr(92)+'n').join([g.content for g in _results[i_f]])}" \
                                                for i_f in range(last_plan_step_index)]),})
        
        result = planning_chain.invoke({"messages": [HumanMessage(plan_str)]})
        matches = self.plan_parser(result)
        
        self.agent_state["steps"] = matches
        self.agent_state["plan_string"] = result.content
        self.agent_state["messages"] = add_messages(state["messages"], [result])
        # print(self.agent_state)
        return self.agent_state

    def react_agent_node(self, state: ReWOO) -> Dict:
        global GLOBAL_LLM
        for state_key in state.keys():
            self.agent_state[state_key] = state[state_key]
        # print(self.agent_state)
        
        llm_with_stop = GLOBAL_LLM.bind(stop=custom_stop['observation'])
        executor_system_prompt = '\n'.join([EXECUTE_STEP_SYSTEM, FUNCTION_PROMPT_TEMPLATE, RESPONSE_FORMAT_SYSTEM])
        last_plan_step_index = self.get_current_step(state)-1
        _results = (state["results"] or {}) if "results" in state else {}
        if last_plan_step_index not in _results.keys():
            _results[last_plan_step_index] = []
        
        if last_plan_step_index is not None:
            previous_steps = "\n\n".join([f"STEP {i+1}. {step} \nOUTPUT for STEP {i+1}: \n{(chr(92)+'n').join([g.content for g in _results[i]])}" \
                                        for i, step in enumerate(state["steps"][:last_plan_step_index])])
            plan_str = STEP_PROMPT_TEMPLATE.format(
                objective=state["task"],
                previous_steps=previous_steps if len(previous_steps.strip())>0 else "No previous steps executed.",
                current_step=state["steps"][last_plan_step_index]
            )
        else:
            plan_str = None

        if len(self.PREVIOUS_CONVERSATION)==0 and self.multi_plan_offset>0:
            print("Using initial planning context...")
            self.PREVIOUS_CONVERSATION = state["messages"][:self.multi_plan_offset]
        react_prompt = ChatPromptTemplate.from_messages([
            ("system", executor_system_prompt),
            MessagesPlaceholder(variable_name="previous_conversation", optional=True),
            ])
        react_prompt_with_tools = react_prompt.partial(tools=self.tools_renderer, tool_names=self.tool_names)
        react_prompt_with_tools_value = react_prompt_with_tools.invoke({
            "previous_conversation": self.PREVIOUS_CONVERSATION
        })
        # print("\n".join([f.content for f in react_prompt_with_tools.invoke({}).messages[1:]]))
        
        if plan_str is not None:
            react_prompt_with_tools_value = ChatPromptValue(messages = \
                react_prompt_with_tools_value.to_messages() + \
                [HumanMessage(plan_str)] + \
                _results[last_plan_step_index]
            )
        else:
            react_prompt_with_tools_value = ChatPromptValue(messages = \
                react_prompt_with_tools_value.to_messages() + \
                [HumanMessage(HUMAN_PROMPT_TEMPLATE.format(input=state["messages"][0].content))] + \
                [f for f in state["messages"][1:]]
            )

        response = llm_with_stop.invoke(react_prompt_with_tools_value)
        response.content = f"{response.content}\n\nObservation: "
        _results[last_plan_step_index].append(response)
        
        self.agent_state["results"] = _results
        self.agent_state["messages"] = add_messages(state["messages"], [response])
        # print(self.agent_state)
        return self.agent_state

    def properties_plan_orchestrator(self, state: ReWOO) -> Literal["planning_node2", "prop_react_agent_node"]:
        for state_key in state.keys():
            if state_key=="messages":
                self.agent_state["messages"] = [state["messages"][-1]]
            else:
                self.agent_state[state_key] = state[state_key]
        self.properties_extractor.agent_state = self.agent_state
        print("PROPERTIES ORCHESTRATOR")
        print(self.agent_state["messages"])
        
        assert self.properties_extractor.plan_prompt1_flag
        if self.properties_extractor.plan_prompt1_flag and not self.properties_extractor.plan_prompt2_flag:
            return "planning_node2"
        return "prop_react_agent_node"

    def kripke_plan_orchestrator(self, state: ReWOO) -> Literal["planning_node2", "kripke_react_agent_node"]:
        for state_key in state.keys():
            if state_key=="messages":
                self.agent_state["messages"] = [state["messages"][-1]]
            else:
                self.agent_state[state_key] = state[state_key]
        self.kripke_extractor.agent_state = self.agent_state
        print("KRIPKE ORCHESTRATOR")
        print(self.agent_state["messages"])
        
        assert self.kripke_extractor.plan_prompt1_flag
        if self.kripke_extractor.plan_prompt1_flag and not self.kripke_extractor.plan_prompt2_flag:
            return "planning_node2"
        return "kripke_react_agent_node"
    
    def nusmv_plan_orchestrator(self, state: ReWOO) -> Literal["planning_node2", "nusmv_react_agent_node"]:
        for state_key in state.keys():
            if state_key=="messages":
                self.agent_state["messages"] = [state["messages"][-1]]
            else:
                self.agent_state[state_key] = state[state_key]
        self.nusmv_planner.agent_state = self.agent_state
        print("NuSMV ORCHESTRATOR")
        print(self.agent_state["messages"])
        
        assert self.nusmv_planner.plan_prompt1_flag
        if self.nusmv_planner.plan_prompt1_flag and not self.nusmv_planner.plan_prompt2_flag:
            return "planning_node2"
        return "nusmv_react_agent_node"
    
    def solver(self, state: ReWOO) -> Literal["react_agent_node", "verification_node", "replanning_node"]:
        last_plan_step_index = self.get_current_step(state)-1
        print(f"Solving step {last_plan_step_index}.")
        print("Total Results: {} | {}".format(len(state["results"]), [len(state["results"][f]) for f in range(len(state["results"]))]))
        if state["result"] is None:
            if len(state["results"][last_plan_step_index])==0 and last_plan_step_index<len(state["steps"])-1 and SOLVER_STRATEGY=="plan-and-execute":
                print("replanning_node")
                return "replanning_node"
            else:
                print("react_agent_node")
                return "react_agent_node"
        print("verification_node")
        return "verification_node"