import os, sys
import copy
from os import listdir
from os.path import isfile, isdir, join

import pandas as pd
import numpy as np
import pickle as pkl
import json, ast
import sqlite3
import http.client
import tempfile, uuid
import pathlib, importlib
import asyncio, subprocess
import PyPDF2

from langchain import hub
from langchain_core.tools import tool
from langchain_core.prompts import ChatPromptTemplate, MessagesPlaceholder
from langchain_core.messages import SystemMessage, HumanMessage, AIMessage, ToolMessage, AnyMessage
from langchain_core.output_parsers import StrOutputParser, JsonOutputParser
from langchain_core.runnables import RunnableLambda
from langchain.text_splitter import RecursiveCharacterTextSplitter
from langchain_community.document_loaders import PyPDFLoader, TextLoader, CSVLoader, Docx2txtLoader, UnstructuredEPubLoader, BSHTMLLoader

from utils.crag_templates import answer_system_template, answer_human_template, sop_answer_human_template
from utils.tool_templates import *
from utils.sql_utils import sql_rag_response
from utils.repl_shell import PythonREPL
from utils.web_scrapers import playwright_search, playwright_save_url
from utils.callbacks import IntermediateStepsCallback
from utils.crag_agent import set_crag_llm, InformationRetrieverAgent

from rag.cmrag import DocumentFile, get_rag, generate_rag_response, process_and_embed_documents
from rag.cmrag import set_corrective_multihop, unset_corrective_multihop

GLOBAL_LLM = None
CODING_LLM = None
REASONING_LLM = None
TOOLS_CAN_UPDATE_RAG = False
GLOBAL_RETRIES = 5
CONVERSATION_BUFFER = []

SOP_FILE = None
CODE_CONTENT = None
SMV_PATHS = []
NUSMV_CODE = []
NUSMV_CODE_OUTPUT = []

filesystem_root = join(f"{os.getenv('CACHE')}", "documents")
downloads_root = join(f"{os.getenv('CACHE')}", "downloads")
if not isdir(filesystem_root):
    os.mkdir(filesystem_root)
if not isdir(downloads_root):
    os.mkdir(downloads_root)

class DocumentLoader:
    supported_extensions = {
            ".pdf": PyPDFLoader,
            ".txt": TextLoader,
            ".py": TextLoader,
            ".smv": TextLoader,
            ".json": TextLoader,
            ".csv": CSVLoader,
            ".epub": UnstructuredEPubLoader,
            ".docx": Docx2txtLoader,
            ".html": BSHTMLLoader,
        }

    @classmethod
    def load_document(cls, file_path: str) -> str:
        ext = pathlib.Path(file_path).suffix.lower()
        loader_class = cls.supported_extensions.get(ext)
        if not loader_class:
            raise ValueError(f"Unsupported file extension: {ext}")
        if not isfile(file_path):
            raise ValueError(f"File not found: {file_path}")
        loader = loader_class(file_path=file_path)
        return loader.load()
    
def set_sop_file(file_path: str) -> bool:
    try:
        print(file_path)
        assert isfile(file_path)
        global SOP_FILE
        SOP_FILE = file_path
    except Exception as e:
        print(f"SOP file could not be read with error: {e}")
        return False
    return True

def set_smv_files(file_path: str) -> bool:
    global SMV_PATHS
    file_paths = [
        join(file_path, "states.smv"),
        join(file_path, "kripke.smv"),
        join(file_path, "properties.smv"),
    ]
    try:
        assert len(file_paths)==3
        assert isfile(file_paths[0])
        assert isfile(file_paths[1])
        assert isfile(file_paths[2])
        SMV_PATHS = [f for f in file_paths]
    except Exception as e:
        print(f"SOP file could not be read with error: {e}")
        return False
    return True

def set_reasoning_llm(reasoning_llm):
    global REASONING_LLM
    REASONING_LLM = reasoning_llm
    return REASONING_LLM

def remove_thinking_tokens(ai_message: AIMessage, start_thought_token: Optional[str]='<think>', end_thought_token: Optional[str]='</think>'):
    start_index = ai_message.content.find(start_thought_token)
    end_index = ai_message.content.rfind(end_thought_token) + len(end_thought_token)
    try:
        thought_message =  f"\n\n{ai_message.content[start_index:end_index]}\n\n"
        if end_index>-1:
            ai_message.content = ai_message.content[end_index:]
        # callback_obj = IntermediateStepsCallback()
        # callback_obj.add_to_global_stream(thought_message)
        print("THINKING TOKENS:", thought_message)
        # print("CONTENT:", ai_message.content)
    except Exception as e:
        print(f"REASONING MODEL PARSING failed with error: {e}")
    return ai_message

def set_tools_llm_state(llm, coding_llm, has_thinking_tokens=False):
    global GLOBAL_LLM, CODING_LLM
    if has_thinking_tokens:
        print("USING REASONING MODEL PARSING")
        GLOBAL_LLM = (llm | RunnableLambda(remove_thinking_tokens))
        CODING_LLM = (coding_llm | RunnableLambda(remove_thinking_tokens))
    else:
        GLOBAL_LLM = llm
        CODING_LLM = coding_llm
    return llm, GLOBAL_LLM, CODING_LLM

def set_filesystem_root(dir_path):
    global filesystem_root
    if isdir(dir_path):
        filesystem_root = dir_path
    else:
        print(f"{dir_path} is not valid")
    return filesystem_root

def get_filesystem_root():
    global filesystem_root
    return filesystem_root

def set_conversation_buffer(buffer):
    global CONVERSATION_BUFFER
    CONVERSATION_BUFFER = buffer
    return CONVERSATION_BUFFER

def verification_function(instruction_prompt, final_answer):
    global GLOBAL_LLM
    global CODING_LLM
    global CONVERSATION_BUFFER
    
    def make_previous_work(previous_work: List[AnyMessage]) -> str:
        if previous_work is None or len(previous_work)<2:
            return ""
        else:
            previous_work = "\n\n".join([f.content for f in previous_work[1:]])
            return f"This is your previous work:\n\n{previous_work}"
    
    verification_prompt = ChatPromptTemplate.from_messages([
        ("system", VERIFICATION_TEMPLATE['system']),
        ("human", VERIFICATION_TEMPLATE['human']),
    ])
    verification_chain = (verification_prompt | GLOBAL_LLM | StrOutputParser())
    
    retry_ctr = 0
    response_dict = {'score': None, 'explanation': None}
    while retry_ctr<GLOBAL_RETRIES:
        try:
            llm_response = verification_chain.invoke({
                "instruction_prompt": instruction_prompt,
                "reasoning_steps": make_previous_work(CONVERSATION_BUFFER),
                "final_answer": final_answer,
            })
            response_dict = json_output_parser(CODING_LLM, llm_response, required_keys=['score', 'explanation'])
            break
        except Exception as e:
            print(f"Verification failed with error: {e}")
            retry_ctr += 1
            if retry_ctr==GLOBAL_RETRIES:
                response_dict = {'score':'None', 'explanation':'Answer could not be verified'}
    return response_dict

def json_output_parser(llm, message_content, required_keys=["action", "action_input"], force_parse:bool=False, max_retries=5):
    def clean_keys(keys_string):
        keys_string = keys_string.strip()
        if keys_string.startswith('[') and keys_string.endswith(']'):
            keys_string = keys_string[1:-1]
        keys_list = keys_string.split(',')
        return [f.strip() for f in keys_list if len(f.strip())>0] 
    
    retry_ctr = 0
    dict_str = message_content[message_content.find("{"):message_content.rfind("}")+1]
    
    while retry_ctr<max_retries:
        try:
            ldict = ast.literal_eval(dict_str)
            ldict = {k:ldict[k] for k in required_keys}
            return ldict
        except Exception as e:
            if force_parse:
                break
            if required_keys is None:
                json_parsing_prompt = ChatPromptTemplate.from_messages([
                    ("system", JSON_PARSER_TEMPLATE['keys']['json_parsing_system']),
                    ("human", JSON_PARSER_TEMPLATE['keys']['json_parsing_template']),
                ])
                json_debugger_chain = (json_parsing_prompt | llm | StrOutputParser())
                keys_string = json_debugger_chain.invoke({
                    "json_string": dict_str,
                })
                required_keys = clean_keys(keys_string)
            
            json_parsing_prompt = ChatPromptTemplate.from_messages([
                ("system", JSON_PARSER_TEMPLATE['value']['json_parsing_system']),
                ("human", JSON_PARSER_TEMPLATE['value']['json_parsing_template']),
            ])
            json_debugger_chain = (json_parsing_prompt | llm | StrOutputParser())
            parsed_dict = {}
            for k in required_keys:
                parsed_dict[k] = json_debugger_chain.invoke({
                "json_string": dict_str,
                "key_name": k,
            })
            dict_str = json.dumps(parsed_dict)
            retry_ctr += 1
    return json.loads(dict_str)

async def get_google_api_results(query, result_page=0, num_results=5):
    api_key = os.environ['SERPER_DEV_KEY']
    conn = http.client.HTTPSConnection("google.serper.dev")
    headers = {
    'X-API-KEY': api_key,
    'Content-Type': 'application/json'
    }
    
    try:
        payload = json.dumps({
            "q": query,
            "num": num_results,
            "page":result_page,
        })
        conn.request("POST", "/search", payload, headers)
        response = conn.getresponse()    
    except Exception as e:
        print(f"Serper failed with error: {e}")
        return []
    
    results = []
    if response.status == 200:
        data = json.loads(response.read().decode("utf-8"))
        urls = []
        try:
            for l_idx in range(len(data['organic'])):
                if data['organic'][l_idx]["link"] is not None and data['organic'][l_idx]["title"] is not None:
                    urls.append(data['organic'][l_idx]["link"])
                    results.append({'link':urls[-1], 'title':data['organic'][l_idx]["title"]})
        except Exception as e:
            print(f"Google scraper failed with error: {e}")
        return results
    else:
        print(f"Request failed with status code: {response.status}")
        return results

async def nusmv_coding_function(code_specifications:Union[str, Dict[str, Any]]) -> str:
    global GLOBAL_LLM, REASONING_LLM
    global CODING_LLM
    global CONVERSATION_BUFFER
    code_script = ""
    
    try:
        assert GLOBAL_LLM is not None
        assert CODING_LLM is not None
        prompt = ChatPromptTemplate.from_messages([
            ("system", nusmv_code_sys_prompt),
            MessagesPlaceholder(variable_name="previous_conversation", optional=True),
            ("human", nusmv_code_prompt)
        ])
        if REASONING_LLM is not None:
            coding_chain = (prompt | REASONING_LLM | RunnableLambda(remove_thinking_tokens) | StrOutputParser())
        else:
            coding_chain = (prompt | CODING_LLM | StrOutputParser())
        callback_obj = IntermediateStepsCallback()
        
        def make_previous_work(conversation_buffer: List[AnyMessage]) -> List[AnyMessage]:
            previous_work = []
            for idx, f in enumerate(conversation_buffer):
                # if idx==0:
                #     previous_work.append(HumanMessage(content=read_nusmvs()))
                if f.type=="ai":
                    content = f.content
                    if content.endswith("\n\nObservation: "):
                        content = content[:content.rfind("\n\nObservation: ")]
                    previous_work.append(HumanMessage(content=content))
                if f.type=="tool":
                    previous_work.append(AIMessage(content=f.content))
            return previous_work
        cb = make_previous_work(CONVERSATION_BUFFER[1:-1])
        print("NUSMV CODING TOOL")
        print("********************************")
        print("\n".join([f"{f.type}:{f.content}" for f in cb]))
        print("********************************")
        
        code_script = await coding_chain.ainvoke({
            'previous_conversation': cb,
            'user_specification': str(code_specifications) + f"\n\n{read_nusmvs()}"
        }, {"callbacks": [callback_obj]})

        return code_script

    except Exception as e:
        return f"Code script generation failed with error: {e}"

@tool(name_or_callable="list_directory")
async def list_directory(dir_path: str) -> str:
    """Tool to list the directory structure relative to `dir_path` inside the current working directory.

    Args:
        dir_path (str): The directory path to start the traversal from inside the current working directory.

    Returns:
        str: The directory structure formatted as a string.
    """
    global filesystem_root
    dir_structure = ""
    try:
        def print_directory_structure(path, dir_structure, indent=''):
            dir_content = os.listdir(path)
            for idx, item in enumerate(dir_content):
                if item.startswith('.') or len(item.strip())==0:
                    continue
                item_path = os.path.join(path, item)
                if os.path.isdir(item_path):
                    dir_structure += f"{indent}├── {os.path.basename(item_path)}\n"
                    dir_structure = print_directory_structure(item_path, dir_structure, indent + '│   ')
                elif idx==len(dir_content)-1:
                    dir_structure += f"{indent}└── {item}\n"
                else:
                    dir_structure += f"{indent}├── {item}\n"
            return dir_structure
        
        dir_structure = print_directory_structure(join(filesystem_root, dir_path), dir_structure)
        # return dir_structure
        # fpath = pathlib.Path(filesystem_root).absolute()
        return f"Current working directory: {filesystem_root} \n\nDirectory structure for the provided `dir_path`: \n\n" + dir_structure
    except Exception as e:
        return f"List directory structure failed with exception: {e}"
        
@tool(name_or_callable="direct_response")
async def direct_response(answer: str) -> str:
    """Tool to respond directly for any given task if no other provided tool is appropriate for solving the task.

    Args:
        answer (str): Response for the given task.

    Returns:
        str: Response as answer for the given task.
    """
    return answer

@tool(name_or_callable="knowledge_search")
async def knowledge_search(query: str) -> str:
    """Tool to respond to any given task with search results based on your internal knowledge base. \
    The knowledge base was built using user-provided/attached/locally-stored documents. \
    The input to this tool should be a well-designed and detailed query that can be used to \
    search for required information in the knowledge base.

    Args:
        query (str): Prompt to search the required information for the given task from your internal knowledge base.

    Returns:
        str: Response for the given task.
    """
    global GLOBAL_LLM
    assert GLOBAL_LLM is not None
    
    answer_prompt = ChatPromptTemplate.from_messages([
        ("system", answer_system_template),
        ("human", answer_human_template),
    ])
    answer_llm = (answer_prompt | GLOBAL_LLM | StrOutputParser())
    
    rag_content = ""
    callback_obj = IntermediateStepsCallback()
    try:
        current_loop = asyncio.get_running_loop()
        rag_task = current_loop.run_in_executor(None, lambda:get_rag(GLOBAL_LLM, callback_manager=[callback_obj]))
        vectorstore, _ = await rag_task
        # vectorstore, _ = await get_rag(GLOBAL_LLM, callback_manager=[callback_obj])
        rag_content = await generate_rag_response(query, vectorstore, [], GLOBAL_LLM, 
                                                [callback_obj], None, use_standalone=False)
    except Exception as e:
        print(f"RAG failed with error: {e}")
        rag_content = f"No results found in knowledge base for the input query '{query}'. \n\n" + \
            "Use 'web_scraper' tool to search the relevant information from the web."
    finally:
        callback_obj.add_to_global_stream("<END_OF_RAG>")
    
    final_answer = await answer_llm.ainvoke({
        "question": query,
        "context": rag_content,
    })
    return "Results from the knowledge base:\n\n" + final_answer

@tool(name_or_callable="final_answer")
async def final_answer(answer: str) -> str:
    """A passthrough function for displaying the final answer to the user.

    Args:
        answer (str): Final answer generated by the AI Agent.

    Returns:
        str: Final answer to display to the user.
    """
    return answer


@staticmethod
def read_docs():
    """Function to read and make the relevant document available for tools to use.
    Returns:
        str: The content required by the tool
    """
    global SOP_FILE
    pth = copy.deepcopy(SOP_FILE)
    text = ""
    with open(pth, "rb") as file:
        pdf = PyPDF2.PdfReader(file)
        for page_num in range(len(pdf.pages)):
            page = pdf.pages[page_num]
            text += page.extract_text()
    return text

def read_nusmvs():
    global SMV_PATHS
    pth1 = SMV_PATHS[0]
    pth2 = SMV_PATHS[1]
    pth3 = SMV_PATHS[2]
    smv_file_map = {
        "states.smv": "The following code represents the operational modes and dynamic variables of the system:\n\n",
        "kripke.smv": "The following code represents the transitions between the operational modes of the system:\n\n",
        "properties.smv": "The following code represents properties used for verification of the system's behavior:\n\n",
    }
    
    with open(pth1, 'r') as fp:
        txt1 = fp.read()
    with open(pth2, 'r') as fp:
        txt2 = fp.read()
    with open(pth3, 'r') as fp:
        txt3 = fp.read()
    text = f"{smv_file_map[os.path.basename(pth1)]}{txt1}\n\n{smv_file_map[os.path.basename(pth2)]}{txt2}\n\n{smv_file_map[os.path.basename(pth3)]}{txt3}"
    return text
    
@tool(name_or_callable="create_states")
async def create_states(candidate_states: str)->str:
    """
    A custom tool to extract finite state machine states from a string of potential terms and given system specification.
    
    Args:
        candidate_states (str): A string containing potential terms which could be states in the final kripke structure of the system.
    Returns:
        str: A string response containing the extracted states.
    """
    global GLOBAL_LLM, REASONING_LLM
    global NUSMV_CODE
    global CONVERSATION_BUFFER
    assert GLOBAL_LLM is not None
    txt = read_docs()
    nlp_prompt = ChatPromptTemplate.from_messages([
        ("system", state_sys_prompt),
        MessagesPlaceholder(variable_name="previous_conversation", optional=True),
        ("human", state_human_prompt),
    ])
    if REASONING_LLM is not None:
        nlp_chain = nlp_prompt | REASONING_LLM | RunnableLambda(remove_thinking_tokens) | StrOutputParser()
    else:
        nlp_chain = nlp_prompt | GLOBAL_LLM | StrOutputParser()
    
    def make_previous_work(conversation_buffer: List[AnyMessage]) -> List[AnyMessage]:
        previous_work = []
        for f in conversation_buffer:
            if f.type=="ai":
                content = f.content
                if content.endswith("\n\nObservation: "):
                    content = content[:content.rfind("\n\nObservation: ")]
                previous_work.append(HumanMessage(content=content))
            if f.type=="tool":
                previous_work.append(AIMessage(content=f.content))
        return previous_work
    cb = make_previous_work(CONVERSATION_BUFFER[1:-1])
    print("CREATE STATES TOOL")
    print("********************************")
    print("\n".join([f"{f.type}:{f.content}" for f in cb]))
    print("********************************")
    
    callback_obj = IntermediateStepsCallback()
    response = await nlp_chain.ainvoke(
        {
            "previous_conversation": cb,
            'doc': txt,
            'candidate_terms': candidate_states
        },
        {"callbacks": [callback_obj]},
    )
    
    # print(response)
    NUSMV_CODE.append(response)
    return 'The following is the output from `create_states` tool: \n\n' + response + \
        '\n\nInitiate the next "Thought-Action-Observation" step and ensure that your response aligns with the specified $JSON_BLOB format.'

@tool(name_or_callable="create_transitions")
async def create_transitions(states: str, candidate_triggers: str | None)->str:
    """Tool to extract, formalize, and validate state transitions for a Kripke structure.

    Args:
        states (str): A string of state objects as produced by create_states(). Each state field should include at least a `state_id` and `name` field.
        candidate_triggers (str): A string of candidate trigger phrases or events (e.g. "on fire alarm","after passenger exchange") to seed the extraction. \
            The LLM will use these hints and also scan SOP for any additional transition cues. If None, the tool will infer triggers solely from the SOP text.

    Returns:
        str: Returns a string populated with all kinds of properties of a system.
    """
    global GLOBAL_LLM, REASONING_LLM
    global NUSMV_CODE
    global CONVERSATION_BUFFER
    assert GLOBAL_LLM is not None
    txt = read_docs()
    trans_prompt = ChatPromptTemplate.from_messages([
        ("system", transition_sys_prompt),
        MessagesPlaceholder(variable_name="previous_conversation", optional=True),
        ("human", transition_human_prompt)
    ])
    if REASONING_LLM is not None:
        trans_chain = trans_prompt | REASONING_LLM | RunnableLambda(remove_thinking_tokens) | StrOutputParser()
    else:    
        trans_chain = trans_prompt | GLOBAL_LLM | StrOutputParser()
    
    def make_previous_work(conversation_buffer: List[AnyMessage]) -> List[AnyMessage]:
        previous_work = []
        for f in conversation_buffer:
            if f.type=="ai":
                content = f.content
                if content.endswith("\n\nObservation: "):
                    content = content[:content.rfind("\n\nObservation: ")]
                previous_work.append(HumanMessage(content=content))
            if f.type=="tool":
                previous_work.append(AIMessage(content=f.content))
        return previous_work
    cb = make_previous_work(CONVERSATION_BUFFER[1:-1])
    print("CREATE TRANSITIONS TOOL")
    print("********************************")
    print("\n".join([f"{f.type}:{f.content}" for f in cb]))
    print("********************************")
    
    callback_obj = IntermediateStepsCallback()
    response = await trans_chain.ainvoke({
        'previous_conversation': cb,
        'doc': txt,
        'states': states,
        'candidate_terms': candidate_triggers,
    }, {"callbacks": [callback_obj]},)
    
    # print(response)
    NUSMV_CODE.append(response)
    return 'The following is the output from `create_transitions` tool: \n\n' + response + \
        '\n\nInitiate the next "Thought-Action-Observation" step and ensure that your response aligns with the specified $JSON_BLOB format.'
        
@tool(name_or_callable="generate_nl_descriptions")
async def generate_nl_descriptions(sys_description: str)-> str:
    """An AI-assisted tool that generates natural language text properties based on system description.

    Args:
        seed (str): A short string containing natural-language description of the system under verification.

    Returns:
        str: A string containing natural language text properties generated from system description.
    """
    global GLOBAL_LLM, REASONING_LLM
    global CONVERSATION_BUFFER
    global NUSMV_CODE
    assert GLOBAL_LLM is not None
    txt = read_docs()
    prop_prompt = ChatPromptTemplate.from_messages([
        ("system", prop_chain_sys),
         MessagesPlaceholder(variable_name="previous_conversation", optional=True),
        ("human", prop_chain_human)
    ])
    if REASONING_LLM is not None:
        prop_chain = (prop_prompt | REASONING_LLM | RunnableLambda(remove_thinking_tokens) | StrOutputParser())
    else:
        prop_chain = (prop_prompt | GLOBAL_LLM | StrOutputParser())
    
    def make_previous_work(conversation_buffer: List[AnyMessage]) -> List[AnyMessage]:
        previous_work = []
        for f in conversation_buffer:
            if f.type=="ai":
                content = f.content
                if content.endswith("\n\nObservation: "):
                    content = content[:content.rfind("\n\nObservation: ")]
                previous_work.append(HumanMessage(content=content))
            if f.type=="tool":
                previous_work.append(AIMessage(content=f.content))
        return previous_work
    cb = make_previous_work(CONVERSATION_BUFFER[1:-1])
    print("NL descriptions TOOL")
    print("********************************")
    print("\n".join([f"{f.type}:{f.content}" for f in cb]))
    print("********************************")
    
    callback_obj = IntermediateStepsCallback()
    response = await prop_chain.ainvoke({
        'previous_conversation': cb,
        'doc': txt,
        'SYSTEM_DESC': sys_description,
    }, {"callbacks": [callback_obj]},)
    NUSMV_CODE.append(response)
    return 'The following is the output from `generate_nl_descriptions` tool: \n\n' + response + \
        '\n\nInitiate the next "Thought-Action-Observation" step and ensure that your response aligns with the specified $JSON_BLOB format.'

@tool(name_or_callable="crag_tool")
async def crag_tool(query: str, sop_flag: str) -> str:
    """Custom tool to provide context to other tools based on your internal knowledge base. \
    The knowledge base was built using user-provided/attached/locally-stored documents. \
    Investigate concepts, retrieve specific information, and locate relevant sections within the provided document. \
    Think broadly about how this tool can aid in understanding the task and gathering necessary context.
    
    Args:
        query (str): Prompt to search the required information for the given task from your internal knowledge base.
        sop_flag (str): 'true' if SOP document is required as part of the context, otherwise 'false'.
         
    Returns:
        str: A string containing response for the input query providing information.
    """
    global GLOBAL_LLM, REASONING_LLM
    global CONVERSATION_BUFFER
    assert GLOBAL_LLM is not None
    def make_previous_work(conversation_buffer: List[AnyMessage]) -> List[AnyMessage]:
        previous_work = []
        for f in conversation_buffer:
            if f.type=="ai":
                content = f.content
                if content.endswith("\n\nObservation: "):
                    content = content[:content.rfind("\n\nObservation: ")]
                previous_work.append(HumanMessage(content=content))
            if f.type=="tool":
                previous_work.append(AIMessage(content=f.content))
        return previous_work
    cb = make_previous_work(CONVERSATION_BUFFER[1:-1])
    
    print("_________________________IN CRAG TOOL_________________________")
    answer_prompt = ChatPromptTemplate.from_messages([
        ("system", answer_system_template),
        MessagesPlaceholder(variable_name="previous_conversation", optional=True),
        # ("human", answer_human_template),
        ("human", sop_answer_human_template),
    ])
    if REASONING_LLM is not None:
        answer_chain = (answer_prompt | REASONING_LLM | RunnableLambda(remove_thinking_tokens) | StrOutputParser())
    else:
        answer_chain = (answer_prompt | GLOBAL_LLM | StrOutputParser())

    sop_flag = True if sop_flag.lower()=='true' else False
    rag_content = ""
    try:
        # set_crag_llm(GLOBAL_LLM)
        agent_obj= InformationRetrieverAgent("Model Checking") 
        rag_content = agent_obj.run(query)
        print(rag_content)
    except Exception as e:
        print(f"Failed to call RAG with error: {e}")
        rag_content = f"No results found in knowledge base for the input query '{query}'. \n\n" + \
            "Use 'web_scraper' tool to search the relevant information from the web."
       
    callback_obj = IntermediateStepsCallback()
    final_answer = answer_chain.invoke({
        "previous_conversation": cb,
        "question": query,
        "context": rag_content,
        "sop_txt": ("SOP Document: " + read_docs() + "\n\n") if sop_flag else "",
    }, {"callbacks": [callback_obj]})
    return 'Results from the crag_tool are:\n\n' + final_answer + \
        '\n\nInitiate the next "Thought-Action-Observation" step and ensure that your response aligns with the specified $JSON_BLOB format.'

@tool(name_or_callable="save_nusmv")
async def save_nusmv(file_path: str) -> str:
    """Tool to save the last generated NuSMV code as a text file at the specified `file_path` inside the current working directory.

    Args:
        file_path (str): The file path relative to the current working directory.

    Returns:
        str: Message describing whether the file was saved successfully or not.
    """
    #global SMV_CONTENT
    global filesystem_root
    global NUSMV_CODE
    try:
        code = NUSMV_CODE[-1]
        assert code is not None
        if not isdir(filesystem_root):
            os.mkdir(filesystem_root)
        with open(join(filesystem_root, file_path), 'w') as fp:
            # content = CODE_CONTENT.encode('utf-8').decode('unicode-escape')
            content = copy.deepcopy(code)
            if "```nuSMV" in content:
                content = content[content.find("```nuSMV\n")+len("```nuSMV\n"):content.rfind("```")]
            elif "```smv" in content:
                content = content[content.find("```smv\n")+len("```smv\n"):content.rfind("```")]
            elif "```nusmv" in content:
                content = content[content.find("```nusmv\n")+len("```nusmv\n"):content.rfind("```")]
            else:
                start_index = content.find("MODULE main")
                end_index = content.rfind("```")
                end_index = end_index if end_index>-1 else len(content)
                content = content[start_index:end_index]
            fp.write(content)
        return f"File has been saved successfully as `{file_path}` in the current working directory."
    except Exception as e:
        return f"Saving file {file_path} failed with exception: {e}"
    
@tool(name_or_callable="read_nusmv")
async def read_nusmv(file_path: str) -> str:
    """Tool to read a NuSMV model file

    Args:
        file_path (str): The NuSMV model file path relative to the current working directory.

    Returns:
        str: Content of the NuSMV model file.
    """
    global NUSMV_CODE
    global filesystem_root
    try:
        assert file_path.endswith('.smv')
        assert isfile(join(filesystem_root, file_path))
        with open(join(filesystem_root, file_path), 'r') as fp:
            content = fp.read()
        NUSMV_CODE.append(content)
        return "```nusmv\n" + content + "\n```" + \
            f"\n\n The NuSMV model file at path '{file_path}' was successfully read from the current working directory and printed to the console."
    except Exception as e:
        return f"Reading NuSMV model file at path {file_path} failed with exception: {e}"

@tool(name_or_callable="execute_nusmv")
async def execute_nusmv(program_pth: str) -> str:
    """
    Tool to execute a NuSMV(.smv) script program located in the current working directory and stream it's standard output.

    Args:
        program_pth (str): The relative path to the NuSMV script located in the current working directory to be executed. This argument should not include any command line instructions.
    
    Yields:
        str: Lines of output from the program.
    """
    global NUSMV_CODE_OUTPUT
    global filesystem_root
    try:
        program_pth = join(filesystem_root, program_pth)
        if not isfile(program_pth):
            raise ValueError(f"The input code file at provided path='{program_pth}' does not exist.")
    except Exception as e:
        return f"Script execution failed with error: {e}"
    
    process = await  asyncio.create_subprocess_exec(
        'NuSMV',
        program_pth,
        stdout=subprocess.PIPE,
        stderr=subprocess.PIPE
    )

    outputs = [f"Streaming standard output for {program_pth}:"]
    try:
        stdout, stderr = await process.communicate()
        stdout = stdout.decode('utf-8', errors='ignore') if stdout else ""
        stderr = stderr.decode('utf-8', errors='ignore') if stderr else ""
        outputs.append(stdout)
    except Exception as e:
        # process.cancel()
        raise ValueError(f"The input code file at provided program path='{program_pth}' did not execute successfully")
    finally:
        if process.returncode !=0:
            outputs.append(stderr)
    outputs = "\n".join(outputs)
    NUSMV_CODE_OUTPUT.append(outputs)
    return outputs + "\n\nEnd of NuSMV program output."

@tool(name_or_callable="debug_nusmv")
async def debug_nusmv(file_path: str) -> str:
    """
    AI-assisted tool for debugging a NuSMV model file using its model-checking CLI output.

    Args:
        file_path (str): The relative path to the NuSMV model file located in the current working directory to be executed.

    Returns:
        str: The revised NuSMV code.
    """
    global NUSMV_CODE
    global NUSMV_CODE_OUTPUT
    global CONVERSATION_BUFFER
    global REASONING_LLM, CODING_LLM
    global filesystem_root
    assert CODING_LLM is not None
    def sanitize_code(code_content):
        if "```nuSMV" in code_content:
            code_content = code_content[code_content.find("```nuSMV\n")+len("```nuSMV\n"):code_content.rfind("```")]
        if "```smv" in code_content:
            code_content = code_content[code_content.find("```smv\n")+len("```smv\n"):code_content.rfind("```")]
        elif "```nusmv" in content:
            code_content = code_content[code_content.find("```nusmv\n")+len("```nusmv\n"):code_content.rfind("```")]
        else:
            start_index = code_content.find("MODULE main")
            end_index = code_content.rfind("```")
            end_index = end_index if end_index>-1 else len(code_content)
            code_content = code_content[start_index:end_index]
        return code_content
    def make_previous_work(conversation_buffer: List[AnyMessage]) -> List[AnyMessage]:
        previous_work = []
        for f in conversation_buffer:
            if f.type=="ai":
                content = f.content
                if content.endswith("\n\nObservation: "):
                    content = content[:content.rfind("\n\nObservation: ")]
                previous_work.append(HumanMessage(content=content))
            if f.type=="tool":
                previous_work.append(AIMessage(content=f.content))
        return previous_work
    cb = make_previous_work(CONVERSATION_BUFFER[1:-1])
    print("DEBUG TOOL")
    print("********************************")
    print("\n".join([f"{f.type}:{f.content}" for f in cb]))
    print("********************************")
    
    if len(NUSMV_CODE_OUTPUT)==0:
        return "Please execute the NuSMV model file using `execute_nusmv` tool."
    
    debug_prompt_template = ChatPromptTemplate.from_messages([
        ("system", nusmv_debug_system),
        MessagesPlaceholder(variable_name="previous_conversation", optional=True),
        ("human", nusmv_debug_prompt),
    ])
    if REASONING_LLM is not None:
        debug_chain = (debug_prompt_template | REASONING_LLM)
    else:
        debug_chain = (debug_prompt_template | CODING_LLM)
    callback_obj = IntermediateStepsCallback()
    
    assert file_path.endswith('.smv')
    assert isfile(join(filesystem_root, file_path))
    with open(join(filesystem_root, file_path), 'r') as fp:
        content = fp.read()
    NUSMV_CODE.append(content)
    code_script = copy.deepcopy(NUSMV_CODE[-1])
    lines = code_script.encode(encoding="utf-8").splitlines(keepends=True)
    width = len(str(len(lines)))
    code_script = "".join([f"{str(i).rjust(width)} | {line.decode('utf-8')}" for i, line in enumerate(lines, start=1)])
    print(code_script)
    
    code_output = copy.deepcopy(NUSMV_CODE_OUTPUT[-1])
    lines = code_output.encode(encoding="utf-8").splitlines(keepends=True)
    code_output = "".join([f.decode('utf-8') for f in lines if not f.decode('utf-8').startswith("***") and len(f.decode('utf-8').strip())!=0])
    print(code_output)
    
    final_answer = await debug_chain.ainvoke(
        {
            "previous_conversation": cb,
            "nusmv_file": code_script,
            "cli_output": code_output,
        },
        {"callbacks": [callback_obj]}
    )
    final_answer = remove_thinking_tokens(final_answer)
    code_content = sanitize_code(final_answer.content)
    NUSMV_CODE.append(code_content)
    return final_answer.content

@tool(name_or_callable="nusmv_codepad")
async def nusmv_codepad(code_specifications:Union[str, Dict[str, Any]]) -> str:
    """
    Generates a complete NuSMV model code based on provided specifications.

    This AI-assisted tool constructs NuSMV code structured into distinct blocks, including `MODULE`, `DEFINE`, `ASSIGN`, `VAR`, `TRANS`, and `SPEC`.

    Args:
        code_specifications (Dict[str, str] | str):
            A detailed blueprint for generating the NuSMV model. Recommended specification keys include:
            - module_name (str, optional): Name of the top-level module. Defaults to "main".
            - purpose (str): Brief description of what the NuSMV model captures or verifies.
            - state_variables (str): Definitions for primary operational mode enumerations or boolean mappings.
            - dynamic_variables (str): Additional dynamic variables, including types, ranges, and default values.
            - init_conditions (str): Initial condition statements (`ASSIGN init(...)`).
            - transition_rules (str): Transition rules (`TRANS` formulas or `next(var)` assignments).
            - guard_conditions (str, optional): Additional predicates governing transitions. Defaults to "None".
            - property_specs (str): Property specifications (`CTL/LTL/SPEC` statements).
            - fairness_constraints (str, optional): Fairness constraints (`FAIRNESS` or `JUSTICE`). Defaults to "None".
            - definitions (str, optional): Aliases or macros defined via `DEFINE`. Defaults to "None".
            - documentation (str, optional): Inline comments or documentation to be included. Defaults to "None".
            - validation_checks (str, optional): Semantic validation rules (e.g., ensuring all SPECs reference existing variables). Defaults to "None".
            - integration_points (str, optional): Instructions detailing how the generated model integrates into external workflows or tools (e.g., test harnesses, continuous integration pipelines). Defaults to "None".

    Returns:
        str: A runnable and complete NuSMV model script string that fulfills the provided `code_specifications`.
            The resulting script can be saved directly as `model.smv` and executed using the NuSMV checker.
    """
    global NUSMV_CODE
    try:
        code_script = await nusmv_coding_function(code_specifications=code_specifications)
        NUSMV_CODE.append(code_script)
        return 'The generated NuSMV model code for the provided specifications is as follows:\n\n' + code_script + \
            '\n\nInitiate the next "Thought-Action-Observation" step and ensure that your response aligns with the specified $JSON_BLOB format.'
    except Exception as e:
        return f"Code script generation failed with error: {e}"

@tool(name_or_callable="generate_spec_properties")
async def generate_spec_properties(nl_text_conditions:str)->str:
    """An AI-assisted tool to inject Kripke Structure knowledge into natural-language property claims into NuSMV SPEC formulas.

    Args:
        nl_text_conditions (str): A string containing natural language text for properties generated from system description.

    Returns:
        str: A string containing NuSMV compatible property SPECs.
    """
    global GLOBAL_LLM, REASONING_LLM
    global CONVERSATION_BUFFER
    global NUSMV_CODE
    assert GLOBAL_LLM is not None
    file_sys = filesystem_root
    state_pth = join(file_sys, 'states.smv')
    trans_pth = join(file_sys, 'kripke.smv')
    state_info = open(state_pth, "r").read()
    transition_info = open(trans_pth, "r").read()
    txt = read_docs()
    
    def make_previous_work(conversation_buffer: List[AnyMessage]) -> List[AnyMessage]:
        previous_work = []
        for f in conversation_buffer:
            if f.type=="ai":
                content = f.content
                if content.endswith("\n\nObservation: "):
                    content = content[:content.rfind("\n\nObservation: ")]
                previous_work.append(HumanMessage(content=content))
            if f.type=="tool":
                previous_work.append(AIMessage(content=f.content))
        return previous_work
    cb = make_previous_work(CONVERSATION_BUFFER[1:-1])
    print("CTL properties TOOL")
    print("********************************")
    print("\n".join([f"{f.type}:{f.content}" for f in cb]))
    print("********************************")

    prop_prompt = ChatPromptTemplate.from_messages([
        ("system", prop_sys_prompt),
        MessagesPlaceholder(variable_name="previous_conversation", optional=True),
        ("human", prop_human_prompt)
    ])
    if REASONING_LLM is not None:
        augment_chain = prop_prompt | REASONING_LLM | RunnableLambda(remove_thinking_tokens) | StrOutputParser()
    else:
        augment_chain = prop_prompt | GLOBAL_LLM | StrOutputParser()
        
    callback_obj = IntermediateStepsCallback()
    response = await augment_chain.ainvoke({
        'previous_conversation': cb,
        'doc':  txt,
        'props': nl_text_conditions,
        'state': state_info,
        'trans': transition_info
    }, {"callbacks": [callback_obj]},)
    
    NUSMV_CODE.append(response)
    return 'The following is the output from `generate_spec_properties` tool for the given system description: \n\n' + response + \
        '\n\nInitiate the next "Thought-Action-Observation" step and ensure that your response aligns with the specified $JSON_BLOB format.'

@tool(name_or_callable="categorize_properties")
async def categorize_properties(properties: str)-> str:
    """A tool to organize NuSMV SPECs into various categories.

    Args:
        properties (str): A string containing NuSMV compatible property SPECs.

    Returns:
        str: A string containing NuSMV SPECs categorized into 3 categories.
    """
    global GLOBAL_LLM, REASONING_LLM
    assert GLOBAL_LLM is not None
    org_prompt = ChatPromptTemplate.from_messages([
        ("system", categorize_sys),
        ("human", categorize_human)
    ])
    if REASONING_LLM is not None:
        org_chain = org_prompt | REASONING_LLM | RunnableLambda(remove_thinking_tokens) | StrOutputParser()
    else:
        org_chain = org_prompt | GLOBAL_LLM | StrOutputParser()
        
    callback_obj = IntermediateStepsCallback()
    res = await org_chain.ainvoke({
        'props': properties
    }, {"callbacks": [callback_obj]},)
    return 'The following is the output from `categorize_properties` tool: \n\n' + res + \
    '\n\nInitiate the next "Thought-Action-Observation" step and ensure that your response aligns with the specified $JSON_BLOB format.'

def get_available_tools():
    return list(tool_dict.keys())

def set_rag_update(update_rag):
    global TOOLS_CAN_UPDATE_RAG
    TOOLS_CAN_UPDATE_RAG = update_rag
    return TOOLS_CAN_UPDATE_RAG

tool_dict = {
    'list_directory': list_directory,
    'direct_response': direct_response,
    'knowledge_search': knowledge_search,
    'crag_tool': crag_tool,
    'create_states': create_states,
    'create_transitions': create_transitions,
    'generate_nl_descriptions': generate_nl_descriptions,
    'generate_spec_properties': generate_spec_properties,
    'categorize_properties': categorize_properties,
    'read_nusmv': read_nusmv,
    'save_nusmv': save_nusmv,
    'execute_nusmv': execute_nusmv,
    'debug_nusmv': debug_nusmv,
    'nusmv_codepad': nusmv_codepad,
}
custom_stop = {
    'thought': ["Thought:", "THOUGHT:", "**Thought**", "# Thought"],
    'action': ["Action:", "ACTION:", "**Action**", "# Action"],
    'observation': ["Observation:", "OBSERVATION:", "**Observation**", "# Observation"], 
    'plan': ["<END_OF_PLAN>"],
}