from pydantic import BaseModel, Field
from typing import TYPE_CHECKING, Any, Dict, List, Optional, Sequence, TypeVar, Union, Literal, Callable

nusmv_code_sys_prompt = """You are an expert AI assistant trained in formal verification and NuSMV syntax. \
Your task is to generate a **complete, runnable NuSMV model** from the following partial artifacts: \
(1) a list of STATES, (2) TRANSITION relationships, and (3) a set of CTL/LTL/SEPC PROPERTIES. \
A valid NuSMV model must contain, at minimum, the sections MODULE main, VAR, ASSIGN (or TRANS), and one or more CTL/LTL/SPEC statements.
"""

nusmv_code_prompt = """Please generate the NuSMV model file for the given NuSMV specifications. \
Ensure the generated NuSMV code is well-commented and formatted properly for readability. \
Include any necessary explanations or context withing the code as "--comments" to clarify the purpose and functionality of the code.

SPECIFICATION: \n---------------\n{user_specification}

Do not simplify, omit, or abstract any part of the NuSMV code-every section of the code should be explicitly populated with all relevant details. \
Do not use any external tools in your response, only output the contents of the required NuSMV model file.
"""

JSON_PARSER_TEMPLATE = {
    'value':{
        'json_parsing_system':"""You are an advanced LLM-based JSON parser. Your task is to correctly parse a given JSON string \
and return the value of the specified key. Always respond with only the value of the specified key and nothing else.
""",
        'json_parsing_template':"""JSON string:

{json_string}

Output the value for the following key:

{key_name}

Please respond with only the value of the specified key and nothing else.
""",
    },
    'keys':{
        'json_parsing_system':"""You are an advanced LLM-based JSON parser. Your task is to correctly parse a given JSON string \
and return a list of all keys present in the JSON string. Always respond with only a comma separated list of keys and nothing else.
""",
        'json_parsing_template':"""JSON string:

{json_string}

Please respond with a comma separated list of keys and nothing else.
""",
    }
}

VERIFICATION_TEMPLATE = {
    'system': """You will receive an instruction, reasoning steps, and final answer. 
Your task is to evaluate how well the reasoning steps and final answer address the given instruction. 

Use the following Likert scale to generate a score: 
- 10: Completely addresses the question with clear and logical reasoning, leading to an accurate final answer. 
- 7-9: Addresses the question well, but there are minor gaps or ambiguities in reasoning or the final answer. 
- 4-6: Partially addresses the question; reasoning or the final answer has notable gaps or issues. 
- 1-3: Barely addresses or does not address the question; reasoning is flawed or missing, and the final answer is inaccurate. 

Always evaluate the provided reasoning steps and the final answer to determine how well they fulfill the instruction. \
Do not consider factors outside the reasoning and final answer provided. \
Ensure your explanation justifies the score based on the quality of reasoning steps and final answer in addressing the instruction.
""",
    'human': """INSTRUCTION:
{instruction_prompt}

REASONING STEPS:
{reasoning_steps}

FINAL ANSWER:
{final_answer}

Evaluate how well the reasoning steps and final answer address the given instruction.

Answer in a $JSON_BLOB, using the following schema:
```json
{{
    "score": <Likert scale score from 1 to 10>,
    "explanation": "<Reason for the selected score>"
}}
```

Reminder to ALWAYS respond with a valid $JSON_BLOB and nothing else.
"""
}


state_sys_prompt = """You are an expert finite state machine (FSM) designer for complex control systems. \
Your task is to extract and formalize:
1. All discrete operational modes/states
2. All dynamic system variables(quantities that evolve over time) 

Respond only in **NuSMV code blocks** (MODULE, VAR, ASSIGN, TRANS / next, optional DEFINE).
"""


state_human_prompt = """System Specification:
{doc}

Initial Candidate Terms:
{candidate_terms}

Instructions:

0. **Analyse the full SOP text** to identify:
   - Phrases indicating operational **modes** or **states** (especially noun-verb combinations).
   - **Dynamic variables** that evolve over time or require monitoring.

   Record any newly discovered items as comments using the format: `--missing_candidates: ...`.

1. **Organize results into NuSMV sections**:

   a) **States Section**:
      - Normalize and semantically cluster the provided `{candidate_terms}` along with any identified `--missing_candidates`.
      - Define an enumeration variable in the `VAR` block for each canonicalized state/mode.
      - Merge synonymous terms as needed and annotate merges with inline comments.

   b) **Dynamic Variables Section**:
      - Declare each variable in the same `VAR` block, specifying appropriate types and value ranges.
      - Include inline comments indicating:
        - The source text reference.
        - The inferred range (if applicable).

2. **ASSIGN Block**:
   - Set initial values for each dynamic variable.
   - If an initial value is not specified in the SOP, select a reasonable default and annotate with `--assumed`.

3. **Coverage Notes**:
   - At the end of the module, list any parts of the SOP that were not mapped to states or variables using:  
     `--uncovered: ...`

4. **Output Requirements**:
   - Generate a single, complete NuSMV `MODULE` containing all declarations, logic, and annotations.
   - Do not simplify, omit, or abstract any part of the NuSMV model code. All states and variables must be explicitly populated with relevant details and all dynamic variables should be properly initialized.
   - The output must include the entire NuSMV script in one block—fully self-contained and ready for execution.
   - Use inline `--` comments for all annotations.
   - Do **not** include JSON, Markdown, or any explanatory text outside the NuSMV code.
"""


transition_sys_prompt = """You are an expert finite-state-machine (FSM) designer for complex control systems. \
Your task is to extract and formalize all valid transitions between known states from a system specification directly as **NuSMV code blocks** (MODULE, VAR, ASSIGN, TRANS / next, optional DEFINE).
"""

transition_human_prompt = """System Specification:
{doc}

Declared VAR Block (from `create_states` tool's output):
{states}

Candidate Events / Trigger Phrases:
{candidate_terms}

Instructions:

0. **Analyse the SOP text** for any additional missing transition cues (e.g., “upon X, go to Y”, “if event E then …”, etc.).
   - Log new candidates as: `--missing_triggers: ...`

1. **Merge duplicates** from `candidate_terms` and `missing_triggers` into a unified set.
   - Document results in: `--merged_triggers: ...`

2. **For each unique trigger phrase**:
   - Identify valid transitions between declared states.
   - Determine updates to dynamic variables that occur during the transition.

   For each dynamic variable:
   - Create a `next(var) := case ... esac;` block.
     - Each `case` branch must include:
       - A **Boolean guard condition** referencing `current_states` and/or other variables.  
       - An **assignment value** for the variable.  
       - Inline comments for each case line, including:
         - `source: "<exact SOP snippet>"`

   - If a variable is **not affected** by a condition, include a default branch:
     ``` 
     TRUE : var;
     ```

3. **Catch-all Default**: Ensure every `case` block ends with a `TRUE : var;` to preserve variable values when no guard matches.

4. **Coverage Notes**: At the end of the module, include:
   - `--uncovered: ...` to document any unmapped SOP portions.

5. **Output Requirements**:
   - Ensure the output contains the **entire set of transition definitions** required for the system to function.
   - The code must be complete, consistent with the declared `VAR` block, and executable as-is in NuSMV. Do not simplify, omit, or abstract any part of the NuSMV model code.
   - Use inline `--` comments for annotations and explanations.
   - Do **not** include JSON, Markdown, or any explanatory text outside the NuSMV code.
"""


crag_tool_system_prompt = """Please provide an answer to the 'Question', using the 'Knowledge Base' provided. \
Ensure Use this tool to investigate concepts, retrieve specific information, and locate relevant sections within the provided document. \
Think broadly about how this tool can aid in understanding the task and gathering necessary context \
Cite specific information from the knowledge base and elaborate on key points if required. \
The input queries are general and not document specific. \
"""

crag_tool_human_prompt = """"Knowledge Base: {context}
Question: {question}
"""

prop_sys_prompt = """You are a formal verification and model-checking expert.  
Your job is to craft a comprehensive suite of CTL/LTL properties (invariants, formulas, etc.) to verify a given Kripke structure.  
Respond only in **NuSMV code syntax**.
"""

prop_human_prompt = """System Specification SOP document:
{doc}

Natural Language text for properties generated from the given SOP document:
{props}

Contents of the file `state.smv`
{state}

Contents of the file `kripke.smv`
{trans}

Instructions:
0. **Analyze the SOP document** and candidate property list for any missing rules, expectations, or norms.
   - If a critical safety or liveness constraint is not captured in `props`, you may add a candidate.  
   - Use `--missing_semantics:` comments to note inferred or assumed properties.

1. **Analyze each property statement** and decide:
   - Which CTL/LTL category it fits into:  
     `safety-invariant`, `safety-transition`, `liveness`, `fairness`, `reachability`, or `temporal-ordering`.
   - Which states, transition conditions, or dynamic variables it refers to.

2. **Ground the symbolic expression** in the actual Kripke model:
   - Replace all informal terms with the exact variable names, constants, and state values used in `state.smv` and `kripke.smv`.
   - Do **not invent new variable names or states**—you must only use what is declared in the FSM model.
   - Ensure the property refers to reachable or declared behaviour (e.g., don't assert a transition to a state that doesn't exist).

3. **Formalize each property as a CTL/LTL/SPEC formula** in NuSMV `SPEC` syntax:
   - `SPEC AG (condition)` — invariant  
   - `SPEC AF (condition)` — eventuality  
   - `SPEC AG (trigger -> AF consequence)` — temporal dependency  
   - Use only valid CTL operators (AG, EF, AX, etc.) and correct parentheses/boolean syntax (`&`, `|`, `->`, `!`, etc.).

4. **Annotate and validate** each property:
   - Include an inline `--` explanation in plain language.
   - Use `-- missing_semantics:` comments only if the SOP implies a rule that is not explicitly modeled.

5. **Output Requirements:**
   - Return a NuSMV code block with:
     • A list of `SPEC` declarations, one per line.
     • Inline `--` comments for explanation, and assumptions.
   - Assume the block will be appended to an existing NuSMV `MODULE`.
   - Do **not** include JSON, Markdown, or use any external tools in your response. Only output the contents of the required NuSMV file.
"""

prop_chain_sys = """You are “Domain-Property Synthesizer.”  
Input: a short system description.  
Output: a bullet list of realistic CTL/LTL/SPEC (NuSMV-style) properties for that system—nothing else.
"""

prop_chain_human = """System Specification SOP document:
{doc}

System description:
{SYSTEM_DESC}

Task
----
Generate **additional candidate properties** for model checking.

What to produce
---------------
- A Markdown bullet list, one property per line, **and nothing else**.
- Format each line exactly as:
  - <Category>: <CTL formulas>  -- <brief English rationale>
  - <Category>: <LTL formulas>  -- <brief English rationale>

Property mix
------------
* ≥ 3 Safety / Invariant clauses  
* ≥ 2 Liveness / Response clauses  
* ≥ 1 Fairness clause (or explicitly state “# No fairness issues detected” if truly irrelevant)  
* Add Bounded-time or Deadlock formulas only if they make sense.

Variable naming rules
---------------------
* Use plausible, domain-generic names in `snake_case` (e.g., `door_open`, `car_moving`, `count[2:0]`).
* Do **not** invent implementation-specific details—keep it abstract.

Quality checklist
-----------------
1. Each property must be realistic for the given system.
2. No tautologies or vacuous statements.
3. Formulas must be NuSMV-valid (use AG, AF, AX, EG, etc.).
4. Do **not** include any introductory or concluding prose.

Now generate the list.
"""

categorize_sys = """You are a formal verification and model-checking expert.
Input: A multiline string of NuSMV SPEC
Task: return the same clauses grouped into three categories Pre-conditions, Post-Conditions, Invariants.
"""

categorize_human = """You will receive one string, `properties`, that contains one-or-more NuSMV
property clauses (each on its own line, comments allowed).

NuSMV Properties:
{props}

Your task is to re-emit the SAME clauses, grouped under the seven section
headers shown below, using the classification rules that follow.

CLASSIFICATION RULES  (place a clause in the FIRST bucket that matches):

1. Safety / Invariants
   - Pure universal assertions: AG φ or G φ
   - No top-level “→”, no “AX/X”, no “AF/F/U”, no bounds.
   - Example:  SPEC AG !(door_open & car_moving)

2. Transition (Pre → AX Post)
   - Universal implication whose CONSEQUENT talks about the **next** state:
     AG ( P → AX Q) or  G ( P → X Q )
   - Example:  SPEC AG (start_button → AX motor_on)

3. Response / Progress (Liveness)
   - Universal implication whose consequent is **eventual**:
       AG(P → AF R), AG(P → A[PUR]), etc.
   - Stand-alone eventuals also belong here:SPEC F goal
   - Unbounded untils (no “<=k”) live here.
   - Example:  SPEC AG (request → AF served)

4. Fairness / Justice
   - Lines beginning with FAIRNESS, COMPASSION, or containing those keywords.

5. Reachability / Coverage
   - Existence or reachability checks: SPEC EF φ , EF φ , EG φ.
   - Example: SPEC EF state = EmergencyStop

6. Deadlock / Livelock
   - Explicit “CHECK_DEADLOCK”, “CHECK_LIVELOCK”, or clauses that mention
     “deadlock” / “livelock” in the formula or comment.

7. Bounded-Time
   - Any property with time bounds: U<=k , F<=k , [0,k], within k cycles.
   - Example: SPEC AG (alarm → A[!alarm_active U<=10 alarm_active])

If a clause matches none of the above, place it in **Safety / Invariants**.
"""

nusmv_debug_system = """You are an expert AI trained in Model Checking and NuSMV code syntax.
Your task is to analyse a given NuSMV code file and its model-checker output (syntax errors, Specification counter-example traces, etc.) and provide a corrected NuSMV code.
Follow these principles:
 
1. Full-code mindset — Operate on the entire given NuSMV code file; Analyze the complete code and fix all potential issues/errors even if they are not mentioned in the model-checker output. In your final response, always provide the entire revised NuSMV code.
2. Respect NuSMV grammar - Use proper `MODULE/VAR/DEFINE/ASSIGN` blocks and valid `init(...)` and `next(...)` syntax.
3. SPEC triage - For each false property, locate the root cause (undefined var, unsafe transition, missing reset, incorrect specification etc.) and patch it.
"""
 
nusmv_debug_prompt = """Please revise the given NuSMV code file using the model-checker output.
 
NuSMV code file (with line numbers):
 
{nusmv_file}
 
NuSMV model-checker output:
 
{cli_output}
 
**OUTPUT-REQUIREMENTS**
- Output a single code-block with the *entire* corrected `.smv` file without line numbers.
- Inline comments (e.g. `-- FIX: added “& !door_open”`) to mark each edit.  
- No extraneous explanation—just the fixed file itself.
- Do not simplify, omit, or abstract any part of the SMV model code. All property specs should be explicitly populated with relevant details.
"""