import torch.nn as nn
import torch.nn.functional as F
import random
import torch
import math
import matplotlib.pyplot as plt
import os
import pickle
import numpy as np
import csv
from itertools import permutations
from utils import mixup_data, get_lambda
from collections import Counter

class VGG(nn.Module):
    def __init__(self, args, features, num_classes=1000, init_weights=True, **kwargs):
        super(VGG, self).__init__()
        self.features = features
        self.args = args
        self.classifier = nn.Sequential(
            nn.Linear(512, 4096),
            nn.ReLU(True),
            nn.Dropout(),
            nn.Linear(4096, 4096),
            nn.ReLU(True),
            nn.Dropout(),
            nn.Linear(4096, num_classes),
        )

        if init_weights:
            self._initialize_weights()

    def forward(self, x):
        feature_vector = self.features(x)
        feature_vector = feature_vector.view(feature_vector.size(0), -1)
        logits = self.classifier(feature_vector)

        return logits

    def _initialize_weights(self):
        for m in self.modules():
            if isinstance(m, nn.Conv2d):
                nn.init.kaiming_normal_(m.weight, mode='fan_out', nonlinearity='relu')
                if m.bias is not None:
                    nn.init.constant_(m.bias, 0)
            elif isinstance(m, nn.BatchNorm2d):
                nn.init.constant_(m.weight, 1)
                nn.init.constant_(m.bias, 0)
            elif isinstance(m, nn.Linear):
                nn.init.normal_(m.weight, 0, 0.01)
                nn.init.constant_(m.bias, 0)


class VGGManifoldMixup(nn.Module):
    def __init__(self, args, features, num_classes=1000, init_weights=True, **kwargs):
        super(VGGManifoldMixup, self).__init__()
        self.features = features
        self.args = args
        self.classifier = nn.Sequential(
            nn.Linear(512, 4096),
            nn.ReLU(True),
            nn.Dropout(),
            nn.Linear(4096, 4096),
            nn.ReLU(True),
            nn.Dropout(),
            nn.Linear(4096, num_classes),
        )

        if init_weights:
            self._initialize_weights()

    def forward(self, x, label=None, device=None):
        feature_vector = self.features(x)
        feature_vector = feature_vector.view(feature_vector.size(0), -1)
        logits = self.classifier(feature_vector)
        if label == None:
            return logits
        out, y_a, y_b, lam = mixup_data(feature_vector, label, alpha=2.0)
        out = self.classifier(out)
        lam = torch.tensor(lam).to(device)
        return out, y_a, y_b, lam

    def _initialize_weights(self):
        for m in self.modules():
            if isinstance(m, nn.Conv2d):
                nn.init.kaiming_normal_(m.weight, mode='fan_out', nonlinearity='relu')
                if m.bias is not None:
                    nn.init.constant_(m.bias, 0)
            elif isinstance(m, nn.BatchNorm2d):
                nn.init.constant_(m.weight, 1)
                nn.init.constant_(m.bias, 0)
            elif isinstance(m, nn.Linear):
                nn.init.normal_(m.weight, 0, 0.01)
                nn.init.constant_(m.bias, 0)


class VGG_AMA(nn.Module):
    def __init__(self, args, features, num_classes=1000, init_weights=True, beta=2./3., **kwargs):
        super(VGG_AMA, self).__init__()
        self.features = features
        self.args = args
        self.classifier = nn.Sequential(
            nn.Linear(512, 4096),
            nn.ReLU(True),
            nn.Dropout(),
            nn.Linear(4096, 4096),
            nn.ReLU(True),
            nn.Dropout(),
            nn.Linear(4096, num_classes),
        )

        self.softmax = nn.Softmax(dim=-1)
        self.beta = beta
        if init_weights:
            self._initialize_weights()

    def forward(self, x, tr_acc=None, label=None):
        feature_vector = self.features(x)
        feature_vector = feature_vector.view(feature_vector.size(0), -1)
        logits = self.classifier(feature_vector)

        if label is None:
            return logits

        p = math.exp(-self.beta * tr_acc)
        virtual_samples, src_labels, tgt_labels = self.create_virtual_sample_balance(feature_vector, label, p)
        if len(virtual_samples) > 0:
            virtual_logits = self.classifier(virtual_samples)
            out = torch.vstack((logits, virtual_logits))
            labels = torch.cat((label, src_labels))
        return out, labels

    def create_virtual_sample_balance(self, feature_vector, label, p):
        virtual_samples, virtual_labels = [], []
        src_labels, tgt_labels = [], []
        label_np = label.detach().cpu().numpy()
        counter = Counter(label_np)
        prob = {k: 1. / (len(counter) * v) for k, v in counter.items()}
        prob = [prob[v] for v in label_np]
        bsz = len(feature_vector)
        indices = np.random.choice(bsz, (bsz, 2), p=prob)
        indices = np.array([[i, j] for i, j in indices if i != j])
        if len(indices) > 0:
            virtual_samples = feature_vector[indices[:, 0]] * p + \
                              feature_vector[indices[:, 1]] * (1. - p)
            src_labels = label[indices[:, 0]]
            tgt_labels = label[indices[:, 1]]

        return virtual_samples, src_labels, tgt_labels

    def _initialize_weights(self):
        for m in self.modules():
            if isinstance(m, nn.Conv2d):
                nn.init.kaiming_normal_(m.weight, mode='fan_out', nonlinearity='relu')
                if m.bias is not None:
                    nn.init.constant_(m.bias, 0)
            elif isinstance(m, nn.BatchNorm2d):
                nn.init.constant_(m.weight, 1)
                nn.init.constant_(m.bias, 0)
            elif isinstance(m, nn.Linear):
                nn.init.normal_(m.weight, 0, 0.01)
                nn.init.constant_(m.bias, 0)


def make_layers(cfg, channels, batch_norm=False):
    layers = []
    in_channels = channels
    for v in cfg:
        if v == 'M':
            layers += [nn.MaxPool2d(kernel_size=2, stride=2)]
        else:
            conv2d = nn.Conv2d(in_channels, v, kernel_size=3, padding=1)
            if batch_norm:
                layers += [conv2d, nn.BatchNorm2d(v), nn.ReLU(inplace=True)]
            else:
                layers += [conv2d, nn.ReLU(inplace=True)]
            in_channels = v
    return nn.Sequential(*layers)

# vgg11
cfg = [64, "M", 128, "M", 256, 256, "M", 512, 512, "M", 512, 512, "M"]

def vgg_origin(args, num_classes, **kwargs):
    return VGG(args, make_layers(cfg, 3, True), num_classes, True)

def vgg_manifold_mixup(args, num_classes, **kwargs):
    return VGGManifoldMixup(args, make_layers(cfg, 3, True), num_classes, True)

def vgg_AMA(args, num_classes, **kwargs):
    return VGG_AMA(args, make_layers(cfg, 3, True), num_classes, True, beta=args.beta)