import torch
import torch.nn as nn
import torch.nn.functional as F
import numpy as np
from itertools import permutations
from utils import mixup_data, get_lambda
import math
from collections import Counter

class BasicBlock(nn.Module):
    expansion = 1

    def __init__(self, in_planes, planes, stride=1):
        super(BasicBlock, self).__init__()
        self.conv1 = nn.Conv2d(
            in_planes, planes, kernel_size=3, stride=stride, padding=1, bias=False)
        self.bn1 = nn.BatchNorm2d(planes)
        self.conv2 = nn.Conv2d(planes, planes, kernel_size=3,
                               stride=1, padding=1, bias=False)
        self.bn2 = nn.BatchNorm2d(planes)

        self.shortcut = nn.Sequential()
        if stride != 1 or in_planes != self.expansion*planes:
            self.shortcut = nn.Sequential(
                nn.Conv2d(in_planes, self.expansion*planes,
                          kernel_size=1, stride=stride, bias=False),
                nn.BatchNorm2d(self.expansion*planes)
            )

    def forward(self, x):
        out = F.relu(self.bn1(self.conv1(x)))
        out = self.bn2(self.conv2(out))
        out += self.shortcut(x)
        out = F.relu(out)
        return out


class Bottleneck(nn.Module):
    expansion = 4

    def __init__(self, in_planes, planes, stride=1):
        super(Bottleneck, self).__init__()
        self.conv1 = nn.Conv2d(in_planes, planes, kernel_size=1, bias=False)
        self.bn1 = nn.BatchNorm2d(planes)
        self.conv2 = nn.Conv2d(planes, planes, kernel_size=3,
                               stride=stride, padding=1, bias=False)
        self.bn2 = nn.BatchNorm2d(planes)
        self.conv3 = nn.Conv2d(planes, self.expansion *
                               planes, kernel_size=1, bias=False)
        self.bn3 = nn.BatchNorm2d(self.expansion*planes)

        self.shortcut = nn.Sequential()
        if stride != 1 or in_planes != self.expansion*planes:
            self.shortcut = nn.Sequential(
                nn.Conv2d(in_planes, self.expansion*planes,
                          kernel_size=1, stride=stride, bias=False),
                nn.BatchNorm2d(self.expansion*planes)
            )

    def forward(self, x):
        out = F.relu(self.bn1(self.conv1(x)))
        out = F.relu(self.bn2(self.conv2(out)))
        out = self.bn3(self.conv3(out))
        out += self.shortcut(x)
        out = F.relu(out)
        return out


class ResNetForSupCon(nn.Module):
    def __init__(self, block, num_blocks, in_channel=3, zero_init_residual=False):
        super(ResNetForSupCon, self).__init__()
        self.in_planes = 64

        self.conv1 = nn.Conv2d(in_channel, 64, kernel_size=3, stride=1, padding=1,
                               bias=False)
        self.bn1 = nn.BatchNorm2d(64)
        self.layer1 = self._make_layer(block, 64, num_blocks[0], stride=1)
        self.layer2 = self._make_layer(block, 128, num_blocks[1], stride=2)
        self.layer3 = self._make_layer(block, 256, num_blocks[2], stride=2)
        self.layer4 = self._make_layer(block, 512, num_blocks[3], stride=2)
        self.avgpool = nn.AdaptiveAvgPool2d((1, 1))

        for m in self.modules():
            if isinstance(m, nn.Conv2d):
                nn.init.kaiming_normal_(m.weight, mode='fan_out', nonlinearity='relu')
            elif isinstance(m, (nn.BatchNorm2d, nn.GroupNorm)):
                nn.init.constant_(m.weight, 1)
                nn.init.constant_(m.bias, 0)

        # Zero-initialize the last BN in each residual branch,
        # so that the residual branch starts with zeros, and each residual block behaves
        # like an identity. This improves the model by 0.2~0.3% according to:
        # https://arxiv.org/abs/1706.02677
        if zero_init_residual:
            for m in self.modules():
                if isinstance(m, Bottleneck):
                    nn.init.constant_(m.bn3.weight, 0)
                elif isinstance(m, BasicBlock):
                    nn.init.constant_(m.bn2.weight, 0)

    def _make_layer(self, block, planes, num_blocks, stride):
        strides = [stride] + [1] * (num_blocks - 1)
        layers = []
        for i in range(num_blocks):
            stride = strides[i]
            layers.append(block(self.in_planes, planes, stride))
            self.in_planes = planes * block.expansion
        return nn.Sequential(*layers)

    def forward(self, x, layer=100):
        out = F.relu(self.bn1(self.conv1(x)))
        out = self.layer1(out)
        out = self.layer2(out)
        out = self.layer3(out)
        out = self.layer4(out)
        out = self.avgpool(out)
        out = torch.flatten(out, 1)
        return out


def resnet18ForSupCon(**kwargs):
    return ResNetForSupCon(BasicBlock, [2, 2, 2, 2], **kwargs)


def resnet34ForSupCon(**kwargs):
    return ResNetForSupCon(BasicBlock, [3, 4, 6, 3], **kwargs)


def resnet50ForSupCon(**kwargs):
    return ResNetForSupCon(Bottleneck, [3, 4, 6, 3], **kwargs)


model_dict = {
    'resnet18': [resnet18ForSupCon, 512],
    'resnet34': [resnet34ForSupCon, 512],
    'resnet50': [resnet50ForSupCon, 2048],
}


class SupCEResNet(nn.Module):
    def __init__(self, name='resnet50', num_classes=10):
        super(SupCEResNet, self).__init__()
        model_fun, dim_in = model_dict[name]
        self.encoder = model_fun()
        self.fc = nn.Linear(dim_in, num_classes)

    def forward(self, x):
        return self.fc(self.encoder(x))


class SupConResNet(nn.Module):
    def __init__(self, name='resnet50', head='linear', feat_dim=128):
        super(SupConResNet, self).__init__()
        model_fun, dim_in = model_dict[name]
        self.encoder = model_fun()
        if head == 'linear':
            self.head = nn.Linear(dim_in, feat_dim)
        elif head == 'mlp':
            self.head = nn.Sequential(
                nn.Linear(dim_in, dim_in),
                nn.ReLU(inplace=True),
                nn.Linear(dim_in, feat_dim)
            )
        else:
            raise NotImplementedError(
                'head not supported: {}'.format(head))

    def forward(self, x):
        feat = self.encoder(x)
        feat = F.normalize(self.head(feat), dim=1)
        return feat


class LinearClassifier(nn.Module):
    def __init__(self, name='resnet50', num_classes=10):
        super(LinearClassifier, self).__init__()
        _, feat_dim = model_dict[name]
        self.fc = nn.Linear(feat_dim, num_classes)

    def forward(self, features):
        return self.fc(features)


class ManifoldMixupResNet(nn.Module):
    def __init__(self, name='resnet50', num_classes=10):
        super(ManifoldMixupResNet, self).__init__()
        model_fun, dim_in = model_dict[name]
        self.encoder = model_fun()
        self.fc = nn.Linear(dim_in, num_classes)

    def forward(self, x, label=None, device=None):
        if label is None:
            return self.fc(self.encoder(x))
        feature_vector = self.encoder(x)
        out, y_a, y_b, lam = mixup_data(feature_vector, label, alpha=2.0)
        out = self.fc(out)
        lam = torch.tensor(lam).to(device)
        return out, y_a, y_b, lam


class AMAResNet(nn.Module):
    def __init__(self, name='resnet50', num_classes=10, beta=2./3., **kwargs):
        super(AMAResNet, self).__init__()
        model_fun, dim_in = model_dict[name]
        self.encoder = model_fun()
        self.fc = nn.Linear(dim_in, num_classes)
        self.beta = beta

    def forward(self,  x, tr_acc=None, label=None):
        feature_vector = self.encoder(x)
        logits = self.fc(feature_vector)
        if label == None:
            return logits

        p = math.exp(-self.beta * tr_acc)
        virtual_samples, src_labels, tgt_labels = self.create_virtual_sample_balance(feature_vector, label, p)
        if len(virtual_samples) > 0:
            virtual_logits = self.fc(virtual_samples)
            out = torch.vstack((logits, virtual_logits))
            labels = torch.cat((label, src_labels))
        return out, labels

    def create_virtual_sample_balance(self, feature_vector, label, p):
        virtual_samples, virtual_labels = [], []
        src_labels, tgt_labels = [], []
        label_np = label.detach().cpu().numpy()
        counter = Counter(label_np)
        prob = {k: 1. / (len(counter) * v) for k, v in counter.items()}
        prob = [prob[v] for v in label_np]
        bsz = len(feature_vector)
        indices = np.random.choice(bsz, (bsz, 2), p=prob)
        indices = np.array([[i, j] for i, j in indices if i != j])
        if len(indices) > 0:
            virtual_samples = feature_vector[indices[:, 0]] * p + \
                              feature_vector[indices[:, 1]] * (1. - p)
            src_labels = label[indices[:, 0]]
            tgt_labels = label[indices[:, 1]]

        return virtual_samples, src_labels, tgt_labels

def resnet_origin(num_classes, **kwargs):
    return SupCEResNet(name='resnet50', num_classes=num_classes)

def resnet_manifold_mixup(num_classes, **kwargs):
    return ManifoldMixupResNet(name='resnet50', num_classes=num_classes)

def resnet_AMA(num_classes, beta, **kwargs):
    return AMAResNet(name='resnet50', num_classes=num_classes, beta=beta)
