import torch
import torch.nn as nn
import torch.nn.functional as F
import torch.nn.init as init
from torch.nn import Parameter
import numpy as np
from itertools import permutations
from utils import mixup_data
import collections
from collections import Counter
import math
__all__ = ['ResNet_s', 'resnet20', 'resnet32', 'resnet44', 'resnet56', 'resnet110', 'resnet1202']


def _weights_init(m):
    classname = m.__class__.__name__
    if isinstance(m, nn.Linear) or isinstance(m, nn.Conv2d):
        init.kaiming_normal_(m.weight)


class NormedLinear(nn.Module):

    def __init__(self, in_features, out_features):
        super(NormedLinear, self).__init__()
        self.weight = Parameter(torch.Tensor(in_features, out_features))
        self.weight.data.uniform_(-1, 1).renorm_(2, 1, 1e-5).mul_(1e5)

    def forward(self, x):
        out = F.normalize(x, dim=1).mm(F.normalize(self.weight, dim=0))
        return out


class LambdaLayer(nn.Module):

    def __init__(self, lambd):
        super(LambdaLayer, self).__init__()
        self.lambd = lambd

    def forward(self, x):
        return self.lambd(x)


class BasicBlock(nn.Module):
    expansion = 1

    def __init__(self, in_planes, planes, stride=1, option='A'):
        super(BasicBlock, self).__init__()
        self.conv1 = nn.Conv2d(in_planes, planes, kernel_size=3, stride=stride, padding=1, bias=False)
        self.bn1 = nn.BatchNorm2d(planes)
        self.conv2 = nn.Conv2d(planes, planes, kernel_size=3, stride=1, padding=1, bias=False)
        self.bn2 = nn.BatchNorm2d(planes)

        self.shortcut = nn.Sequential()
        if stride != 1 or in_planes != planes:
            if option == 'A':
                """
                For CIFAR10 ResNet paper uses option A.
                """
                self.shortcut = LambdaLayer(lambda x:
                                            F.pad(x[:, :, ::2, ::2], (0, 0, 0, 0, planes // 4, planes // 4), "constant",
                                                  0))
            elif option == 'B':
                self.shortcut = nn.Sequential(
                    nn.Conv2d(in_planes, self.expansion * planes, kernel_size=1, stride=stride, bias=False),
                    nn.BatchNorm2d(self.expansion * planes)
                )

    def forward(self, x):
        out = F.relu(self.bn1(self.conv1(x)))
        out = self.bn2(self.conv2(out))
        out += self.shortcut(x)
        out = F.relu(out)
        return out


class ResNet_s(nn.Module):

    def __init__(self, block, num_blocks, num_classes=10, use_norm=False, return_features=False):
        super(ResNet_s, self).__init__()
        factor = 1
        self.in_planes = 16 * factor

        self.conv1 = nn.Conv2d(3, 16 * factor, kernel_size=3, stride=1, padding=1, bias=False)
        self.bn1 = nn.BatchNorm2d(16 * factor)
        self.layer1 = self._make_layer(block, 16 * factor, num_blocks[0], stride=1)
        self.layer2 = self._make_layer(block, 32 * factor, num_blocks[1], stride=2)
        self.layer3 = self._make_layer(block, 64 * factor, num_blocks[2], stride=2)
        self.avgpool = nn.AdaptiveAvgPool2d((1, 1))
        if use_norm:
            self.fc = NormedLinear(64 * factor, num_classes)
        else:
            self.fc = nn.Linear(64 * factor, num_classes)
        self.apply(_weights_init)
        self.return_encoding = return_features

    def _make_layer(self, block, planes, num_blocks, stride):
        strides = [stride] + [1] * (num_blocks - 1)
        layers = []
        for stride in strides:
            layers.append(block(self.in_planes, planes, stride))
            self.in_planes = planes * block.expansion

        return nn.Sequential(*layers)

    def forward(self, x):
        out = F.relu(self.bn1(self.conv1(x)))
        out = self.layer1(out)
        out = self.layer2(out)
        out = self.layer3(out)
        out = self.avgpool(out)
        encoding = out.view(out.size(0), -1)
        out = self.fc(encoding)
        if self.return_encoding:
            return out, encoding
        else:
            return out


class ResNet_manifoldMixup(nn.Module):
    def __init__(self, block, num_blocks, num_classes=10, use_norm=False, return_features=False):
        super(ResNet_manifoldMixup, self).__init__()
        factor = 1
        self.in_planes = 16 * factor

        self.conv1 = nn.Conv2d(3, 16 * factor, kernel_size=3, stride=1, padding=1, bias=False)
        self.bn1 = nn.BatchNorm2d(16 * factor)
        self.layer1 = self._make_layer(block, 16 * factor, num_blocks[0], stride=1)
        self.layer2 = self._make_layer(block, 32 * factor, num_blocks[1], stride=2)
        self.layer3 = self._make_layer(block, 64 * factor, num_blocks[2], stride=2)
        self.avgpool = nn.AdaptiveAvgPool2d((1, 1))
        if use_norm:
            self.fc = NormedLinear(64 * factor, num_classes)
        else:
            self.fc = nn.Linear(64 * factor, num_classes)
        self.apply(_weights_init)
        self.return_encoding = return_features

    def _make_layer(self, block, planes, num_blocks, stride):
        strides = [stride] + [1] * (num_blocks - 1)
        layers = []
        for stride in strides:
            layers.append(block(self.in_planes, planes, stride))
            self.in_planes = planes * block.expansion

        return nn.Sequential(*layers)

    def forward(self, x, label=None, device=None):
        out = F.relu(self.bn1(self.conv1(x)))
        out = self.layer1(out)
        out = self.layer2(out)
        out = self.layer3(out)
        out = self.avgpool(out)
        feature_vector = out.view(out.size(0), -1)
        logits = self.fc(feature_vector)
        if label is None:
            if self.return_encoding:
                return logits, feature_vector
            else:
                return logits
        out, y_a, y_b, lam = mixup_data(feature_vector, label, alpha=2.0)
        out = self.fc(out)
        lam = torch.tensor(lam).to(device)
        return out, y_a, y_b, lam


class ResNet_AMA(nn.Module):
    def __init__(self, block, num_blocks, num_classes=10, use_norm=False, return_features=False, beta=2./3., **kwargs):
        super(ResNet_AMA, self).__init__()
        factor = 1
        self.in_planes = 16 * factor

        self.conv1 = nn.Conv2d(3, 16 * factor, kernel_size=3, stride=1, padding=1, bias=False)
        self.bn1 = nn.BatchNorm2d(16 * factor)
        self.layer1 = self._make_layer(block, 16 * factor, num_blocks[0], stride=1)
        self.layer2 = self._make_layer(block, 32 * factor, num_blocks[1], stride=2)
        self.layer3 = self._make_layer(block, 64 * factor, num_blocks[2], stride=2)
        self.avgpool = nn.AdaptiveAvgPool2d((1, 1))
        if use_norm:
            self.fc = NormedLinear(64 * factor, num_classes)
        else:
            self.fc = nn.Linear(64 * factor, num_classes)
        self.apply(_weights_init)
        self.return_encoding = return_features

        self.beta = beta

    def _make_layer(self, block, planes, num_blocks, stride):
        strides = [stride] + [1] * (num_blocks - 1)
        layers = []
        for stride in strides:
            layers.append(block(self.in_planes, planes, stride))
            self.in_planes = planes * block.expansion

        return nn.Sequential(*layers)

    def forward(self, x, tr_acc=None, label=None):
        out = F.relu(self.bn1(self.conv1(x)))
        out = self.layer1(out)
        out = self.layer2(out)
        out = self.layer3(out)
        out = self.avgpool(out)
        feature_vector = out.view(out.size(0), -1)
        logits = self.fc(feature_vector)
        if label == None:
            if self.return_encoding:
                return logits, feature_vector
            else:
                return logits

        p = math.exp(-self.beta * tr_acc)
        virtual_samples, src_labels, tgt_labels = self.create_virtual_sample_balance(feature_vector, label, p)
        if len(virtual_samples) > 0:
            virtual_logits = self.fc(virtual_samples)
            output = torch.vstack((logits, virtual_logits))
            labels = torch.cat((label, src_labels))
        return output, labels

    def create_virtual_sample_balance(self, feature_vector, label, p):
        virtual_samples, virtual_labels = [], []
        src_labels, tgt_labels = [], []
        label_np = label.detach().cpu().numpy()
        counter = Counter(label_np)
        prob = {k: 1. / (len(counter) * v) for k, v in counter.items()}
        prob = [prob[v] for v in label_np]
        bsz = len(feature_vector)
        indices = np.random.choice(bsz, (bsz, 2), p=prob)
        indices = np.array([[i, j] for i, j in indices if i != j])
        if len(indices) > 0:
            virtual_samples = feature_vector[indices[:, 0]] * p + \
                              feature_vector[indices[:, 1]] * (1. - p)
            src_labels = label[indices[:, 0]]
            tgt_labels = label[indices[:, 1]]

        return virtual_samples, src_labels, tgt_labels


def resnet20():
    return ResNet_s(BasicBlock, [3, 3, 3])


def resnet32(num_classes=100, use_norm=False, return_features=False):
    return ResNet_s(BasicBlock, [5, 5, 5], num_classes=num_classes, use_norm=use_norm, return_features=return_features)

def resnet32_manifold_mixup(num_classes=100, use_norm=False, return_features=False):
    return ResNet_manifoldMixup(BasicBlock, [5, 5, 5], num_classes=num_classes, use_norm=use_norm, return_features=return_features)

def resnet32_AMA(num_classes=100, beta=2./3., use_norm=False, return_features=False):
    return ResNet_AMA(BasicBlock, [5, 5, 5], num_classes=num_classes, use_norm=use_norm, return_features=return_features, beta=beta)

def resnet44():
    return ResNet_s(BasicBlock, [7, 7, 7])


def resnet56():
    return ResNet_s(BasicBlock, [9, 9, 9])


def resnet110():
    return ResNet_s(BasicBlock, [18, 18, 18])


def resnet1202():
    return ResNet_s(BasicBlock, [200, 200, 200])
