using Random, LinearAlgebra, Pickle;
using Statistics, StatsPlots, ColorSchemes;
import Distributions;
include("../binary_search.jl");
include("../expfam.jl");

## Getting identification strategy

function everybody(expe, wstar)
    if expe == "fast"
        iss = [(AdaTopTwo("cst", "EB", "TCI", false), GLRT("GK16")),
               (AdaTopTwo("cst", "EB", "TCI", true), GLRT("GK16")),
               (AdaTopTwo("cst", "EB", "TCIs", true), GLRT("GK16")),
               (AdaTopTwo("cst", "EB", "TCIsp", true), GLRT("GK16")),
               (AdaTopTwo("cst", "TS", "TC", false), GLRT("GK16")),
               (AdaTopTwo("cst", "TS", "RS", false), GLRT("GK16")),
               (AdaTopTwo("cst", "UCBI", "TC", true), GLRT("GK16")),
               (LUCB(), GLRT("GK16")),
               (LUCBhalf(), GLRT("GK16")),
               (RoundRobin(), GLRT("GK16"))];
    elseif expe == "random"
        iss = [(AdaTopTwo("cst", "EB", "TCI", false), GLRT("GK16")),
               (AdaTopTwo("cst", "EB", "TCI", true), GLRT("GK16")),
               (AdaTopTwo("cst", "EB", "TCIs+", true), GLRT("GK16")),
               (AdaTopTwo("cst", "EB", "TCIs", true), GLRT("GK16")),
               (AdaTopTwo("cst", "EB", "TCIsp+", true), GLRT("GK16")),
               (AdaTopTwo("cst", "EB", "TCIsp", true), GLRT("GK16")),
               (AdaTopTwo("BC", "EB", "TCI", false), GLRT("GK16")),
               (AdaTopTwo("cst", "TS", "TC", false), GLRT("GK16")),
               (AdaTopTwo("cst", "TS", "TC", true), GLRT("GK16")),
               (AdaTopTwo("BC", "TS", "TC", false), GLRT("GK16")),
               (AdaTopTwo("cst", "TS", "RS", false), GLRT("GK16")),
               (AdaTopTwo("cst", "TS", "RS", true), GLRT("GK16")),
               (AdaTopTwo("BC", "TS", "RS", false), GLRT("GK16")),
               (AdaTopTwo("cst", "UCBI", "TC", true), GLRT("GK16")),
               (AdaTopTwo("cst", "UCBI", "TC", false), GLRT("GK16")),
               (AdaTopTwo("BC", "UCBI", "TC", true), GLRT("GK16")),
               (FWSampling(CTracking), GLRT("GK16")),
               (DKM(CTracking), GLRT("GK16")),
               (LUCB(), GLRT("GK16")),
               (LUCBhalf(), GLRT("GK16")),
               (TaS(CTracking), GLRT("GK16")),
               (RoundRobin(), GLRT("GK16"))];
    elseif expe == "penaTCI"
        iss = [(AdaTopTwo("cst", "EB", "TCI", false), GLRT("GK16")),
               (AdaTopTwo("cst", "EB", "TCI", true), GLRT("GK16")),
               (AdaTopTwo("cst", "EB", "TCIs+", true), GLRT("GK16")),
               (AdaTopTwo("cst", "EB", "TCIs", true), GLRT("GK16")),
               (AdaTopTwo("cst", "EB", "TCIsp+", true), GLRT("GK16")),
               (AdaTopTwo("cst", "EB", "TCIsp", true), GLRT("GK16"))];
    elseif expe == "fastAdapt"
        iss = [(AdaTopTwo("cst", "EB", "TCI", false), GLRT("GK16")),
               (AdaTopTwo("BC", "EB", "TCI", false), GLRT("GK16")),
               (AdaTopTwo("cst", "TS", "TC", false), GLRT("GK16")),
               (AdaTopTwo("BC", "TS", "TC", false), GLRT("GK16")),
               (AdaTopTwo("cst", "TS", "RS", false), GLRT("GK16")),
               (AdaTopTwo("BC", "TS", "RS", false), GLRT("GK16")),
               (AdaTopTwo("cst", "UCBI", "TC", true), GLRT("GK16")),
               (AdaTopTwo("BC", "UCBI", "TC", true), GLRT("GK16")),
               (LUCB(), GLRT("GK16")),
               (LUCBhalf(), GLRT("GK16")),
               (RoundRobin(), GLRT("GK16"))];
    elseif expe == "test"
        iss = [(AdaTopTwo("cst", "EB", "TCI", false), GLRT("GK16")),
                (LUCB(), GLRT("GK16")),
                (LUCBhalf(), GLRT("GK16")),
                (RoundRobin(), GLRT("GK16"))];
    else
        @error "Not implemented";
    end
    return iss;
end


## Instances

function get_instance_experiment(param_inst)
    K = param_inst["nK"];
    if param_inst["inst"] == "sparse"
        μs = [(k == 1) ? 0.25 : 0. for k in 1:K];
        dists = [Gaussian() for μ in μs];
        return μs, dists;
    elseif param_inst["inst"] == "alpha3"
        μs = [1. - ((k - 1) / K)^(0.3) for k in 1:K];
        dists = [Gaussian() for μ in μs];
        return μs, dists;
    elseif param_inst["inst"] == "alpha6"
        μs = [1. - ((k - 1) / K)^(0.6) for k in 1:K];
        dists = [Gaussian() for μ in μs];
        return μs, dists;
    elseif param_inst["inst"] == "eqgap"
        μs = [(k == 1) ? 0 : -0.1 for k in 1:K];
        dists = [Gaussian() for μ in μs];
        return μs, dists;
    elseif param_inst["inst"] == "eqgapE"
        μs = [(k == 1) ? 0 : -0.5 for k in 1:K];
        dists = [Gaussian() for μ in μs];
        return μs, dists;
    elseif param_inst["inst"] == "trivial"
        μs = [(k == 1) ? 0 : -1 - k for k in 1:K];
        dists = [Gaussian() for μ in μs];
        return μs, dists;
    else
        @error "Not Implemented";
    end
end

## Summary

function print_summary(pep, dists, μs, Tstar, wstar, δs, iss, datas, Nruns, file)
    nK = nanswers(pep, μs);
    astar = istar(pep, μs);

    ios = (stdout, open(file, "a"));
    for i in 1:length(δs)
        δ = δs[i];
        data = getindex.(datas, i);

        # lower bound
        kl = (1 - 2 * δ) * log((1 - δ) / δ);
        lbd = kl * Tstar;

        rule = repeat("-", 90);
        for io in ios
            println(io, "");
            println(io, rule);
            println(io, long(pep));
            println(io, rule);
            println(io, @sprintf("%10s", "a*"), "  ",
                    @sprintf("%10s", "δ"), "  ",
                    @sprintf("%10s", "K"), "  ",
                    @sprintf("%10s", "LBD"));
            println(io, @sprintf("%10.3f", astar), "  ",
                    @sprintf("%10.3f", δ), "  ",
                    @sprintf("%10.0f", nK), "  ",
                    @sprintf("%10.0f", lbd));
            println(io, rule);
            println(io, @sprintf("%-30s", "Arm"),
                    join(map(k -> @sprintf("%6s", k), 1:nK)), " ",
                    @sprintf("%7s", "total"), "  ",
                    @sprintf("%7s", "std"), "  ",
                    @sprintf("%7s", "err"), "  ",
                    @sprintf("%7s", "time"));
            println(io, @sprintf("%-30s", "μs"),
                    join(map(x -> @sprintf("%6.2f", x), μs)));
            println(io, @sprintf("%-30s", "w*"),
                    join(map(x -> @sprintf("%6.2f", x), wstar)));
            println(io, rule);
            println(io, @sprintf("%-30s", "Sampling rule"));
        end

        # iss
        for r in eachindex(iss)
            τs = [sum(x[2]) for x in data[r,:]];
            Eτ = mean(τs);
            στ = std(τs);
            err = sum(x->x[1] .!= astar, data[r,:])/Nruns;
            tim = sum(x->x[3], data[r,:])/Nruns;

            for io in ios
                println(io, @sprintf("%-30s", abbrev(iss[r][1]) * " & " * abbrev(iss[r][2])),
                        join(map(k -> @sprintf("%6.0f", sum(x->x[2][k], data[r,:])/Nruns), 1:nK)), " ",
                        @sprintf("%7.0f", Eτ), "  ",
                        @sprintf("%7.0f", στ), "  ",
                        @sprintf("%7.5f", err), "  ",
                        @sprintf("%7.5f", tim/1e6)
                        );
            end
            if err > δ
                @warn "too many errors for $(iss[r])";
            end
        end

        for io in ios
            println(io, rule);
        end
    end
end

function print_rand_summary(δs, iss, data, iss_index, param_inst, Nruns, file)
    ios = (stdout, open(file, "a"));
    δ = δs[1];
    K = param_inst["nK"];

    rule = repeat("-", 90);
    for io in ios
        println(io, "");
        println(io, rule);
        println(io, @sprintf("%10s", "δ"), "  ",
                    @sprintf("%10s", "K"), "  ",
                    @sprintf("%10s", "Δmin"), "  ",
                    @sprintf("%10s", "Δmax"), "  ",
                    @sprintf("%10s", "μ1"));
        println(io, @sprintf("%10.3f", δ), "  ",
                    @sprintf("%10.0f", K), "  ",
                    @sprintf("%10.0f", param_inst["gapmin"]), "  ",
                    @sprintf("%10.0f", param_inst["gapmax"]), "  ",
                    @sprintf("%10.0f", param_inst["mu1"]));
        println(io, rule);
        println(io, @sprintf("%-30s", ""),
                    @sprintf("%7s", "total"), "  ",
                    @sprintf("%7s", "std"), "  ",
                    @sprintf("%7s", "err"), "  ",
                    @sprintf("%7s", "time"));
        println(io, rule);
        println(io, @sprintf("%-30s", "Sampling rule"));
    end

    for is in iss
        r = iss_index[is];
        results = getindex.(data[r, :], 1);
        instances = getindex.(data[r, :], 2);
        τs = [sum(x[2]) for x in results];
        Eτ = mean(τs);
        στ = std(τs);
        err = sum(x->x[1] .!= 1, results)/Nruns;
        tim = sum(x->x[3], results)/Nruns;

        for io in ios
            println(io, @sprintf("%-30s", abbrev_is(is)),
                        @sprintf("%7.0f", Eτ), "  ",
                        @sprintf("%7.0f", στ), "  ",
                        @sprintf("%7.5f", err), "  ",
                        @sprintf("%7.5f", tim/1e6));
        end

        if err > δ
            @warn "too many errors for " * abbrev_is(is);
        end
    end
    for io in ios
        println(io, rule);
    end
end

abbrev_is(is) = typeof(is) == ExpGap ? abbrev(is) : abbrev(is[1]) * "-" * abbrev(is[2]);

## Plots

name_algos_plots = Dict("opt-G-GK16" => "Fixed",
                        "DKM-C-G-GK16" => "DKM",
                        "cst-EB-TCI-G-GK16" => "EB-TCI",
                        "cst-UCB-TC-G-GK16" => "S-TTUCB",
                        "cst-UCBI-TC-G-GK16" => "S-TTUCB",
                        "cst-TS-TC-G-GK16" => "T3C",
                        "cst-TS-RS-G-GK16" => "TTTS",
                        "cst-IMED-TC-G-GK16" => "S-IMED-TC",
                        "cst-EB-TCI-T-G-GK16" => "T-EB-TCI",
                        "cst-UCB-TC-T-G-GK16" => "TTUCB",
                        "cst-UCBI-TC-T-G-GK16" => "TTUCB",
                        "cst-TS-TC-T-G-GK16" => "T-T3C",
                        "cst-TS-RS-T-G-GK16" => "T-TTTS",
                        "cst-IMED-TC-T-G-GK16" => "IMED-TC",
                        "BC-EB-TCI-G-GK16" => "A-EB-TCI",
                        "BC-UCB-TC-G-GK16" => "AS-TTUCB",
                        "BC-UCBI-TC-G-GK16" => "AS-TTUCB",
                        "BC-TS-TC-G-GK16" => "A-T3C",
                        "BC-TS-RS-G-GK16" => "A-TTTS",
                        "BC-IMED-TC-G-GK16" => "AS-IMED-TC",
                        "BC-EB-TCI-T-G-GK16" => "AT-EB-TCI",
                        "BC-UCB-TC-T-G-GK16" => "A-TTUCB",
                        "BC-UCBI-TC-T-G-GK16" => "A-TTUCB",
                        "BC-TS-TC-T-G-GK16" => "AT-T3C",
                        "BC-TS-RS-T-G-GK16" => "AT-TTTS",
                        "BC-IMED-TC-T-G-GK16" => "A-IMED-TC",
                        "FWS-C-G-GK16" => "FWS",
                        "TaS-C-G-GK16" => "TaS",
                        "LUCB-G-GK16" => "LUCB",
                        "LUCBhalf-G-GK16" => "1/2-LUCB",
                        "RR-G-GK16" => "Uniform");

name_algos_plots_bis = Dict("opt-G-GK16" => "Fixed",
                            "DKM-C-G-GK16" => "DKM",
                            "cst-EB-TCI-G-GK16" => "EB-TCI",
                            "cst-UCBI-TC-T-G-GK16" => "TTUCB",
                            "cst-TS-TC-G-GK16" => "T3C",
                            "cst-TS-RS-G-GK16" => "TTTS",
                            "FWS-C-G-GK16" => "FWS",
                            "TaS-C-G-GK16" => "TaS",
                            "LUCB-G-GK16" => "LUCB",
                            "LUCBhalf-G-GK16" => "1/2-LUCB",
                            "RR-G-GK16" => "Uniform");

pal_colors = palette(:tab20);
dict_colors = Dict("EB-TCI" => pal_colors[1],
                   "T-EB-TCI" => pal_colors[2],
                   "A-EB-TCI" => pal_colors[2],
                   "AT-EB-TCI" => pal_colors[2],
                   "Original" => pal_colors[5],
                   "κ = 1.2" => pal_colors[1],
                   "κ = 2" => pal_colors[2],
                   "α = 2" => pal_colors[3],
                   "α = 1.2" => pal_colors[4],
                   "T3C" => pal_colors[19],
                   "T-T3C" => pal_colors[20],
                   "A-T3C" => pal_colors[20],
                   "AT-T3C" => pal_colors[20],
                   "TTTS" => pal_colors[17],
                   "T-TTTS" => pal_colors[18],
                   "A-TTTS" => pal_colors[18],
                   "AT-TTTS" => pal_colors[18],
                   "TTUCB" => pal_colors[7],
                   "S-TTUCB" => pal_colors[8],
                   "A-TTUCB" => pal_colors[8],
                   "AS-TTUCB" => pal_colors[8],
                   "IMED-TC" => pal_colors[10],
                   "IMED-TC-T" => pal_colors[11],
                   "BC-IMED-TC" => pal_colors[10],
                   "BC-IMED-TC-T" => pal_colors[11],
                   "TaS" => pal_colors[3],
                   "FWS" => pal_colors[15],
                   "DKM" => pal_colors[5],
                   "LUCB" => pal_colors[16],
                   "1/2-LUCB" => pal_colors[17],
                   "Fixed" => pal_colors[12],
                   "Uniform" => pal_colors[14]);

function plot_samp(pep, μs, Tstar, wstar, δ, iss, data, N, file, iss_to_keep, name_iss)
    keep = filter(i -> filter_plot_samp(iss[i], iss_to_keep), 1:length(iss));
    _plot_samp(pep, μs, Tstar, wstar, δ, iss[keep], data[keep,:], N, file, iss_to_keep, name_iss);
end

function filter_plot_samp(is, iss_to_keep)
    sr, rsp = is;
    if abbrev(sr) * "-" * abbrev(rsp) in iss_to_keep
        return true;
    else
        return false;
    end
end

function _plot_samp(pep, μs, Tstar, wstar, δ, iss, data, N, file, iss_to_keep, name_iss)
    # lower bound
    kl = (1 - 2 * δ) * log((1 - δ) / δ);
    lbd = kl * Tstar;

    for r in eachindex(iss)
        _τs = [sum(x[2]) for x in data[r, :]];
        τs = [(τ <= 15 * lbd) ? τ : 15 * lbd for τ in _τs];
        x = name_algos_plots[abbrev(iss[r][1]) * "-" * abbrev(iss[r][2])];
        if r == 1
            boxplot(
                [x],
                [τs],
                label="",
                notch=true,
                xtickfontsize=8,
                ytickfontsize=12,
                color=dict_colors[x],
                outliers=true);
        else
            boxplot!(
                [x],
                [τs],
                label="",
                notch=true,
                xtickfontsize=8,
                ytickfontsize=12,
                color=dict_colors[x],
                outliers=true);
        end
        plot!([x], [mean(τs)], marker=(:star4,10,:black), label="");
    end

    plot!([lbd], seriestype=:hline, legend=:topright, label="lbd");

    savefig(file);
end

function plot_arms(pep, μs, Tstar, wstar, δ, iss, data, N, file, iss_to_keep, name_iss)
    keep = filter(i -> filter_plot_arms(iss[i], iss_to_keep), 1:length(iss));
    _plot_arms(pep, μs, Tstar, wstar, δ, iss[keep], data[keep,:], N, file, iss_to_keep, name_iss);
end

function filter_plot_arms(is, iss_to_keep)
    sr, rsp = is;
    if abbrev(sr) * "-" * abbrev(rsp) in iss_to_keep
        return true;
    else
        return false;
    end
end

function _plot_arms(pep, μs, Tstar, wstar, δ, iss, data, N, file, iss_to_keep, name_iss)
    # xlabel for identification strategies
    _xs = permutedims(vcat(collect((abbrev(sr) * "-" * abbrev(rsp) for (sr, rsp) in iss))...));
    xs = [name_algos_plots[x] for x in _xs];

    K = length(μs);

    ps = Any[];
    for i in 1:K
        # Mean empirical sampling proportions for arm i
        means = sum(getindex.(getindex.(data,2), i) ./ sum.(getindex.(data,2)), dims=2)/N;

        p = boxplot(
            xs,
            map(x -> x[2][i] / sum(x[2]), data)',  # the ' calls the adjoint
            label="",
            notch=true,
            outliers=true);

        # plot means
        plot!(p, xs, means', marker=(:star4,10,:black), label="", ylabel="w_" * string(i));

        push!(ps, p);
    end

    plot(ps..., layout=K);
    savefig(file);
end

function plot_cput(pep, μs, Tstar, wstar, δ, iss, data, N, file)
    keep = filter(i -> filter_plot_cput(iss[i]), 1:length(iss));
    _plot_cput(pep, μs, Tstar, wstar, δ, iss[keep], data[keep,:], N, file);
end

function filter_plot_cput(is)
    sr, rsp = is;
    if false
        return false;
    else
        return true;
    end
end

function _plot_cput(pep, μs, Tstar, wstar, δ, iss, data, N, file)
    # xlabel for identification strategies
    _xs = permutedims(vcat(collect((abbrev(sr) * "-" * abbrev(rsp) for (sr, rsp) in iss))...));
    xs = [name_algos_plots[x] for x in _xs];

    # Median CPU time
    medians = median(getindex.(data, 3), dims=2)/1e6;

    boxplot(
        xs,
        map(x -> x[3]/1e6, data)',  # the ' calls the adjoint
        label=:none,
        yaxis=:log,
        notch=true,
        outliers=true);

    savefig(file);
end

function plot_rand_samp(iss, data, iss_index, Nruns, file, iss_to_keep, name_iss)
    keep = filter(i -> filter_plot_rand_samp(iss[i], iss_to_keep), 1:length(iss));
    _plot_rand_samp(iss[keep], data, iss_index, Nruns, file, iss_to_keep, name_iss);
end

function filter_plot_rand_samp(is, iss_to_keep)
    sr, rsp = is;
    if abbrev(sr) * "-" * abbrev(rsp) in iss_to_keep
        return true;
    else
        return false;
    end
end

function _plot_rand_samp(iss, data, iss_index, Nruns, file, iss_to_keep, name_iss)
    med_τs = zeros(length(iss));
    is_dict = Dict();
    for (i, is) in enumerate(iss)
        r = iss_index[is];
        results = getindex.(data[r, :], 1);
        med_τs[i] = median([sum(x[2]) for x in results]);
        is_dict[abbrev(is[1]) * "-" * abbrev(is[2])] = is;
    end
    med_τ = 8 * median(med_τs);
    for (i, is_idx) in enumerate(iss_to_keep)
        is = is_dict[is_idx];
        r = iss_index[is];
        results = getindex.(data[r, :], 1);
        τs = [sum(x[2]) for x in results];
        x = name_iss[i];
        if i == 1
            boxplot(
                [x],
                [τs],
                label="",
                notch=true,
                ylim=(0, med_τ),
                #ylim=(0, 35000),
                xtickfontsize=8,
                ytickfontsize=12,
                color=dict_colors[x],
                outliers=true);
        else
            boxplot!(
                [x],
                [τs],
                label="",
                notch=true,
                ylim=(0, med_τ),
                #ylim=(0, 35000),
                xtickfontsize=8,
                ytickfontsize=12,
                color=dict_colors[x],
                outliers=true);
        end
        plot!([x], [mean(τs)], marker=(:star4,10,:black), label="");
    end

    savefig(file);
end

function plot_rand_perror(iss, data, iss_index, Nruns, file, iss_to_keep, name_iss)
    keep = filter(i -> filter_plot_rand_perror(iss[i], iss_to_keep), 1:length(iss));
    _plot_rand_perror(iss[keep], data, iss_index, Nruns, file, iss_to_keep, name_iss);
end

function filter_plot_rand_perror(is, iss_to_keep)
    if abbrev(is[1]) * "-" * abbrev(is[2]) in iss_to_keep
        return true;
    else
        return false;
    end
end

function _plot_rand_perror(iss, data, iss_index, Nruns, file, iss_to_keep, name_iss)
    # Hard-coded default parameter
    freqHist = 200;
    history = "partial";

    med_list = Float64[];
    is_dict = Dict();
    for (i, is) in enumerate(iss)
        r = iss_index[is];
        results = getindex.(data[r, :], 1);
        push!(med_list, median([sum(x[2]) for x in results]));
        is_dict[abbrev(is[1]) * "-" * abbrev(is[2])] = is;
    end
    budget = minimum(med_list);

    α = 0.05;
    k = Distributions.quantile(Distributions.Normal(), 1 - α / 2);
    start = 1;

    for (i, is_idx) in enumerate(iss_to_keep)
        is = is_dict[is_idx];
        r = iss_index[is];
        xname = name_iss[i];
        results = getindex.(data[r, :], 1);

        # Compute probability of errors: Mean and Wilson CI
        perrors = zeros(1, Int64(ceil(budget / freqHist)));
        real_Nruns = zeros(1, Int64(ceil(budget / freqHist)));
        times = [i*freqHist for i in eachindex(perrors)];
        times[end] = Int64(ceil(budget));

        for x in results
            if length(x[4]) > length(perrors)
                _recos = x[4][1:length(perrors)];
            else
                _recos = x[4];
            end
            perrors[1:length(_recos)] .+= _recos .!= 1;
            real_Nruns[1:length(_recos)] .+= 1;
        end

        p0_errors = (perrors .+ k^2) ./ (real_Nruns .+ k^2);
        perrors ./= real_Nruns;
        CIs_errors = k * sqrt.(real_Nruns .* perrors .* (1 .- perrors) .+ k^2 / 4) ./ (real_Nruns .+ k^2);

        if i == 1
            plot(times[start:end], p0_errors[start:end], ribbon=CIs_errors[start:end], label="", linecolor = invisible(), fillalpha=.1, color=dict_colors[xname]);
            plot!(times[start:end], perrors[start:end], label=xname, color=dict_colors[xname], linestyle=:auto);
        else
            plot!(times[start:end], p0_errors[start:end], ribbon=CIs_errors[start:end], label="", linecolor = invisible(), fillalpha=.1, color=dict_colors[xname]);
            plot!(times[start:end], perrors[start:end], label=xname, color=dict_colors[xname], linestyle=:auto);
        end
    end
    savefig(file);
end


function plot_rand_cput(iss, data, iss_index, Nruns, file, iss_to_keep, name_iss)
    keep = filter(i -> filter_plot_rand_cput(iss[i], iss_to_keep), 1:length(iss));
    _plot_rand_cput(iss[keep], data, iss_index, Nruns, file, iss_to_keep, name_iss);
end

function filter_plot_rand_cput(is, iss_to_keep)
    sr, rsp = is;
    if abbrev(sr) * "-" * abbrev(rsp) in iss_to_keep
        return true;
    else
        return false;
    end
end

function _plot_rand_cput(iss, data, iss_index, Nruns, file, iss_to_keep, name_iss)
    med_τs = zeros(length(iss));
    is_dict = Dict();
    for (i, is) in enumerate(iss)
        r = iss_index[is];
        results = getindex.(data[r, :], 1);
        med_τs[i] = median([sum(x[3]) for x in results]);
        is_dict[abbrev(is[1]) * "-" * abbrev(is[2])] = is;
    end
    med_τ = 8 * median(med_τs);
    for (i, is_idx) in enumerate(iss_to_keep)
        is = is_dict[is_idx];
        r = iss_index[is];
        results = getindex.(data[r, :], 1);
        τs = [sum(x[3]) for x in results];
        x = name_iss[i];
        println("Algo: ", x);
        println("Mean cput: ", mean(τs)/1e6);
        println("Std cput: ", std(τs)/1e6);
        println("----")
        if i == 1
            boxplot(
                [x],
                [τs],
                label="",
                notch=true,
                #ylim=(0, med_τ),
                #ylim=(0, 35000),
                xtickfontsize=8,
                ytickfontsize=12,
                color=dict_colors[x],
                outliers=true);
        else
            boxplot!(
                [x],
                [τs],
                label="",
                notch=true,
                #ylim=(0, med_τ),
                #ylim=(0, 35000),
                xtickfontsize=8,
                ytickfontsize=12,
                color=dict_colors[x],
                outliers=true);
        end
        plot!([x], [mean(τs)], marker=(:star4,10,:black), label="");
    end

    savefig(file);
end


function plot_instances_dists(data, iss, iss_index, file_plot_charactimes, file_plot_allocbest)
    # Get the instances for the first iss since they are all identical
    r = iss_index[iss[1]];
    instances = getindex.(data[r, :], 2);

    Tstars = getindex.(instances, 2);
    Tstar_betas = getindex.(instances, 4);

    exp_ratios_time = exp.(Tstar_betas ./ Tstars);
    histogram(exp_ratios_time, bins=:scott);
    savefig(file_plot_charactimes);

    wstars = getindex.(instances, 3);
    wstars_astar = getindex.(wstars, 1);
    histogram(wstars_astar, bins=:scott);
    savefig(file_plot_allocbest);
end
