class Yarn:
    """Represents the yarns that a yarn store sells"""
    
    def __init__(self, purchase_price: int, sell_price: int, color: str):
        self.purchase_price = purchase_price
        self.sell_price = sell_price
        self.color = color

class BankAccount:
    """Represents the bank account of this yarn store"""

    def __init__(self, balance: int):
        self.balance = balance

    def reduce_balance(self, quantity: int):
        """Reduces balance of this account if possible"""
        if quantity > self.balance:
            raise ValueError
        else:
            self.balance -= quantity
    
    def add_balance(self, quantity: int):
        """Adds to this account's balacne"""
        self.balance += quantity

    
    def get_balance(self):
        """Returns the balance of this account"""
        return self.balance

class WareHouse:
    """Represents a warehouse that stores the yarn stock of this yarn store."""
    
    def __init__(self):
        self.stock = {}

    def stock_of(self, item: Yarn):
        """Gets the stock of the yarn given"""
        if item not in self.stock:
            raise ValueError
        else:
            return self.stock[item]
        
    def add_stock(self, items: dict[Yarn, int]):
        """Adds stock to this warehouse"""
        for item, quant in items.items():
            if item in self.stock:
                self.stock[item] += quant
            else:
                self.stock[item] = quant
    
    def reduce_stock(self, items: dict[Yarn, int]):
        """Reduces the stock of this warehouse"""
        for item, quant in items.items():
            if item in self.stock and self.stock[item] >= quant:
                self.stock[item] -= quant
            else:
                raise ValueError

class Store:
    
    def __init__(self, starting_balance: int):
        self.bank = BankAccount(starting_balance)
        self.warehouse = WareHouse()

    def buy_yarn(self, order: dict[Yarn, int]):
        """Buy the quantity of yarn specified by the order"""
        self.warehouse.add_stock(order)
        self.bank.reduce_balance(self.calculate_cost(order, True))
    
    def sell_yarn(self, order: dict[Yarn, int]):
        """Sell the quantity of yarn specified by the order"""
        self.warehouse.reduce_stock(order)
        self.bank.add_balance(self.calculate_cost(order, False))

    def calculate_cost(self, order: dict[Yarn, int], is_purchase: bool):
        """Calcualtes the cost of this order, depending on if we are buying or selling yarn"""
        total = 0
        for item in order:
            if is_purchase:
                total += item.purchase_price * order[item]
            else:
                total += item.sell_price * order[item]
        return total