#!/bin/bash
# this script makes sure that all the tests in the benchmarks actually run on the ground truth
# solutions.

# -1. pushd here
pushd `dirname $0` > /dev/null

# 0. get temp dir to do shananigans in
TEMP_DIR=$(mktemp -d)
TEMP_VENV=/tmp/sanity_check_venv # so we can reuse the venv
echo "+ temp dir: $TEMP_DIR"
echo "+ temp venv: $TEMP_VENV"

# 1. initialize python venv, if it doesn't exist
if [ ! -d "$TEMP_VENV" ]; then
  echo "+ initializing python venv"
  python3 -m venv $TEMP_VENV/venv
  source $TEMP_VENV/venv/bin/activate

  # 2. install requirements
  echo "+ installing requirements from requirements.txt"
  pip install -r requirements.txt
else
  source $TEMP_VENV/venv/bin/activate
fi

# run one specific benchmark, if specified
SPECIFIC_BENCHMARK=${1:-none}

# 3. run sanity check on each benchmark
for benchmark in ./*; do
    # if SPECIFIC_BENCHMARK is not none, then only run that benchmark;
    # continue here if not matching (using * to match all surrounding text)
    if [ "$SPECIFIC_BENCHMARK" != "none" ] && [[ "$benchmark" != *"$SPECIFIC_BENCHMARK"* ]]; then
      continue
    fi

    # if it's a directory and has "_" in the name, then it's a benchmark;
    if [ -d "$benchmark" ] && [[ "$benchmark" == *_* ]]; then
      echo "+ running sanity check on $benchmark"
      # cat after.py with tests.py into a temp file
      rm -f $TEMP_DIR/temp.py
      cp $benchmark/after.py $TEMP_DIR/temp.py
      printf "\n\n\n" >> $TEMP_DIR/temp.py
      cat $benchmark/tests.py >> $TEMP_DIR/temp.py
      # run the temp file
      timeout 15 python $TEMP_DIR/temp.py
      RESULT=$?
      # if the result is not 0, then the sanity check failed
      if [ $RESULT -ne 0 ]; then
        echo "+ sanity check failed on $benchmark - tests did not pass on ground truth solution"
        if [ $RESULT -eq 124 ]; then
          echo "+ reason of fail: timeout"
        fi
      fi

      # now, we need to make sure that the tests fail on the before.py
      cp $benchmark/before.py $TEMP_DIR/temp.py
      printf "\n\n\n" >> $TEMP_DIR/temp.py
      cat $benchmark/tests.py >> $TEMP_DIR/temp.py
      # run the temp file with a timeout of 15 seconds, because some of the tests create infinite loops
      timeout 15 python $TEMP_DIR/temp.py 2> /dev/null
      RESULT=$?
      # if the result is not 0, then the sanity check failed
      if [ $RESULT -eq 0 ]; then
        echo "+ sanity check failed on $benchmark - tests passed on the before solution"
      fi

      # check that coverage is 100%
      cp $benchmark/after.py $TEMP_DIR/temp.py
      printf "\n\n\n" >> $TEMP_DIR/temp.py
      cat $benchmark/tests.py >> $TEMP_DIR/temp.py
      coverage run --rcfile=./.coveragerc --data-file $TEMP_DIR/temp_coverage.data $TEMP_DIR/temp.py
      coverage report --fail-under=100 -m --data-file $TEMP_DIR/temp_coverage.data 1> /tmp/coverage_report.txt
      RESULT=$?
      if [ $RESULT -ne 0 ]; then
        echo "+ sanity check failed on $benchmark - coverage is not 100%"
        echo "+ coverage report:"
        cat /tmp/coverage_report.txt
      fi
    fi
done

# remove temp dir
rm -rf $TEMP_DIR
popd > /dev/null
