function [div,w_star] = one_side_max_sliced_bures_path(A,B,verbose)
% Calculates the one sided max sliced Bures for linear kernel 
% computes w^* = argmin_{w: ||w||<=1}  g(w)-h(w) 
%      and div = h(w^*) - g(w^*) = max_{w: ||w||<=1} h(w)-g(w), where
%         g^2(w) = w'*A*w
%         h^2(w) = w'*B*w
%      Using the gradient for g(w^*)>0
%      there exists a gamma \in (0,1]
%       such that  w^* = argmax_{w: ||w||<=1}  gamma h^2(w) - g^2(w)
%      If g(w^*) = 0, then w^* = argmax_{w: ||w||<=1}  w'*b*w

if nargin<3
    verbose =0;
end

g = @(w) sqrt(max(0,w'*B*w));
h = @(w) sqrt(max(0,w'*A*w));
% Maximize
obj = @(w) (h(w) - g(w))/sqrt(w'*w);
% or equivalently -\min_w { -obj(w) = (g(w) - h(w))/\sqrt(w'w)}

% solution for g(w)>0
obj_gamma = @(gamma) -eigv_eval(gamma*A-B,obj);
[gamma_star,f_gamma_star] = fminbnd(obj_gamma, 0,1);

obj_not_null = -f_gamma_star;
% solution for g(w)=0 => w in null(B)
V = null(B);
if size(V,2)>0
    [beta,~]=eigs(V'*A*V, 1,'largestreal');
    w = V*beta;
    w = w/sqrt(w'*w);
    obj_null = obj(w);
else
    obj_null = -inf;
end

if obj_null > obj_not_null
    div = obj_null;
    gamma_star = inf;
    w_star = w;
else
    div = obj_not_null;
    [w_star,~]=eigs(gamma_star*A-B,1,'largestreal');
end

if verbose
    fprintf('Obj= %.7f @ %.7f \n\n', div,gamma_star)
end
end

function val = eigv_eval(A,obj)
[u,~]=eigs(A, 1,'largestreal');
val = obj(u);
end
