import os
import random
from re import A
from typing import Iterable, List, Optional, Union
from functools import lru_cache

from pymoo.util.ref_dirs import get_reference_directions
import matplotlib.pyplot as plt
import numpy as np
import seaborn as sns
import torch as th
import torch.nn.functional as F
from torch import nn


@th.no_grad()
def layer_init(layer, method="orthogonal", weight_gain: float = 1, bias_const: float = 0) -> None:
    if isinstance(layer, (nn.Linear, nn.Conv2d)):
        if method == "xavier":
            th.nn.init.xavier_uniform_(layer.weight, gain=weight_gain)
        elif method == "orthogonal":
            th.nn.init.orthogonal_(layer.weight, gain=weight_gain)
        th.nn.init.constant_(layer.bias, bias_const)


@th.no_grad()
def polyak_update(
    params: Iterable[th.nn.Parameter],
    target_params: Iterable[th.nn.Parameter],
    tau: float,
) -> None:
    for param, target_param in zip(params, target_params):
        if tau == 1:
            target_param.data.copy_(param.data)
        else:
            target_param.data.mul_(1.0 - tau)
            th.add(target_param.data, param.data, alpha=tau, out=target_param.data)


def disable_gradients(params: Iterable[th.nn.Parameter]):
    for param in params:
        param.requires_grad = False


def huber(x, min_priority=0.01):
    return th.where(x < min_priority, 0.5 * x.pow(2), min_priority * x).mean()


def quantile_huber_loss(
    current_quantiles: th.Tensor,
    target_quantiles: th.Tensor,
    cum_prob: Optional[th.Tensor] = None,
    sum_over_quantiles: bool = True,
) -> th.Tensor:
    """
    The quantile-regression loss, as described in the QR-DQN and TQC papers.
    Taken from https://github.com/Stable-Baselines-Team/stable-baselines3-contrib/blob/master/sb3_contrib/common/utils.py
    :param current_quantiles: current estimate of quantiles, must be either
        (batch_size, n_quantiles) or (batch_size, n_critics, n_quantiles)
    :param target_quantiles: target of quantiles, must be either (batch_size, n_target_quantiles),
        (batch_size, 1, n_target_quantiles), or (batch_size, n_critics, n_target_quantiles)
    :param cum_prob: cumulative probabilities to calculate quantiles (also called midpoints in QR-DQN paper),
        must be either (batch_size, n_quantiles), (batch_size, 1, n_quantiles), or (batch_size, n_critics, n_quantiles).
        (if None, calculating unit quantiles)
    :param sum_over_quantiles: if summing over the quantile dimension or not
    :return: the loss
    """
    if current_quantiles.ndim != target_quantiles.ndim:
        raise ValueError(
            f"Error: The dimension of curremt_quantile ({current_quantiles.ndim}) needs to match "
            f"the dimension of target_quantiles ({target_quantiles.ndim})."
        )
    if current_quantiles.shape[0] != target_quantiles.shape[0]:
        raise ValueError(
            f"Error: The batch size of curremt_quantile ({current_quantiles.shape[0]}) needs to match "
            f"the batch size of target_quantiles ({target_quantiles.shape[0]})."
        )
    if current_quantiles.ndim not in (2, 3):
        raise ValueError(f"Error: The dimension of current_quantiles ({current_quantiles.ndim}) needs to be either 2 or 3.")

    if cum_prob is None:
        n_quantiles = current_quantiles.shape[-1]
        # Cumulative probabilities to calculate quantiles.
        cum_prob = (th.arange(n_quantiles, device=current_quantiles.device, dtype=th.float) + 0.5) / n_quantiles
        if current_quantiles.ndim == 2:
            # For QR-DQN, current_quantiles have a shape (batch_size, n_quantiles), and make cum_prob
            # broadcastable to (batch_size, n_quantiles, n_target_quantiles)
            cum_prob = cum_prob.view(1, -1, 1)
        elif current_quantiles.ndim == 3:
            # For TQC, current_quantiles have a shape (batch_size, n_critics, n_quantiles), and make cum_prob
            # broadcastable to (batch_size, n_critics, n_quantiles, n_target_quantiles)
            cum_prob = cum_prob.view(1, 1, -1, 1)

    # QR-DQN
    # target_quantiles: (batch_size, n_target_quantiles) -> (batch_size, 1, n_target_quantiles)
    # current_quantiles: (batch_size, n_quantiles) -> (batch_size, n_quantiles, 1)
    # pairwise_delta: (batch_size, n_target_quantiles, n_quantiles)
    # TQC
    # target_quantiles: (batch_size, 1, n_target_quantiles) -> (batch_size, 1, 1, n_target_quantiles)
    # current_quantiles: (batch_size, n_critics, n_quantiles) -> (batch_size, n_critics, n_quantiles, 1)
    # pairwise_delta: (batch_size, n_critics, n_quantiles, n_target_quantiles)
    # Note: in both cases, the loss has the same shape as pairwise_delta
    pairwise_delta = target_quantiles.unsqueeze(-2) - current_quantiles.unsqueeze(-1)
    abs_pairwise_delta = th.abs(pairwise_delta)
    huber_loss = th.where(abs_pairwise_delta > 1, abs_pairwise_delta - 0.5, pairwise_delta**2 * 0.5)
    loss = th.abs(cum_prob - (pairwise_delta.detach() < 0).float()) * huber_loss
    if sum_over_quantiles:
        loss = loss.sum(dim=-2).mean()
    else:
        loss = loss.mean()
    return loss


def get_grad_norm(params: Iterable[th.nn.Parameter]) -> th.Tensor:
    """This is how the grad norm is computed inside torch.nn.clip_grad_norm_()"""
    parameters = [p for p in params if p.grad is not None]
    if len(parameters) == 0:
        return th.tensor(0.)
    device = parameters[0].grad.device
    total_norm = th.norm(th.stack([th.norm(p.grad.detach(), 2.0).to(device) for p in parameters]), 2.0)
    return total_norm


def avgL1Norm(x, eps=1e-8):
	return x/x.abs().mean(-1, keepdim=True).clamp(min=eps)


def extrema_weights(dim) -> List[np.ndarray]:
    extrema_weights = []
    for i in range(dim):
        w = np.zeros(dim, dtype=np.float32)
        w[i] = 1.0
        extrema_weights.append(w)
    return extrema_weights


def unique_tol(a: List[np.ndarray], tol=1e-4) -> List[np.ndarray]:
    """ Returns unique elements of a list of arrays, with a tolerance."""
    if len(a) == 0:
        return a
    delete = np.array([False] * len(a))
    a = np.array(a)
    for i in range(len(a)):
        if delete[i]:
            continue
        for j in range(i + 1, len(a)):
            if np.allclose(a[i], a[j], tol):
                delete[j] = True
    return list(a[~delete])


def remove_array_from_list(arr_list, target_array):
    # Loop over each item in the list and check for array equality
    for i, arr in enumerate(arr_list):
        if np.array_equal(arr, target_array):
            del arr_list[i]
            return arr_list  # Return the modified list after removal
    return arr_list  # Return the list as-is if the array is not found


def unique_tol_inds(a: List[np.ndarray], tol=1e-4) -> List[np.ndarray]:
    """ Returns indices unique elements of a list of arrays, with a tolerance."""
    if len(a) == 0:
        return a
    delete = np.array([False] * len(a))
    a = np.array(a)
    for i in range(len(a)):
        if delete[i]:
            continue
        for j in range(i + 1, len(a)):
            if np.allclose(a[i], a[j], tol):
                delete[j] = True
    return (~delete).nonzero()


def filter_from_list(a: List[np.ndarray], b = List[np.ndarray], tol=1e-4) -> List[np.ndarray]:
    """ Returns elements of a that are not in b."""
    if len(a) == 0:
        return a
    delete = np.array([False] * len(a))
    a = np.array(a)
    b = np.array(b)
    for i in range(len(a)):
        for j in range(len(b)):
            if np.allclose(a[i], b[j], tol):
                delete[i] = True
    return list(a[~delete])


def generate_weights(count=1, n=3, m=1):
    """Source: https://github.com/axelabels/DynMORL/blob/db15c29bc2cf149c9bda6b8890fee05b1ac1e19e/utils.py#L281"""
    all_weights = []

    target = np.random.dirichlet(np.ones(n), 1)[0]
    prev_t = target
    for _ in range(count // m):
        target = np.random.dirichlet(np.ones(n), 1)[0]
        if m == 1:
            all_weights.append(target)
        else:
            for i in range(m):
                i_w = target * (i + 1) / float(m) + prev_t * (m - i - 1) / float(m)
                all_weights.append(i_w)
        prev_t = target + 0.0

    return all_weights


def random_weights(dim, seed=None, n=1):
    """Generate random normalized weights from a Dirichlet distribution alpha=1
    Args:
        dim: size of the weight vector
    """
    if seed is not None:
        rng = np.random.default_rng(seed)
    else:
        rng = np.random
    weights = []
    for _ in range(n):
        w = rng.dirichlet(np.ones(dim))
        weights.append(w)
    if n == 1:
        return weights[0]
    return weights

@lru_cache
def incremental_weights(dim: int, n_partitions: int = 3) -> List[np.ndarray]:
    # Incremental lattice design of weight vector set: https://dl.acm.org/doi/10.1145/3377929.3398082 and https://pymoo.org/misc/reference_directions.html
    return list(get_reference_directions("incremental", dim, n_partitions=n_partitions))

@lru_cache
def equally_spaced_weights(dim: int, num_weights: int, seed: int = 42) -> List[np.ndarray]:
    if dim == 4 and num_weights == 64:
        return list(np.array([[0.        , 0.        , 0.        , 1.        ],
                            [0.        , 0.        , 0.1825182 , 0.8174818 ],
                            [0.        , 0.        , 0.42941872, 0.57058128],
                            [0.        , 0.        , 0.63166691, 0.36833309],
                            [0.        , 0.        , 0.80891937, 0.19108063],
                            [0.        , 0.        , 1.        , 0.        ],
                            [0.        , 0.18782132, 0.        , 0.81217868],
                            [0.        , 0.19233038, 0.17102803, 0.63664159],
                            [0.        , 0.19314541, 0.34403213, 0.46282246],
                            [0.        , 0.19563706, 0.80436294, 0.        ],
                            [0.        , 0.19730236, 0.62828414, 0.1744135 ],
                            [0.        , 0.30820203, 0.42240375, 0.26939423],
                            [0.        , 0.37564887, 0.18888592, 0.43546521],
                            [0.        , 0.3908159 , 0.        , 0.6091841 ],
                            [0.        , 0.4086043 , 0.5913957 , 0.        ],
                            [0.        , 0.48591584, 0.33171283, 0.18237134],
                            [0.        , 0.59801417, 0.        , 0.40198583],
                            [0.        , 0.6203081 , 0.3796919 , 0.        ],
                            [0.        , 0.64205541, 0.16993197, 0.18801262],
                            [0.        , 0.79924348, 0.        , 0.20075652],
                            [0.        , 0.80674339, 0.19325661, 0.        ],
                            [0.        , 1.        , 0.        , 0.        ],
                            [0.10054392, 0.11271933, 0.4806242 , 0.30611255],
                            [0.12125426, 0.47232476, 0.12254403, 0.28387695],
                            [0.13166065, 0.        , 0.23723336, 0.63110598],
                            [0.15714423, 0.27629407, 0.46084116, 0.10572053],
                            [0.17747903, 0.19887567, 0.18385637, 0.43978893],
                            [0.17780999, 0.45161026, 0.37057975, 0.        ],
                            [0.18123668, 0.18050442, 0.63825889, 0.        ],
                            [0.18229082, 0.18601377, 0.        , 0.63169541],
                            [0.18312735, 0.81687265, 0.        , 0.        ],
                            [0.18701696, 0.61955729, 0.19342575, 0.        ],
                            [0.18762836, 0.        , 0.62775828, 0.18461336],
                            [0.19150382, 0.62845637, 0.        , 0.18003981],
                            [0.19676478, 0.        , 0.        , 0.80323522],
                            [0.19958147, 0.366198  , 0.        , 0.43422053],
                            [0.20288033, 0.        , 0.79711967, 0.        ],
                            [0.21668429, 0.        , 0.36955509, 0.41376063],
                            [0.2444585 , 0.37235089, 0.22382167, 0.15936893],
                            [0.25220487, 0.17690948, 0.32525312, 0.24563253],
                            [0.31742306, 0.        , 0.17821411, 0.50436282],
                            [0.339946  , 0.43649252, 0.        , 0.22356147],
                            [0.35029777, 0.17742291, 0.47227932, 0.        ],
                            [0.35596691, 0.64403309, 0.        , 0.        ],
                            [0.35684806, 0.17888956, 0.        , 0.46426238],
                            [0.3605863 , 0.        , 0.44484434, 0.19456935],
                            [0.37198294, 0.20668976, 0.13109536, 0.29023194],
                            [0.38159102, 0.        , 0.        , 0.61840898],
                            [0.3901447 , 0.47102974, 0.13882555, 0.        ],
                            [0.39150593, 0.        , 0.60849407, 0.        ],
                            [0.39992945, 0.30503155, 0.29503901, 0.        ],
                            [0.43548953, 0.        , 0.24314381, 0.32136666],
                            [0.4672209 , 0.12111282, 0.291216  , 0.12045029],
                            [0.48389694, 0.29943011, 0.08904761, 0.12762534],
                            [0.57667358, 0.        , 0.        , 0.42332642],
                            [0.60326838, 0.15819132, 0.        , 0.2385403 ],
                            [0.61095541, 0.        , 0.38904459, 0.        ],
                            [0.61414477, 0.38585523, 0.        , 0.        ],
                            [0.62716667, 0.        , 0.18426891, 0.18856442],
                            [0.63002336, 0.18680604, 0.1831706 , 0.        ],
                            [0.79486646, 0.        , 0.        , 0.20513354],
                            [0.79690639, 0.        , 0.20309361, 0.        ],
                            [0.8086597 , 0.1913403 , 0.        , 0.        ],
                            [1.        , 0.        , 0.        , 0.        ]]))

    elif dim == 8 and num_weights == 64:
        return list(np.array([[0.        , 0.        , 0.        , 0.        , 0.        ,
                            0.        , 0.        , 1.        ],
                        [0.        , 0.        , 0.        , 0.        , 0.        ,
                            0.        , 0.34730282, 0.65269718],
                        [0.        , 0.        , 0.        , 0.        , 0.        ,
                            0.        , 1.        , 0.        ],
                        [0.        , 0.        , 0.        , 0.        , 0.        ,
                            0.33227373, 0.66772627, 0.        ],
                        [0.        , 0.        , 0.        , 0.        , 0.        ,
                            0.36177213, 0.        , 0.63822787],
                        [0.        , 0.        , 0.        , 0.        , 0.        ,
                            0.65537871, 0.34462129, 0.        ],
                        [0.        , 0.        , 0.        , 0.        , 0.        ,
                            1.        , 0.        , 0.        ],
                        [0.        , 0.        , 0.        , 0.        , 0.24445771,
                            0.5307161 , 0.        , 0.2248262 ],
                        [0.        , 0.        , 0.        , 0.        , 0.35956755,
                            0.        , 0.64043245, 0.        ],
                        [0.        , 0.        , 0.        , 0.        , 0.61570435,
                            0.        , 0.        , 0.38429565],
                        [0.        , 0.        , 0.        , 0.        , 0.63265454,
                            0.36734546, 0.        , 0.        ],
                        [0.        , 0.        , 0.        , 0.        , 1.        ,
                            0.        , 0.        , 0.        ],
                        [0.        , 0.        , 0.        , 0.30591955, 0.        ,
                            0.        , 0.36207911, 0.33200134],
                        [0.        , 0.        , 0.        , 0.32493148, 0.        ,
                            0.34326211, 0.33180641, 0.        ],
                        [0.        , 0.        , 0.        , 0.33620309, 0.        ,
                            0.        , 0.        , 0.66379691],
                        [0.        , 0.        , 0.        , 0.34507374, 0.32676639,
                            0.        , 0.        , 0.32815988],
                        [0.        , 0.        , 0.        , 0.36228357, 0.63771643,
                            0.        , 0.        , 0.        ],
                        [0.        , 0.        , 0.        , 0.61802238, 0.        ,
                            0.        , 0.38197762, 0.        ],
                        [0.        , 0.        , 0.        , 0.63883001, 0.        ,
                            0.36116999, 0.        , 0.        ],
                        [0.        , 0.        , 0.        , 0.65975088, 0.        ,
                            0.        , 0.        , 0.34024912],
                        [0.        , 0.        , 0.        , 1.        , 0.        ,
                            0.        , 0.        , 0.        ],
                        [0.        , 0.        , 0.32266868, 0.        , 0.31136291,
                            0.        , 0.        , 0.36596841],
                        [0.        , 0.        , 0.3281344 , 0.        , 0.        ,
                            0.33890445, 0.        , 0.33296114],
                        [0.        , 0.        , 0.33039151, 0.32845412, 0.        ,
                            0.        , 0.        , 0.34115438],
                        [0.        , 0.        , 0.33854192, 0.        , 0.        ,
                            0.66145808, 0.        , 0.        ],
                        [0.        , 0.        , 0.36761406, 0.        , 0.63238594,
                            0.        , 0.        , 0.        ],
                        [0.        , 0.        , 0.3772902 , 0.        , 0.        ,
                            0.        , 0.6227098 , 0.        ],
                        [0.        , 0.        , 0.52202962, 0.        , 0.24413818,
                            0.        , 0.2338322 , 0.        ],
                        [0.        , 0.        , 0.63009429, 0.36990571, 0.        ,
                            0.        , 0.        , 0.        ],
                        [0.        , 0.        , 0.6538837 , 0.        , 0.        ,
                            0.        , 0.        , 0.3461163 ],
                        [0.        , 0.        , 0.65769286, 0.        , 0.        ,
                            0.34230714, 0.        , 0.        ],
                        [0.        , 0.        , 1.        , 0.        , 0.        ,
                            0.        , 0.        , 0.        ],
                        [0.        , 0.12508197, 0.        , 0.29851692, 0.29145544,
                            0.28494567, 0.        , 0.        ],
                        [0.        , 0.25669118, 0.        , 0.2171804 , 0.        ,
                            0.        , 0.52612842, 0.        ],
                        [0.        , 0.27797623, 0.        , 0.        , 0.1822315 ,
                            0.25847933, 0.28131294, 0.        ],
                        [0.        , 0.28279819, 0.28267197, 0.27728329, 0.        ,
                            0.        , 0.15724654, 0.        ],
                        [0.        , 0.33099358, 0.66900642, 0.        , 0.        ,
                            0.        , 0.        , 0.        ],
                        [0.        , 0.3366947 , 0.        , 0.        , 0.6633053 ,
                            0.        , 0.        , 0.        ],
                        [0.        , 0.34735974, 0.33692229, 0.        , 0.        ,
                            0.31571797, 0.        , 0.        ],
                        [0.        , 0.3632401 , 0.        , 0.        , 0.        ,
                            0.        , 0.        , 0.6367599 ],
                        [0.        , 0.36464433, 0.        , 0.63535567, 0.        ,
                            0.        , 0.        , 0.        ],
                        [0.        , 0.37489845, 0.        , 0.        , 0.        ,
                            0.62510155, 0.        , 0.        ],
                        [0.        , 0.45825872, 0.        , 0.26813056, 0.        ,
                            0.        , 0.        , 0.27361073],
                        [0.        , 0.62903271, 0.        , 0.        , 0.        ,
                            0.        , 0.37096729, 0.        ],
                        [0.        , 0.66003808, 0.        , 0.        , 0.33996192,
                            0.        , 0.        , 0.        ],
                        [0.        , 1.        , 0.        , 0.        , 0.        ,
                            0.        , 0.        , 0.        ],
                        [0.20521611, 0.        , 0.        , 0.        , 0.53827931,
                            0.        , 0.25650459, 0.        ],
                        [0.21810273, 0.        , 0.23562026, 0.54627701, 0.        ,
                            0.        , 0.        , 0.        ],
                        [0.2186478 , 0.56384165, 0.        , 0.        , 0.        ,
                            0.21751055, 0.        , 0.        ],
                        [0.23195387, 0.        , 0.        , 0.23653814, 0.        ,
                            0.26672378, 0.        , 0.26478421],
                        [0.24486163, 0.26494542, 0.21286202, 0.        , 0.27733093,
                            0.        , 0.        , 0.        ],
                        [0.24609642, 0.24460528, 0.        , 0.        , 0.        ,
                            0.        , 0.22821937, 0.28107893],
                        [0.25606698, 0.        , 0.25766333, 0.        , 0.        ,
                            0.23862229, 0.2476474 , 0.        ],
                        [0.28297601, 0.        , 0.        , 0.28369691, 0.17991601,
                            0.        , 0.25341108, 0.        ],
                        [0.34015212, 0.        , 0.        , 0.        , 0.        ,
                            0.65984788, 0.        , 0.        ],
                        [0.37470084, 0.        , 0.        , 0.        , 0.        ,
                            0.        , 0.        , 0.62529916],
                        [0.38770825, 0.        , 0.61229175, 0.        , 0.        ,
                            0.        , 0.        , 0.        ],
                        [0.40779241, 0.        , 0.        , 0.        , 0.        ,
                            0.        , 0.59220759, 0.        ],
                        [0.53992671, 0.        , 0.22541048, 0.        , 0.        ,
                            0.        , 0.        , 0.23466281],
                        [0.6066836 , 0.        , 0.        , 0.        , 0.3933164 ,
                            0.        , 0.        , 0.        ],
                        [0.61548895, 0.38451105, 0.        , 0.        , 0.        ,
                            0.        , 0.        , 0.        ],
                        [0.6192454 , 0.        , 0.        , 0.3807546 , 0.        ,
                            0.        , 0.        , 0.        ],
                        [0.66308485, 0.        , 0.        , 0.        , 0.        ,
                            0.33691515, 0.        , 0.        ],
                        [1.        , 0.        , 0.        , 0.        , 0.        ,
                            0.        , 0.        , 0.        ]]))

    elif dim == 3 and num_weights == 32:
        return list(np.array([[0.        , 0.        , 1.        ],
                            [0.        , 0.16833405, 0.83166595],
                            [0.        , 0.33482796, 0.66517204],
                            [0.        , 0.50423339, 0.49576661],
                            [0.        , 0.66417035, 0.33582965],
                            [0.        , 0.83344246, 0.16655754],
                            [0.        , 1.        , 0.        ],
                            [0.14839575, 0.70240172, 0.14920253],
                            [0.14914059, 0.55641195, 0.29444746],
                            [0.15249299, 0.41676095, 0.43074606],
                            [0.15429261, 0.28013565, 0.56557175],
                            [0.15512314, 0.14147763, 0.70339924],
                            [0.16205058, 0.        , 0.83794942],
                            [0.16636617, 0.83363383, 0.        ],
                            [0.28995868, 0.55987025, 0.15017107],
                            [0.29772425, 0.41436625, 0.2879095 ],
                            [0.30145955, 0.28234616, 0.41619429],
                            [0.30542977, 0.1417119 , 0.55285833],
                            [0.33089884, 0.        , 0.66910116],
                            [0.34293709, 0.65706291, 0.        ],
                            [0.42739503, 0.41769437, 0.1549106 ],
                            [0.43886948, 0.15669865, 0.40443186],
                            [0.49494756, 0.        , 0.50505244],
                            [0.49699492, 0.50300508, 0.        ],
                            [0.55714767, 0.29077287, 0.15207946],
                            [0.56829115, 0.15299163, 0.27871722],
                            [0.6646443 , 0.        , 0.3353557 ],
                            [0.66654002, 0.33345998, 0.        ],
                            [0.70664144, 0.15188099, 0.14147757],
                            [0.83124437, 0.        , 0.16875563],
                            [0.83515074, 0.16484926, 0.        ],
                            [1.        , 0.        , 0.        ]]))
    elif dim == 4 and num_weights == 32:
        return list(np.array([[0.        , 0.        , 0.        , 1.        ],
                            [0.        , 0.        , 0.26329573, 0.73670427],
                            [0.        , 0.        , 0.50650682, 0.49349318],
                            [0.        , 0.        , 0.74850828, 0.25149172],
                            [0.        , 0.        , 1.        , 0.        ],
                            [0.        , 0.25043115, 0.        , 0.74956885],
                            [0.        , 0.25215606, 0.74784394, 0.        ],
                            [0.        , 0.26215278, 0.2490784 , 0.48876882],
                            [0.        , 0.27794804, 0.47414135, 0.24791061],
                            [0.        , 0.49603471, 0.        , 0.50396529],
                            [0.        , 0.51530232, 0.48469768, 0.        ],
                            [0.        , 0.73885817, 0.        , 0.26114183],
                            [0.        , 0.75027404, 0.24972596, 0.        ],
                            [0.        , 1.        , 0.        , 0.        ],
                            [0.17782928, 0.45601854, 0.17931625, 0.18683593],
                            [0.24258098, 0.2705678 , 0.48685122, 0.        ],
                            [0.24948638, 0.        , 0.4962022 , 0.25431143],
                            [0.25042635, 0.26815303, 0.        , 0.48142062],
                            [0.25238529, 0.        , 0.25254926, 0.49506545],
                            [0.25329737, 0.74670263, 0.        , 0.        ],
                            [0.25997849, 0.        , 0.        , 0.74002151],
                            [0.26354374, 0.        , 0.73645626, 0.        ],
                            [0.46936843, 0.28046375, 0.25016783, 0.        ],
                            [0.47886557, 0.27229334, 0.        , 0.24884109],
                            [0.48542195, 0.51457805, 0.        , 0.        ],
                            [0.49450989, 0.        , 0.25291047, 0.25257964],
                            [0.50040314, 0.        , 0.        , 0.49959686],
                            [0.50331124, 0.        , 0.49668876, 0.        ],
                            [0.73446916, 0.26553084, 0.        , 0.        ],
                            [0.74161543, 0.        , 0.        , 0.25838457],
                            [0.74232488, 0.        , 0.25767512, 0.        ],
                            [1.        , 0.        , 0.        , 0.        ]]))
    elif dim == 2 and num_weights == 100:
        return list(np.array([[0.        , 1.        ],
                            [0.00989782, 0.99010218],
                            [0.01993052, 0.98006948],
                            [0.0300275 , 0.9699725 ],
                            [0.03995353, 0.96004647],
                            [0.05011469, 0.94988531],
                            [0.06011285, 0.93988715],
                            [0.07004192, 0.92995808],
                            [0.08018663, 0.91981337],
                            [0.09010646, 0.90989354],
                            [0.10024172, 0.89975828],
                            [0.11024928, 0.88975072],
                            [0.12022687, 0.87977313],
                            [0.13033179, 0.86966821],
                            [0.14033374, 0.85966626],
                            [0.15033916, 0.84966084],
                            [0.16045562, 0.83954438],
                            [0.17040724, 0.82959276],
                            [0.18052552, 0.81947448],
                            [0.19048563, 0.80951437],
                            [0.20061826, 0.79938174],
                            [0.21058282, 0.78941718],
                            [0.22071204, 0.77928796],
                            [0.23067493, 0.76932507],
                            [0.24080451, 0.75919549],
                            [0.25078636, 0.74921364],
                            [0.26092089, 0.73907911],
                            [0.27088089, 0.72911911],
                            [0.28103846, 0.71896154],
                            [0.29100805, 0.70899195],
                            [0.30116884, 0.69883116],
                            [0.31118544, 0.68881456],
                            [0.32119786, 0.67880214],
                            [0.33134588, 0.66865412],
                            [0.34134114, 0.65865886],
                            [0.35149886, 0.64850114],
                            [0.36151955, 0.63848045],
                            [0.37155448, 0.62844552],
                            [0.38171579, 0.61828421],
                            [0.39170356, 0.60829644],
                            [0.40186367, 0.59813633],
                            [0.41194626, 0.58805374],
                            [0.42197219, 0.57802781],
                            [0.43204495, 0.56795505],
                            [0.44220168, 0.55779832],
                            [0.45222491, 0.54777509],
                            [0.4623879 , 0.5376121 ],
                            [0.47240432, 0.52759568],
                            [0.48256657, 0.51743343],
                            [0.492647  , 0.507353  ],
                            [0.50271275, 0.49728725],
                            [0.5128868 , 0.4871132 ],
                            [0.52292235, 0.47707765],
                            [0.53310901, 0.46689099],
                            [0.5431523 , 0.4568477 ],
                            [0.55331945, 0.44668055],
                            [0.56340749, 0.43659251],
                            [0.5735081 , 0.4264919 ],
                            [0.58366832, 0.41633168],
                            [0.59375722, 0.40624278],
                            [0.60385202, 0.39614798],
                            [0.61405645, 0.38594355],
                            [0.6240944 , 0.3759056 ],
                            [0.63432711, 0.36567289],
                            [0.64436995, 0.35563005],
                            [0.65456693, 0.34543307],
                            [0.66466478, 0.33533522],
                            [0.67480455, 0.32519545],
                            [0.68495378, 0.31504622],
                            [0.69512206, 0.30487794],
                            [0.70522724, 0.29477276],
                            [0.71543895, 0.28456105],
                            [0.72549551, 0.27450449],
                            [0.73576036, 0.26423964],
                            [0.74579   , 0.25421   ],
                            [0.75609728, 0.24390272],
                            [0.76613531, 0.23386469],
                            [0.77638384, 0.22361616],
                            [0.78648495, 0.21351505],
                            [0.79667611, 0.20332389],
                            [0.8068197 , 0.1931803 ],
                            [0.81699905, 0.18300095],
                            [0.82715268, 0.17284732],
                            [0.83732119, 0.16267881],
                            [0.84748478, 0.15251522],
                            [0.85763991, 0.14236009],
                            [0.8678227 , 0.1321773 ],
                            [0.87800645, 0.12199355],
                            [0.88812614, 0.11187386],
                            [0.89837508, 0.10162492],
                            [0.90845873, 0.09154127],
                            [0.91874825, 0.08125175],
                            [0.92881356, 0.07118644],
                            [0.9390955 , 0.0609045 ],
                            [0.94923429, 0.05076571],
                            [0.95934918, 0.04065082],
                            [0.96961006, 0.03038994],
                            [0.97970368, 0.02029632],
                            [0.98997425, 0.01002575],
                            [1.        , 0.        ]]))
    elif dim == 3 and num_weights == 100:
        return np.array([[0.        , 0.        , 1.        ],
                        [0.        , 0.07854098, 0.92145902],
                        [0.        , 0.15729154, 0.84270846],
                        [0.        , 0.23298768, 0.76701232],
                        [0.        , 0.30506327, 0.69493673],
                        [0.        , 0.37817914, 0.62182086],
                        [0.        , 0.45332046, 0.54667954],
                        [0.        , 0.52860599, 0.47139401],
                        [0.        , 0.60405589, 0.39594411],
                        [0.        , 0.68127395, 0.31872605],
                        [0.        , 0.75854087, 0.24145913],
                        [0.        , 0.83787577, 0.16212423],
                        [0.        , 0.91817169, 0.08182831],
                        [0.        , 1.        , 0.        ],
                        [0.07764226, 0.60747473, 0.31488301],
                        [0.0779515 , 0.68547196, 0.23657655],
                        [0.07822555, 0.76369412, 0.15808034],
                        [0.07844974, 0.5291152 , 0.39243506],
                        [0.07941488, 0.841831  , 0.07875413],
                        [0.08002235, 0.45144307, 0.46853458],
                        [0.081342  , 0.07747892, 0.84117908],
                        [0.08160973, 0.91839027, 0.        ],
                        [0.08318566, 0.15213169, 0.76468265],
                        [0.0833961 , 0.37442301, 0.54218089],
                        [0.08469622, 0.        , 0.91530378],
                        [0.08516568, 0.22603991, 0.68879441],
                        [0.08704019, 0.29898015, 0.61397967],
                        [0.15512119, 0.53209824, 0.31278057],
                        [0.15532345, 0.60988903, 0.23478752],
                        [0.15637759, 0.6870804 , 0.15654201],
                        [0.15724737, 0.45288932, 0.38986331],
                        [0.1582605 , 0.76346765, 0.07827185],
                        [0.16299592, 0.83700408, 0.        ],
                        [0.16337944, 0.37324141, 0.46337915],
                        [0.16374302, 0.        , 0.83625698],
                        [0.16589282, 0.07442741, 0.75967977],
                        [0.16779957, 0.14921166, 0.68298877],
                        [0.17260152, 0.22119685, 0.60620163],
                        [0.17291603, 0.29549224, 0.53159173],
                        [0.23265813, 0.4562365 , 0.31110537],
                        [0.23273531, 0.53367383, 0.23359086],
                        [0.23415641, 0.60994357, 0.15590002],
                        [0.23657114, 0.68532602, 0.07810284],
                        [0.23822677, 0.3759903 , 0.38578293],
                        [0.24189489, 0.75810511, 0.        ],
                        [0.24897491, 0.        , 0.75102509],
                        [0.24964473, 0.07568577, 0.6746695 ],
                        [0.25742166, 0.22129048, 0.52128786],
                        [0.2578984 , 0.14577328, 0.59632832],
                        [0.25996448, 0.29498468, 0.44505084],
                        [0.30970374, 0.45685314, 0.23344312],
                        [0.31003735, 0.37931747, 0.31064518],
                        [0.31197009, 0.53254806, 0.15548185],
                        [0.31446233, 0.60761496, 0.07792271],
                        [0.31927717, 0.68072283, 0.        ],
                        [0.3282323 , 0.        , 0.6717677 ],
                        [0.34118021, 0.14677414, 0.51204564],
                        [0.34220389, 0.06843516, 0.58936095],
                        [0.34288181, 0.22295632, 0.43416187],
                        [0.34939514, 0.29585337, 0.35475149],
                        [0.38413376, 0.37714751, 0.23871873],
                        [0.38841538, 0.45389284, 0.15769178],
                        [0.39196292, 0.52973769, 0.07829939],
                        [0.39604416, 0.60395584, 0.        ],
                        [0.42509888, 0.07700426, 0.49789686],
                        [0.42623721, 0.15318373, 0.42057906],
                        [0.42797394, 0.        , 0.57202606],
                        [0.43128834, 0.22756907, 0.3411426 ],
                        [0.44407277, 0.29692624, 0.25900099],
                        [0.46160583, 0.37437781, 0.16401636],
                        [0.4672655 , 0.45171928, 0.08101522],
                        [0.47254388, 0.52745612, 0.        ],
                        [0.50794335, 0.07763478, 0.41442187],
                        [0.50862575, 0.        , 0.49137425],
                        [0.51136889, 0.15369288, 0.33493822],
                        [0.52185624, 0.22531538, 0.25282838],
                        [0.53152308, 0.29828988, 0.17018704],
                        [0.54101954, 0.37567097, 0.08330949],
                        [0.54502656, 0.45497344, 0.        ],
                        [0.58946874, 0.        , 0.41053126],
                        [0.59047325, 0.07775393, 0.33177282],
                        [0.59890841, 0.1510777 , 0.2500139 ],
                        [0.60747647, 0.22547136, 0.16705217],
                        [0.61467158, 0.30141514, 0.08391327],
                        [0.61912649, 0.38087351, 0.        ],
                        [0.66808467, 0.        , 0.33191533],
                        [0.67560565, 0.07472195, 0.24967241],
                        [0.68256419, 0.15065568, 0.16678013],
                        [0.68988089, 0.22691881, 0.0832003 ],
                        [0.69409976, 0.30590024, 0.        ],
                        [0.75318262, 0.        , 0.24681738],
                        [0.75834239, 0.07639165, 0.16526597],
                        [0.76522219, 0.15177366, 0.08300415],
                        [0.76897163, 0.23102837, 0.        ],
                        [0.83207205, 0.        , 0.16792795],
                        [0.84143216, 0.07574533, 0.08282252],
                        [0.84259116, 0.15740884, 0.        ],
                        [0.91724666, 0.        , 0.08275334],
                        [0.92024869, 0.07975131, 0.        ],
                        [1.        , 0.        , 0.        ]])
    elif dim == 4 and num_weights == 100:
        return np.array([[0.        , 0.        , 0.        , 1.        ],
                    [0.        , 0.        , 0.16116397, 0.83883603],
                    [0.        , 0.        , 0.33274207, 0.66725793],
                    [0.        , 0.        , 0.49493671, 0.50506329],
                    [0.        , 0.        , 0.68350305, 0.31649695],
                    [0.        , 0.        , 0.83718214, 0.16281786],
                    [0.        , 0.        , 1.        , 0.        ],
                    [0.        , 0.14197601, 0.54162341, 0.31640058],
                    [0.        , 0.15010475, 0.15043062, 0.69946463],
                    [0.        , 0.15104696, 0.33050928, 0.51844376],
                    [0.        , 0.15950121, 0.        , 0.84049879],
                    [0.        , 0.16575846, 0.83424154, 0.        ],
                    [0.        , 0.16785224, 0.67413083, 0.15801693],
                    [0.        , 0.26101151, 0.38946932, 0.34951916],
                    [0.        , 0.28109565, 0.16181526, 0.55708908],
                    [0.        , 0.31818039, 0.68181961, 0.        ],
                    [0.        , 0.31929409, 0.50782312, 0.17288279],
                    [0.        , 0.41519501, 0.        , 0.58480499],
                    [0.        , 0.41715296, 0.28955562, 0.29329141],
                    [0.        , 0.42574157, 0.14547731, 0.42878112],
                    [0.        , 0.51929272, 0.48070728, 0.        ],
                    [0.        , 0.53079171, 0.32977517, 0.13943311],
                    [0.        , 0.56326186, 0.1504879 , 0.28625023],
                    [0.        , 0.56540712, 0.        , 0.43459288],
                    [0.        , 0.67671298, 0.32328702, 0.        ],
                    [0.        , 0.70067203, 0.15370636, 0.14562161],
                    [0.        , 0.70805717, 0.        , 0.29194283],
                    [0.        , 0.84247666, 0.15752334, 0.        ],
                    [0.        , 0.85279815, 0.        , 0.14720185],
                    [0.        , 1.        , 0.        , 0.        ],
                    [0.11226047, 0.26397776, 0.2242778 , 0.39948396],
                    [0.11405443, 0.35467172, 0.53127385, 0.        ],
                    [0.12771839, 0.35950806, 0.35998852, 0.15278503],
                    [0.13995024, 0.        , 0.16432966, 0.6957201 ],
                    [0.14003578, 0.11950127, 0.37570026, 0.36476269],
                    [0.14320643, 0.15100979, 0.        , 0.70578378],
                    [0.1435711 , 0.68831787, 0.16811103, 0.        ],
                    [0.14459785, 0.        , 0.69387844, 0.16152371],
                    [0.14505707, 0.53745388, 0.17171771, 0.14577135],
                    [0.14593699, 0.        , 0.53159199, 0.32247103],
                    [0.14618596, 0.14370937, 0.16239064, 0.54771403],
                    [0.14638417, 0.        , 0.85361583, 0.        ],
                    [0.14823637, 0.        , 0.        , 0.85176363],
                    [0.14861707, 0.7063419 , 0.        , 0.14504103],
                    [0.148786  , 0.56654864, 0.        , 0.28466536],
                    [0.14918551, 0.16844059, 0.68237391, 0.        ],
                    [0.15042974, 0.28573262, 0.        , 0.56383764],
                    [0.15076435, 0.84923565, 0.        , 0.        ],
                    [0.15160728, 0.42571407, 0.        , 0.42267865],
                    [0.16051061, 0.17391578, 0.49995336, 0.16562025],
                    [0.16129843, 0.        , 0.3218731 , 0.51682848],
                    [0.16943829, 0.49513661, 0.3354251 , 0.        ],
                    [0.1742619 , 0.38950485, 0.15505096, 0.2811823 ],
                    [0.23432885, 0.22543023, 0.2985792 , 0.24166173],
                    [0.26548209, 0.31407405, 0.42044386, 0.        ],
                    [0.28120943, 0.        , 0.55383579, 0.16495478],
                    [0.28185718, 0.15654646, 0.16837935, 0.39321701],
                    [0.28278328, 0.37826813, 0.22053144, 0.11841715],
                    [0.28350596, 0.54624879, 0.17024525, 0.        ],
                    [0.2843874 , 0.        , 0.1606294 , 0.5549832 ],
                    [0.28560916, 0.        , 0.71439084, 0.        ],
                    [0.29502901, 0.        , 0.        , 0.70497099],
                    [0.29649571, 0.70350429, 0.        , 0.        ],
                    [0.2969212 , 0.        , 0.37498943, 0.32808937],
                    [0.29825637, 0.15274906, 0.54899457, 0.        ],
                    [0.29854472, 0.55714403, 0.        , 0.14431126],
                    [0.29926266, 0.14294478, 0.        , 0.55779256],
                    [0.30087321, 0.28355721, 0.        , 0.41556958],
                    [0.30324922, 0.41767179, 0.        , 0.27907899],
                    [0.34816567, 0.16666074, 0.38027352, 0.10490007],
                    [0.37237275, 0.26485548, 0.12640904, 0.23636274],
                    [0.3994639 , 0.09867884, 0.25363212, 0.24822515],
                    [0.41041071, 0.29850106, 0.29108823, 0.        ],
                    [0.41194319, 0.        , 0.1793511 , 0.40870571],
                    [0.4191893 , 0.        , 0.42140702, 0.15940368],
                    [0.42391435, 0.        , 0.57608565, 0.        ],
                    [0.42812154, 0.42620791, 0.14567054, 0.        ],
                    [0.43614898, 0.56385102, 0.        , 0.        ],
                    [0.43727215, 0.        , 0.        , 0.56272785],
                    [0.44936276, 0.15309332, 0.        , 0.39754393],
                    [0.45329876, 0.39227175, 0.        , 0.15442949],
                    [0.5340006 , 0.15859122, 0.16121972, 0.14618846],
                    [0.53628949, 0.15048332, 0.31322719, 0.        ],
                    [0.54059018, 0.22414946, 0.        , 0.23526037],
                    [0.55777615, 0.        , 0.28974194, 0.15248191],
                    [0.55970172, 0.        , 0.44029828, 0.        ],
                    [0.56317871, 0.        , 0.14594353, 0.29087776],
                    [0.56564333, 0.28658036, 0.1477763 , 0.        ],
                    [0.57359995, 0.42640005, 0.        , 0.        ],
                    [0.57469661, 0.        , 0.        , 0.42530339],
                    [0.69912575, 0.14504326, 0.155831  , 0.        ],
                    [0.70045812, 0.        , 0.29954188, 0.        ],
                    [0.70204379, 0.15152517, 0.        , 0.14643104],
                    [0.70445371, 0.        , 0.14904808, 0.14649821],
                    [0.7086286 , 0.        , 0.        , 0.2913714 ],
                    [0.71066328, 0.28933672, 0.        , 0.        ],
                    [0.8497198 , 0.        , 0.1502802 , 0.        ],
                    [0.85145592, 0.        , 0.        , 0.14854408],
                    [0.85413807, 0.14586193, 0.        , 0.        ],
                    [1.        , 0.        , 0.        , 0.        ]])
    elif dim == 8 and num_weights == 100:
        return np.array([[0.        , 0.        , 0.        , 0.        , 0.        ,
                    0.        , 0.        , 1.        ],
                [0.        , 0.        , 0.        , 0.        , 0.        ,
                    0.        , 0.33949971, 0.66050029],
                [0.        , 0.        , 0.        , 0.        , 0.        ,
                    0.        , 0.66212812, 0.33787188],
                [0.        , 0.        , 0.        , 0.        , 0.        ,
                    0.        , 1.        , 0.        ],
                [0.        , 0.        , 0.        , 0.        , 0.        ,
                    0.32495212, 0.33706387, 0.33798401],
                [0.        , 0.        , 0.        , 0.        , 0.        ,
                    0.33227236, 0.66772764, 0.        ],
                [0.        , 0.        , 0.        , 0.        , 0.        ,
                    0.33611564, 0.        , 0.66388436],
                [0.        , 0.        , 0.        , 0.        , 0.        ,
                    0.66188429, 0.33811571, 0.        ],
                [0.        , 0.        , 0.        , 0.        , 0.        ,
                    0.66928673, 0.        , 0.33071327],
                [0.        , 0.        , 0.        , 0.        , 0.        ,
                    1.        , 0.        , 0.        ],
                [0.        , 0.        , 0.        , 0.        , 0.32511411,
                    0.3464099 , 0.        , 0.32847599],
                [0.        , 0.        , 0.        , 0.        , 0.33504776,
                    0.        , 0.        , 0.66495224],
                [0.        , 0.        , 0.        , 0.        , 0.33518959,
                    0.        , 0.33671571, 0.32809471],
                [0.        , 0.        , 0.        , 0.        , 0.33546692,
                    0.        , 0.66453308, 0.        ],
                [0.        , 0.        , 0.        , 0.        , 0.34047242,
                    0.33882986, 0.32069772, 0.        ],
                [0.        , 0.        , 0.        , 0.        , 0.65021854,
                    0.34978146, 0.        , 0.        ],
                [0.        , 0.        , 0.        , 0.        , 0.66018364,
                    0.        , 0.33981636, 0.        ],
                [0.        , 0.        , 0.        , 0.        , 0.67076614,
                    0.        , 0.        , 0.32923386],
                [0.        , 0.        , 0.        , 0.        , 1.        ,
                    0.        , 0.        , 0.        ],
                [0.        , 0.        , 0.        , 0.31907282, 0.34466465,
                    0.        , 0.        , 0.33626254],
                [0.        , 0.        , 0.        , 0.32879469, 0.67120531,
                    0.        , 0.        , 0.        ],
                [0.        , 0.        , 0.        , 0.32884166, 0.        ,
                    0.        , 0.33841786, 0.33274048],
                [0.        , 0.        , 0.        , 0.33579761, 0.        ,
                    0.33441333, 0.32978907, 0.        ],
                [0.        , 0.        , 0.        , 0.33671961, 0.3360384 ,
                    0.        , 0.32724199, 0.        ],
                [0.        , 0.        , 0.        , 0.34057163, 0.3254604 ,
                    0.33396797, 0.        , 0.        ],
                [0.        , 0.        , 0.        , 0.34227034, 0.        ,
                    0.65772966, 0.        , 0.        ],
                [0.        , 0.        , 0.        , 0.34669574, 0.        ,
                    0.        , 0.65330426, 0.        ],
                [0.        , 0.        , 0.        , 0.34985852, 0.        ,
                    0.        , 0.        , 0.65014148],
                [0.        , 0.        , 0.        , 0.66477273, 0.33522727,
                    0.        , 0.        , 0.        ],
                [0.        , 0.        , 0.        , 0.66705696, 0.        ,
                    0.        , 0.        , 0.33294304],
                [0.        , 0.        , 0.        , 0.66774834, 0.        ,
                    0.        , 0.33225166, 0.        ],
                [0.        , 0.        , 0.        , 1.        , 0.        ,
                    0.        , 0.        , 0.        ],
                [0.        , 0.        , 0.32137115, 0.3423554 , 0.        ,
                    0.33627344, 0.        , 0.        ],
                [0.        , 0.        , 0.32877946, 0.67122054, 0.        ,
                    0.        , 0.        , 0.        ],
                [0.        , 0.        , 0.32931796, 0.        , 0.33448844,
                    0.        , 0.3361936 , 0.        ],
                [0.        , 0.        , 0.33466964, 0.3360219 , 0.        ,
                    0.        , 0.        , 0.32930846],
                [0.        , 0.        , 0.33614123, 0.        , 0.        ,
                    0.33945283, 0.32440594, 0.        ],
                [0.        , 0.        , 0.33883621, 0.        , 0.        ,
                    0.        , 0.        , 0.66116379],
                [0.        , 0.        , 0.34293717, 0.        , 0.        ,
                    0.32543894, 0.        , 0.33162389],
                [0.        , 0.        , 0.34312028, 0.3425833 , 0.31429641,
                    0.        , 0.        , 0.        ],
                [0.        , 0.        , 0.34436106, 0.        , 0.        ,
                    0.        , 0.65563894, 0.        ],
                [0.        , 0.        , 0.64260304, 0.        , 0.35739696,
                    0.        , 0.        , 0.        ],
                [0.        , 0.        , 0.66187741, 0.        , 0.        ,
                    0.        , 0.        , 0.33812259],
                [0.        , 0.        , 0.66312632, 0.        , 0.        ,
                    0.33687368, 0.        , 0.        ],
                [0.        , 0.        , 0.66857767, 0.        , 0.        ,
                    0.        , 0.33142233, 0.        ],
                [0.        , 0.        , 1.        , 0.        , 0.        ,
                    0.        , 0.        , 0.        ],
                [0.        , 0.21488155, 0.36003524, 0.22188469, 0.        ,
                    0.        , 0.20319852, 0.        ],
                [0.        , 0.22890064, 0.24873201, 0.        , 0.52236735,
                    0.        , 0.        , 0.        ],
                [0.        , 0.26627253, 0.26305134, 0.        , 0.17216463,
                    0.        , 0.        , 0.29851151],
                [0.        , 0.32809301, 0.33337382, 0.        , 0.        ,
                    0.33853317, 0.        , 0.        ],
                [0.        , 0.33000652, 0.66999348, 0.        , 0.        ,
                    0.        , 0.        , 0.        ],
                [0.        , 0.33021281, 0.        , 0.        , 0.        ,
                    0.33505217, 0.33473502, 0.        ],
                [0.        , 0.33245185, 0.        , 0.34530326, 0.        ,
                    0.        , 0.        , 0.32224489],
                [0.        , 0.33272938, 0.        , 0.        , 0.        ,
                    0.        , 0.33203607, 0.33523455],
                [0.        , 0.33276866, 0.        , 0.33137906, 0.33585228,
                    0.        , 0.        , 0.        ],
                [0.        , 0.33577192, 0.        , 0.        , 0.        ,
                    0.        , 0.        , 0.66422808],
                [0.        , 0.33611418, 0.        , 0.        , 0.        ,
                    0.66388582, 0.        , 0.        ],
                [0.        , 0.33615139, 0.        , 0.        , 0.        ,
                    0.33818425, 0.        , 0.32566436],
                [0.        , 0.33775723, 0.        , 0.32678071, 0.        ,
                    0.        , 0.33546206, 0.        ],
                [0.        , 0.33906584, 0.        , 0.66093416, 0.        ,
                    0.        , 0.        , 0.        ],
                [0.        , 0.33953859, 0.        , 0.        , 0.        ,
                    0.        , 0.66046141, 0.        ],
                [0.        , 0.33992983, 0.        , 0.        , 0.3396515 ,
                    0.32041867, 0.        , 0.        ],
                [0.        , 0.64686463, 0.35313537, 0.        , 0.        ,
                    0.        , 0.        , 0.        ],
                [0.        , 0.65819224, 0.        , 0.        , 0.        ,
                    0.        , 0.        , 0.34180776],
                [0.        , 0.6609039 , 0.        , 0.3390961 , 0.        ,
                    0.        , 0.        , 0.        ],
                [0.        , 0.660962  , 0.        , 0.        , 0.339038  ,
                    0.        , 0.        , 0.        ],
                [0.        , 0.66159837, 0.        , 0.        , 0.        ,
                    0.33840163, 0.        , 0.        ],
                [0.        , 0.66869336, 0.        , 0.        , 0.        ,
                    0.        , 0.33130664, 0.        ],
                [0.        , 1.        , 0.        , 0.        , 0.        ,
                    0.        , 0.        , 0.        ],
                [0.20703253, 0.        , 0.18647648, 0.        , 0.23832494,
                    0.36816604, 0.        , 0.        ],
                [0.22077261, 0.        , 0.36351925, 0.        , 0.        ,
                    0.        , 0.20340588, 0.21230227],
                [0.321612  , 0.33961167, 0.        , 0.33877633, 0.        ,
                    0.        , 0.        , 0.        ],
                [0.32286086, 0.34128216, 0.        , 0.        , 0.        ,
                    0.        , 0.33585698, 0.        ],
                [0.32405324, 0.33162558, 0.34432118, 0.        , 0.        ,
                    0.        , 0.        , 0.        ],
                [0.32495549, 0.        , 0.67504451, 0.        , 0.        ,
                    0.        , 0.        , 0.        ],
                [0.32690965, 0.        , 0.33759588, 0.33549447, 0.        ,
                    0.        , 0.        , 0.        ],
                [0.32749835, 0.        , 0.        , 0.34042987, 0.        ,
                    0.        , 0.33207178, 0.        ],
                [0.3293453 , 0.33755619, 0.        , 0.        , 0.33309851,
                    0.        , 0.        , 0.        ],
                [0.32960401, 0.        , 0.        , 0.        , 0.33908867,
                    0.        , 0.        , 0.33130732],
                [0.33057548, 0.66942452, 0.        , 0.        , 0.        ,
                    0.        , 0.        , 0.        ],
                [0.33193149, 0.3425293 , 0.        , 0.        , 0.        ,
                    0.3255392 , 0.        , 0.        ],
                [0.33436313, 0.        , 0.        , 0.        , 0.        ,
                    0.3423702 , 0.        , 0.32326667],
                [0.33462623, 0.32985088, 0.        , 0.        , 0.        ,
                    0.        , 0.        , 0.33552289],
                [0.33721991, 0.        , 0.        , 0.33680363, 0.        ,
                    0.        , 0.        , 0.32597646],
                [0.3383095 , 0.        , 0.        , 0.3223375 , 0.33935301,
                    0.        , 0.        , 0.        ],
                [0.33951434, 0.        , 0.        , 0.        , 0.        ,
                    0.        , 0.34102622, 0.31945944],
                [0.34177188, 0.        , 0.        , 0.        , 0.65822812,
                    0.        , 0.        , 0.        ],
                [0.34355669, 0.        , 0.        , 0.        , 0.        ,
                    0.        , 0.65644331, 0.        ],
                [0.34429476, 0.        , 0.        , 0.        , 0.        ,
                    0.65570524, 0.        , 0.        ],
                [0.34468201, 0.        , 0.        , 0.        , 0.        ,
                    0.32283728, 0.33248071, 0.        ],
                [0.34519146, 0.        , 0.33908543, 0.        , 0.31572311,
                    0.        , 0.        , 0.        ],
                [0.34617499, 0.        , 0.        , 0.32897795, 0.        ,
                    0.32484706, 0.        , 0.        ],
                [0.34692255, 0.        , 0.        , 0.        , 0.        ,
                    0.        , 0.        , 0.65307745],
                [0.65273992, 0.34726008, 0.        , 0.        , 0.        ,
                    0.        , 0.        , 0.        ],
                [0.66203267, 0.        , 0.        , 0.33796733, 0.        ,
                    0.        , 0.        , 0.        ],
                [0.66245812, 0.        , 0.33754188, 0.        , 0.        ,
                    0.        , 0.        , 0.        ],
                [0.664246  , 0.        , 0.        , 0.        , 0.        ,
                    0.        , 0.        , 0.335754  ],
                [0.66608776, 0.        , 0.        , 0.        , 0.33391224,
                    0.        , 0.        , 0.        ],
                [0.6764983 , 0.        , 0.        , 0.        , 0.        ,
                    0.        , 0.3235017 , 0.        ],
                [1.        , 0.        , 0.        , 0.        , 0.        ,
                    0.        , 0.        , 0.        ]])
    elif dim == 12 and num_weights == 100:
        return np.array([[0.        , 0.        , 0.        , 0.        , 0.        ,
                    0.        , 0.        , 0.        , 0.        , 0.        ,
                    0.        , 1.        ],
                [0.        , 0.        , 0.        , 0.        , 0.        ,
                    0.        , 0.        , 0.        , 0.        , 0.        ,
                    0.59884126, 0.40115874],
                [0.        , 0.        , 0.        , 0.        , 0.        ,
                    0.        , 0.        , 0.        , 0.        , 0.        ,
                    1.        , 0.        ],
                [0.        , 0.        , 0.        , 0.        , 0.        ,
                    0.        , 0.        , 0.        , 0.        , 0.38938866,
                    0.61061134, 0.        ],
                [0.        , 0.        , 0.        , 0.        , 0.        ,
                    0.        , 0.        , 0.        , 0.        , 0.53825047,
                    0.        , 0.46174953],
                [0.        , 0.        , 0.        , 0.        , 0.        ,
                    0.        , 0.        , 0.        , 0.        , 1.        ,
                    0.        , 0.        ],
                [0.        , 0.        , 0.        , 0.        , 0.        ,
                    0.        , 0.        , 0.        , 0.35836093, 0.        ,
                    0.64163907, 0.        ],
                [0.        , 0.        , 0.        , 0.        , 0.        ,
                    0.        , 0.        , 0.        , 0.38364992, 0.        ,
                    0.        , 0.61635008],
                [0.        , 0.        , 0.        , 0.        , 0.        ,
                    0.        , 0.        , 0.        , 0.54693608, 0.45306392,
                    0.        , 0.        ],
                [0.        , 0.        , 0.        , 0.        , 0.        ,
                    0.        , 0.        , 0.        , 1.        , 0.        ,
                    0.        , 0.        ],
                [0.        , 0.        , 0.        , 0.        , 0.        ,
                    0.        , 0.        , 0.38281998, 0.        , 0.        ,
                    0.61718002, 0.        ],
                [0.        , 0.        , 0.        , 0.        , 0.        ,
                    0.        , 0.        , 0.56868309, 0.        , 0.        ,
                    0.        , 0.43131691],
                [0.        , 0.        , 0.        , 0.        , 0.        ,
                    0.        , 0.        , 0.58695507, 0.41304493, 0.        ,
                    0.        , 0.        ],
                [0.        , 0.        , 0.        , 0.        , 0.        ,
                    0.        , 0.        , 0.63119536, 0.        , 0.36880464,
                    0.        , 0.        ],
                [0.        , 0.        , 0.        , 0.        , 0.        ,
                    0.        , 0.        , 1.        , 0.        , 0.        ,
                    0.        , 0.        ],
                [0.        , 0.        , 0.        , 0.        , 0.        ,
                    0.        , 0.28021101, 0.        , 0.24941019, 0.21262722,
                    0.        , 0.25775158],
                [0.        , 0.        , 0.        , 0.        , 0.        ,
                    0.        , 0.39046529, 0.        , 0.        , 0.        ,
                    0.        , 0.60953471],
                [0.        , 0.        , 0.        , 0.        , 0.        ,
                    0.        , 0.39391871, 0.60608129, 0.        , 0.        ,
                    0.        , 0.        ],
                [0.        , 0.        , 0.        , 0.        , 0.        ,
                    0.        , 0.40511334, 0.        , 0.        , 0.59488666,
                    0.        , 0.        ],
                [0.        , 0.        , 0.        , 0.        , 0.        ,
                    0.        , 0.60773257, 0.        , 0.        , 0.        ,
                    0.39226743, 0.        ],
                [0.        , 0.        , 0.        , 0.        , 0.        ,
                    0.        , 0.63342082, 0.        , 0.36657918, 0.        ,
                    0.        , 0.        ],
                [0.        , 0.        , 0.        , 0.        , 0.        ,
                    0.        , 1.        , 0.        , 0.        , 0.        ,
                    0.        , 0.        ],
                [0.        , 0.        , 0.        , 0.        , 0.        ,
                    0.23152266, 0.24533968, 0.24837326, 0.        , 0.        ,
                    0.2747644 , 0.        ],
                [0.        , 0.        , 0.        , 0.        , 0.        ,
                    0.37475245, 0.62524755, 0.        , 0.        , 0.        ,
                    0.        , 0.        ],
                [0.        , 0.        , 0.        , 0.        , 0.        ,
                    0.39551188, 0.        , 0.        , 0.        , 0.        ,
                    0.60448812, 0.        ],
                [0.        , 0.        , 0.        , 0.        , 0.        ,
                    0.4460859 , 0.        , 0.        , 0.        , 0.        ,
                    0.        , 0.5539141 ],
                [0.        , 0.        , 0.        , 0.        , 0.        ,
                    0.52053419, 0.        , 0.47946581, 0.        , 0.        ,
                    0.        , 0.        ],
                [0.        , 0.        , 0.        , 0.        , 0.        ,
                    0.580253  , 0.        , 0.        , 0.        , 0.419747  ,
                    0.        , 0.        ],
                [0.        , 0.        , 0.        , 0.        , 0.        ,
                    0.64208371, 0.        , 0.        , 0.35791629, 0.        ,
                    0.        , 0.        ],
                [0.        , 0.        , 0.        , 0.        , 0.        ,
                    1.        , 0.        , 0.        , 0.        , 0.        ,
                    0.        , 0.        ],
                [0.        , 0.        , 0.        , 0.        , 0.257268  ,
                    0.2586618 , 0.2254854 , 0.        , 0.25858481, 0.        ,
                    0.        , 0.        ],
                [0.        , 0.        , 0.        , 0.        , 0.37953297,
                    0.        , 0.        , 0.        , 0.        , 0.        ,
                    0.        , 0.62046703],
                [0.        , 0.        , 0.        , 0.        , 0.38146688,
                    0.        , 0.        , 0.61853312, 0.        , 0.        ,
                    0.        , 0.        ],
                [0.        , 0.        , 0.        , 0.        , 0.42915462,
                    0.        , 0.        , 0.        , 0.        , 0.57084538,
                    0.        , 0.        ],
                [0.        , 0.        , 0.        , 0.        , 0.50801295,
                    0.        , 0.49198705, 0.        , 0.        , 0.        ,
                    0.        , 0.        ],
                [0.        , 0.        , 0.        , 0.        , 0.58304782,
                    0.        , 0.        , 0.        , 0.41695218, 0.        ,
                    0.        , 0.        ],
                [0.        , 0.        , 0.        , 0.        , 0.58878737,
                    0.41121263, 0.        , 0.        , 0.        , 0.        ,
                    0.        , 0.        ],
                [0.        , 0.        , 0.        , 0.        , 0.61001784,
                    0.        , 0.        , 0.        , 0.        , 0.        ,
                    0.38998216, 0.        ],
                [0.        , 0.        , 0.        , 0.        , 1.        ,
                    0.        , 0.        , 0.        , 0.        , 0.        ,
                    0.        , 0.        ],
                [0.        , 0.        , 0.        , 0.17103564, 0.        ,
                    0.23352824, 0.        , 0.        , 0.59543612, 0.        ,
                    0.        , 0.        ],
                [0.        , 0.        , 0.        , 0.25071797, 0.        ,
                    0.        , 0.23933616, 0.24817109, 0.26177477, 0.        ,
                    0.        , 0.        ],
                [0.        , 0.        , 0.        , 0.38637546, 0.        ,
                    0.        , 0.        , 0.        , 0.        , 0.61362454,
                    0.        , 0.        ],
                [0.        , 0.        , 0.        , 0.42029501, 0.        ,
                    0.        , 0.        , 0.57970499, 0.        , 0.        ,
                    0.        , 0.        ],
                [0.        , 0.        , 0.        , 0.48420188, 0.51579812,
                    0.        , 0.        , 0.        , 0.        , 0.        ,
                    0.        , 0.        ],
                [0.        , 0.        , 0.        , 0.50047657, 0.        ,
                    0.        , 0.        , 0.        , 0.        , 0.        ,
                    0.49952343, 0.        ],
                [0.        , 0.        , 0.        , 0.5014989 , 0.        ,
                    0.        , 0.        , 0.        , 0.        , 0.        ,
                    0.        , 0.4985011 ],
                [0.        , 0.        , 0.        , 0.54620758, 0.        ,
                    0.        , 0.45379242, 0.        , 0.        , 0.        ,
                    0.        , 0.        ],
                [0.        , 0.        , 0.        , 0.57656141, 0.        ,
                    0.        , 0.        , 0.        , 0.42343859, 0.        ,
                    0.        , 0.        ],
                [0.        , 0.        , 0.        , 0.61840657, 0.        ,
                    0.38159343, 0.        , 0.        , 0.        , 0.        ,
                    0.        , 0.        ],
                [0.        , 0.        , 0.        , 1.        , 0.        ,
                    0.        , 0.        , 0.        , 0.        , 0.        ,
                    0.        , 0.        ],
                [0.        , 0.        , 0.18642743, 0.48820527, 0.        ,
                    0.        , 0.        , 0.13739279, 0.        , 0.18797451,
                    0.        , 0.        ],
                [0.        , 0.        , 0.18802524, 0.        , 0.48287438,
                    0.        , 0.        , 0.        , 0.        , 0.1307344 ,
                    0.        , 0.19836597],
                [0.        , 0.        , 0.22606496, 0.        , 0.        ,
                    0.2608532 , 0.26108058, 0.        , 0.        , 0.25200125,
                    0.        , 0.        ],
                [0.        , 0.        , 0.25642053, 0.        , 0.24185789,
                    0.        , 0.23250704, 0.26921453, 0.        , 0.        ,
                    0.        , 0.        ],
                [0.        , 0.        , 0.26406192, 0.        , 0.        ,
                    0.28245193, 0.        , 0.        , 0.23683558, 0.        ,
                    0.        , 0.21665058],
                [0.        , 0.        , 0.43431989, 0.        , 0.        ,
                    0.        , 0.        , 0.        , 0.        , 0.        ,
                    0.        , 0.56568011],
                [0.        , 0.        , 0.4770854 , 0.        , 0.        ,
                    0.        , 0.        , 0.        , 0.        , 0.5229146 ,
                    0.        , 0.        ],
                [0.        , 0.        , 0.54998544, 0.        , 0.        ,
                    0.        , 0.        , 0.        , 0.        , 0.        ,
                    0.45001456, 0.        ],
                [0.        , 0.        , 0.55184347, 0.        , 0.        ,
                    0.        , 0.        , 0.        , 0.44815653, 0.        ,
                    0.        , 0.        ],
                [0.        , 0.        , 0.61342195, 0.        , 0.        ,
                    0.        , 0.38657805, 0.        , 0.        , 0.        ,
                    0.        , 0.        ],
                [0.        , 0.        , 0.61617376, 0.38382624, 0.        ,
                    0.        , 0.        , 0.        , 0.        , 0.        ,
                    0.        , 0.        ],
                [0.        , 0.        , 0.61932687, 0.        , 0.        ,
                    0.        , 0.        , 0.38067313, 0.        , 0.        ,
                    0.        , 0.        ],
                [0.        , 0.        , 0.61994567, 0.        , 0.38005433,
                    0.        , 0.        , 0.        , 0.        , 0.        ,
                    0.        , 0.        ],
                [0.        , 0.        , 0.63348504, 0.        , 0.        ,
                    0.36651496, 0.        , 0.        , 0.        , 0.        ,
                    0.        , 0.        ],
                [0.        , 0.        , 1.        , 0.        , 0.        ,
                    0.        , 0.        , 0.        , 0.        , 0.        ,
                    0.        , 0.        ],
                [0.        , 0.20593024, 0.        , 0.18724589, 0.        ,
                    0.44306694, 0.        , 0.        , 0.        , 0.        ,
                    0.16375693, 0.        ],
                [0.        , 0.21994049, 0.22890055, 0.        , 0.        ,
                    0.        , 0.        , 0.        , 0.27678358, 0.        ,
                    0.27437537, 0.        ],
                [0.        , 0.22789081, 0.        , 0.        , 0.24511887,
                    0.        , 0.        , 0.        , 0.        , 0.26218848,
                    0.26480183, 0.        ],
                [0.        , 0.23398902, 0.        , 0.        , 0.        ,
                    0.        , 0.        , 0.23590111, 0.2674208 , 0.        ,
                    0.        , 0.26268908],
                [0.        , 0.25274723, 0.        , 0.        , 0.        ,
                    0.        , 0.24726775, 0.        , 0.        , 0.        ,
                    0.26437597, 0.23560905],
                [0.        , 0.41093524, 0.58906476, 0.        , 0.        ,
                    0.        , 0.        , 0.        , 0.        , 0.        ,
                    0.        , 0.        ],
                [0.        , 0.42855545, 0.        , 0.        , 0.        ,
                    0.        , 0.        , 0.        , 0.57144455, 0.        ,
                    0.        , 0.        ],
                [0.        , 0.430671  , 0.        , 0.569329  , 0.        ,
                    0.        , 0.        , 0.        , 0.        , 0.        ,
                    0.        , 0.        ],
                [0.        , 0.46984962, 0.        , 0.        , 0.        ,
                    0.        , 0.        , 0.53015038, 0.        , 0.        ,
                    0.        , 0.        ],
                [0.        , 0.52785955, 0.        , 0.        , 0.        ,
                    0.        , 0.        , 0.        , 0.        , 0.        ,
                    0.        , 0.47214045],
                [0.        , 0.56242837, 0.        , 0.        , 0.43757163,
                    0.        , 0.        , 0.        , 0.        , 0.        ,
                    0.        , 0.        ],
                [0.        , 0.5861462 , 0.        , 0.        , 0.        ,
                    0.        , 0.        , 0.        , 0.        , 0.4138538 ,
                    0.        , 0.        ],
                [0.        , 0.58764157, 0.        , 0.        , 0.        ,
                    0.        , 0.41235843, 0.        , 0.        , 0.        ,
                    0.        , 0.        ],
                [0.        , 0.61951673, 0.        , 0.        , 0.        ,
                    0.        , 0.        , 0.        , 0.        , 0.        ,
                    0.38048327, 0.        ],
                [0.        , 0.63731169, 0.        , 0.        , 0.        ,
                    0.36268831, 0.        , 0.        , 0.        , 0.        ,
                    0.        , 0.        ],
                [0.        , 1.        , 0.        , 0.        , 0.        ,
                    0.        , 0.        , 0.        , 0.        , 0.        ,
                    0.        , 0.        ],
                [0.21457319, 0.        , 0.        , 0.        , 0.        ,
                    0.        , 0.        , 0.21828073, 0.        , 0.56714608,
                    0.        , 0.        ],
                [0.21981891, 0.        , 0.        , 0.        , 0.17280842,
                    0.        , 0.        , 0.22262247, 0.        , 0.        ,
                    0.17215135, 0.21259885],
                [0.23439492, 0.25726661, 0.25304445, 0.        , 0.        ,
                    0.        , 0.25529402, 0.        , 0.        , 0.        ,
                    0.        , 0.        ],
                [0.24366821, 0.        , 0.24611789, 0.        , 0.        ,
                    0.        , 0.        , 0.        , 0.25148868, 0.25872522,
                    0.        , 0.        ],
                [0.24373108, 0.19622287, 0.        , 0.26390058, 0.13737481,
                    0.        , 0.        , 0.        , 0.15877066, 0.        ,
                    0.        , 0.        ],
                [0.25875581, 0.        , 0.        , 0.        , 0.26532407,
                    0.23572073, 0.        , 0.        , 0.        , 0.24019939,
                    0.        , 0.        ],
                [0.35925944, 0.        , 0.        , 0.        , 0.        ,
                    0.64074056, 0.        , 0.        , 0.        , 0.        ,
                    0.        , 0.        ],
                [0.36463292, 0.        , 0.        , 0.        , 0.        ,
                    0.        , 0.        , 0.        , 0.        , 0.        ,
                    0.        , 0.63536708],
                [0.45601795, 0.        , 0.        , 0.        , 0.        ,
                    0.        , 0.        , 0.        , 0.        , 0.        ,
                    0.54398205, 0.        ],
                [0.49762352, 0.        , 0.        , 0.        , 0.        ,
                    0.        , 0.        , 0.50237648, 0.        , 0.        ,
                    0.        , 0.        ],
                [0.5168204 , 0.        , 0.        , 0.        , 0.        ,
                    0.24300886, 0.        , 0.        , 0.        , 0.        ,
                    0.        , 0.24017074],
                [0.52759511, 0.47240489, 0.        , 0.        , 0.        ,
                    0.        , 0.        , 0.        , 0.        , 0.        ,
                    0.        , 0.        ],
                [0.54835491, 0.        , 0.        , 0.        , 0.        ,
                    0.        , 0.45164509, 0.        , 0.        , 0.        ,
                    0.        , 0.        ],
                [0.55898993, 0.        , 0.44101007, 0.        , 0.        ,
                    0.        , 0.        , 0.        , 0.        , 0.        ,
                    0.        , 0.        ],
                [0.56770231, 0.        , 0.        , 0.        , 0.        ,
                    0.        , 0.        , 0.        , 0.43229769, 0.        ,
                    0.        , 0.        ],
                [0.61203464, 0.        , 0.        , 0.        , 0.38796536,
                    0.        , 0.        , 0.        , 0.        , 0.        ,
                    0.        , 0.        ],
                [0.62258806, 0.        , 0.        , 0.        , 0.        ,
                    0.        , 0.        , 0.        , 0.        , 0.37741194,
                    0.        , 0.        ],
                [0.64776397, 0.        , 0.        , 0.35223603, 0.        ,
                    0.        , 0.        , 0.        , 0.        , 0.        ,
                    0.        , 0.        ],
                [1.        , 0.        , 0.        , 0.        , 0.        ,
                    0.        , 0.        , 0.        , 0.        , 0.        ,
                    0.        , 0.        ]])
    elif dim == 8 and num_weights == 256:
        return np.array([[0.        , 0.        , 0.        , 0.        , 0.        ,
        0.        , 0.        , 1.        ],
       [0.        , 0.        , 0.        , 0.        , 0.        ,
        0.        , 0.49121137, 0.50878863],
       [0.        , 0.        , 0.        , 0.        , 0.        ,
        0.        , 0.73382054, 0.26617946],
       [0.        , 0.        , 0.        , 0.        , 0.        ,
        0.        , 1.        , 0.        ],
       [0.        , 0.        , 0.        , 0.        , 0.        ,
        0.25356093, 0.2617167 , 0.48472237],
       [0.        , 0.        , 0.        , 0.        , 0.        ,
        0.2567518 , 0.        , 0.7432482 ],
       [0.        , 0.        , 0.        , 0.        , 0.        ,
        0.260648  , 0.739352  , 0.        ],
       [0.        , 0.        , 0.        , 0.        , 0.        ,
        0.50027512, 0.2585382 , 0.24118668],
       [0.        , 0.        , 0.        , 0.        , 0.        ,
        0.50170037, 0.49829963, 0.        ],
       [0.        , 0.        , 0.        , 0.        , 0.        ,
        0.5035349 , 0.        , 0.4964651 ],
       [0.        , 0.        , 0.        , 0.        , 0.        ,
        0.74164972, 0.        , 0.25835028],
       [0.        , 0.        , 0.        , 0.        , 0.        ,
        0.75105592, 0.24894408, 0.        ],
       [0.        , 0.        , 0.        , 0.        , 0.        ,
        1.        , 0.        , 0.        ],
       [0.        , 0.        , 0.        , 0.        , 0.24759867,
        0.24185857, 0.26020859, 0.25033417],
       [0.        , 0.        , 0.        , 0.        , 0.25012525,
        0.        , 0.2486489 , 0.50122585],
       [0.        , 0.        , 0.        , 0.        , 0.2513199 ,
        0.25263587, 0.        , 0.49604423],
       [0.        , 0.        , 0.        , 0.        , 0.25133459,
        0.        , 0.74866541, 0.        ],
       [0.        , 0.        , 0.        , 0.        , 0.25191851,
        0.25618454, 0.49189695, 0.        ],
       [0.        , 0.        , 0.        , 0.        , 0.25312034,
        0.49948052, 0.24739914, 0.        ],
       [0.        , 0.        , 0.        , 0.        , 0.25710415,
        0.49045712, 0.        , 0.25243872],
       [0.        , 0.        , 0.        , 0.        , 0.49125959,
        0.        , 0.50874041, 0.        ],
       [0.        , 0.        , 0.        , 0.        , 0.49306089,
        0.        , 0.        , 0.50693911],
       [0.        , 0.        , 0.        , 0.        , 0.49737986,
        0.2537852 , 0.24883494, 0.        ],
       [0.        , 0.        , 0.        , 0.        , 0.49742741,
        0.24788547, 0.        , 0.25468712],
       [0.        , 0.        , 0.        , 0.        , 0.50035735,
        0.        , 0.24894587, 0.25069679],
       [0.        , 0.        , 0.        , 0.        , 0.50082387,
        0.49917613, 0.        , 0.        ],
       [0.        , 0.        , 0.        , 0.        , 0.73566435,
        0.        , 0.26433565, 0.        ],
       [0.        , 0.        , 0.        , 0.        , 0.74316627,
        0.        , 0.        , 0.25683373],
       [0.        , 0.        , 0.        , 0.        , 0.74513779,
        0.25486221, 0.        , 0.        ],
       [0.        , 0.        , 0.        , 0.        , 1.        ,
        0.        , 0.        , 0.        ],
       [0.        , 0.        , 0.        , 0.24812926, 0.24339223,
        0.        , 0.        , 0.50847851],
       [0.        , 0.        , 0.        , 0.24876659, 0.        ,
        0.49562953, 0.25560388, 0.        ],
       [0.        , 0.        , 0.        , 0.24964895, 0.        ,
        0.        , 0.48948046, 0.26087059],
       [0.        , 0.        , 0.        , 0.24994356, 0.25916894,
        0.        , 0.25035267, 0.24053484],
       [0.        , 0.        , 0.        , 0.25280197, 0.        ,
        0.2564694 , 0.        , 0.49072863],
       [0.        , 0.        , 0.        , 0.25346348, 0.48708448,
        0.        , 0.        , 0.25945205],
       [0.        , 0.        , 0.        , 0.25385813, 0.        ,
        0.25071739, 0.49542447, 0.        ],
       [0.        , 0.        , 0.        , 0.25490588, 0.74509412,
        0.        , 0.        , 0.        ],
       [0.        , 0.        , 0.        , 0.25566403, 0.49443439,
        0.24990159, 0.        , 0.        ],
       [0.        , 0.        , 0.        , 0.25613053, 0.25421834,
        0.48965113, 0.        , 0.        ],
       [0.        , 0.        , 0.        , 0.25877775, 0.        ,
        0.        , 0.        , 0.74122225],
       [0.        , 0.        , 0.        , 0.26060475, 0.25450977,
        0.24138254, 0.24350293, 0.        ],
       [0.        , 0.        , 0.        , 0.26083045, 0.26146819,
        0.        , 0.47770136, 0.        ],
       [0.        , 0.        , 0.        , 0.26310663, 0.        ,
        0.        , 0.73689337, 0.        ],
       [0.        , 0.        , 0.        , 0.26736691, 0.        ,
        0.73263309, 0.        , 0.        ],
       [0.        , 0.        , 0.        , 0.47000997, 0.        ,
        0.28532084, 0.        , 0.24466919],
       [0.        , 0.        , 0.        , 0.48566162, 0.        ,
        0.        , 0.25084167, 0.26349672],
       [0.        , 0.        , 0.        , 0.4947924 , 0.        ,
        0.25473679, 0.25047081, 0.        ],
       [0.        , 0.        , 0.        , 0.49716094, 0.        ,
        0.        , 0.        , 0.50283906],
       [0.        , 0.        , 0.        , 0.49924592, 0.50075408,
        0.        , 0.        , 0.        ],
       [0.        , 0.        , 0.        , 0.50055571, 0.25349701,
        0.        , 0.        , 0.24594728],
       [0.        , 0.        , 0.        , 0.5042369 , 0.        ,
        0.        , 0.4957631 , 0.        ],
       [0.        , 0.        , 0.        , 0.71556438, 0.        ,
        0.28443562, 0.        , 0.        ],
       [0.        , 0.        , 0.        , 0.74150628, 0.        ,
        0.        , 0.        , 0.25849372],
       [0.        , 0.        , 0.        , 0.74349186, 0.25650814,
        0.        , 0.        , 0.        ],
       [0.        , 0.        , 0.        , 0.74518773, 0.        ,
        0.        , 0.25481227, 0.        ],
       [0.        , 0.        , 0.        , 1.        , 0.        ,
        0.        , 0.        , 0.        ],
       [0.        , 0.        , 0.09898138, 0.24418558, 0.21426742,
        0.21366625, 0.        , 0.22889937],
       [0.        , 0.        , 0.2462976 , 0.        , 0.        ,
        0.        , 0.25063266, 0.50306973],
       [0.        , 0.        , 0.24769045, 0.49737286, 0.2549367 ,
        0.        , 0.        , 0.        ],
       [0.        , 0.        , 0.24788076, 0.47130914, 0.        ,
        0.28081009, 0.        , 0.        ],
       [0.        , 0.        , 0.24893973, 0.        , 0.        ,
        0.        , 0.49706812, 0.25399214],
       [0.        , 0.        , 0.24945123, 0.        , 0.        ,
        0.50396904, 0.        , 0.24657973],
       [0.        , 0.        , 0.25001736, 0.25311305, 0.        ,
        0.        , 0.        , 0.49686959],
       [0.        , 0.        , 0.25012876, 0.        , 0.        ,
        0.74987124, 0.        , 0.        ],
       [0.        , 0.        , 0.25043816, 0.74956184, 0.        ,
        0.        , 0.        , 0.        ],
       [0.        , 0.        , 0.25362163, 0.25052764, 0.        ,
        0.        , 0.24153652, 0.25431421],
       [0.        , 0.        , 0.25534291, 0.49717825, 0.        ,
        0.        , 0.24747885, 0.        ],
       [0.        , 0.        , 0.25541621, 0.        , 0.25117622,
        0.        , 0.        , 0.49340756],
       [0.        , 0.        , 0.25621385, 0.49092353, 0.        ,
        0.        , 0.        , 0.25286262],
       [0.        , 0.        , 0.25638639, 0.        , 0.48943659,
        0.        , 0.        , 0.25417702],
       [0.        , 0.        , 0.25662264, 0.25366733, 0.48971003,
        0.        , 0.        , 0.        ],
       [0.        , 0.        , 0.25680718, 0.        , 0.        ,
        0.25800792, 0.        , 0.4851849 ],
       [0.        , 0.        , 0.25721306, 0.        , 0.        ,
        0.        , 0.74278694, 0.        ],
       [0.        , 0.        , 0.25769361, 0.        , 0.49423488,
        0.        , 0.24807152, 0.        ],
       [0.        , 0.        , 0.25953143, 0.        , 0.25385119,
        0.        , 0.48661738, 0.        ],
       [0.        , 0.        , 0.26195553, 0.25125084, 0.        ,
        0.        , 0.48679362, 0.        ],
       [0.        , 0.        , 0.26312648, 0.        , 0.        ,
        0.        , 0.        , 0.73687352],
       [0.        , 0.        , 0.26344303, 0.        , 0.73655697,
        0.        , 0.        , 0.        ],
       [0.        , 0.        , 0.26354593, 0.        , 0.457974  ,
        0.27848007, 0.        , 0.        ],
       [0.        , 0.        , 0.28964554, 0.        , 0.        ,
        0.29434005, 0.41601441, 0.        ],
       [0.        , 0.        , 0.29928281, 0.17504853, 0.17087565,
        0.35479302, 0.        , 0.        ],
       [0.        , 0.        , 0.44744447, 0.19812001, 0.        ,
        0.16195798, 0.19247753, 0.        ],
       [0.        , 0.        , 0.48998112, 0.        , 0.        ,
        0.        , 0.25446646, 0.25555242],
       [0.        , 0.        , 0.49146804, 0.        , 0.        ,
        0.50853196, 0.        , 0.        ],
       [0.        , 0.        , 0.49293863, 0.25779861, 0.24926276,
        0.        , 0.        , 0.        ],
       [0.        , 0.        , 0.49579016, 0.25261306, 0.        ,
        0.        , 0.        , 0.25159678],
       [0.        , 0.        , 0.49589178, 0.        , 0.        ,
        0.26108587, 0.        , 0.24302235],
       [0.        , 0.        , 0.49650692, 0.        , 0.25008785,
        0.        , 0.        , 0.25340523],
       [0.        , 0.        , 0.50104653, 0.        , 0.        ,
        0.        , 0.        , 0.49895347],
       [0.        , 0.        , 0.50189559, 0.        , 0.        ,
        0.        , 0.49810441, 0.        ],
       [0.        , 0.        , 0.50401986, 0.49598014, 0.        ,
        0.        , 0.        , 0.        ],
       [0.        , 0.        , 0.50620844, 0.        , 0.49379156,
        0.        , 0.        , 0.        ],
       [0.        , 0.        , 0.50675902, 0.        , 0.25187812,
        0.        , 0.24136286, 0.        ],
       [0.        , 0.        , 0.73386384, 0.        , 0.        ,
        0.26613616, 0.        , 0.        ],
       [0.        , 0.        , 0.73985647, 0.26014353, 0.        ,
        0.        , 0.        , 0.        ],
       [0.        , 0.        , 0.74197693, 0.        , 0.        ,
        0.        , 0.        , 0.25802307],
       [0.        , 0.        , 0.74731093, 0.        , 0.        ,
        0.        , 0.25268907, 0.        ],
       [0.        , 0.        , 0.75249986, 0.        , 0.24750014,
        0.        , 0.        , 0.        ],
       [0.        , 0.        , 1.        , 0.        , 0.        ,
        0.        , 0.        , 0.        ],
       [0.        , 0.12677265, 0.2770548 , 0.        , 0.21617872,
        0.185595  , 0.19439883, 0.        ],
       [0.        , 0.16826595, 0.        , 0.44233484, 0.20482959,
        0.        , 0.18456961, 0.        ],
       [0.        , 0.18211139, 0.20494572, 0.        , 0.        ,
        0.45316485, 0.15977803, 0.        ],
       [0.        , 0.19332173, 0.25343972, 0.15297025, 0.21396653,
        0.        , 0.        , 0.18630178],
       [0.        , 0.19482084, 0.        , 0.34506826, 0.        ,
        0.4601109 , 0.        , 0.        ],
       [0.        , 0.19864791, 0.        , 0.20168575, 0.        ,
        0.        , 0.16202372, 0.43764262],
       [0.        , 0.20925465, 0.        , 0.10146144, 0.        ,
        0.46555442, 0.        , 0.22372948],
       [0.        , 0.20963805, 0.        , 0.28802884, 0.        ,
        0.17221811, 0.19030795, 0.13980705],
       [0.        , 0.23840467, 0.25321261, 0.25334482, 0.        ,
        0.        , 0.25503789, 0.        ],
       [0.        , 0.23989792, 0.2502516 , 0.        , 0.        ,
        0.25760802, 0.        , 0.25224245],
       [0.        , 0.24085471, 0.        , 0.        , 0.24227784,
        0.51686745, 0.        , 0.        ],
       [0.        , 0.24123483, 0.51055358, 0.        , 0.24821159,
        0.        , 0.        , 0.        ],
       [0.        , 0.24141427, 0.        , 0.        , 0.25065117,
        0.        , 0.25358981, 0.25434474],
       [0.        , 0.24389267, 0.        , 0.        , 0.        ,
        0.25763548, 0.2560047 , 0.24246715],
       [0.        , 0.24480292, 0.        , 0.        , 0.        ,
        0.25570788, 0.49948919, 0.        ],
       [0.        , 0.24481515, 0.        , 0.        , 0.        ,
        0.        , 0.50149985, 0.253685  ],
       [0.        , 0.24518135, 0.        , 0.50083068, 0.        ,
        0.        , 0.        , 0.25398797],
       [0.        , 0.24573194, 0.        , 0.        , 0.49697113,
        0.        , 0.25729693, 0.        ],
       [0.        , 0.24598158, 0.        , 0.75401842, 0.        ,
        0.        , 0.        , 0.        ],
       [0.        , 0.24621828, 0.25766765, 0.24667839, 0.        ,
        0.24943568, 0.        , 0.        ],
       [0.        , 0.24663386, 0.        , 0.26224661, 0.2487279 ,
        0.24239163, 0.        , 0.        ],
       [0.        , 0.24768385, 0.        , 0.        , 0.23893588,
        0.        , 0.        , 0.51338028],
       [0.        , 0.24784482, 0.        , 0.        , 0.        ,
        0.        , 0.        , 0.75215518],
       [0.        , 0.24886014, 0.75113986, 0.        , 0.        ,
        0.        , 0.        , 0.        ],
       [0.        , 0.24895498, 0.        , 0.        , 0.25089366,
        0.24712641, 0.        , 0.25302495],
       [0.        , 0.25026396, 0.        , 0.        , 0.2513398 ,
        0.        , 0.49839625, 0.        ],
       [0.        , 0.25040845, 0.        , 0.26571021, 0.48388134,
        0.        , 0.        , 0.        ],
       [0.        , 0.2508069 , 0.        , 0.        , 0.        ,
        0.        , 0.7491931 , 0.        ],
       [0.        , 0.25195943, 0.47966162, 0.        , 0.        ,
        0.        , 0.        , 0.26837894],
       [0.        , 0.25250269, 0.        , 0.        , 0.        ,
        0.24302229, 0.        , 0.50447502],
       [0.        , 0.25304726, 0.49273996, 0.25421279, 0.        ,
        0.        , 0.        , 0.        ],
       [0.        , 0.25321786, 0.24529409, 0.50148805, 0.        ,
        0.        , 0.        , 0.        ],
       [0.        , 0.25350069, 0.24404785, 0.        , 0.        ,
        0.        , 0.25177762, 0.25067383],
       [0.        , 0.25464202, 0.        , 0.        , 0.74535798,
        0.        , 0.        , 0.        ],
       [0.        , 0.25633357, 0.49198425, 0.        , 0.        ,
        0.        , 0.25168218, 0.        ],
       [0.        , 0.25724627, 0.        , 0.        , 0.48962887,
        0.25312486, 0.        , 0.        ],
       [0.        , 0.25839143, 0.25169321, 0.        , 0.        ,
        0.        , 0.48991536, 0.        ],
       [0.        , 0.2586288 , 0.        , 0.        , 0.48365971,
        0.        , 0.        , 0.25771149],
       [0.        , 0.25910842, 0.2536643 , 0.        , 0.48722728,
        0.        , 0.        , 0.        ],
       [0.        , 0.25947763, 0.        , 0.        , 0.2348493 ,
        0.2624558 , 0.24321728, 0.        ],
       [0.        , 0.41874764, 0.27232303, 0.        , 0.17418153,
        0.        , 0.1347478 , 0.        ],
       [0.        , 0.45862268, 0.        , 0.27807996, 0.        ,
        0.        , 0.26329736, 0.        ],
       [0.        , 0.46587596, 0.26846337, 0.        , 0.        ,
        0.26566067, 0.        , 0.        ],
       [0.        , 0.47736095, 0.        , 0.26036428, 0.        ,
        0.        , 0.        , 0.26227478],
       [0.        , 0.48089712, 0.        , 0.        , 0.        ,
        0.51910288, 0.        , 0.        ],
       [0.        , 0.48262867, 0.        , 0.        , 0.26142677,
        0.        , 0.25594456, 0.        ],
       [0.        , 0.48518758, 0.51481242, 0.        , 0.        ,
        0.        , 0.        , 0.        ],
       [0.        , 0.48532659, 0.        , 0.26687286, 0.        ,
        0.24780055, 0.        , 0.        ],
       [0.        , 0.48547451, 0.        , 0.        , 0.        ,
        0.26556571, 0.        , 0.24895978],
       [0.        , 0.48939187, 0.        , 0.51060813, 0.        ,
        0.        , 0.        , 0.        ],
       [0.        , 0.48965888, 0.        , 0.        , 0.        ,
        0.        , 0.        , 0.51034112],
       [0.        , 0.49240816, 0.        , 0.25447452, 0.25311733,
        0.        , 0.        , 0.        ],
       [0.        , 0.4929668 , 0.        , 0.        , 0.        ,
        0.        , 0.25024854, 0.25678466],
       [0.        , 0.49506226, 0.        , 0.        , 0.        ,
        0.25542892, 0.24950882, 0.        ],
       [0.        , 0.49560919, 0.        , 0.        , 0.        ,
        0.        , 0.50439081, 0.        ],
       [0.        , 0.49679388, 0.        , 0.        , 0.50320612,
        0.        , 0.        , 0.        ],
       [0.        , 0.49729963, 0.24616574, 0.25653464, 0.        ,
        0.        , 0.        , 0.        ],
       [0.        , 0.49849623, 0.        , 0.        , 0.24944053,
        0.        , 0.        , 0.25206324],
       [0.        , 0.49940262, 0.        , 0.        , 0.24499766,
        0.25559972, 0.        , 0.        ],
       [0.        , 0.50198764, 0.24802128, 0.        , 0.        ,
        0.        , 0.        , 0.24999107],
       [0.        , 0.72507892, 0.27492108, 0.        , 0.        ,
        0.        , 0.        , 0.        ],
       [0.        , 0.73739162, 0.        , 0.        , 0.        ,
        0.        , 0.        , 0.26260838],
       [0.        , 0.73749864, 0.        , 0.        , 0.        ,
        0.        , 0.26250136, 0.        ],
       [0.        , 0.7389399 , 0.        , 0.        , 0.2610601 ,
        0.        , 0.        , 0.        ],
       [0.        , 0.74242827, 0.        , 0.        , 0.        ,
        0.25757173, 0.        , 0.        ],
       [0.        , 0.7447764 , 0.        , 0.2552236 , 0.        ,
        0.        , 0.        , 0.        ],
       [0.        , 1.        , 0.        , 0.        , 0.        ,
        0.        , 0.        , 0.        ],
       [0.15988693, 0.1936855 , 0.        , 0.29076761, 0.18953066,
        0.        , 0.        , 0.1661293 ],
       [0.1652404 , 0.        , 0.        , 0.        , 0.        ,
        0.19568696, 0.45492833, 0.1841443 ],
       [0.17118294, 0.        , 0.        , 0.1498568 , 0.        ,
        0.29822994, 0.12331584, 0.25741448],
       [0.18714442, 0.        , 0.45684316, 0.        , 0.18133544,
        0.17467698, 0.        , 0.        ],
       [0.2002489 , 0.21989169, 0.        , 0.12834624, 0.22241004,
        0.        , 0.22910313, 0.        ],
       [0.20053757, 0.        , 0.24007431, 0.19734626, 0.18203935,
        0.        , 0.18000251, 0.        ],
       [0.20058727, 0.        , 0.25772879, 0.21275025, 0.        ,
        0.14935732, 0.        , 0.17957638],
       [0.20726877, 0.        , 0.15716155, 0.        , 0.25398163,
        0.20222843, 0.        , 0.17935962],
       [0.21089534, 0.19086324, 0.1332355 , 0.        , 0.        ,
        0.17806542, 0.28694049, 0.        ],
       [0.2367743 , 0.25365977, 0.        , 0.25426659, 0.        ,
        0.25529934, 0.        , 0.        ],
       [0.2379375 , 0.25267255, 0.25273177, 0.        , 0.25665817,
        0.        , 0.        , 0.        ],
       [0.23954642, 0.24484097, 0.        , 0.        , 0.        ,
        0.        , 0.25556193, 0.26005068],
       [0.24095785, 0.24805875, 0.24251861, 0.        , 0.        ,
        0.        , 0.        , 0.26846479],
       [0.24158079, 0.        , 0.        , 0.        , 0.        ,
        0.        , 0.25010447, 0.50831474],
       [0.24340937, 0.        , 0.        , 0.        , 0.25316412,
        0.5034265 , 0.        , 0.        ],
       [0.24454925, 0.4885479 , 0.        , 0.        , 0.        ,
        0.        , 0.        , 0.26690285],
       [0.24713715, 0.        , 0.        , 0.49950407, 0.        ,
        0.        , 0.25335878, 0.        ],
       [0.24725854, 0.        , 0.26946473, 0.48327673, 0.        ,
        0.        , 0.        , 0.        ],
       [0.24819988, 0.        , 0.50146377, 0.        , 0.        ,
        0.        , 0.25033635, 0.        ],
       [0.24851738, 0.        , 0.        , 0.        , 0.25189106,
        0.        , 0.49959155, 0.        ],
       [0.24902456, 0.49573624, 0.        , 0.        , 0.        ,
        0.        , 0.2552392 , 0.        ],
       [0.24967236, 0.        , 0.        , 0.25172723, 0.        ,
        0.        , 0.49860041, 0.        ],
       [0.24969201, 0.        , 0.        , 0.25129178, 0.        ,
        0.        , 0.2507179 , 0.24829832],
       [0.25015812, 0.50147441, 0.        , 0.        , 0.24836746,
        0.        , 0.        , 0.        ],
       [0.2504826 , 0.        , 0.        , 0.48754472, 0.        ,
        0.        , 0.        , 0.26197268],
       [0.25076731, 0.        , 0.25174052, 0.        , 0.        ,
        0.        , 0.49749217, 0.        ],
       [0.25101347, 0.        , 0.24605802, 0.        , 0.        ,
        0.        , 0.25483389, 0.24809461],
       [0.25181631, 0.        , 0.        , 0.25233505, 0.49584863,
        0.        , 0.        , 0.        ],
       [0.25221241, 0.        , 0.        , 0.        , 0.24875431,
        0.2514825 , 0.24755077, 0.        ],
       [0.25227027, 0.        , 0.        , 0.        , 0.24867882,
        0.        , 0.25544364, 0.24360728],
       [0.25296132, 0.        , 0.        , 0.2486854 , 0.        ,
        0.24124874, 0.25710454, 0.        ],
       [0.25324467, 0.        , 0.        , 0.        , 0.        ,
        0.48970833, 0.257047  , 0.        ],
       [0.25348339, 0.49111114, 0.25540547, 0.        , 0.        ,
        0.        , 0.        , 0.        ],
       [0.25378804, 0.25066223, 0.        , 0.        , 0.49554973,
        0.        , 0.        , 0.        ],
       [0.25434697, 0.        , 0.        , 0.        , 0.        ,
        0.        , 0.        , 0.74565303],
       [0.25449975, 0.        , 0.        , 0.49035998, 0.25514026,
        0.        , 0.        , 0.        ],
       [0.25467525, 0.24362568, 0.50169907, 0.        , 0.        ,
        0.        , 0.        , 0.        ],
       [0.25474962, 0.74525038, 0.        , 0.        , 0.        ,
        0.        , 0.        , 0.        ],
       [0.25500531, 0.        , 0.        , 0.        , 0.49271101,
        0.        , 0.25228368, 0.        ],
       [0.25649946, 0.        , 0.25947144, 0.        , 0.        ,
        0.4840291 , 0.        , 0.        ],
       [0.25665395, 0.24242082, 0.        , 0.        , 0.24047522,
        0.        , 0.        , 0.26045001],
       [0.2567887 , 0.        , 0.        , 0.        , 0.7432113 ,
        0.        , 0.        , 0.        ],
       [0.25680138, 0.24998025, 0.2461029 , 0.24711547, 0.        ,
        0.        , 0.        , 0.        ],
       [0.25744933, 0.        , 0.        , 0.        , 0.        ,
        0.        , 0.74255067, 0.        ],
       [0.25857018, 0.2265241 , 0.26920609, 0.        , 0.        ,
        0.24569963, 0.        , 0.        ],
       [0.2586659 , 0.2619691 , 0.        , 0.        , 0.23041826,
        0.24894674, 0.        , 0.        ],
       [0.25916642, 0.25448566, 0.        , 0.        , 0.        ,
        0.48634792, 0.        , 0.        ],
       [0.25947738, 0.        , 0.        , 0.        , 0.47834972,
        0.        , 0.        , 0.2621729 ],
       [0.25949324, 0.49241305, 0.        , 0.24809371, 0.        ,
        0.        , 0.        , 0.        ],
       [0.25964446, 0.        , 0.        , 0.        , 0.        ,
        0.74035554, 0.        , 0.        ],
       [0.26083171, 0.        , 0.        , 0.73916829, 0.        ,
        0.        , 0.        , 0.        ],
       [0.26103345, 0.        , 0.24640322, 0.        , 0.        ,
        0.        , 0.        , 0.49256333],
       [0.26136248, 0.        , 0.        , 0.        , 0.        ,
        0.24318626, 0.        , 0.49545126],
       [0.26180915, 0.        , 0.        , 0.48228832, 0.        ,
        0.25590253, 0.        , 0.        ],
       [0.26187803, 0.        , 0.4886886 , 0.        , 0.        ,
        0.        , 0.        , 0.24943337],
       [0.26224578, 0.25297512, 0.        , 0.4847791 , 0.        ,
        0.        , 0.        , 0.        ],
       [0.27471408, 0.        , 0.        , 0.16768802, 0.15861962,
        0.        , 0.        , 0.39897827],
       [0.28932769, 0.        , 0.        , 0.23422727, 0.        ,
        0.47644504, 0.        , 0.        ],
       [0.42778969, 0.        , 0.        , 0.        , 0.        ,
        0.37423166, 0.        , 0.19797865],
       [0.44050274, 0.        , 0.27616269, 0.        , 0.28333457,
        0.        , 0.        , 0.        ],
       [0.47066739, 0.        , 0.20882423, 0.        , 0.        ,
        0.10822462, 0.        , 0.21228376],
       [0.47757358, 0.2530415 , 0.        , 0.        , 0.        ,
        0.        , 0.26938492, 0.        ],
       [0.48468095, 0.        , 0.26794343, 0.24737562, 0.        ,
        0.        , 0.        , 0.        ],
       [0.48485115, 0.        , 0.        , 0.        , 0.25538183,
        0.25976701, 0.        , 0.        ],
       [0.48518736, 0.        , 0.        , 0.        , 0.        ,
        0.25454989, 0.26026275, 0.        ],
       [0.48897729, 0.        , 0.        , 0.        , 0.        ,
        0.        , 0.51102271, 0.        ],
       [0.49015497, 0.        , 0.25528623, 0.        , 0.        ,
        0.        , 0.25455879, 0.        ],
       [0.49072759, 0.        , 0.        , 0.2543653 , 0.        ,
        0.        , 0.25490711, 0.        ],
       [0.49147062, 0.        , 0.        , 0.        , 0.        ,
        0.        , 0.25333985, 0.25518953],
       [0.49262548, 0.25487479, 0.        , 0.        , 0.        ,
        0.        , 0.        , 0.25249973],
       [0.49601382, 0.        , 0.        , 0.24817161, 0.25581458,
        0.        , 0.        , 0.        ],
       [0.49637373, 0.        , 0.        , 0.        , 0.25179232,
        0.        , 0.25183396, 0.        ],
       [0.49714105, 0.        , 0.        , 0.        , 0.50285895,
        0.        , 0.        , 0.        ],
       [0.49733045, 0.25084386, 0.25182569, 0.        , 0.        ,
        0.        , 0.        , 0.        ],
       [0.49778873, 0.25724083, 0.        , 0.        , 0.24497044,
        0.        , 0.        , 0.        ],
       [0.49793501, 0.        , 0.50206499, 0.        , 0.        ,
        0.        , 0.        , 0.        ],
       [0.49906197, 0.50093803, 0.        , 0.        , 0.        ,
        0.        , 0.        , 0.        ],
       [0.49915177, 0.        , 0.        , 0.25802775, 0.        ,
        0.        , 0.        , 0.24282048],
       [0.50104611, 0.        , 0.        , 0.49895389, 0.        ,
        0.        , 0.        , 0.        ],
       [0.50900499, 0.        , 0.        , 0.        , 0.        ,
        0.        , 0.        , 0.49099501],
       [0.51090541, 0.        , 0.        , 0.        , 0.25046314,
        0.        , 0.        , 0.23863145],
       [0.5129691 , 0.16335315, 0.        , 0.16549313, 0.        ,
        0.15818462, 0.        , 0.        ],
       [0.72011888, 0.        , 0.        , 0.        , 0.        ,
        0.27988112, 0.        , 0.        ],
       [0.72814268, 0.        , 0.        , 0.        , 0.        ,
        0.        , 0.27185732, 0.        ],
       [0.74012066, 0.        , 0.        , 0.25987934, 0.        ,
        0.        , 0.        , 0.        ],
       [0.74728141, 0.        , 0.25271859, 0.        , 0.        ,
        0.        , 0.        , 0.        ],
       [0.74851946, 0.        , 0.        , 0.        , 0.        ,
        0.        , 0.        , 0.25148054],
       [0.76091231, 0.23908769, 0.        , 0.        , 0.        ,
        0.        , 0.        , 0.        ],
       [1.        , 0.        , 0.        , 0.        , 0.        ,
        0.        , 0.        , 0.        ]])
    elif dim == 12 and num_weights == 256:
        return np.array([[0.        , 0.        , 0.        , 0.        , 0.        ,
        0.        , 0.        , 0.        , 0.        , 0.        ,
        0.        , 1.        ],
       [0.        , 0.        , 0.        , 0.        , 0.        ,
        0.        , 0.        , 0.        , 0.        , 0.        ,
        0.33296206, 0.66703794],
       [0.        , 0.        , 0.        , 0.        , 0.        ,
        0.        , 0.        , 0.        , 0.        , 0.        ,
        0.67398528, 0.32601472],
       [0.        , 0.        , 0.        , 0.        , 0.        ,
        0.        , 0.        , 0.        , 0.        , 0.        ,
        1.        , 0.        ],
       [0.        , 0.        , 0.        , 0.        , 0.        ,
        0.        , 0.        , 0.        , 0.        , 0.33420259,
        0.34036169, 0.32543572],
       [0.        , 0.        , 0.        , 0.        , 0.        ,
        0.        , 0.        , 0.        , 0.        , 0.66387804,
        0.33612196, 0.        ],
       [0.        , 0.        , 0.        , 0.        , 0.        ,
        0.        , 0.        , 0.        , 0.        , 0.66452056,
        0.        , 0.33547944],
       [0.        , 0.        , 0.        , 0.        , 0.        ,
        0.        , 0.        , 0.        , 0.        , 1.        ,
        0.        , 0.        ],
       [0.        , 0.        , 0.        , 0.        , 0.        ,
        0.        , 0.        , 0.        , 0.33006092, 0.32818164,
        0.34175744, 0.        ],
       [0.        , 0.        , 0.        , 0.        , 0.        ,
        0.        , 0.        , 0.        , 0.33455646, 0.        ,
        0.3289616 , 0.33648194],
       [0.        , 0.        , 0.        , 0.        , 0.        ,
        0.        , 0.        , 0.        , 0.33476789, 0.        ,
        0.66523211, 0.        ],
       [0.        , 0.        , 0.        , 0.        , 0.        ,
        0.        , 0.        , 0.        , 0.33636364, 0.        ,
        0.        , 0.66363636],
       [0.        , 0.        , 0.        , 0.        , 0.        ,
        0.        , 0.        , 0.        , 0.66182231, 0.        ,
        0.33817769, 0.        ],
       [0.        , 0.        , 0.        , 0.        , 0.        ,
        0.        , 0.        , 0.        , 0.6666444 , 0.3333556 ,
        0.        , 0.        ],
       [0.        , 0.        , 0.        , 0.        , 0.        ,
        0.        , 0.        , 0.        , 1.        , 0.        ,
        0.        , 0.        ],
       [0.        , 0.        , 0.        , 0.        , 0.        ,
        0.        , 0.        , 0.33183711, 0.        , 0.33814014,
        0.        , 0.33002275],
       [0.        , 0.        , 0.        , 0.        , 0.        ,
        0.        , 0.        , 0.33306442, 0.66693558, 0.        ,
        0.        , 0.        ],
       [0.        , 0.        , 0.        , 0.        , 0.        ,
        0.        , 0.        , 0.33456281, 0.        , 0.        ,
        0.        , 0.66543719],
       [0.        , 0.        , 0.        , 0.        , 0.        ,
        0.        , 0.        , 0.33897592, 0.32337309, 0.        ,
        0.33765099, 0.        ],
       [0.        , 0.        , 0.        , 0.        , 0.        ,
        0.        , 0.        , 0.6490477 , 0.        , 0.3509523 ,
        0.        , 0.        ],
       [0.        , 0.        , 0.        , 0.        , 0.        ,
        0.        , 0.        , 0.65845497, 0.        , 0.        ,
        0.        , 0.34154503],
       [0.        , 0.        , 0.        , 0.        , 0.        ,
        0.        , 0.        , 0.65916494, 0.        , 0.        ,
        0.34083506, 0.        ],
       [0.        , 0.        , 0.        , 0.        , 0.        ,
        0.        , 0.        , 0.66105007, 0.33894993, 0.        ,
        0.        , 0.        ],
       [0.        , 0.        , 0.        , 0.        , 0.        ,
        0.        , 0.        , 1.        , 0.        , 0.        ,
        0.        , 0.        ],
       [0.        , 0.        , 0.        , 0.        , 0.        ,
        0.        , 0.3149628 , 0.31851349, 0.        , 0.        ,
        0.36652371, 0.        ],
       [0.        , 0.        , 0.        , 0.        , 0.        ,
        0.        , 0.32592755, 0.        , 0.67407245, 0.        ,
        0.        , 0.        ],
       [0.        , 0.        , 0.        , 0.        , 0.        ,
        0.        , 0.327781  , 0.33253291, 0.        , 0.        ,
        0.        , 0.33968609],
       [0.        , 0.        , 0.        , 0.        , 0.        ,
        0.        , 0.33048106, 0.        , 0.        , 0.66951894,
        0.        , 0.        ],
       [0.        , 0.        , 0.        , 0.        , 0.        ,
        0.        , 0.33074395, 0.        , 0.33200608, 0.        ,
        0.        , 0.33724997],
       [0.        , 0.        , 0.        , 0.        , 0.        ,
        0.        , 0.33208044, 0.33087705, 0.33704251, 0.        ,
        0.        , 0.        ],
       [0.        , 0.        , 0.        , 0.        , 0.        ,
        0.        , 0.33471736, 0.        , 0.3239365 , 0.34134614,
        0.        , 0.        ],
       [0.        , 0.        , 0.        , 0.        , 0.        ,
        0.        , 0.33933355, 0.        , 0.        , 0.33243021,
        0.        , 0.32823624],
       [0.        , 0.        , 0.        , 0.        , 0.        ,
        0.        , 0.34418927, 0.65581073, 0.        , 0.        ,
        0.        , 0.        ],
       [0.        , 0.        , 0.        , 0.        , 0.        ,
        0.        , 0.65593128, 0.        , 0.        , 0.        ,
        0.        , 0.34406872],
       [0.        , 0.        , 0.        , 0.        , 0.        ,
        0.        , 0.6561946 , 0.        , 0.        , 0.        ,
        0.3438054 , 0.        ],
       [0.        , 0.        , 0.        , 0.        , 0.        ,
        0.        , 0.65931696, 0.        , 0.        , 0.34068304,
        0.        , 0.        ],
       [0.        , 0.        , 0.        , 0.        , 0.        ,
        0.        , 0.66418623, 0.33581377, 0.        , 0.        ,
        0.        , 0.        ],
       [0.        , 0.        , 0.        , 0.        , 0.        ,
        0.        , 0.66665661, 0.        , 0.33334339, 0.        ,
        0.        , 0.        ],
       [0.        , 0.        , 0.        , 0.        , 0.        ,
        0.        , 1.        , 0.        , 0.        , 0.        ,
        0.        , 0.        ],
       [0.        , 0.        , 0.        , 0.        , 0.        ,
        0.32540211, 0.        , 0.        , 0.        , 0.33776631,
        0.        , 0.33683158],
       [0.        , 0.        , 0.        , 0.        , 0.        ,
        0.32878342, 0.        , 0.        , 0.        , 0.34073616,
        0.33048042, 0.        ],
       [0.        , 0.        , 0.        , 0.        , 0.        ,
        0.32933833, 0.32595312, 0.        , 0.        , 0.        ,
        0.        , 0.34470855],
       [0.        , 0.        , 0.        , 0.        , 0.        ,
        0.32992033, 0.        , 0.        , 0.        , 0.        ,
        0.        , 0.67007967],
       [0.        , 0.        , 0.        , 0.        , 0.        ,
        0.33015171, 0.        , 0.        , 0.        , 0.        ,
        0.33055577, 0.33929252],
       [0.        , 0.        , 0.        , 0.        , 0.        ,
        0.33272037, 0.34019618, 0.        , 0.32708344, 0.        ,
        0.        , 0.        ],
       [0.        , 0.        , 0.        , 0.        , 0.        ,
        0.33308916, 0.33339166, 0.33351918, 0.        , 0.        ,
        0.        , 0.        ],
       [0.        , 0.        , 0.        , 0.        , 0.        ,
        0.33460524, 0.33681605, 0.        , 0.        , 0.32857871,
        0.        , 0.        ],
       [0.        , 0.        , 0.        , 0.        , 0.        ,
        0.33601433, 0.33405208, 0.        , 0.        , 0.        ,
        0.32993359, 0.        ],
       [0.        , 0.        , 0.        , 0.        , 0.        ,
        0.33630077, 0.        , 0.        , 0.66369923, 0.        ,
        0.        , 0.        ],
       [0.        , 0.        , 0.        , 0.        , 0.        ,
        0.33756374, 0.        , 0.        , 0.        , 0.66243626,
        0.        , 0.        ],
       [0.        , 0.        , 0.        , 0.        , 0.        ,
        0.34132354, 0.        , 0.3305351 , 0.32814136, 0.        ,
        0.        , 0.        ],
       [0.        , 0.        , 0.        , 0.        , 0.        ,
        0.34199791, 0.        , 0.        , 0.3255943 , 0.        ,
        0.33240779, 0.        ],
       [0.        , 0.        , 0.        , 0.        , 0.        ,
        0.34416084, 0.        , 0.65583916, 0.        , 0.        ,
        0.        , 0.        ],
       [0.        , 0.        , 0.        , 0.        , 0.        ,
        0.34503562, 0.65496438, 0.        , 0.        , 0.        ,
        0.        , 0.        ],
       [0.        , 0.        , 0.        , 0.        , 0.        ,
        0.36350403, 0.        , 0.        , 0.32233128, 0.3141647 ,
        0.        , 0.        ],
       [0.        , 0.        , 0.        , 0.        , 0.        ,
        0.5485144 , 0.        , 0.21974088, 0.        , 0.        ,
        0.        , 0.23174473],
       [0.        , 0.        , 0.        , 0.        , 0.        ,
        0.66222739, 0.        , 0.        , 0.        , 0.        ,
        0.33777261, 0.        ],
       [0.        , 0.        , 0.        , 0.        , 0.        ,
        0.67215092, 0.32784908, 0.        , 0.        , 0.        ,
        0.        , 0.        ],
       [0.        , 0.        , 0.        , 0.        , 0.        ,
        1.        , 0.        , 0.        , 0.        , 0.        ,
        0.        , 0.        ],
       [0.        , 0.        , 0.        , 0.        , 0.23312389,
        0.        , 0.25809052, 0.        , 0.        , 0.25471912,
        0.25406648, 0.        ],
       [0.        , 0.        , 0.        , 0.        , 0.24325544,
        0.24461522, 0.        , 0.        , 0.25212826, 0.        ,
        0.        , 0.26000108],
       [0.        , 0.        , 0.        , 0.        , 0.32968341,
        0.        , 0.        , 0.        , 0.34667043, 0.32364616,
        0.        , 0.        ],
       [0.        , 0.        , 0.        , 0.        , 0.3320165 ,
        0.        , 0.        , 0.33247244, 0.        , 0.33551106,
        0.        , 0.        ],
       [0.        , 0.        , 0.        , 0.        , 0.33268514,
        0.        , 0.34030211, 0.32701275, 0.        , 0.        ,
        0.        , 0.        ],
       [0.        , 0.        , 0.        , 0.        , 0.33326307,
        0.        , 0.        , 0.        , 0.        , 0.        ,
        0.33480108, 0.33193585],
       [0.        , 0.        , 0.        , 0.        , 0.33379309,
        0.        , 0.32175778, 0.        , 0.34444914, 0.        ,
        0.        , 0.        ],
       [0.        , 0.        , 0.        , 0.        , 0.33388415,
        0.        , 0.        , 0.        , 0.32755008, 0.        ,
        0.33856577, 0.        ],
       [0.        , 0.        , 0.        , 0.        , 0.33424274,
        0.        , 0.        , 0.        , 0.        , 0.        ,
        0.66575726, 0.        ],
       [0.        , 0.        , 0.        , 0.        , 0.33601582,
        0.33084759, 0.        , 0.        , 0.        , 0.33313659,
        0.        , 0.        ],
       [0.        , 0.        , 0.        , 0.        , 0.33630192,
        0.33149175, 0.        , 0.33220633, 0.        , 0.        ,
        0.        , 0.        ],
       [0.        , 0.        , 0.        , 0.        , 0.33696342,
        0.        , 0.        , 0.        , 0.        , 0.66303658,
        0.        , 0.        ],
       [0.        , 0.        , 0.        , 0.        , 0.33720875,
        0.        , 0.        , 0.        , 0.        , 0.33058889,
        0.        , 0.33220236],
       [0.        , 0.        , 0.        , 0.        , 0.33822447,
        0.33454535, 0.        , 0.        , 0.        , 0.        ,
        0.32723018, 0.        ],
       [0.        , 0.        , 0.        , 0.        , 0.34214035,
        0.        , 0.65785965, 0.        , 0.        , 0.        ,
        0.        , 0.        ],
       [0.        , 0.        , 0.        , 0.        , 0.34390781,
        0.65609219, 0.        , 0.        , 0.        , 0.        ,
        0.        , 0.        ],
       [0.        , 0.        , 0.        , 0.        , 0.35008526,
        0.        , 0.        , 0.64991474, 0.        , 0.        ,
        0.        , 0.        ],
       [0.        , 0.        , 0.        , 0.        , 0.65106855,
        0.        , 0.        , 0.        , 0.        , 0.        ,
        0.        , 0.34893145],
       [0.        , 0.        , 0.        , 0.        , 0.6619849 ,
        0.        , 0.        , 0.        , 0.3380151 , 0.        ,
        0.        , 0.        ],
       [0.        , 0.        , 0.        , 0.        , 0.66246637,
        0.        , 0.        , 0.        , 0.        , 0.        ,
        0.33753363, 0.        ],
       [0.        , 0.        , 0.        , 0.        , 0.66957358,
        0.        , 0.33042642, 0.        , 0.        , 0.        ,
        0.        , 0.        ],
       [0.        , 0.        , 0.        , 0.        , 0.67039973,
        0.        , 0.        , 0.32960027, 0.        , 0.        ,
        0.        , 0.        ],
       [0.        , 0.        , 0.        , 0.        , 1.        ,
        0.        , 0.        , 0.        , 0.        , 0.        ,
        0.        , 0.        ],
       [0.        , 0.        , 0.        , 0.25796442, 0.50522245,
        0.        , 0.        , 0.        , 0.        , 0.23681313,
        0.        , 0.        ],
       [0.        , 0.        , 0.        , 0.28867661, 0.        ,
        0.19888077, 0.        , 0.26157971, 0.        , 0.        ,
        0.        , 0.25086291],
       [0.        , 0.        , 0.        , 0.31996034, 0.        ,
        0.32998783, 0.        , 0.        , 0.        , 0.        ,
        0.35005183, 0.        ],
       [0.        , 0.        , 0.        , 0.32379303, 0.        ,
        0.        , 0.        , 0.33181192, 0.        , 0.        ,
        0.34439505, 0.        ],
       [0.        , 0.        , 0.        , 0.32700885, 0.        ,
        0.        , 0.32754364, 0.        , 0.        , 0.        ,
        0.34544751, 0.        ],
       [0.        , 0.        , 0.        , 0.32704204, 0.        ,
        0.        , 0.        , 0.33271661, 0.34024136, 0.        ,
        0.        , 0.        ],
       [0.        , 0.        , 0.        , 0.32777453, 0.        ,
        0.        , 0.        , 0.33001806, 0.        , 0.34220741,
        0.        , 0.        ],
       [0.        , 0.        , 0.        , 0.32891238, 0.        ,
        0.        , 0.        , 0.        , 0.        , 0.        ,
        0.        , 0.67108762],
       [0.        , 0.        , 0.        , 0.33146887, 0.        ,
        0.        , 0.32925262, 0.        , 0.33927851, 0.        ,
        0.        , 0.        ],
       [0.        , 0.        , 0.        , 0.33161636, 0.        ,
        0.        , 0.        , 0.        , 0.        , 0.        ,
        0.33076403, 0.33761961],
       [0.        , 0.        , 0.        , 0.33166401, 0.33403679,
        0.        , 0.        , 0.        , 0.        , 0.        ,
        0.3342992 , 0.        ],
       [0.        , 0.        , 0.        , 0.33214129, 0.32191289,
        0.        , 0.        , 0.        , 0.        , 0.        ,
        0.        , 0.34594582],
       [0.        , 0.        , 0.        , 0.33222481, 0.        ,
        0.        , 0.33312779, 0.        , 0.        , 0.3346474 ,
        0.        , 0.        ],
       [0.        , 0.        , 0.        , 0.33354246, 0.        ,
        0.        , 0.3237435 , 0.        , 0.        , 0.        ,
        0.        , 0.34271404],
       [0.        , 0.        , 0.        , 0.33373613, 0.        ,
        0.        , 0.        , 0.        , 0.        , 0.33901753,
        0.        , 0.32724634],
       [0.        , 0.        , 0.        , 0.33462031, 0.33425747,
        0.        , 0.        , 0.        , 0.33112221, 0.        ,
        0.        , 0.        ],
       [0.        , 0.        , 0.        , 0.33463247, 0.        ,
        0.        , 0.33025772, 0.33510982, 0.        , 0.        ,
        0.        , 0.        ],
       [0.        , 0.        , 0.        , 0.33611159, 0.        ,
        0.        , 0.        , 0.        , 0.        , 0.66388841,
        0.        , 0.        ],
       [0.        , 0.        , 0.        , 0.33740431, 0.        ,
        0.        , 0.        , 0.        , 0.32895508, 0.33364061,
        0.        , 0.        ],
       [0.        , 0.        , 0.        , 0.34126369, 0.        ,
        0.        , 0.65873631, 0.        , 0.        , 0.        ,
        0.        , 0.        ],
       [0.        , 0.        , 0.        , 0.34500248, 0.        ,
        0.65499752, 0.        , 0.        , 0.        , 0.        ,
        0.        , 0.        ],
       [0.        , 0.        , 0.        , 0.3453118 , 0.        ,
        0.        , 0.        , 0.6546882 , 0.        , 0.        ,
        0.        , 0.        ],
       [0.        , 0.        , 0.        , 0.35098972, 0.        ,
        0.32794938, 0.3210609 , 0.        , 0.        , 0.        ,
        0.        , 0.        ],
       [0.        , 0.        , 0.        , 0.65442396, 0.        ,
        0.        , 0.        , 0.        , 0.        , 0.        ,
        0.34557604, 0.        ],
       [0.        , 0.        , 0.        , 0.65483598, 0.34516402,
        0.        , 0.        , 0.        , 0.        , 0.        ,
        0.        , 0.        ],
       [0.        , 0.        , 0.        , 0.6586165 , 0.        ,
        0.        , 0.        , 0.        , 0.3413835 , 0.        ,
        0.        , 0.        ],
       [0.        , 0.        , 0.        , 0.66021741, 0.        ,
        0.        , 0.        , 0.        , 0.        , 0.        ,
        0.        , 0.33978259],
       [0.        , 0.        , 0.        , 0.66669676, 0.        ,
        0.        , 0.        , 0.33330324, 0.        , 0.        ,
        0.        , 0.        ],
       [0.        , 0.        , 0.        , 0.66754081, 0.        ,
        0.        , 0.33245919, 0.        , 0.        , 0.        ,
        0.        , 0.        ],
       [0.        , 0.        , 0.        , 1.        , 0.        ,
        0.        , 0.        , 0.        , 0.        , 0.        ,
        0.        , 0.        ],
       [0.        , 0.        , 0.20421349, 0.22149237, 0.        ,
        0.19580077, 0.        , 0.        , 0.37849337, 0.        ,
        0.        , 0.        ],
       [0.        , 0.        , 0.26246001, 0.        , 0.        ,
        0.23263427, 0.        , 0.25911063, 0.        , 0.24579509,
        0.        , 0.        ],
       [0.        , 0.        , 0.32081428, 0.        , 0.        ,
        0.        , 0.        , 0.        , 0.        , 0.        ,
        0.33389567, 0.34529006],
       [0.        , 0.        , 0.32381881, 0.        , 0.33189907,
        0.        , 0.        , 0.        , 0.        , 0.34428213,
        0.        , 0.        ],
       [0.        , 0.        , 0.32621711, 0.33933355, 0.        ,
        0.        , 0.        , 0.33444934, 0.        , 0.        ,
        0.        , 0.        ],
       [0.        , 0.        , 0.32657636, 0.        , 0.        ,
        0.        , 0.3397132 , 0.        , 0.33371045, 0.        ,
        0.        , 0.        ],
       [0.        , 0.        , 0.32678577, 0.        , 0.33453837,
        0.        , 0.        , 0.        , 0.        , 0.        ,
        0.        , 0.33867586],
       [0.        , 0.        , 0.32759735, 0.        , 0.32651075,
        0.        , 0.        , 0.        , 0.        , 0.        ,
        0.3458919 , 0.        ],
       [0.        , 0.        , 0.32834585, 0.33701698, 0.        ,
        0.        , 0.        , 0.        , 0.        , 0.        ,
        0.33463717, 0.        ],
       [0.        , 0.        , 0.32921274, 0.        , 0.3325215 ,
        0.33826576, 0.        , 0.        , 0.        , 0.        ,
        0.        , 0.        ],
       [0.        , 0.        , 0.32945759, 0.        , 0.        ,
        0.        , 0.        , 0.        , 0.        , 0.32950293,
        0.        , 0.34103948],
       [0.        , 0.        , 0.33015609, 0.        , 0.        ,
        0.        , 0.        , 0.        , 0.3266764 , 0.        ,
        0.34316751, 0.        ],
       [0.        , 0.        , 0.33052629, 0.3337002 , 0.        ,
        0.33577352, 0.        , 0.        , 0.        , 0.        ,
        0.        , 0.        ],
       [0.        , 0.        , 0.33066761, 0.        , 0.        ,
        0.        , 0.        , 0.        , 0.        , 0.33125344,
        0.33807895, 0.        ],
       [0.        , 0.        , 0.33082957, 0.        , 0.33139157,
        0.        , 0.        , 0.        , 0.33777886, 0.        ,
        0.        , 0.        ],
       [0.        , 0.        , 0.33150526, 0.        , 0.        ,
        0.        , 0.        , 0.        , 0.31699971, 0.        ,
        0.        , 0.35149504],
       [0.        , 0.        , 0.33200434, 0.        , 0.3320584 ,
        0.        , 0.33593727, 0.        , 0.        , 0.        ,
        0.        , 0.        ],
       [0.        , 0.        , 0.33268161, 0.        , 0.        ,
        0.        , 0.        , 0.32097473, 0.34634366, 0.        ,
        0.        , 0.        ],
       [0.        , 0.        , 0.3339864 , 0.        , 0.        ,
        0.3353327 , 0.        , 0.        , 0.        , 0.        ,
        0.        , 0.33068089],
       [0.        , 0.        , 0.33414105, 0.        , 0.33376113,
        0.        , 0.        , 0.33209782, 0.        , 0.        ,
        0.        , 0.        ],
       [0.        , 0.        , 0.33459395, 0.        , 0.        ,
        0.        , 0.        , 0.        , 0.33624757, 0.32915849,
        0.        , 0.        ],
       [0.        , 0.        , 0.33474713, 0.        , 0.        ,
        0.        , 0.        , 0.66525287, 0.        , 0.        ,
        0.        , 0.        ],
       [0.        , 0.        , 0.33518581, 0.34003979, 0.3247744 ,
        0.        , 0.        , 0.        , 0.        , 0.        ,
        0.        , 0.        ],
       [0.        , 0.        , 0.3365236 , 0.        , 0.        ,
        0.33024475, 0.33323165, 0.        , 0.        , 0.        ,
        0.        , 0.        ],
       [0.        , 0.        , 0.33666182, 0.        , 0.        ,
        0.        , 0.66333818, 0.        , 0.        , 0.        ,
        0.        , 0.        ],
       [0.        , 0.        , 0.33734253, 0.33849912, 0.        ,
        0.        , 0.32415835, 0.        , 0.        , 0.        ,
        0.        , 0.        ],
       [0.        , 0.        , 0.33916726, 0.        , 0.        ,
        0.        , 0.33141998, 0.        , 0.        , 0.        ,
        0.        , 0.32941276],
       [0.        , 0.        , 0.33962456, 0.        , 0.        ,
        0.66037544, 0.        , 0.        , 0.        , 0.        ,
        0.        , 0.        ],
       [0.        , 0.        , 0.34033875, 0.        , 0.        ,
        0.        , 0.32451592, 0.33514533, 0.        , 0.        ,
        0.        , 0.        ],
       [0.        , 0.        , 0.34058062, 0.        , 0.        ,
        0.33163883, 0.        , 0.        , 0.        , 0.        ,
        0.32778055, 0.        ],
       [0.        , 0.        , 0.34158191, 0.32664015, 0.        ,
        0.        , 0.        , 0.        , 0.        , 0.33177794,
        0.        , 0.        ],
       [0.        , 0.        , 0.65356015, 0.        , 0.        ,
        0.        , 0.        , 0.        , 0.        , 0.        ,
        0.        , 0.34643985],
       [0.        , 0.        , 0.65573607, 0.        , 0.        ,
        0.        , 0.        , 0.34426393, 0.        , 0.        ,
        0.        , 0.        ],
       [0.        , 0.        , 0.65684924, 0.        , 0.        ,
        0.        , 0.        , 0.        , 0.        , 0.34315076,
        0.        , 0.        ],
       [0.        , 0.        , 0.66130328, 0.        , 0.        ,
        0.        , 0.        , 0.        , 0.33869672, 0.        ,
        0.        , 0.        ],
       [0.        , 0.        , 0.66205853, 0.        , 0.33794147,
        0.        , 0.        , 0.        , 0.        , 0.        ,
        0.        , 0.        ],
       [0.        , 0.        , 0.66578779, 0.33421221, 0.        ,
        0.        , 0.        , 0.        , 0.        , 0.        ,
        0.        , 0.        ],
       [0.        , 0.        , 0.66609084, 0.        , 0.        ,
        0.33390916, 0.        , 0.        , 0.        , 0.        ,
        0.        , 0.        ],
       [0.        , 0.        , 0.6662309 , 0.        , 0.        ,
        0.        , 0.        , 0.        , 0.        , 0.        ,
        0.3337691 , 0.        ],
       [0.        , 0.        , 1.        , 0.        , 0.        ,
        0.        , 0.        , 0.        , 0.        , 0.        ,
        0.        , 0.        ],
       [0.        , 0.17526148, 0.        , 0.        , 0.33858552,
        0.22889789, 0.25725512, 0.        , 0.        , 0.        ,
        0.        , 0.        ],
       [0.        , 0.22545134, 0.36933715, 0.20446529, 0.        ,
        0.        , 0.        , 0.        , 0.        , 0.        ,
        0.        , 0.20074621],
       [0.        , 0.29033951, 0.        , 0.23538978, 0.        ,
        0.        , 0.        , 0.        , 0.2350704 , 0.        ,
        0.23920031, 0.        ],
       [0.        , 0.31354062, 0.        , 0.3535237 , 0.        ,
        0.        , 0.        , 0.        , 0.        , 0.33293568,
        0.        , 0.        ],
       [0.        , 0.32089143, 0.        , 0.        , 0.        ,
        0.        , 0.32256017, 0.        , 0.        , 0.        ,
        0.3565484 , 0.        ],
       [0.        , 0.32524858, 0.        , 0.        , 0.        ,
        0.        , 0.        , 0.        , 0.        , 0.34083159,
        0.        , 0.33391982],
       [0.        , 0.3256997 , 0.        , 0.34707304, 0.        ,
        0.32722726, 0.        , 0.        , 0.        , 0.        ,
        0.        , 0.        ],
       [0.        , 0.32613426, 0.        , 0.        , 0.        ,
        0.        , 0.        , 0.        , 0.33605971, 0.        ,
        0.        , 0.33780602],
       [0.        , 0.3263657 , 0.        , 0.        , 0.        ,
        0.        , 0.        , 0.32505993, 0.        , 0.34857437,
        0.        , 0.        ],
       [0.        , 0.32646885, 0.        , 0.        , 0.        ,
        0.        , 0.        , 0.        , 0.        , 0.34090722,
        0.33262393, 0.        ],
       [0.        , 0.3272065 , 0.        , 0.        , 0.        ,
        0.33853309, 0.        , 0.        , 0.33426041, 0.        ,
        0.        , 0.        ],
       [0.        , 0.32857889, 0.        , 0.        , 0.        ,
        0.33124489, 0.        , 0.        , 0.        , 0.        ,
        0.        , 0.34017622],
       [0.        , 0.32863797, 0.        , 0.        , 0.        ,
        0.        , 0.        , 0.33287342, 0.        , 0.        ,
        0.        , 0.33848861],
       [0.        , 0.32972239, 0.32909275, 0.        , 0.        ,
        0.34118486, 0.        , 0.        , 0.        , 0.        ,
        0.        , 0.        ],
       [0.        , 0.3301367 , 0.        , 0.        , 0.        ,
        0.6698633 , 0.        , 0.        , 0.        , 0.        ,
        0.        , 0.        ],
       [0.        , 0.33074711, 0.        , 0.        , 0.33159788,
        0.        , 0.        , 0.        , 0.        , 0.        ,
        0.        , 0.33765502],
       [0.        , 0.33122024, 0.        , 0.        , 0.        ,
        0.        , 0.        , 0.33055193, 0.33822783, 0.        ,
        0.        , 0.        ],
       [0.        , 0.33178386, 0.        , 0.        , 0.        ,
        0.        , 0.        , 0.        , 0.32695538, 0.34126076,
        0.        , 0.        ],
       [0.        , 0.33199892, 0.33528982, 0.        , 0.        ,
        0.        , 0.        , 0.        , 0.33271125, 0.        ,
        0.        , 0.        ],
       [0.        , 0.33215064, 0.33178599, 0.        , 0.33606337,
        0.        , 0.        , 0.        , 0.        , 0.        ,
        0.        , 0.        ],
       [0.        , 0.33310797, 0.        , 0.        , 0.33192846,
        0.        , 0.        , 0.33496357, 0.        , 0.        ,
        0.        , 0.        ],
       [0.        , 0.33355097, 0.        , 0.        , 0.        ,
        0.33402424, 0.        , 0.33242479, 0.        , 0.        ,
        0.        , 0.        ],
       [0.        , 0.33369083, 0.        , 0.33591097, 0.        ,
        0.        , 0.3303982 , 0.        , 0.        , 0.        ,
        0.        , 0.        ],
       [0.        , 0.33403171, 0.3403218 , 0.        , 0.        ,
        0.        , 0.32564649, 0.        , 0.        , 0.        ,
        0.        , 0.        ],
       [0.        , 0.33412182, 0.        , 0.        , 0.        ,
        0.31728512, 0.        , 0.        , 0.        , 0.        ,
        0.34859306, 0.        ],
       [0.        , 0.33416319, 0.        , 0.        , 0.        ,
        0.        , 0.        , 0.        , 0.        , 0.        ,
        0.        , 0.66583681],
       [0.        , 0.3354997 , 0.        , 0.        , 0.        ,
        0.        , 0.33014764, 0.        , 0.33435266, 0.        ,
        0.        , 0.        ],
       [0.        , 0.33574291, 0.        , 0.        , 0.        ,
        0.        , 0.32837784, 0.        , 0.        , 0.        ,
        0.        , 0.33587925],
       [0.        , 0.33585201, 0.        , 0.        , 0.33293353,
        0.        , 0.        , 0.        , 0.        , 0.33121446,
        0.        , 0.        ],
       [0.        , 0.33617346, 0.        , 0.33136055, 0.33246599,
        0.        , 0.        , 0.        , 0.        , 0.        ,
        0.        , 0.        ],
       [0.        , 0.33672437, 0.        , 0.        , 0.        ,
        0.34444967, 0.        , 0.        , 0.        , 0.31882596,
        0.        , 0.        ],
       [0.        , 0.3367975 , 0.        , 0.        , 0.        ,
        0.        , 0.        , 0.        , 0.        , 0.        ,
        0.34131537, 0.32188713],
       [0.        , 0.33687331, 0.        , 0.32937682, 0.        ,
        0.        , 0.        , 0.33374988, 0.        , 0.        ,
        0.        , 0.        ],
       [0.        , 0.33735798, 0.        , 0.        , 0.        ,
        0.        , 0.33325933, 0.        , 0.        , 0.32938269,
        0.        , 0.        ],
       [0.        , 0.33755171, 0.        , 0.        , 0.        ,
        0.        , 0.        , 0.66244829, 0.        , 0.        ,
        0.        , 0.        ],
       [0.        , 0.33806293, 0.        , 0.33341868, 0.        ,
        0.        , 0.        , 0.        , 0.        , 0.        ,
        0.        , 0.32851839],
       [0.        , 0.33826866, 0.33524681, 0.        , 0.        ,
        0.        , 0.        , 0.        , 0.        , 0.        ,
        0.32648452, 0.        ],
       [0.        , 0.34030557, 0.        , 0.        , 0.        ,
        0.        , 0.        , 0.        , 0.65969443, 0.        ,
        0.        , 0.        ],
       [0.        , 0.34077934, 0.        , 0.        , 0.        ,
        0.        , 0.3281604 , 0.33106026, 0.        , 0.        ,
        0.        , 0.        ],
       [0.        , 0.34300874, 0.        , 0.        , 0.        ,
        0.        , 0.65699126, 0.        , 0.        , 0.        ,
        0.        , 0.        ],
       [0.        , 0.34393082, 0.        , 0.        , 0.65606918,
        0.        , 0.        , 0.        , 0.        , 0.        ,
        0.        , 0.        ],
       [0.        , 0.55430274, 0.        , 0.        , 0.22335331,
        0.        , 0.22234394, 0.        , 0.        , 0.        ,
        0.        , 0.        ],
       [0.        , 0.65492004, 0.        , 0.34507996, 0.        ,
        0.        , 0.        , 0.        , 0.        , 0.        ,
        0.        , 0.        ],
       [0.        , 0.65536495, 0.        , 0.        , 0.        ,
        0.34463505, 0.        , 0.        , 0.        , 0.        ,
        0.        , 0.        ],
       [0.        , 0.65694344, 0.        , 0.        , 0.        ,
        0.        , 0.        , 0.        , 0.        , 0.34305656,
        0.        , 0.        ],
       [0.        , 0.65884529, 0.34115471, 0.        , 0.        ,
        0.        , 0.        , 0.        , 0.        , 0.        ,
        0.        , 0.        ],
       [0.        , 0.66025582, 0.        , 0.        , 0.        ,
        0.        , 0.        , 0.        , 0.33974418, 0.        ,
        0.        , 0.        ],
       [0.        , 0.66178717, 0.        , 0.        , 0.        ,
        0.        , 0.        , 0.        , 0.        , 0.        ,
        0.33821283, 0.        ],
       [0.        , 0.66645899, 0.        , 0.        , 0.        ,
        0.        , 0.        , 0.        , 0.        , 0.        ,
        0.        , 0.33354101],
       [0.        , 1.        , 0.        , 0.        , 0.        ,
        0.        , 0.        , 0.        , 0.        , 0.        ,
        0.        , 0.        ],
       [0.19866282, 0.        , 0.        , 0.        , 0.24013028,
        0.        , 0.        , 0.33417036, 0.        , 0.        ,
        0.        , 0.22703654],
       [0.20501159, 0.        , 0.        , 0.27640913, 0.25369479,
        0.26488449, 0.        , 0.        , 0.        , 0.        ,
        0.        , 0.        ],
       [0.23183374, 0.19259018, 0.        , 0.        , 0.        ,
        0.36169258, 0.21388351, 0.        , 0.        , 0.        ,
        0.        , 0.        ],
       [0.2546962 , 0.        , 0.        , 0.        , 0.        ,
        0.        , 0.        , 0.24837374, 0.        , 0.23663896,
        0.26029109, 0.        ],
       [0.25931177, 0.        , 0.        , 0.        , 0.22974449,
        0.        , 0.27021328, 0.        , 0.        , 0.        ,
        0.        , 0.24073046],
       [0.31142561, 0.08553145, 0.        , 0.        , 0.        ,
        0.        , 0.        , 0.        , 0.29335945, 0.        ,
        0.3096835 , 0.        ],
       [0.31809425, 0.33449624, 0.        , 0.        , 0.        ,
        0.        , 0.34740951, 0.        , 0.        , 0.        ,
        0.        , 0.        ],
       [0.32206444, 0.        , 0.33717078, 0.        , 0.        ,
        0.        , 0.        , 0.        , 0.        , 0.        ,
        0.34076478, 0.        ],
       [0.32469812, 0.        , 0.        , 0.        , 0.33941052,
        0.        , 0.        , 0.        , 0.33589136, 0.        ,
        0.        , 0.        ],
       [0.32676046, 0.        , 0.        , 0.        , 0.        ,
        0.        , 0.        , 0.        , 0.34085035, 0.        ,
        0.        , 0.33238919],
       [0.32750204, 0.        , 0.        , 0.        , 0.        ,
        0.33858299, 0.        , 0.        , 0.        , 0.33391497,
        0.        , 0.        ],
       [0.32934142, 0.        , 0.32904252, 0.        , 0.        ,
        0.        , 0.        , 0.        , 0.        , 0.34161606,
        0.        , 0.        ],
       [0.32981228, 0.        , 0.        , 0.        , 0.        ,
        0.32848624, 0.        , 0.        , 0.        , 0.        ,
        0.34170148, 0.        ],
       [0.33078109, 0.        , 0.        , 0.        , 0.        ,
        0.        , 0.        , 0.        , 0.3376437 , 0.3315752 ,
        0.        , 0.        ],
       [0.33119477, 0.        , 0.        , 0.        , 0.        ,
        0.34038571, 0.        , 0.        , 0.        , 0.        ,
        0.        , 0.32841952],
       [0.33160584, 0.        , 0.33545266, 0.        , 0.        ,
        0.        , 0.        , 0.        , 0.        , 0.        ,
        0.        , 0.3329415 ],
       [0.33182333, 0.        , 0.        , 0.        , 0.        ,
        0.        , 0.        , 0.        , 0.        , 0.        ,
        0.66817667, 0.        ],
       [0.33234066, 0.        , 0.33406657, 0.        , 0.33359277,
        0.        , 0.        , 0.        , 0.        , 0.        ,
        0.        , 0.        ],
       [0.33326035, 0.33277149, 0.        , 0.        , 0.33396816,
        0.        , 0.        , 0.        , 0.        , 0.        ,
        0.        , 0.        ],
       [0.33330473, 0.        , 0.        , 0.        , 0.        ,
        0.        , 0.        , 0.        , 0.66669527, 0.        ,
        0.        , 0.        ],
       [0.33384138, 0.        , 0.        , 0.66615862, 0.        ,
        0.        , 0.        , 0.        , 0.        , 0.        ,
        0.        , 0.        ],
       [0.33459226, 0.        , 0.        , 0.        , 0.        ,
        0.        , 0.        , 0.        , 0.        , 0.32909145,
        0.        , 0.33631629],
       [0.33462044, 0.        , 0.        , 0.        , 0.        ,
        0.33319739, 0.        , 0.33218217, 0.        , 0.        ,
        0.        , 0.        ],
       [0.33487278, 0.32963007, 0.        , 0.33549716, 0.        ,
        0.        , 0.        , 0.        , 0.        , 0.        ,
        0.        , 0.        ],
       [0.33499252, 0.66500748, 0.        , 0.        , 0.        ,
        0.        , 0.        , 0.        , 0.        , 0.        ,
        0.        , 0.        ],
       [0.33556437, 0.        , 0.        , 0.        , 0.3393851 ,
        0.        , 0.        , 0.        , 0.        , 0.32505054,
        0.        , 0.        ],
       [0.33671264, 0.        , 0.33420078, 0.        , 0.        ,
        0.        , 0.        , 0.        , 0.32908658, 0.        ,
        0.        , 0.        ],
       [0.33678163, 0.        , 0.        , 0.        , 0.33421091,
        0.        , 0.        , 0.        , 0.        , 0.        ,
        0.32900745, 0.        ],
       [0.33717818, 0.        , 0.        , 0.        , 0.66282182,
        0.        , 0.        , 0.        , 0.        , 0.        ,
        0.        , 0.        ],
       [0.33736102, 0.        , 0.        , 0.        , 0.        ,
        0.66263898, 0.        , 0.        , 0.        , 0.        ,
        0.        , 0.        ],
       [0.33780682, 0.        , 0.        , 0.        , 0.        ,
        0.        , 0.        , 0.        , 0.        , 0.66219318,
        0.        , 0.        ],
       [0.33799981, 0.        , 0.        , 0.        , 0.        ,
        0.        , 0.        , 0.32941594, 0.33258425, 0.        ,
        0.        , 0.        ],
       [0.33925744, 0.        , 0.        , 0.        , 0.        ,
        0.        , 0.33516515, 0.        , 0.        , 0.32557741,
        0.        , 0.        ],
       [0.33934617, 0.        , 0.        , 0.        , 0.        ,
        0.        , 0.66065383, 0.        , 0.        , 0.        ,
        0.        , 0.        ],
       [0.34011214, 0.33145595, 0.32843191, 0.        , 0.        ,
        0.        , 0.        , 0.        , 0.        , 0.        ,
        0.        , 0.        ],
       [0.34016863, 0.33187638, 0.        , 0.        , 0.        ,
        0.        , 0.        , 0.        , 0.        , 0.        ,
        0.        , 0.32795498],
       [0.34040754, 0.        , 0.        , 0.34234016, 0.        ,
        0.        , 0.        , 0.        , 0.3172523 , 0.        ,
        0.        , 0.        ],
       [0.34044373, 0.        , 0.65955627, 0.        , 0.        ,
        0.        , 0.        , 0.        , 0.        , 0.        ,
        0.        , 0.        ],
       [0.34105643, 0.        , 0.        , 0.        , 0.        ,
        0.        , 0.3364094 , 0.        , 0.        , 0.        ,
        0.32253417, 0.        ],
       [0.34136231, 0.32792256, 0.        , 0.        , 0.        ,
        0.        , 0.        , 0.33071513, 0.        , 0.        ,
        0.        , 0.        ],
       [0.3417461 , 0.        , 0.        , 0.        , 0.        ,
        0.        , 0.        , 0.        , 0.        , 0.        ,
        0.        , 0.6582539 ],
       [0.34323945, 0.        , 0.        , 0.33317162, 0.        ,
        0.        , 0.        , 0.        , 0.        , 0.32358893,
        0.        , 0.        ],
       [0.34467483, 0.        , 0.        , 0.        , 0.        ,
        0.        , 0.33213616, 0.323189  , 0.        , 0.        ,
        0.        , 0.        ],
       [0.34695556, 0.        , 0.        , 0.        , 0.        ,
        0.        , 0.3265068 , 0.        , 0.32653763, 0.        ,
        0.        , 0.        ],
       [0.34918245, 0.        , 0.32143033, 0.        , 0.        ,
        0.        , 0.32938722, 0.        , 0.        , 0.        ,
        0.        , 0.        ],
       [0.35295561, 0.        , 0.        , 0.        , 0.        ,
        0.        , 0.        , 0.64704439, 0.        , 0.        ,
        0.        , 0.        ],
       [0.35517308, 0.        , 0.        , 0.32732239, 0.        ,
        0.        , 0.        , 0.31750454, 0.        , 0.        ,
        0.        , 0.        ],
       [0.55878299, 0.        , 0.        , 0.        , 0.        ,
        0.        , 0.        , 0.20466594, 0.        , 0.        ,
        0.        , 0.23655107],
       [0.56219746, 0.        , 0.        , 0.        , 0.23433558,
        0.20346695, 0.        , 0.        , 0.        , 0.        ,
        0.        , 0.        ],
       [0.66071444, 0.        , 0.        , 0.        , 0.        ,
        0.        , 0.        , 0.        , 0.        , 0.        ,
        0.33928556, 0.        ],
       [0.6654359 , 0.        , 0.        , 0.        , 0.        ,
        0.        , 0.        , 0.        , 0.        , 0.3345641 ,
        0.        , 0.        ],
       [0.6663558 , 0.        , 0.3336442 , 0.        , 0.        ,
        0.        , 0.        , 0.        , 0.        , 0.        ,
        0.        , 0.        ],
       [0.66978023, 0.        , 0.        , 0.        , 0.        ,
        0.        , 0.        , 0.        , 0.33021977, 0.        ,
        0.        , 0.        ],
       [1.        , 0.        , 0.        , 0.        , 0.        ,
        0.        , 0.        , 0.        , 0.        , 0.        ,
        0.        , 0.        ]])
    else:
        return list(get_reference_directions("energy", dim, num_weights, seed=seed))


def moving_average(interval: Union[np.array, List], window_size: int) -> np.array:
    if window_size == 1:
        return interval
    window = np.ones(int(window_size)) / float(window_size)
    return np.convolve(interval, window, "same")


def linearly_decaying_epsilon(initial_epsilon, decay_period, step, warmup_steps, final_epsilon):
    """Returns the current epsilon for the agent's epsilon-greedy policy.
    This follows the Nature DQN schedule of a linearly decaying epsilon (Mnih et
    al., 2015). The schedule is as follows:
    Begin at 1. until warmup_steps steps have been taken; then
    Linearly decay epsilon from 1. to epsilon in decay_period steps; and then
    Use epsilon from there on.
    Args:
    decay_period: float, the period over which epsilon is decayed.
    step: int, the number of training steps completed so far.
    warmup_steps: int, the number of steps taken before epsilon is decayed.
    epsilon: float, the final value to which to decay the epsilon parameter.
    Returns:
    A float, the current epsilon value computed according to the schedule.
    """
    steps_left = decay_period + warmup_steps - step
    bonus = (initial_epsilon - final_epsilon) * steps_left / decay_period
    bonus = np.clip(bonus, 0.0, 1.0 - final_epsilon)
    return final_epsilon + bonus


def update_mean_var_count_from_moments(
    mean, var, count, batch_mean, batch_var, batch_count
):
    """Updates the mean, var and count using the previous mean, var, count and batch values."""
    delta = batch_mean - mean
    tot_count = count + batch_count

    new_mean = mean + delta * batch_count / tot_count
    m_a = var * count
    m_b = batch_var * batch_count
    M2 = m_a + m_b + np.square(delta) * count * batch_count / tot_count
    new_var = M2 / tot_count
    new_count = tot_count

    return new_mean, new_var, new_count

class ObservationNormalizer:
    """Tracks the mean, variance and count of values."""

    # https://en.wikipedia.org/wiki/Algorithms_for_calculating_variance#Parallel_algorithm
    def __init__(self, epsilon=1e-4, shape=(), dtype=np.float32):
        """Tracks the mean, variance and count of values."""
        self.mean = np.zeros(shape, dtype=dtype)
        self.var = np.ones(shape, dtype=dtype)
        self.count = epsilon

    def update(self, x):
        """Updates the mean, var and count from a batch of samples."""
        batch_mean = np.mean(x, axis=0)
        batch_var = np.var(x, axis=0)
        batch_count = x.shape[0]
        self.update_from_moments(batch_mean, batch_var, batch_count)

    def update_from_moments(self, batch_mean, batch_var, batch_count):
        """Updates from batch mean, variance and count moments."""
        self.mean, self.var, self.count = update_mean_var_count_from_moments(
            self.mean, self.var, self.count, batch_mean, batch_var, batch_count
        )

    def normalize(self, observations):
        return (observations - self.mean) / np.sqrt(self.var + 1e-8)


def seed_everything(seed: int = 42):
    if seed is None:
        return
    random.seed(seed)
    os.environ["PYTHONHASHSEED"] = str(seed)
    np.random.seed(seed)
    th.manual_seed(seed)
    th.cuda.manual_seed(seed)
    th.backends.cudnn.deterministic = True
    th.backends.cudnn.benchmark = True


def config_plots():
    params = {
        "text.latex.preamble": r"\usepackage{amsmath}",
        "mathtext.fontset": "cm",
        "figure.figsize": (1.5 * 4.5, 1.5 * 3),
        "xtick.major.pad": 0,
        "ytick.major.pad": 0,
    }
    plt.rcParams.update(params)
    sns.set_style("white", rc={"xtick.bottom": False, "ytick.left": False})
    sns.set_context(
        "paper",
        rc={
            "text.usetex": True,
            "lines.linewidth": 2,
            "font.size": 15,
            "figure.autolayout": True,
            "xtick.labelsize": 12,
            "ytick.labelsize": 12,
            "axes.titlesize": 12,
            "axes.labelsize": 15,
            "lines.markersize": 12,
            "legend.fontsize": 14,
        },
    )
    sns.despine()
    plt.grid(alpha=0.25)
    plt.tight_layout()


# Code courtesy of JPH: https://github.com/jparkerholder
""" def make_gif(policy, env, step_count, state_filter, maxsteps=1000):
    envname = env.spec.id
    gif_name = '_'.join([envname, str(step_count)])
    state = env.reset()
    done = False
    steps = []
    rewards = []
    t = 0
    while (not done) & (t < maxsteps):
        s = env.render('rgb_array')
        steps.append(s)
        action = policy.get_action(state, state_filter=state_filter, deterministic=True)
        action = np.clip(action, env.action_space.low[0], env.action_space.high[0])
        action = action.reshape(len(action), )
        state, reward, done, _ = env.step(action)
        rewards.append(reward)
        t +=1
    print('Final reward :', np.sum(rewards))
    clip = ImageSequenceClip(steps, fps=30)
    if not os.path.isdir('gifs'):
        os.makedirs('gifs')
    clip.write_gif('gifs/{}.gif'.format(gif_name), fps=30) """


if __name__ == "__main__":

    # print(generate_weights(10,3,10))
    print(random_weights())
