
def get_parameters(function_call):
    '''
    Gets the parameters from the application of a function call.

    Returns parameters as a list.

    Requires {function_call} to be the function call or an assert test case.

    Ex: test([1,2],[[1,2]],1) => [[1,2],[[1,2]],1]
    Ex: assert test(2,3) == 2 => [2,3]

    Returns [[1,2], [[1,2]], 1].

    '''

    # Get just the parameters by getting the outer parentheses
    num_paren, num_bracket, start, end = 0, 0, 0, 0

    start = function_call.find('(')
    end = start + 1
    num_paren = 1

    while num_paren > 0 and end < len(function_call):
        if function_call[end] == '(':
            num_paren += 1
        elif function_call[end] == ')':
            num_paren -= 1
        end += 1

    # If no outer matching parentheses, parameters are deformed
    if end == len(function_call) and num_paren > 0:
        return None

    parameter_string = function_call[start + 1:end - 1]
    parameters = []
    num_paren, num_bracket, start, end = 0, 0, 0, 0

    while end < len(parameter_string):
        char = parameter_string[end]

        # Add to parameters if outside of parentheses + brackets
        if char == "," and num_paren == 0 and num_bracket == 0:
            # Strip white spaces from each parameter
            parameters.append(parameter_string[start:end].strip())
            start = end + 1

        # Counting for internal parentheses and brackets
        num_paren = num_paren + \
            1 if char == '(' else num_paren - 1 if char == ')' else num_paren

        num_bracket = num_bracket + \
            1 if char == '[' else num_bracket - \
            1 if char == ']' else num_bracket

        end += 1

        # if end is at the end of parameter_string => add it to list if nonempty
        if end == len(parameter_string) and parameter_string[start:end]:
            parameters.append(parameter_string[start:end].strip())

    return parameters


def get_parameter_typing(param):
    '''
    Return parameter type (in type hinting format) given {param} as a string.

    WARNING - Uses EVAL(). Operate at your own expense!
    '''
    try:
        param_eval = eval(param)

        return type(param_eval).__name__
    except:
        # Deformed parameter
        return None


def eval_param(param):
    '''
    Returns a string evaluated to an object.

    WARNING - Uses Eval().
    '''
    try:
        return eval(param)
    except:
        # Deformed parameter
        return None


def get_parameter_detailed_typing(param):
    '''
    Return parameter type (in type hinting format) given {param} as a string.

    This is a more detailed parameter typing.
    Ex:
    Given [[[3]]], should return List[List[List[int]]] instead of just list.

    Requires:
    {param} is syntatically correct.

    WARNING - Makes use of {eval_param}, which uses EVAL(). Operate at your own expense!
    '''

    # Custom representation of types because so many types are dealt with differnetly
    class ID:
        '''
        The "Any" object.
        '''
        name = "Any"

        def __eq__(self, object):
            return type(self) == type(object)

        def __hash__(self):
            return hash(self.name)

        def __str__(self):
            return self.name

        def __repr__(self):
            return self.name

    class BasicID(ID):
        name = "Any"

    class StrID(BasicID):
        name = "str"

    class IntID(BasicID):
        name = "int"

    class FloatID(BasicID):
        name = "float"

    class ComplexID(BasicID):
        name = "complex"

    class TypeID(BasicID):
        name = "type"

    # Iterables

    # Names have changed to lowercase in version 3.9+
    class TupleID(ID):
        name = "Tuple"

        @staticmethod
        def if_one_type(contents_list):
            '''
            Gets if there is only one type in a list.
            '''
            whole_set = set(sum(contents_list, []))

            if len(whole_set) == 1:
                return list(whole_set)[0]

            return None

        def __init__(self, contents):
            self.contents = contents

        def __repr__(self):
            return self.name + f"{self.contents}"

        def __str__(self):
            return self.name + f"{self.contents}"

        def __len__(self):
            return len(self.contents)

    class DictID(ID):
        name = "Dict"

        def __init__(self, key, value):
            self.key = key
            self.value = value

        def __repr__(self):
            return self.name + f"[{self.key},{self.value}]"

        def __str__(self):
            return self.name + f"[{self.key},{self.value}]"

    class SetID(ID):
        name = "Set"

        def __init__(self, contents):
            self.contents = contents

        def __repr__(self):
            return self.name + f"[{self.contents}]"

        def __str__(self):
            return self.name + f"[{self.contents}]"

    class ListID(ID):
        name = "List"

    def get_first_non_similar(x):
        '''
        Returns a list of the longest similar objects of a list {x} of lists.


        Returns None if no lists exist.
        Adds on "Any" if the lists in the list {x} are not all the same.
        '''

        if len(x) == 0:
            return None

        type_1 = x[0]

        length_x = [len(n) for n in x]

        # Check if should return "Any" at the end
        is_the_same = len(set(length_x)) <= 1

        for i in range(len(type_1)):
            for j in range(1, len(x)):

                # If any part is different
                if i >= len(x[j]) or x[j][i] != type_1[i]:
                    return type_1[:i] + [ID()]

            # Dealing with tuples

            if isinstance(type_1[i], TupleID):
                l_tuples = []

                for j in range(len(x)):
                    l_tuples.append(x[j][i].contents)

                common_tuple = get_first_non_similar(l_tuples)

                # if exists a non_similar
                if ID() in common_tuple:
                    is_one_type = TupleID.if_one_type(l_tuples)

                    if is_one_type:
                        return type_1[:i] + [TupleID([is_one_type, ID()])]
                    else:
                        return type_1[:i] + [TupleID([ID()])]

                return type_1[:i] + [TupleID(common_tuple)]

            # Dealing with dicts
            elif isinstance(type_1[i], DictID):
                keys_list = []
                values_list = []

                for j in range(len(x)):
                    keys_list.append(x[j][i].key)
                    values_list.append(x[j][i].value)

                common_keys = get_first_non_similar(keys_list)
                common_values = get_first_non_similar(values_list)

                return type_1[:i] + [DictID(common_keys, common_values)]

            # Dealing with sets
            elif isinstance(type_1[i], SetID):
                val_list = []

                for j in range(len(x)):
                    val_list.append(x[j][i].contents)

                common_keys = get_first_non_similar(val_list)

                return type_1[:i] + [SetID(common_keys)]

        if is_the_same:
            return type_1
        else:
            return type_1 + [ID()]

    def helper(n):
        # If basic type => just return type
        if isinstance(n, str):
            return [StrID()]

        elif isinstance(n, int):
            return [IntID()]

        elif isinstance(n, float):
            return [FloatID()]

        elif isinstance(n, complex):
            return [ComplexID()]

        elif isinstance(n, type):
            return [TypeID()]

        # Iterables have to be treated differently
        elif isinstance(n, list):
            types_in_n = [helper(x) for x in n]

            if len(types_in_n) == 0:
                return [ListID()]

            # Perform comparison to find where the lists are no longer alike
            for i in types_in_n:
                return [ListID()] + get_first_non_similar(types_in_n)

        # Tuples are unique
        elif isinstance(n, tuple):
            types_in_n = []
            for i in n:
                types_in_n.extend(helper(i))

            if len(types_in_n) == 0:
                return [TupleID([])]

            return [TupleID(types_in_n)]

        # Sets are like tuples mixed with dicts
        elif isinstance(n, set):
            vals = list(n)

            return [SetID(helper(vals)[1:])]

        # If NoneType
        elif n is None:
            return [ID()]

        # Dicts also need to be dealt with uniquely
        elif isinstance(n, dict):
            keys = list(n.keys())
            values = list(n.values())

            return [DictID(helper(keys)[1:], helper(values)[1:])]

        # Should not occur
        else:
            raise ValueError()

    def convert_intermed_repr(n):
        if isinstance(n, ID):
            return str(n)

        if n:
            cur = n.pop(0)

            if isinstance(cur, ListID):
                return f"{ListID.name}[{convert_intermed_repr(n)}]"
            elif isinstance(cur, TupleID):
                elements = [convert_intermed_repr(x) for x in cur.contents]
                # If need ellipses
                if "Any" in elements and elements[0] != "Any":
                    return f"{TupleID.name}[{elements[0]}, ...]"
                elif elements[0] == "Any":
                    return f"{TupleID.name}[Any]"
                else:
                    return f"{TupleID.name}[{', '.join(elements)}]"
            elif isinstance(cur, DictID):
                key = convert_intermed_repr(cur.key)

                if key:
                    return f"{DictID.name}[{key},{convert_intermed_repr(cur.value)}]"
                else:
                    return f"{DictID.name}"
            elif isinstance(cur, SetID):
                element = convert_intermed_repr(cur.contents)

                return f"{SetID.name}[{element}]"

            else:
                return str(cur)

    evaled_param = eval_param(param)

    print("TYPE", evaled_param, type(evaled_param))
    if evaled_param is None:
        return ""

    # Intermediate representation
    intermed_repr = helper(evaled_param)
    print(intermed_repr, "YAY")

    return convert_intermed_repr(intermed_repr)
