import logging
import numpy as np
import torch
import torch.nn as nn

from transformers import BertPreTrainedModel, BertModel
from transformers.file_utils import (
    add_start_docstrings,
    add_start_docstrings_to_callable,
)
from transformers.modeling_bert import BERT_START_DOCSTRING, BERT_INPUTS_DOCSTRING

logger = logging.getLogger(__name__)


@add_start_docstrings(
    """Bert Model transformer with a sequence classification/regression head on top (a linear layer on top of
    the pooled output) e.g. for GLUE tasks. """,
    BERT_START_DOCSTRING,
)
class BertForSequenceClassification(BertPreTrainedModel):
    def __init__(self, config):
        super().__init__(config)

        self.bert = BertModel(config)
        self.dropout = nn.Dropout(config.hidden_dropout_prob)

        self.finetuning_task = config.finetuning_task
        if isinstance(self.finetuning_task, str):
            module = nn.Linear(config.hidden_size, config.num_labels)
            setattr(self, "classifier", module)
            self.num_labels = config.num_labels
        elif isinstance(self.finetuning_task, list):
            for task, n_labels in zip(self.finetuning_task, config.n_labels):
                module = nn.Linear(config.hidden_size, n_labels)
                setattr(self, "classifier_%s" % task, module)
            self.num_labels = config.n_labels

        self.init_weights()
        logger.info("BertForSequenceClassification initialized from adapter_bert.py")

    def get_classifier(self, task):

        return getattr(self, "classifier_%s" % task)

    @add_start_docstrings_to_callable(BERT_INPUTS_DOCSTRING)
    def forward(
        self,
        guid=None,
        input_ids=None,
        attention_mask=None,
        token_type_ids=None,
        position_ids=None,
        head_mask=None,
        inputs_embeds=None,
        labels=None,
        task=None,
        replay=False,
    ):
        r"""
            labels (:obj:`torch.LongTensor` of shape :obj:`(batch_size,)`, `optional`, defaults to :obj:`None`):
                Labels for computing the sequence classification/regression loss.
                Indices should be in :obj:`[0, ..., config.num_labels - 1]`.
                If :obj:`config.num_labels == 1` a regression loss is computed (Mean-Square loss),
                If :obj:`config.num_labels > 1` a classification loss is computed (Cross-Entropy).
        Returns:
            :obj:`tuple(torch.FloatTensor)` comprising various elements depending on the configuration (:class:`~transformers.BertConfig`) and inputs:
            loss (:obj:`torch.FloatTensor` of shape :obj:`(1,)`, `optional`, returned when :obj:`label` is provided):
                Classification (or regression if config.num_labels==1) loss.
            logits (:obj:`torch.FloatTensor` of shape :obj:`(batch_size, config.num_labels)`):
                Classification (or regression if config.num_labels==1) scores (before SoftMax).
            hidden_states (:obj:`tuple(torch.FloatTensor)`, `optional`, returned when ``config.output_hidden_states=True``):
                Tuple of :obj:`torch.FloatTensor` (one for the output of the embeddings + one for the output of each layer)
                of shape :obj:`(batch_size, sequence_length, hidden_size)`.
                Hidden-states of the model at the output of each layer plus the initial embedding outputs.
            attentions (:obj:`tuple(torch.FloatTensor)`, `optional`, returned when ``config.output_attentions=True``):
                Tuple of :obj:`torch.FloatTensor` (one for each layer) of shape
                :obj:`(batch_size, num_heads, sequence_length, sequence_length)`.
                Attentions weights after the attention softmax, used to compute the weighted average in the self-attention
                heads.
        """

        outputs = self.bert(
            input_ids,
            attention_mask=attention_mask,
            token_type_ids=token_type_ids,
            position_ids=position_ids,
            head_mask=head_mask,
            inputs_embeds=inputs_embeds,
        )

        pooled_output = outputs[1]

        pooled_output = self.dropout(pooled_output)

        if task is None:
            logits = self.classifier(pooled_output)
            num_labels = self.num_labels
        elif isinstance(task, list) and labels is not None:

            # import ipdb; ipdb.set_trace()
            loss_fct = nn.CrossEntropyLoss()
            overall_loss = None
            for idx, _task in enumerate(task):
                classifier = self.get_classifier(task=_task)
                logits = classifier(pooled_output[idx])
                num_labels = self.num_labels[self.finetuning_task.index(_task)]
                loss = loss_fct(logits.view(-1, num_labels), labels[idx].view(-1))
                if overall_loss is None:
                    overall_loss = loss
                else:
                    overall_loss += loss

            return (overall_loss,)
        else:
            classifier = self.get_classifier(task=task)
            logits = classifier(pooled_output)
            num_labels = self.num_labels[self.finetuning_task.index(task)]

        outputs = (logits, pooled_output) + outputs[
            2:
        ]  # add hidden states and attention if they are here

        if labels is not None:
            if num_labels == 1:
                #  We are doing regression
                loss_fct = nn.MSELoss()
                loss = loss_fct(logits.view(-1), labels.view(-1))
            else:
                if replay:  #  Instead of returning 'mean' loss, return 'sum' for replay
                    loss_fct = nn.CrossEntropyLoss(reduction="sum")
                else:
                    loss_fct = nn.CrossEntropyLoss()
                loss = loss_fct(logits.view(-1, num_labels), labels.view(-1))
            outputs = (loss,) + outputs

        return outputs
