import logging
import numpy as np
import torch
import torch.nn as nn

from transformers import DistilBertPreTrainedModel, DistilBertModel
from transformers.file_utils import (
    add_start_docstrings,
    add_start_docstrings_to_callable,
)
from transformers.modeling_distilbert import DISTILBERT_INPUTS_DOCSTRING

logger = logging.getLogger(__name__)


class DistilBertForSequenceClassification(DistilBertPreTrainedModel):
    def __init__(self, config):
        super().__init__(config)
        # self.num_labels = config.num_labels

        self.distilbert = DistilBertModel(config)
        self.dropout = nn.Dropout(config.seq_classif_dropout)

        # Note that DistillBertModel does not have pooler layer
        self.pre_classifier = nn.Linear(config.dim, config.dim)
        # self.classifier = nn.Linear(config.dim, config.num_labels)

        self.finetuning_task = config.finetuning_task
        if isinstance(self.finetuning_task, str):
            module = nn.Linear(config.dim, config.num_labels)
            # setattr(self, "classifier_%s" % finetuning_task, module)
            setattr(self, "classifier", module)
            self.num_labels = config.num_labels
        elif isinstance(self.finetuning_task, list):
            for task, n_labels in zip(self.finetuning_task, config.n_labels):
                module = nn.Linear(config.dim, n_labels)
                setattr(self, "classifier_%s" % task, module)
            self.num_labels = config.n_labels

        self.init_weights()

        logger.info("DistillBertForSequenceClassification initialized")

    def _custom_init_weights(self, module):
        """Initialize the weights."""
        if isinstance(module, nn.Embedding):
            logger.info("Skipping embeddings!")
            # if module.weight.requires_grad:
            #     module.weight.data.normal_(mean=0.0, std=self.config.initializer_range)
        if isinstance(module, nn.Linear):
            module.weight.data.normal_(mean=0.0, std=self.config.initializer_range)
        elif isinstance(module, nn.LayerNorm):
            module.bias.data.zero_()
            module.weight.data.fill_(1.0)
        if isinstance(module, nn.Linear) and module.bias is not None:
            module.bias.data.zero_()

    def rndm_init_weights(self):
        """Initialize and prunes weights if needed."""
        # Initialize weights
        logger.info("Applying custom init weights!")
        self.apply(self._custom_init_weights)

    def _custom_xyz_init_weights(self, module):
        """Initialize the weights."""
        if isinstance(module, nn.Embedding):
            logger.info("Skipping embeddings!")
            # if module.weight.requires_grad:
            #     module.weight.data.normal_(mean=0.0, std=self.config.initializer_range)
        if isinstance(module, nn.Linear):
            # module.weight.data.normal_(mean=0.0, std=self.config.initializer_range)
            module.weight.data.normal_(mean=0.04, std=self.config.initializer_range)
            rand_tensor = torch.rand(module.weight.data.size())
            rand_tensor[rand_tensor < 0.5] = -1.0
            rand_tensor[rand_tensor >= 0.5] = 1.0
            module.weight.data *= rand_tensor
        elif isinstance(module, nn.LayerNorm):
            module.bias.data.zero_()
            module.weight.data.fill_(1.0)
        if isinstance(module, nn.Linear) and module.bias is not None:
            module.bias.data.zero_()

    def xyz_init_weights(self):
        """Initialize and prunes weights if needed."""
        # Initialize weights
        logger.info("Applying custom xyz init weights!")
        self.apply(self._custom_xyz_init_weights)

    def get_classifier(self, task):

        return getattr(self, "classifier_%s" % task)

    @add_start_docstrings_to_callable(DISTILBERT_INPUTS_DOCSTRING)
    def forward(
        self,
        guid=None,
        input_ids=None,
        attention_mask=None,
        token_type_ids=None,
        position_ids=None,
        head_mask=None,
        inputs_embeds=None,
        labels=None,
        task=None,
        replay=False,
        reduction="mean",
        device=None,
    ):
        r"""
            labels (:obj:`torch.LongTensor` of shape :obj:`(batch_size,)`, `optional`, defaults to :obj:`None`):
                Labels for computing the sequence classification/regression loss.
                Indices should be in :obj:`[0, ..., config.num_labels - 1]`.
                If :obj:`config.num_labels == 1` a regression loss is computed (Mean-Square loss),
                If :obj:`config.num_labels > 1` a classification loss is computed (Cross-Entropy).
        Returns:
            :obj:`tuple(torch.FloatTensor)` comprising various elements depending on the configuration (:class:`~transformers.DistilBertConfig`) and inputs:
            loss (:obj:`torch.FloatTensor` of shape :obj:`(1,)`, `optional`, returned when :obj:`label` is provided):
                Classification (or regression if config.num_labels==1) loss.
            logits (:obj:`torch.FloatTensor` of shape :obj:`(batch_size, config.num_labels)`):
                Classification (or regression if config.num_labels==1) scores (before SoftMax).
            hidden_states (:obj:`tuple(torch.FloatTensor)`, `optional`, returned when ``config.output_hidden_states=True``):
                Tuple of :obj:`torch.FloatTensor` (one for the output of the embeddings + one for the output of each layer)
                of shape :obj:`(batch_size, sequence_length, hidden_size)`.
                Hidden-states of the model at the output of each layer plus the initial embedding outputs.
            attentions (:obj:`tuple(torch.FloatTensor)`, `optional`, returned when ``config.output_attentions=True``):
                Tuple of :obj:`torch.FloatTensor` (one for each layer) of shape
                :obj:`(batch_size, num_heads, sequence_length, sequence_length)`.
                Attentions weights after the attention softmax, used to compute the weighted average in the self-attention
                heads.
        """
        distilbert_output = self.distilbert(
            input_ids=input_ids,
            attention_mask=attention_mask,
            head_mask=head_mask,
            inputs_embeds=inputs_embeds,
        )
        hidden_state = distilbert_output[0]  # (bs, seq_len, dim)
        pooled_output = hidden_state[:, 0]  # (bs, dim)
        pooled_output = self.pre_classifier(pooled_output)  # (bs, dim)
        pooled_output = nn.ReLU()(pooled_output)  # (bs, dim)
        pooled_output = self.dropout(pooled_output)  # (bs, dim)

        if task is None:
            logits = self.classifier(pooled_output)  # (bs, dim)
            num_labels = self.num_labels
        elif isinstance(task, list) and labels is not None:
            # import ipdb; ipdb.set_trace()
            loss_fct = nn.CrossEntropyLoss()
            overall_loss = None
            for idx, _task in enumerate(task):
                classifier = self.get_classifier(task=_task)
                logits = classifier(pooled_output[idx])
                num_labels = self.num_labels[self.finetuning_task.index(_task)]
                loss = loss_fct(logits.view(-1, num_labels), labels[idx].view(-1))
                if overall_loss is None:
                    overall_loss = loss
                else:
                    overall_loss += loss

            return (overall_loss,)
        else:
            classifier = self.get_classifier(task=task)
            logits = classifier(pooled_output)
            num_labels = self.num_labels[self.finetuning_task.index(task)]

        outputs = (logits,) + distilbert_output[1:]
        if labels is not None:
            if self.num_labels == 1:
                #  We are doing regression
                loss_fct = nn.MSELoss()
                loss = loss_fct(logits.view(-1), labels.view(-1))
            else:
                if replay:  # Instead of returning 'mean' loss, return 'sum' for replay
                    if device is not None:
                        loss_fct = nn.CrossEntropyLoss(reduction="sum").to(device)
                    else:
                        loss_fct = nn.CrossEntropyLoss(reduction="sum")
                else:
                    if device is not None:
                        loss_fct = nn.CrossEntropyLoss(reduction=reduction).to(device)
                    else:
                        loss_fct = nn.CrossEntropyLoss(reduction=reduction)
                loss = loss_fct(logits.view(-1, num_labels), labels.view(-1))
            outputs = (loss,) + outputs

        return outputs
