import torch
from loguru import logger
import time
import datetime
from utils.utils import *
from utils.training_utils import *
from utils.model_utils import *
from utils.data_utils import *
import utils.loss_utils as lu
from omegaconf import OmegaConf
import wandb




from models.DIFUSCO_baselines import DIFUSCODataset

import tqdm


def test():
    model = get_class(f"{conf.model.classPath}.{conf.model.name}")(conf).to(conf.training.device)
    ckpt = list(filter(lambda x: conf.dataset.name in x, all_ckpts))
    assert len(ckpt) == 1, f"Multiple or no checkpoints found for {conf.dataset.name}"
    ckpt = ckpt[0]
    model.load_state_dict(torch.load(os.path.join(CKPT_PATH, ckpt))["model_state_dict"])
    test_data = DIFUSCODataset(conf, "test", logger.info)

    start_time = time.time()
    decoder_std, decoder_mse, decoder_ratio, non_decoder_std, non_decoder_mse, non_decoder_ratio = baseline_evaluate(model, test_data, is_test=True)

    assert conf.model.SUP == False
    if conf.model.SUP:
        mse = non_decoder_mse
        ratio = non_decoder_ratio
        std = non_decoder_std
    else:
        mse = decoder_mse
        ratio = decoder_ratio
        std = decoder_std

    logger.info(f"TEST mse: {mse:.6f} ratio: {ratio:.6f} std: {std:.6f} Time: {time.time()-start_time:.6f}")
    wandb.log({'test_mse_loss': mse, 'test_ratio': ratio, 'test_std': std})
        



def baseline_evaluate(model, sampler, is_test=False):
    model.eval()

    pred_list = []
    non_decoder_pred_list = [] 
    targets = []
    if is_test:
        sample_loader = test_dataloader(model.args, sampler)
    else: 
        sample_loader = val_dataloader(model.args, sampler)
    # n_batches = sampler.create_batches(shuffle=False)
    for batch_idx, batch_data in tqdm.tqdm(enumerate(sample_loader)):
        batch_data = [x.to(model.device) for x in batch_data]
        pred, non_decoder_pred = model.test_step(batch_data, batch_idx)
        pred_list.append(pred)
        non_decoder_pred_list.append(non_decoder_pred)
        targets.append(batch_data[1].x.sum().item())
    all_pred = torch.tensor(pred_list, device=model.device)
    all_non_decoder_pred = torch.tensor(non_decoder_pred_list, device=model.device)
    all_target = torch.tensor(targets, device=model.device)
    solver_mse_std = (((all_target - all_pred)**2).std() / np.sqrt(len(all_target))).item()

    mse = torch.nn.functional.mse_loss(all_target, all_pred, reduction="mean").item()
    ratio = (torch.round(all_pred) / all_target).mean()
    non_decoder_mse = torch.nn.functional.mse_loss(all_target, all_non_decoder_pred, reduction="mean").item()
    non_decoder_ratio = (torch.round(all_non_decoder_pred) / all_target).mean()
    sum_p_std = (((all_target - all_non_decoder_pred)**2).std() / np.sqrt(len(all_target))).item()

    return  solver_mse_std, mse, ratio, sum_p_std, non_decoder_mse, non_decoder_ratio


from torch_geometric.data import DataLoader as GraphDataLoader

def train_dataloader(args,dataset):
    batch_size = args.batch_size
    train_dataloader = GraphDataLoader(
        dataset, batch_size=batch_size, shuffle=True,
        num_workers=args.num_workers, pin_memory=True,
        persistent_workers=True, drop_last=True)
    return train_dataloader

def test_dataloader(args, dataset):
    batch_size = 1
    test_dataloader = GraphDataLoader(dataset, batch_size=batch_size, shuffle=False)
    return test_dataloader

def val_dataloader(args,dataset):
    batch_size = 1
    val_dataset = torch.utils.data.Subset(dataset, range(args.validation_examples))
    val_dataloader = GraphDataLoader(val_dataset, batch_size=batch_size, shuffle=False)
    return val_dataloader


if __name__ == "__main__":
    main_conf = OmegaConf.load("configs/config.yaml")
    cli_conf = OmegaConf.from_cli()
    data_conf = OmegaConf.load(f"configs/data_configs/{cli_conf.dataset.name}.yaml")
    model_conf = OmegaConf.load(f"configs/model_configs/{cli_conf.model.name}.yaml")
    conf = OmegaConf.merge(main_conf, data_conf, model_conf, cli_conf)
    conf = OmegaConf.merge(conf, cli_conf)
    run_time = "{date:%Y-%m-%d||%H:%M:%S}".format(date=datetime.datetime.now())
    print(OmegaConf.to_yaml(conf))
    
    task_name = ",".join("{}={}".format(*i) for i in conf.model.items() if (i[0] != 'classPath' and i[0] != 'name' and i[0] != 'EQ'))

    if conf.model.EQ:
        conf.task.name = f"{conf.model.name}_EQ_{conf.dataset.name}_{task_name}_{conf.training.loss_fn}_{run_time}"
    else:
        conf.task.name = f"{conf.model.name}_{conf.dataset.name}_{task_name}_{conf.training.loss_fn}_{run_time}"

        
    conf.log.dir = 'test_logs'
    os.makedirs(conf.log.dir, exist_ok=True)

    # **
    decoder_steps = conf.model.parallel_sampling

    conf.task.name = f"{conf.model.name}_{conf.model.diffusion_type}_{conf.dataset.name}_decoder_steps={decoder_steps}"

    print(OmegaConf.to_yaml(conf))

    if conf.model.diffusion_type == 'gaussian':

        CKPT_PATH = 'best_weights/DIFUSCO (G)'
        all_ckpts = os.listdir(CKPT_PATH)
    else:
        CKPT_PATH = 'best_weights/DIFUSCO (C)'
        all_ckpts = os.listdir(CKPT_PATH)

    
    
    open(f"{conf.log.dir}/{conf.task.name}.log", "w").close()  # Clear log file
    logger.add(f"{conf.log.dir}/{conf.task.name}.log")
    logger.info(OmegaConf.to_yaml(conf))
    torch.backends.cuda.matmul.allow_tf32 = False
    torch.backends.cudnn.allow_tf32 = False

    wandb.init(
        project=conf.task.wandb_project,
        name=conf.task.name,
        group=conf.task.name,
        config={

            'learning_rate': conf.training.learning_rate,
            'weight_decay': conf.training.weight_decay,
            'dropout': conf.training.dropout,
            'num_epochs': conf.training.num_epochs,
            'seed': conf.training.seed,
            'batch_size': conf.training.batch_size,
            'dataset_name': conf.dataset.name,
            'dataset_max_node_set_size': conf.dataset.max_node_set_size,
            'dataset_max_edge_set_size': conf.dataset.max_edge_set_size,
            'model_name': conf.model.name,
        }
    )


    set_seed(conf.training.seed, conf)
    test()
