import torch
from loguru import logger
import time
import datetime
from utils.utils import *
from utils.training_utils import *
from utils.model_utils import *
from utils.data_utils import *
import utils.loss_utils as lu
from omegaconf import OmegaConf
import wandb
from collections import defaultdict 
from erdos.utils import get_diracs


def baseline_evaluate(model, sampler):
    model.eval()

    pred_list = []
    targets = []

    n_batches = sampler.create_batches(shuffle=False)
    for i in range(n_batches):
        (
            corpus_batch_data, 
            corpus_batch_data_node_sizes, 
            corpus_batch_data_edge_sizes, 
            batch_target, 
            corpus_batch_adj
        ) = sampler.fetch_batched_data_by_id(i)
        _, pred = model(sampler.packed_query_graphs, 
            sampler.query_graph_node_sizes, 
            sampler.query_graph_edge_sizes, 
            sampler.query_adj_list, 
            corpus_batch_data, 
            corpus_batch_data_node_sizes, 
            corpus_batch_data_edge_sizes,
            corpus_batch_adj)
        #pred_list.append(torch.stack(pred,dim=0).data)
        pred_list.append(pred)
        targets.append(batch_target)
    all_pred = torch.cat(pred_list, dim=0)
    all_target = torch.cat(targets, dim=0)
    mse = torch.nn.functional.mse_loss(all_target, all_pred, reduction="mean").item()
    ratio = (torch.round(all_pred) / all_target).mean()
    sum_p_std = (((all_target - all_pred)**2).std() / np.sqrt(len(all_target))).item()


    return  sum_p_std, mse, ratio

def test():
    test_data = CliqueDataset(conf, "test", logger.info)
    test_data.data_type = "pyg"
    run = 'testing'

    ckpt = list(filter(lambda x: conf.dataset.name in x, all_ckpts))
    assert len(ckpt) == 1, f"Multiple or no checkpoints found for {conf.dataset.name}"
    ckpt = ckpt[0]
    model = get_class(f"{conf.model.classPath}.{conf.model.name}")(conf).to(conf.training.device)
    model.load_state_dict(torch.load(os.path.join(CKPT_PATH, ckpt))['model_state_dict'])
    start_time = time.time()
    std, mse, ratio = baseline_evaluate(model, test_data)
    logger.info(f"Run: {run} TEST mse: {mse:.6f} ratio: {ratio:.6f} std: {std:.6f} Time: {time.time()-start_time:.6f}")
    wandb.log({'test_mse_loss': mse, 'test_ratio': ratio, 'test_std': std})



if __name__ == "__main__":
    main_conf = OmegaConf.load("configs/config.yaml")
    cli_conf = OmegaConf.from_cli()
    data_conf = OmegaConf.load(f"configs/data_configs/{cli_conf.dataset.name}.yaml")
    model_conf = OmegaConf.load(f"configs/model_configs/{cli_conf.model.name}.yaml")
    conf = OmegaConf.merge(main_conf, data_conf, model_conf, cli_conf)
    conf = OmegaConf.merge(conf, cli_conf)



    run_time = "{date:%Y-%m-%d||%H:%M:%S}".format(date=datetime.datetime.now())

    
    task_name = ",".join("{}={}".format(*i) for i in conf.model.items() if (i[0] != 'classPath' and i[0] != 'name' and i[0] != 'EQ'))

    if conf.model.EQ:
        conf.task.name = f"{conf.model.name}_EQ_{conf.dataset.name}_{task_name}_{conf.training.loss_fn}_{run_time}"
    else:
        conf.task.name = f"{conf.model.name}_{conf.dataset.name}_{task_name}_{conf.training.loss_fn}_{run_time}"
    conf.log.dir = 'test_logs'
    os.makedirs(conf.log.dir, exist_ok=True)

    # **

    conf.task.name = f"{conf.model.name}_{conf.dataset.name}"

    print(OmegaConf.to_yaml(conf))

    CKPT_PATH = f'best_weights/{conf.model.name}'
    all_ckpts = os.listdir(CKPT_PATH)

    open(f"{conf.log.dir}/{conf.task.name}.log", "w").close()  # Clear log file
    logger.add(f"{conf.log.dir}/{conf.task.name}.log")
    logger.info(OmegaConf.to_yaml(conf))
    torch.backends.cuda.matmul.allow_tf32 = False
    torch.backends.cudnn.allow_tf32 = False

    wandb.init(
        project=conf.task.wandb_project,
        name=conf.task.name,
        group=conf.task.name,
        config={
            # 'grouper': conf.model.group_name,
            'learning_rate': conf.training.learning_rate,
            'weight_decay': conf.training.weight_decay,
            'dropout': conf.training.dropout,
            'num_epochs': conf.training.num_epochs,
            'seed': conf.training.seed,
            'batch_size': conf.training.batch_size,
            'dataset_name': conf.dataset.name,
            'dataset_max_node_set_size': conf.dataset.max_node_set_size,
            'dataset_max_edge_set_size': conf.dataset.max_edge_set_size,
            'model_name': conf.model.name,
            'problem': conf.model.problem,
        }
    )


    set_seed(conf.training.seed, conf)
    # gmn_config = modify_gmn_main_config(get_default_gmn_config(conf), conf, logger)
    test()
