import torch
from loguru import logger
import time
import datetime
from utils.utils import *
from utils.training_utils import *
from utils.model_utils import *
from utils.data_utils import *
import utils.loss_utils as lu
from omegaconf import OmegaConf
import wandb
from collections import defaultdict 
from erdos.utils import get_diracs

def baseline_evaluate(model, sampler):
    model.eval()

    pred_list = []
    targets = []

    n_batches = sampler.create_batches(shuffle=False)
    for i in range(n_batches):
        (
            corpus_batch_data, 
            corpus_batch_data_node_sizes, 
            corpus_batch_data_edge_sizes, 
            batch_target, 
            corpus_batch_adj
        ) = sampler.fetch_batched_data_by_id(i)
        _, pred = model(sampler.packed_query_graphs, 
            sampler.query_graph_node_sizes, 
            sampler.query_graph_edge_sizes, 
            sampler.query_adj_list, 
            corpus_batch_data, 
            corpus_batch_data_node_sizes, 
            corpus_batch_data_edge_sizes,
            corpus_batch_adj)

        pred_list.append(pred)
        targets.append(batch_target)
    all_pred = torch.cat(pred_list, dim=0)
    all_target = torch.cat(targets, dim=0)
    mse = torch.nn.functional.mse_loss(all_target, all_pred, reduction="mean").item()
    ratio = (torch.round(all_pred) / all_target).mean()

    return  mse, ratio


def train():
    train_data = CliqueDataset(conf, "train", logger.info)
    val_data = CliqueDataset(conf, "val", logger.info)
    train_data.data_type = "pyg"
    val_data.data_type = "pyg"
    es = EarlyStoppingModule(conf.base_dir, conf.task.name, logger=logger)
    logger.info(es.__dict__)
    logger.info(f"This uses the {conf.model.classPath}.{conf.model.name} model")
    logger.info(conf)
    model = get_class(f"{conf.model.classPath}.{conf.model.name}")(conf).to(conf.training.device)
    logger.info(model)
    if conf.training.wandb_watch:
        wandb.watch(model, log_freq=1)
    logger.info(f"no. of params in model: {sum([p.numel() for p in model.parameters()])}")
    optimizer = torch.optim.Adam(
        model.parameters(),
        lr=conf.training.learning_rate,
        weight_decay=conf.training.weight_decay,
    )
    if conf.training.overwrite:
        checkpoint = None
    else:
        checkpoint = es.load_latest_model()

    if not checkpoint:
        es.save_initial_model(model)
        run = 0
    else:
        if es.should_stop_now:
            logger.info("Training has been completed. This logfile can be deleted.")
            return
        else:
            model.load_state_dict(checkpoint["model_state_dict"])
            optimizer.load_state_dict(checkpoint["optim_state_dict"])
            run = checkpoint["epoch"] + 1

    best_val_mse = 1e5
    best_val_ratio = -1




    while conf.training.run_till_early_stopping and run < conf.training.num_epochs:
        model.train()
        start_time = time.time()
        n_batches = train_data.create_batches(shuffle=True)
        epoch_loss = 0


        start_time = time.time()
        for i in range(n_batches):
            (   
                corpus_batch_data, 
                corpus_batch_data_node_sizes, 
                corpus_batch_data_edge_sizes, 
                batch_target, 
                corpus_batch_adj
            ) = train_data.fetch_batched_data_by_id(i)

            optimizer.zero_grad()
            losses, pred = model(train_data.packed_query_graphs, 
                        train_data.query_graph_node_sizes, 
                        train_data.query_graph_edge_sizes, 
                        train_data.query_adj_list, 
                        corpus_batch_data, 
                        corpus_batch_data_node_sizes, 
                        corpus_batch_data_edge_sizes,
                        corpus_batch_adj)
            if model.conf.model.SUP:
                if model.conf.model.name in ['REINFORCE', 'SFE', 'NSFE', 'ST'] : 
                    sup_loss = torch.nn.functional.mse_loss(pred,batch_target)
                    losses = losses + sup_loss
                else:
                    raise NotImplementedError()
            losses.backward()
            optimizer.step()
            epoch_loss = epoch_loss + losses.item()

        logger.info(
            f"Run: {run} train loss: {epoch_loss/n_batches:.2f} Time: {time.time()-start_time:.2f}",
        )
        start_time = time.time()
        mse, ratio = baseline_evaluate(model, val_data)
        if mse < best_val_mse:
            best_val_mse = mse
            best_val_ratio = ratio

        logger.info(f"Run: {run} VAL mse: {mse:.6f}  ratio: {ratio:.6f} Time: {time.time()-start_time:.6f}")

        log_dict = {'val_mse': mse,
                    'best_val_mse': best_val_mse,
                    'best_val_ratio': best_val_ratio,
                    'train_loss': epoch_loss/n_batches
                    }           
        wandb.log(log_dict)

        if conf.training.run_till_early_stopping:
            es_score = -mse
            if es.check([es_score], model, run, optimizer):
                break
            run += 1
        
    ckpt = es.load_best_model()
    model.load_state_dict(ckpt["model_state_dict"])
    test_data = CliqueDataset(conf, "test", logger.info)
    test_data.data_type = "pyg"
    
    mse, ratio= baseline_evaluate(model, val_data) 
    logger.info(f"Run: {run} VAL mse: {mse:.6f} ratio: {ratio:.6f} Time: {time.time()-start_time:.6f}")
    mse, ratio= baseline_evaluate(model, test_data) 
    logger.info(f"Run: {run} TEST mse: {mse:.6f} ratio: {ratio:.6f} Time: {time.time()-start_time:.6f}")
    wandb.log({'test_mse_loss': mse, 'test_ratio': ratio})



if __name__ == "__main__":
    main_conf = OmegaConf.load("configs/config.yaml")
    cli_conf = OmegaConf.from_cli()
    data_conf = OmegaConf.load(f"configs/data_configs/{cli_conf.dataset.name}.yaml")
    model_conf = OmegaConf.load(f"configs/model_configs/{cli_conf.model.name}.yaml")
    conf = OmegaConf.merge(main_conf, data_conf, model_conf, cli_conf)
    conf = OmegaConf.merge(conf, cli_conf)
    run_time = "{date:%Y-%m-%d||%H:%M:%S}".format(date=datetime.datetime.now())
    print(OmegaConf.to_yaml(conf))
    
    task_name = ",".join("{}={}".format(*i) for i in conf.model.items() if (i[0] != 'classPath' and i[0] != 'name' and i[0] != 'EQ'))

    if conf.model.EQ:
        conf.task.name = f"{conf.model.name}_EQ_{conf.dataset.name}_{task_name}_{conf.training.loss_fn}_{run_time}"
    else:
        conf.task.name = f"{conf.model.name}_{conf.dataset.name}_{task_name}_{conf.training.loss_fn}_{run_time}"

    open(f"{conf.log.dir}/{conf.task.name}.log", "w").close()  # Clear log file
    logger.add(f"{conf.log.dir}/{conf.task.name}.log")
    logger.info(OmegaConf.to_yaml(conf))
    torch.backends.cuda.matmul.allow_tf32 = False
    torch.backends.cudnn.allow_tf32 = False

    wandb.init(
        project=conf.task.wandb_project,
        name=conf.task.name,
        group=conf.task.wandb_group,
        config={
            'learning_rate': conf.training.learning_rate,
            'weight_decay': conf.training.weight_decay,
            'dropout': conf.training.dropout,
            'num_epochs': conf.training.num_epochs,
            'seed': conf.training.seed,
            'batch_size': conf.training.batch_size,
            'dataset_name': conf.dataset.name,
            'dataset_max_node_set_size': conf.dataset.max_node_set_size,
            'dataset_max_edge_set_size': conf.dataset.max_edge_set_size,
            'model_name': conf.model.name,
            'problem': conf.model.problem,
        }
    )


    set_seed(conf.training.seed, conf)
    train()
