# coding=utf-8
# Copyright 2020-present the HuggingFace Inc. team.
#
# Licensed under the Apache License, Version 2.0 (the "License");
# you may not use this file except in compliance with the License.
# You may obtain a copy of the License at
#
#     http://www.apache.org/licenses/LICENSE-2.0
#
# Unless required by applicable law or agreed to in writing, software
# distributed under the License is distributed on an "AS IS" BASIS,
# WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
# See the License for the specific language governing permissions and
# limitations under the License.
"""
The Trainer class, to easily train a 🤗 Transformers from scratch or finetune it on a new task.
"""

import contextlib
import functools
import glob
import inspect
import math
import os
import random
import re
import shutil
import sys
import time
import warnings
from collections.abc import Mapping
from pathlib import Path
from typing import TYPE_CHECKING, Any, Callable, Dict, List, Optional, Tuple, Union
import copy
from metrics import f1
import numpy as np

from tqdm.auto import tqdm
from transformers import Trainer
from sklearn.linear_model import LinearRegression, LogisticRegression, LogisticRegressionCV

# Integrations must be imported before ML frameworks:
from transformers.integrations import (  # isort: split
    default_hp_search_backend,
    get_reporting_integration_callbacks,
    hp_params,
    is_fairscale_available,
    is_optuna_available,
    is_ray_tune_available,
    is_sigopt_available,
    is_wandb_available,
    run_hp_search_optuna,
    run_hp_search_ray,
    run_hp_search_sigopt,
    run_hp_search_wandb,
)

import numpy as np
import torch
import torch.distributed as dist
from packaging import version
from torch import nn
from torch.utils.data import DataLoader, Dataset, RandomSampler, SequentialSampler
from torch.utils.data.distributed import DistributedSampler

from huggingface_hub import Repository

from transformers import __version__
from transformers.configuration_utils import PretrainedConfig
from transformers.data.data_collator import DataCollator, DataCollatorWithPadding, default_data_collator
from transformers.debug_utils import DebugOption, DebugUnderflowOverflow
from transformers.deepspeed import deepspeed_init, is_deepspeed_zero3_enabled
from transformers.dependency_versions_check import dep_version_check
from transformers.modelcard import TrainingSummary
from transformers.modeling_utils import PreTrainedModel, load_sharded_checkpoint, unwrap_model
from transformers.models.auto.modeling_auto import MODEL_FOR_CAUSAL_LM_MAPPING_NAMES, MODEL_MAPPING_NAMES
from transformers.optimization import Adafactor, get_scheduler
from transformers.pytorch_utils import ALL_LAYERNORM_LAYERS, is_torch_greater_or_equal_than_1_10, is_torch_less_than_1_11
from transformers.tokenization_utils_base import PreTrainedTokenizerBase
from transformers.trainer_callback import (
    CallbackHandler,
    DefaultFlowCallback,
    PrinterCallback,
    ProgressCallback,
    TrainerCallback,
    TrainerControl,
    TrainerState,
)
from transformers.trainer_pt_utils import (
    DistributedLengthGroupedSampler,
    DistributedSamplerWithLoop,
    DistributedTensorGatherer,
    IterableDatasetShard,
    LabelSmoother,
    LengthGroupedSampler,
    SequentialDistributedSampler,
    ShardSampler,
    distributed_broadcast_scalars,
    distributed_concat,
    find_batch_size,
    get_module_class_from_name,
    get_parameter_names,
    nested_concat,
    nested_detach,
    nested_numpify,
    nested_truncate,
    nested_xla_mesh_reduce,
    reissue_pt_warnings,
)
from transformers.trainer_utils import (
    PREFIX_CHECKPOINT_DIR,
    BestRun,
    EvalLoopOutput,
    EvalPrediction,
    FSDPOption,
    HPSearchBackend,
    HubStrategy,
    IntervalStrategy,
    PredictionOutput,
    RemoveColumnsCollator,
    ShardedDDPOption,
    TrainerMemoryTracker,
    TrainOutput,
    default_compute_objective,
    default_hp_space,
    denumpify_detensorize,
    enable_full_determinism,
    find_executable_batch_size,
    get_last_checkpoint,
    has_length,
    number_of_arguments,
    seed_worker,
    set_seed,
    speed_metrics,
)
from transformers.training_args import OptimizerNames, ParallelMode, TrainingArguments
from transformers.utils import (
    CONFIG_NAME,
    WEIGHTS_INDEX_NAME,
    WEIGHTS_NAME,
    find_labels,
    get_full_repo_name,
    is_apex_available,
    is_datasets_available,
    is_in_notebook,
    is_ipex_available,
    is_sagemaker_dp_enabled,
    is_sagemaker_mp_enabled,
    is_torch_tensorrt_fx_available,
    is_torch_tpu_available,
    is_torchdynamo_available,
    logging,
)
from transformers.utils.generic import ContextManagers
import logging as py_logging
from torch.utils.data import Subset

from torch.nn import CrossEntropyLoss
import torch.nn.functional as F


_is_native_cpu_amp_available = is_torch_greater_or_equal_than_1_10

DEFAULT_CALLBACKS = [DefaultFlowCallback]
DEFAULT_PROGRESS_CALLBACK = ProgressCallback

if is_in_notebook():
    from .utils.notebook import NotebookProgressCallback

    DEFAULT_PROGRESS_CALLBACK = NotebookProgressCallback

if is_apex_available():
    from apex import amp

if is_datasets_available():
    import datasets

if is_torch_tpu_available(check_device=False):
    import torch_xla.core.xla_model as xm
    import torch_xla.debug.metrics as met
    import torch_xla.distributed.parallel_loader as pl

if is_fairscale_available():
    dep_version_check("fairscale")
    import fairscale
    from fairscale.nn.data_parallel import FullyShardedDataParallel as FullyShardedDDP
    from fairscale.nn.data_parallel import ShardedDataParallel as ShardedDDP
    from fairscale.nn.wrap import auto_wrap
    from fairscale.optim import OSS
    from fairscale.optim.grad_scaler import ShardedGradScaler


if is_sagemaker_mp_enabled():
    import smdistributed.modelparallel.torch as smp
    from smdistributed.modelparallel import __version__ as SMP_VERSION

    IS_SAGEMAKER_MP_POST_1_10 = version.parse(SMP_VERSION) >= version.parse("1.10")

    from .trainer_pt_utils import smp_forward_backward, smp_forward_only, smp_gather, smp_nested_concat
else:
    IS_SAGEMAKER_MP_POST_1_10 = False


if TYPE_CHECKING:
    import optuna

logger = logging.get_logger(__name__)


# Name of the files used for checkpointing
TRAINING_ARGS_NAME = "training_args.bin"
TRAINER_STATE_NAME = "trainer_state.json"
OPTIMIZER_NAME = "optimizer.pt"
SCHEDULER_NAME = "scheduler.pt"
SCALER_NAME = "scaler.pt"

def forward_wrap_with_option_len(self, input_ids=None, labels=None, option_len=None, num_options=None, return_dict=None, **kwargs):
    """
    This is to replace the original forward function of Transformer models to enable:
    (1) Partial target sequence: loss will only be calculated on part of the sequence
    (2) Classification-style training: a classification loss (CE) will be calculated over several options
    Input:
    - input_ids, labels: same as the original forward function
    - option_len: a list of int indicating the option lengths, and loss will be calculated only on the
    last option_len tokens 
    - num_options: a list of int indicating the number of options for each example (this will be #label
    words for classification tasks and #choices for multiple choice tasks), and a classification loss
    will be calculated.
    """
    outputs = self.original_forward(input_ids=input_ids, **kwargs)
    if labels is None:
        return outputs
    logits = outputs.logits

    loss = None
    # Shift so that tokens < n predict n
    shift_logits = logits[..., :-1, :].contiguous()
    # Here we use input_ids (which should always = labels) bc sometimes labels are correct candidate IDs
    shift_labels = torch.clone(input_ids)[..., 1:].contiguous()
    shift_labels[shift_labels == self.config.pad_token_id] = -100

    # Apply option len (do not calculate loss on the non-option part)
    for _i, _len in enumerate(option_len):
        shift_labels[_i, :-_len] = -100
        
        
    ################## classifier logits #####################
    accs = []
    ##########################################################

    # Calculate the loss
    loss_fct = CrossEntropyLoss(ignore_index=-100)
    if num_options is not None: 
        # Train as a classification tasks
        log_probs = F.log_softmax(shift_logits, dim=-1)
        mask = shift_labels != -100 # Option part
        shift_labels[~mask] = 0 # So that it doesn't mess up with indexing

        selected_log_probs = torch.gather(log_probs, dim=-1, index=shift_labels.unsqueeze(-1)).squeeze(-1) # (bsz x num_options, len)
        selected_log_probs = (selected_log_probs * mask).sum(-1) / mask.sum(-1) # (bsz x num_options)

        if any([x != num_options[0] for x in num_options]):
            # Multi choice tasks with different number of options
            loss = 0
            start_id = 0
            count = 0
            while start_id < len(num_options):
                end_id = start_id + num_options[start_id]
                _logits = selected_log_probs[start_id:end_id].unsqueeze(0) # (1, num_options)
                _labels = labels[start_id:end_id][0].unsqueeze(0) # (1)
                loss = loss_fct(_logits, _labels) + loss
                accs.append((_logits.argmax() == _labels).sum() / len(_labels))
                count += 1
                start_id = end_id
            loss = loss / count
        else:
            num_options = num_options[0]
            selected_log_probs = selected_log_probs.view(-1, num_options) # (bsz, num_options)
            labels = labels.view(-1, num_options)[:, 0] # Labels repeat so we only take the first one
            loss = loss_fct(selected_log_probs, labels)
            acc = (selected_log_probs.argmax(-1) == labels).sum() / len(labels)
            accs.append(acc)
    else:
        shift_logits = shift_logits.view(-1, self.config.vocab_size)
        shift_labels = shift_labels.view(-1)
        loss = loss_fct(shift_logits, shift_labels)
        accs = [(shift_logits.argmax(-1).sum() == shift_labels).sum() / len(shift_labels)]

    if not return_dict:
        output = (logits,) + outputs[1:]
        # return (loss,) + output if loss is not None else output
        return (loss, accs)

    return CausalLMOutputWithPast(
        loss=loss,
        # clogits=clogits,
        logits=logits,
        # logits=clogits,
        past_key_values=outputs.past_key_values,
        hidden_states=outputs.hidden_states,
        attentions=outputs.attentions,
    )


class OurTrainer(Trainer):
    from transformers.trainer_pt_utils import _get_learning_rate, log_metrics, metrics_format, save_metrics, save_state

    def _inner_training_loop(
        self, batch_size=None, args=None, resume_from_checkpoint=None, trial=None, ignore_keys_for_eval=None
    ):
        # 创建一个文件处理器，设置日志文件路径
        file_handler = py_logging.FileHandler(os.path.join(self.args.output_dir, self.args.log_file))
        file_handler.setLevel(py_logging.INFO)
        # 将处理器添加到 logger
        logger.addHandler(file_handler)
        
        if self.args.fed_num == 0: # no federated
            return self._inner_training_loop_MeZO(batch_size, args, resume_from_checkpoint, trial, ignore_keys_for_eval)
        elif self.args.fed_num > 0: # federated
            if self.args.use_CeZO: # CeZO
                logger.info("-------------------------------------------------------------- Using CeZO --------------------------------------------------------------")
                return self._inner_training_loop_CeZO(batch_size, args, resume_from_checkpoint, trial, ignore_keys_for_eval)
            else:
                logger.info("-------------------------------------------------------------- Using Federated --------------------------------------------------------------")
                if self.args.aggregation_freq == "batch":
                    return self._inner_training_loop_FedSGD(batch_size, args, resume_from_checkpoint, trial, ignore_keys_for_eval)
                else:
                    raise NotImplementedError()
                    return self._inner_training_loop_FedAvg(batch_size, args, resume_from_checkpoint, trial, ignore_keys_for_eval)
        else:
            raise ValueError("fed_num should be non-negative")

    ############## CeZO ##############
    def split_dataloader(self, dataloader, fed_num):
        # 获取原始DataLoader中的所有数据和标签
        dataset = dataloader.dataset
        batch_size = dataloader.batch_size
        indices = list(range(len(dataset)))
        # 将数据索引均分为fed_num份
        splitted_indices = [indices[i::fed_num] for i in range(fed_num)]
        # 确保每份中数据数量相同
        sample_num = max(len(indices) for indices in splitted_indices)
        # 为每一份数据创建新的DataLoader
        dataloaders = []
        for indices in splitted_indices:
            if len(indices) < sample_num:
                temp_slice = indices[0: sample_num - len(indices)]
                indices = indices + temp_slice
            subset = Subset(dataset, indices)
            new_dataloader = DataLoader(subset, batch_size=batch_size, shuffle=True, collate_fn=dataloader.collate_fn)
            dataloaders.append(new_dataloader)
        return dataloaders
    
    def split_dataloader_non_iid(self, dataloader, fed_num, alpha):
        # 获取原始DataLoader中的所有数据和标签
        K = fed_num
        dstrain = dataloader.dataset
        batch_size = dataloader.batch_size
        
        def dirichlet_split_noniid(train_labels, alpha, n_clients):
            '''
            按照参数为alpha的Dirichlet分布将样本索引集合划分为n_clients个子集
            '''
            n_classes = train_labels.max()+1
            # (K, N) 类别标签分布矩阵X，记录每个类别划分到每个client去的比例
            label_distribution = np.random.dirichlet([alpha]*n_clients, n_classes)
            # (K, ...) 记录K个类别对应的样本索引集合
            class_idcs = [np.argwhere(train_labels == y).flatten()
                        for y in range(n_classes)]

            # 记录N个client分别对应的样本索引集合
            client_idcs = [[] for _ in range(n_clients)]
            for k_idcs, fracs in zip(class_idcs, label_distribution):
                # np.split按照比例fracs将类别为k的样本索引k_idcs划分为了N个子集
                # i表示第i个client，idcs表示其对应的样本索引集合idcs
                for i, idcs in enumerate(np.split(k_idcs,
                                                (np.cumsum(fracs)[:-1]*len(k_idcs)).
                                                astype(int))):
                    client_idcs[i] += [idcs]

            client_idcs = [np.concatenate(idcs) for idcs in client_idcs]

            return client_idcs
        
        print('Making subsets according a Dirichlet distribution with alpha=%.4f' % alpha)
        labels = np.array([dstrain.__getitem__(_)[0]['labels'] for _ in range(len(dstrain))])
        client_idcs = dirichlet_split_noniid(labels, alpha=alpha, n_clients=K)
        splitted_indices = [Subset(dstrain, [int(_) for _ in np.random.choice(client_idcs[ik], len(dstrain) // K, replace=True)]) for ik in range(K)]
        
        # 为每一份数据创建新的DataLoader
        dataloaders = []
        for indices in splitted_indices:
            new_dataloader = DataLoader(indices, batch_size=batch_size, shuffle=True, collate_fn=dataloader.collate_fn)
            dataloaders.append(new_dataloader)
        return dataloaders

    def is_batch_nums_equal(self, data_loaders):
        batch_num = len(data_loaders[0])
        for data_loader in data_loaders:
            if len(data_loader) != batch_num:
                return False
        return True

    def _maybe_log_save_evaluate(self, tr_loss, model, trial, epoch, ignore_keys_for_eval):
        if self.control.should_log:
            if is_torch_tpu_available():
                xm.mark_step()

            logs: Dict[str, float] = {}

            # all_gather + mean() to get average loss over all processes
            tr_loss_scalar = self._nested_gather(tr_loss).mean().item()

            # reset tr_loss to zero
            tr_loss -= tr_loss

            logs["loss"] = round(tr_loss_scalar / (self.state.global_step - self._globalstep_last_logged), 4)
            logs["learning_rate"] = self._get_learning_rate()

            self._total_loss_scalar += tr_loss_scalar
            self._globalstep_last_logged = self.state.global_step
            self.store_flos()

            # self.log(logs)
            logger.info(logs)
            
        self.original_forward = self.model.forward
        metrics = None
        if self.control.should_evaluate:
            if isinstance(self.eval_dataset, dict):
                metrics = {}
                for eval_dataset_name, eval_dataset in self.eval_dataset.items():
                    dataset_metrics = self.evaluate(
                        eval_dataset=eval_dataset,
                        ignore_keys=ignore_keys_for_eval,
                        metric_key_prefix=f"eval_{eval_dataset_name}",
                    )
                    metrics.update(dataset_metrics)
                    print('bbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbb')
            else:
                metrics = self.evaluate(ignore_keys=ignore_keys_for_eval)
                dleval = self.get_eval_dataloader(self.eval_dataset)
                
                # self.model.original_forward = self.model.forward
                self.model.forward = forward_wrap_with_option_len.__get__(self.model, type(self.model))
                # print(len(self.eval_dataset))
                # print(len(dleval))
                
                total_acc = 0.0
                total_samples = 0
                for i, x in enumerate(dleval):
                    with torch.inference_mode():
                        bs = x['input_ids'].shape[0]
                        x = self._prepare_inputs(x)
                        fx = self.model(**x)
                        if isinstance(fx, tuple):
                            loss = fx[0].detach()
                            acc = sum(fx[1]).detach() / len(fx[1])
                        else:
                            loss = fx.loss.detach()
                            acc = (fx.logits.argmax(-1) == x['labels']).sum() / torch.tensor(x['labels'].shape).prod()
                        # print(acc)
                        
                        total_acc += acc.item() * bs
                        total_samples += bs

                # 计算整个数据集的平均准确率
                average_acc = total_acc / total_samples
                metrics["acc"] = average_acc
                
                self.model.forward = self.model.original_forward
                
            self._report_to_hp_search(trial, self.state.global_step, metrics)
            logger.info(metrics)
            self.model.forward = self.original_forward

        if self.control.should_save:
            self._save_checkpoint(model, trial, metrics=metrics)
            self.control = self.callback_handler.on_save(self.args, self.state, self.control)

    def train_for_single_model_CeZO_batch(self, args, model, epoch, step, inputs, tr_loss, steps_in_epoch):
        # MeZO added: estimate gradient
        random_seed = 0
        projected_grad = 0
        
        if args.trainer == "zo":
            # random_seed = np.random.randint(1000000000)
            random_seed = self.seed_
            tr_loss_step, projected_grad = self.zo_step_CeZO(model, inputs, random_seed)
        else:
            raise ValueError("trainer should be 'zo'")

        if (
            args.logging_nan_inf_filter
            and not is_torch_tpu_available()
            and (torch.isnan(tr_loss_step) or torch.isinf(tr_loss_step))
        ):
            # if loss is nan or inf simply add the average of previous logged losses
            tr_loss += tr_loss / (1 + self.state.global_step - self._globalstep_last_logged)
        else:
            tr_loss += tr_loss_step

        self.current_flos += float(self.floating_point_ops(inputs))

        # Optimizer step for deepspeed must be called on every step regardless of the value of gradient_accumulation_steps
        if self.deepspeed:
            self.deepspeed.step()

        if (step + 1) % args.gradient_accumulation_steps == 0 or (
            # last step in epoch but step is always smaller than gradient_accumulation_steps
            steps_in_epoch <= args.gradient_accumulation_steps
            and (step + 1) == steps_in_epoch
        ):
            # MeZO added: update model with the estimated gradient
            if args.trainer == "zo":
                self.zo_update_CeZO(model, random_seed, projected_grad)
            else:
                raise ValueError("trainer should be 'zo'")

            self.state.global_step += 1
            self.state.epoch = epoch + (step + 1) / steps_in_epoch
            self.control = self.callback_handler.on_step_end(args, self.state, self.control) # 关系着训练的进度条

            # self._maybe_log_save_evaluate(tr_loss, model, trial, epoch, ignore_keys_for_eval)
        else:
            self.control = self.callback_handler.on_substep_end(args, self.state, self.control)
        
        
        if (self.args.logging_steps > 0 and self.state.global_step % self.args.logging_steps == 0):
            logs = {}
            logs["loss"] = tr_loss_step.item()
            logs["learning_rate"] = self.args.learning_rate
            logs["global_step"] = self.state.global_step
            logs["max_steps"] = self.args.max_steps
            logs["time"] = int(time.time())
            self.log(logs)
            logger.info(str(logs))
        
        
        return model, projected_grad, random_seed, tr_loss
    
    def train_for_single_model_CeZO_epoch(self, args, model, train_dataloader, epoch, epochs_trained,
                                    tr_loss, steps_trained_in_current_epoch, len_dataloader,
                                    resume_from_checkpoint, steps_trained_progress_bar):
        
        if isinstance(train_dataloader, DataLoader) and isinstance(train_dataloader.sampler, DistributedSampler):
            train_dataloader.sampler.set_epoch(epoch)
        elif hasattr(train_dataloader, "dataset") and isinstance(train_dataloader.dataset, IterableDatasetShard):
            train_dataloader.dataset.set_epoch(epoch)

        if is_torch_tpu_available():
            parallel_loader = pl.ParallelLoader(train_dataloader, [args.device]).per_device_loader(args.device)
            epoch_iterator = parallel_loader
        else:
            epoch_iterator = train_dataloader

        # Reset the past mems state at the beginning of each epoch if necessary.
        if args.past_index >= 0:
            self._past = None

        steps_in_epoch = (
            len(epoch_iterator)
            if len_dataloader is not None
            else args.max_steps * args.gradient_accumulation_steps
        )
        self.control = self.callback_handler.on_epoch_begin(args, self.state, self.control)

        if epoch == epochs_trained and resume_from_checkpoint is not None and steps_trained_in_current_epoch == 0:
            self._load_rng_state(resume_from_checkpoint)

        random_seed_epoch = [] # 存储每个batch的随机种子
        projected_grad_epoch = [] # 存储每个batch的 projected_grad
        step = -1
        for step, inputs in enumerate(epoch_iterator):

            # Skip past any already trained steps if resuming training
            if steps_trained_in_current_epoch > 0:
                steps_trained_in_current_epoch -= 1
                if steps_trained_progress_bar is not None:
                    steps_trained_progress_bar.update(1)
                if steps_trained_in_current_epoch == 0:
                    self._load_rng_state(resume_from_checkpoint)
                continue
            elif steps_trained_progress_bar is not None:
                steps_trained_progress_bar.close()
                steps_trained_progress_bar = None

            if step % args.gradient_accumulation_steps == 0:
                self.control = self.callback_handler.on_step_begin(args, self.state, self.control)

            # MeZO added: estimate gradient
            random_seed = 0
            projected_grad = 0
            
            if args.trainer == "zo":
                random_seed = np.random.randint(1000000000)
                tr_loss_step, projected_grad = self.zo_step_CeZO(model, inputs, random_seed)
                random_seed_epoch.append(random_seed)
                projected_grad_epoch.append(projected_grad)
            else:
                raise ValueError("trainer should be 'zo'")
            
            if (
                args.logging_nan_inf_filter
                and not is_torch_tpu_available()
                and (torch.isnan(tr_loss_step) or torch.isinf(tr_loss_step))
            ):
                # if loss is nan or inf simply add the average of previous logged losses
                tr_loss += tr_loss / (1 + self.state.global_step - self._globalstep_last_logged)
            else:
                tr_loss += tr_loss_step

            self.current_flos += float(self.floating_point_ops(inputs))

            # Optimizer step for deepspeed must be called on every step regardless of the value of gradient_accumulation_steps
            if self.deepspeed:
                self.deepspeed.step()

            if (step + 1) % args.gradient_accumulation_steps == 0 or (
                # last step in epoch but step is always smaller than gradient_accumulation_steps
                steps_in_epoch <= args.gradient_accumulation_steps
                and (step + 1) == steps_in_epoch
            ):
                # MeZO added: update model with the estimated gradient
                if args.trainer == "zo":
                    self.zo_update_CeZO(model, random_seed, projected_grad)
                else:
                    raise ValueError("trainer should be 'zo'")

                self.state.global_step += 1
                self.state.epoch = epoch + (step + 1) / steps_in_epoch
                self.control = self.callback_handler.on_step_end(args, self.state, self.control) # 关系着训练的进度条

                # self._maybe_log_save_evaluate(tr_loss, model, trial, epoch, ignore_keys_for_eval)
            else:
                self.control = self.callback_handler.on_substep_end(args, self.state, self.control)
            
            
            if self.control.should_epoch_stop or self.control.should_training_stop:
                break
        if step < 0:
            logger.warning(
                "There seems to be not a single sample in your epoch_iterator, stopping training at step"
                f" {self.state.global_step}! This is expected if you're using an IterableDataset and set"
                f" num_steps ({max_steps}) higher than the number of available samples."
            )
            self.control.should_training_stop = True
        return model, projected_grad_epoch, random_seed_epoch, tr_loss

    def federated_averaging_CeZO_batch(self, model, fed_num, projected_grad_batches, random_seed_batches):
        with torch.no_grad():
            print(projected_grad_batches)
            print(random_seed_batches)
            print([projected_grad_batches[q :: self.Q] for q in range(self.Q)])
            print([1 if sum(projected_grad_batches[q :: self.Q]) > 0 else -1 for q in range(self.Q)])
            
            for q in range(self.Q):
                projected_grad = 1 if sum(projected_grad_batches[q :: self.Q]) > 0 else -1
                random_seed = random_seed_batches[q]
                
                # 保存当前随机数生成器的状态
                initial_rng_state = torch.get_rng_state()
                # 更新参数
                self.zo_update_CeZO(model, random_seed, projected_grad, self.Q)
                # 恢复到初始的随机数生成器状态
                torch.set_rng_state(initial_rng_state)
    
    # federated averaging for multiple batchs
    def federated_averaging_CeZO_epoch(self, model, fed_num, projected_grad_epoches, random_seed_epoches):
        with torch.no_grad():
            for i in range(len(projected_grad_epoches)):
                for j in range(len(projected_grad_epoches[i])): # 遍历所有batch
                    # 确定更新参数
                    projected_grad = projected_grad_epoches[i][j]
                    random_seed = random_seed_epoches[i][j]
                    # 保存当前随机数生成器的状态
                    initial_rng_state = torch.get_rng_state()
                    # 更新参数
                    self.zo_update_CeZO(model, random_seed, projected_grad, fed_num)
                    # 恢复到初始的随机数生成器状态
                    torch.set_rng_state(initial_rng_state)
    
    # 2倍显存实现。
    def _inner_training_loop_CeZO(
        self, batch_size=None, args=None, resume_from_checkpoint=None, trial=None, ignore_keys_for_eval=None
    ):
        """
        We overload the original training loop to add linear probing and MeZO. Search key word "MeZO added"
        for those updates.
        """
        assert args.trainer == "zo", "This is only for ZO"
        
        # self.Q = 5
        self.Q = self.args.onebit_Q
        self.seed_ = -1
        
        self.byzantine_flag = [False] * self.args.fed_num
        for i in range(self.args.byzantine_num):
            self.byzantine_flag[i] = True
        
        # 创建一个文件处理器，设置日志文件路径
        file_handler = py_logging.FileHandler(os.path.join(self.args.output_dir, self.args.log_file))
        file_handler.setLevel(py_logging.INFO)
        # 将处理器添加到 logger
        logger.addHandler(file_handler)
        
        self._train_batch_size = batch_size
        fed_num = self.args.fed_num
        # Data loader and number of training steps
        train_dataloader = self.get_train_dataloader()

        # MeZO added: Linear probing
        if self.args.linear_probing:

            def _get_token_prediction_layer(model):
                if model.config.model_type == "opt":
                    return model.lm_head
                else:
                    raise NotImplementedError(model.config.model_type)

            def _extract_features(model, *args, **kwargs):
                """some magic for getting features pre last layer"""
                features = {}
                def __hook(model_, input_, output_):
                    features["features"] = input_[0].detach()

                _get_token_prediction_layer(model).register_forward_hook(__hook)
                model.forward(*args, **kwargs)
                return features["features"]

            logger.info("Linear probing")
            logger.info("Starting to get features for training dataset")
            targets = []
            features = []
            with torch.inference_mode():
                for step, inputs in enumerate(tqdm(train_dataloader)):
                    for k, v in inputs.items():
                        if isinstance(v, torch.Tensor):
                            inputs[k] = v.to(self.model.device)
                        
                    feature = _extract_features(self.model, **inputs)
                    target = inputs["labels"]

                    # Shift the target (bc it's autoregressive LM) and add the corresponding part
                    assert not self.args.train_as_classification and self.args.only_train_option
                    feature, target = feature[:, :-1], target[:, 1:]
                    for _i, _len in enumerate(inputs["option_len"]):
                        features.append(feature[_i, -_len:])
                        targets.append(target[_i, -_len:])

            logger.info("Finished getting features for training dataset")

            features = torch.cat(features, dim=0).cpu().numpy()
            targets = torch.cat(targets, dim=0).cpu().numpy()
            # Whether to use bias
            if self.model.config.model_type in ["opt", "gpt2"]:
                use_bias = False
            else:
                raise NotImplementedError
            # Set early stopping
            tol = 0.01 if self.args.lp_early_stopping else 1e-4 # 1e-4 is scipy default
            max_iter = 1000 if self.args.lp_early_stopping else 5000

            logger.info("Fitting logistic regression...")
            reg = LogisticRegressionCV(max_iter=max_iter, fit_intercept=use_bias, multi_class="multinomial", random_state=0, tol=tol, n_jobs=-1).fit(features, targets)
            logger.info("Done")

            logger.info("Assigning weights to model")
            decoder = _get_token_prediction_layer(self.model)
            coef_torch = torch.tensor(reg.coef_, device=decoder.weight.device, dtype=decoder.weight.dtype)
            if use_bias:
                bias_torch = torch.tensor(reg.intercept_, device=decoder.weight.device, dtype=decoder.weight.dtype)
            if coef_torch.shape[0] == 1: # The regressor only detects two classes
                assert len(reg.classes_) == 2
                coef_torch = torch.cat([-coef_torch / 2, coef_torch / 2], dim=0)
                if use_bias:
                    bias_torch = torch.cat([-bias_torch / 2, bias_torch / 2], dim=0)

            for _i, token_id in enumerate(reg.classes_):
                decoder.weight.data[token_id] = coef_torch[_i]
                if use_bias:
                    decoder.bias.data[token_id] = bias_torch[_i]

            return None

        # Setting up training control variables:
        # number of training epochs: num_train_epochs
        # number of training steps per epoch: num_update_steps_per_epoch
        # total number of training steps to execute: max_steps
        total_train_batch_size = args.train_batch_size * args.gradient_accumulation_steps * args.world_size

        len_dataloader = None
        # 获取数据集大小，并计算训练epoch数
        if has_length(train_dataloader):
            len_dataloader = len(train_dataloader)
            num_update_steps_per_epoch = len_dataloader // args.gradient_accumulation_steps
            num_update_steps_per_epoch = max(num_update_steps_per_epoch, 1)
            num_examples = self.num_examples(train_dataloader)
            if args.max_steps > 0:
                max_steps = args.max_steps
                num_train_epochs = args.max_steps // num_update_steps_per_epoch + int(
                    args.max_steps % num_update_steps_per_epoch > 0
                )
                # May be slightly incorrect if the last batch in the training dataloader has a smaller size but it's
                # the best we can do.
                num_train_samples = args.max_steps * total_train_batch_size
            else:
                max_steps = math.ceil(args.num_train_epochs * num_update_steps_per_epoch)
                num_train_epochs = math.ceil(args.num_train_epochs)
                num_train_samples = self.num_examples(train_dataloader) * args.num_train_epochs
        elif args.max_steps > 0:  # Rely on max_steps when dataloader does not have a working size
            raise NotImplementedError(
                "The training dataloader does not have a length, please set max_steps to avoid an incorrect number of training steps."
            )
            max_steps = args.max_steps
            # Setting a very large number of epochs so we go as many times as necessary over the iterator.
            num_train_epochs = sys.maxsize
            num_update_steps_per_epoch = max_steps
            num_examples = total_train_batch_size * args.max_steps
            num_train_samples = args.max_steps * total_train_batch_size
        else:
            raise ValueError(
                "args.max_steps must be set to a positive value if dataloader does not have a length, was"
                f" {args.max_steps}"
            )

        if DebugOption.UNDERFLOW_OVERFLOW in self.args.debug:
            if self.args.n_gpu > 1:
                # nn.DataParallel(model) replicates the model, creating new variables and module
                # references registered here no longer work on other gpus, breaking the module
                raise ValueError(
                    "Currently --debug underflow_overflow is not supported under DP. Please use DDP"
                    " (torch.distributed.launch)."
                )
            else:
                debug_overflow = DebugUnderflowOverflow(self.model)  # noqa

        delay_optimizer_creation = (
            self.sharded_ddp is not None
            and self.sharded_ddp != ShardedDDPOption.SIMPLE
            or is_sagemaker_mp_enabled()
            or self.fsdp is not None
        )
        if args.deepspeed:
            deepspeed_engine, optimizer, lr_scheduler = deepspeed_init(
                self, num_training_steps=max_steps, resume_from_checkpoint=resume_from_checkpoint
            )
            self.model = deepspeed_engine.module
            self.model_wrapped = deepspeed_engine
            self.deepspeed = deepspeed_engine
            self.optimizer = optimizer
            self.lr_scheduler = lr_scheduler
        elif not delay_optimizer_creation:
            self.create_optimizer_and_scheduler(num_training_steps=max_steps)

        self.state = TrainerState()
        self.state.is_hyper_param_search = trial is not None

        # Activate gradient checkpointing if needed
        if args.gradient_checkpointing:
            self.model.gradient_checkpointing_enable()

        model = self._wrap_model(self.model_wrapped)

        if is_sagemaker_mp_enabled() and resume_from_checkpoint is not None:
            self._load_from_checkpoint(resume_from_checkpoint, model)

        # for the rest of this function `model` is the outside model, whether it was wrapped or not
        if model is not self.model:
            self.model_wrapped = model

        if delay_optimizer_creation:
            self.create_optimizer_and_scheduler(num_training_steps=max_steps)

        # Check if saved optimizer or scheduler states exist
        self._load_optimizer_and_scheduler(resume_from_checkpoint)

        # important: at this point:
        # self.model         is the Transformers Model
        # self.model_wrapped is DDP(Transformers Model), Deepspeed(Transformers Model), etc.

        # Train!
        logger.info("***** Running training *****")
        logger.info(f"  Num examples = {num_examples}")
        logger.info(f"  Num Epochs = {num_train_epochs}")
        logger.info(f"  Instantaneous batch size per device = {args.per_device_train_batch_size}")
        logger.info(f"  Total train batch size (w. parallel, distributed & accumulation) = {total_train_batch_size}")
        logger.info(f"  Gradient Accumulation steps = {args.gradient_accumulation_steps}")
        logger.info(f"  Total optimization steps = {max_steps}")
        logger.info(
            f"  Number of trainable parameters = {sum(p.numel() for p in model.parameters() if p.requires_grad)}"
        )

        self.state.epoch = 0
        start_time = time.time()
        epochs_trained = 0
        steps_trained_in_current_epoch = 0
        steps_trained_progress_bar = None

        # Check if continuing training from a checkpoint
        if resume_from_checkpoint is not None and os.path.isfile(
            os.path.join(resume_from_checkpoint, TRAINER_STATE_NAME)
        ):
            self.state = TrainerState.load_from_json(os.path.join(resume_from_checkpoint, TRAINER_STATE_NAME))
            epochs_trained = self.state.global_step // num_update_steps_per_epoch
            if not args.ignore_data_skip:
                steps_trained_in_current_epoch = self.state.global_step % (num_update_steps_per_epoch)
                steps_trained_in_current_epoch *= args.gradient_accumulation_steps
            else:
                steps_trained_in_current_epoch = 0

            logger.info("  Continuing training from checkpoint, will skip to saved global_step")
            logger.info(f"  Continuing training from epoch {epochs_trained}")
            logger.info(f"  Continuing training from global step {self.state.global_step}")
            if not args.ignore_data_skip:
                logger.info(
                    f"  Will skip the first {epochs_trained} epochs then the first {steps_trained_in_current_epoch} "
                    "batches in the first epoch. If this takes a lot of time, you can add the `--ignore_data_skip` "
                    "flag to your launch command, but you will resume the training on data already seen by your model."
                )
                if self.is_local_process_zero() and not args.disable_tqdm:
                    steps_trained_progress_bar = tqdm(total=steps_trained_in_current_epoch)
                    steps_trained_progress_bar.set_description("Skipping the first batches")

        # Update the references
        self.callback_handler.model = self.model
        self.callback_handler.optimizer = self.optimizer
        self.callback_handler.lr_scheduler = self.lr_scheduler
        self.callback_handler.train_dataloader = train_dataloader
        if self.hp_name is not None and self._trial is not None:
            # use self._trial because the SigOpt/Optuna hpo only call `_hp_search_setup(trial)` instead of passing trial
            # parameter to Train when using DDP.
            self.state.trial_name = self.hp_name(self._trial)
        if trial is not None:
            assignments = trial.assignments if self.hp_search_backend == HPSearchBackend.SIGOPT else trial
            self.state.trial_params = hp_params(assignments)
        else:
            self.state.trial_params = None
        # This should be the same if the state has been saved but in case the training arguments changed, it's safer
        # to set this after the load.
        self.state.max_steps = max_steps
        self.state.num_train_epochs = num_train_epochs
        self.state.is_local_process_zero = self.is_local_process_zero()
        self.state.is_world_process_zero = self.is_world_process_zero()

        # tr_loss is a tensor to avoid synchronization of TPUs through .item()
        tr_loss = torch.tensor(0.0).to(args.device)
        # _total_loss_scalar is updated everytime .item() has to be called on tr_loss and stores the sum of all losses
        self._total_loss_scalar = 0.0
        self._globalstep_last_logged = self.state.global_step
        model.zero_grad()

        self.control = self.callback_handler.on_train_begin(args, self.state, self.control)

        # Skip the first epochs_trained epochs to get the random state of the dataloader at the right point.
        if not args.ignore_data_skip:
            for epoch in range(epochs_trained):
                is_random_sampler = hasattr(train_dataloader, "sampler") and isinstance(
                    train_dataloader.sampler, RandomSampler
                )
                if is_torch_less_than_1_11 or not is_random_sampler:
                    # We just need to begin an iteration to create the randomization of the sampler.
                    # That was before PyTorch 1.11 however...
                    for _ in train_dataloader:
                        break
                else:
                    # Otherwise we need to call the whooooole sampler cause there is some random operation added
                    # AT THE VERY END!
                    _ = list(train_dataloader.sampler)

        # ------------------------- CeZO added -------------------------
        # 为 CeZO 创建对应的 dataloaders
        self.is_parallel = isinstance(model, torch.nn.DataParallel) or isinstance(model, torch.nn.parallel.DistributedDataParallel) # 检查模型是否被 DataParallel 包装
        # train_dataloaders = self.split_dataloader(train_dataloader, fed_num) # 将原始DataLoader均分为fed_num份
        if self.args.non_iid is True:
            train_dataloaders = self.split_dataloader_non_iid(train_dataloader, fed_num, self.args.non_iid_alpha)
        else:
            train_dataloaders = self.split_dataloader(train_dataloader, fed_num) # 将原始DataLoader均分为fed_num份
        
        # # What parameters to optimize 
        # self.named_parameters_to_optim = []
        # for name, param in model.named_parameters():
        #     if param.requires_grad:
        #         self.named_parameters_to_optim.append((name, param))
        
        # 各个client的参数
        model_state_dicts = {"original": copy.deepcopy(model.module.state_dict() if self.is_parallel else model.state_dict())}
        
        # ------------------------- 开始训练 -------------------------
        for epoch in range(epochs_trained, num_train_epochs):
            print(f'----------------------epoch {epoch}-----------------------------')
            tr_losses = []
            # training for federated learning
            if self.args.aggregation_freq == "batch": # 逐batch聚合
                if not self.is_batch_nums_equal(train_dataloaders):
                    raise ValueError("All train_dataloaders must have the same number of batches.")
                # 训练前的配置
                epoch_iterators = []
                steps_in_epochs = []
                for i in range(fed_num):
                    train_dataloader = train_dataloaders[i]
                    
                    if isinstance(train_dataloader, DataLoader) and isinstance(train_dataloader.sampler, DistributedSampler):
                        train_dataloader.sampler.set_epoch(epoch)
                    elif hasattr(train_dataloader, "dataset") and isinstance(train_dataloader.dataset, IterableDatasetShard):
                        train_dataloader.dataset.set_epoch(epoch)

                    if is_torch_tpu_available():
                        parallel_loader = pl.ParallelLoader(train_dataloader, [args.device]).per_device_loader(args.device)
                        epoch_iterator = parallel_loader
                    else:
                        epoch_iterator = train_dataloader
                    
                    epoch_iterators.append(epoch_iterator)
                    
                    steps_in_epoch = (
                        len(epoch_iterator)
                        if len_dataloader is not None
                        else args.max_steps * args.gradient_accumulation_steps
                    )
                    steps_in_epochs.append(steps_in_epoch)

                # Reset the past mems state at the beginning of each epoch if necessary.
                if args.past_index >= 0:
                    self._past = None
                self.control = self.callback_handler.on_epoch_begin(args, self.state, self.control)
                if epoch == epochs_trained and resume_from_checkpoint is not None and steps_trained_in_current_epoch == 0:
                    self._load_rng_state(resume_from_checkpoint)
                
                # 遍历每个batch
                for step, epoch_inputs in enumerate(zip(*epoch_iterators)):
                    # Skip past any already trained steps if resuming training
                    if steps_trained_in_current_epoch > 0:
                        steps_trained_in_current_epoch -= 1
                        if steps_trained_progress_bar is not None:
                            steps_trained_progress_bar.update(1)
                        if steps_trained_in_current_epoch == 0:
                            self._load_rng_state(resume_from_checkpoint)
                        continue
                    elif steps_trained_progress_bar is not None:
                        steps_trained_progress_bar.close()
                        steps_trained_progress_bar = None

                    if step % args.gradient_accumulation_steps == 0:
                        self.control = self.callback_handler.on_step_begin(args, self.state, self.control)
                    
                    # 逐batches训练
                    projected_grad_batches = []
                    random_seed_batches = []
                    for i in range(fed_num):
                        # -------------- 初始化模型参数 --------------
                        if self.is_parallel: # 获取模型参数
                            model.module.load_state_dict(model_state_dicts["original"])
                        else:
                            model.load_state_dict(model_state_dicts["original"])
                        torch.cuda.empty_cache()
                        # -------------- 训练模型 --------------
                        inputs = epoch_inputs[i]
                        epoch_iterator = epoch_iterators[i]
                        steps_in_epoch = steps_in_epochs[i]
                        for q in range(self.Q):
                            self.seed_ += 1
                            model, projected_grad, random_seed, tr_loss = self.train_for_single_model_CeZO_batch(
                                args, model, epoch, step, inputs, tr_loss, steps_in_epoch)
                            if self.byzantine_flag[i] is True:
                                projected_grad = -projected_grad
                            projected_grad_batches.append(projected_grad)
                            random_seed_batches.append(random_seed)
                        self.seed_ -= self.Q
                    self.seed_ += self.Q
                        
                    # aggregation
                    self.federated_averaging_CeZO_batch(model, fed_num, projected_grad_batches, random_seed_batches)
                    model_state_dicts["original"] = copy.deepcopy(model.module.state_dict() if self.is_parallel else model.state_dict()) # 下发给K个client
                    torch.cuda.empty_cache()
                    
                tr_losses.append(tr_loss)
                
                # 后处理
                self.state.epoch = epoch + 1
                
                self.control = self.callback_handler.on_epoch_end(args, self.state, self.control)
                
                if (epoch) % self.args.eval_epoch == 0:
                    should_evaluate = self.control.should_evaluate
                    should_save = self.control.should_save
                    self.control.should_evaluate = True
                    self.control.should_save = False
                    self._maybe_log_save_evaluate(tr_loss, model, trial, epoch, ignore_keys_for_eval)
                    self.control.should_evaluate = should_evaluate
                    self.control.should_save = should_save
                
                # 本model的本epoch是否训练结束
                if self.control.should_epoch_stop or self.control.should_training_stop:
                    break
                if step < 0:
                    logger.warning(
                        "There seems to be not a single sample in your epoch_iterator, stopping training at step"
                        f" {self.state.global_step}! This is expected if you're using an IterableDataset and set"
                        f" num_steps ({max_steps}) higher than the number of available samples."
                    )
                    self.control.should_training_stop = True
                if DebugOption.TPU_METRICS_DEBUG in self.args.debug:
                    if is_torch_tpu_available():
                        # tpu-comment: Logging debug metrics for PyTorch/XLA (compile, execute times, ops, etc.)
                        xm.master_print(met.metrics_report())
                    else:
                        logger.warning(
                            "You enabled PyTorch/XLA debug metrics but you don't have a TPU "
                            "configured. Check your training configuration if this is unexpected."
                        )
            elif re.search(r'(\d+)batch', self.args.aggregation_freq): # 形如 "[L]batch" 的聚合频率
                raise NotImplementedError()
                L = int(re.search(r'(\d+)batch', self.args.aggregation_freq).group(1))
                if not self.is_batch_nums_equal(train_dataloaders):
                    raise ValueError("All train_dataloaders must have the same number of batches.")
                # 训练前的配置
                epoch_iterators = []
                steps_in_epochs = []
                projected_grad_multi_batches, random_seed_multi_batches = [], []
                for i in range(fed_num):
                    train_dataloader = train_dataloaders[i]
                    
                    if isinstance(train_dataloader, DataLoader) and isinstance(train_dataloader.sampler, DistributedSampler):
                        train_dataloader.sampler.set_epoch(epoch)
                    elif hasattr(train_dataloader, "dataset") and isinstance(train_dataloader.dataset, IterableDatasetShard):
                        train_dataloader.dataset.set_epoch(epoch)

                    if is_torch_tpu_available():
                        parallel_loader = pl.ParallelLoader(train_dataloader, [args.device]).per_device_loader(args.device)
                        epoch_iterator = parallel_loader
                    else:
                        epoch_iterator = train_dataloader
                    
                    epoch_iterators.append(epoch_iterator)
                    
                    steps_in_epoch = (
                        len(epoch_iterator)
                        if len_dataloader is not None
                        else args.max_steps * args.gradient_accumulation_steps
                    )
                    steps_in_epochs.append(steps_in_epoch)

                # Reset the past mems state at the beginning of each epoch if necessary.
                if args.past_index >= 0:
                    self._past = None
                self.control = self.callback_handler.on_epoch_begin(args, self.state, self.control)
                
                if epoch == epochs_trained and resume_from_checkpoint is not None and steps_trained_in_current_epoch == 0:
                    self._load_rng_state(resume_from_checkpoint)
                
                # 初始化
                iterators = [iter(dl) for dl in epoch_iterators] # 每个 DataLoader 的迭代器
                step_counters = [0] * fed_num  # step 计数器列表
                exhausted_iterators = [False] * fed_num # 跟踪每个 DataLoader 是否已经遍历完
                # 遍历所有 dataLoader
                while not all(exhausted_iterators):
                    for i, it in enumerate(iterators):
                        if not exhausted_iterators[i]:
                            # 训练该 dataLoader 中的 L 个 batch
                            steps_in_epoch = steps_in_epochs[i]
                            # -------------- 初始化模型参数 --------------
                            if self.is_parallel: # 获取模型参数
                                model.module.load_state_dict(model_state_dicts["original"])
                            else:
                                model.load_state_dict(model_state_dicts["original"])
                            torch.cuda.empty_cache()
                            # -------------- 初始化 list，用来存放训练后的参数变动 --------------
                            projected_grad_batches = []
                            random_seed_batches = []
                            # -------------- 训练 L 个 batch --------------
                            try:
                                # 尝试获取该 DataLoader 的 L 个 batch
                                for _ in range(L):
                                    # 获取 batch inputs
                                    step = step_counters[i]
                                    step_counters[i] += 1  # 增加 step 计数器
                                    inputs = next(it)
                                    # Skip past any already trained steps if resuming training
                                    if steps_trained_in_current_epoch > 0:
                                        steps_trained_in_current_epoch -= 1
                                        if steps_trained_progress_bar is not None:
                                            steps_trained_progress_bar.update(1)
                                        if steps_trained_in_current_epoch == 0:
                                            self._load_rng_state(resume_from_checkpoint)
                                        continue
                                    elif steps_trained_progress_bar is not None:
                                        steps_trained_progress_bar.close()
                                        steps_trained_progress_bar = None

                                    if step % args.gradient_accumulation_steps == 0:
                                        self.control = self.callback_handler.on_step_begin(args, self.state, self.control)
                                    # -------------- 训练模型 --------------
                                    model, projected_grad, random_seed, tr_loss = self.train_for_single_model_CeZO_batch(
                                        args, model, epoch, step, inputs, tr_loss, steps_in_epoch)
                                    projected_grad_batches.append(projected_grad)
                                    random_seed_batches.append(random_seed)
                            except StopIteration:
                                # 如果 DataLoader 中的数据已经被完全遍历，则标记为已耗尽
                                exhausted_iterators[i] = True
                            except Exception as e:
                                # 其他异常就是真的异常了
                                raise e
                            # -------------- 保存训练后的参数变动（无论满不满 L 个 batch ） --------------
                            projected_grad_multi_batches.append(projected_grad_batches)
                            random_seed_multi_batches.append(random_seed_batches)
                            
                    # 每个 dataloader 都跑完 L 个 batch 了，聚合分发新模型
                    self.federated_averaging_CeZO_epoch(model, fed_num, projected_grad_multi_batches, random_seed_multi_batches)
                    projected_grad_multi_batches, random_seed_multi_batches = [], []
                    model_state_dicts["original"] = copy.deepcopy(model.module.state_dict() if self.is_parallel else model.state_dict()) # 下发给K个client
                    torch.cuda.empty_cache()

                tr_losses.append(tr_loss)
                
                # 后处理
                self.state.epoch = epoch + 1
                
                self.control = self.callback_handler.on_epoch_end(args, self.state, self.control)
                
                if (epoch + 1) % self.args.eval_epoch == 0:
                    should_evaluate = self.control.should_evaluate
                    should_save = self.control.should_save
                    self.control.should_evaluate = True
                    self.control.should_save = False
                    self._maybe_log_save_evaluate(tr_loss, model, trial, epoch, ignore_keys_for_eval)
                    self.control.should_evaluate = should_evaluate
                    self.control.should_save = should_save
                
                # 本model的本epoch是否训练结束
                if self.control.should_epoch_stop or self.control.should_training_stop:
                    break
                if step < 0:
                    logger.warning(
                        "There seems to be not a single sample in your epoch_iterator, stopping training at step"
                        f" {self.state.global_step}! This is expected if you're using an IterableDataset and set"
                        f" num_steps ({max_steps}) higher than the number of available samples."
                    )
                    self.control.should_training_stop = True
                if DebugOption.TPU_METRICS_DEBUG in self.args.debug:
                    if is_torch_tpu_available():
                        # tpu-comment: Logging debug metrics for PyTorch/XLA (compile, execute times, ops, etc.)
                        xm.master_print(met.metrics_report())
                    else:
                        logger.warning(
                            "You enabled PyTorch/XLA debug metrics but you don't have a TPU "
                            "configured. Check your training configuration if this is unexpected."
                        )
            elif self.args.aggregation_freq == "epoch": # 逐epoch聚合
                raise NotImplementedError()
                projected_grad_epoches = []
                random_seed_epoches = []
                for i in range(fed_num):
                    # -------------- 初始化模型参数 --------------
                    if self.is_parallel: # 获取模型参数
                        model.module.load_state_dict(model_state_dicts["original"])
                    else:
                        model.load_state_dict(model_state_dicts["original"])
                    torch.cuda.empty_cache()
                    # -------------- 训练模型 --------------
                    model, projected_grad, random_seed, tr_loss = self.train_for_single_model_CeZO_epoch(
                        args, model, train_dataloaders[i], epoch, epochs_trained,
                        tr_loss, steps_trained_in_current_epoch, len_dataloader,
                        resume_from_checkpoint, steps_trained_progress_bar)
                    projected_grad_epoches.append(projected_grad)
                    random_seed_epoches.append(random_seed)
                tr_losses.append(tr_loss)
                
                # federated aggregation
                self.federated_averaging_CeZO_epoch(model, fed_num, projected_grad_epoches, random_seed_epoches)
                model_state_dicts["original"] = copy.deepcopy(model.module.state_dict() if self.is_parallel else model.state_dict()) # 下发给K个client
                torch.cuda.empty_cache()
                
                # 后处理
                self.state.epoch = epoch + 1
                
                self.control = self.callback_handler.on_epoch_end(args, self.state, self.control)
                
                if (epoch + 1) % self.args.eval_epoch == 0:
                    should_evaluate = self.control.should_evaluate
                    should_save = self.control.should_save
                    self.control.should_evaluate = True
                    self.control.should_save = False
                    self._maybe_log_save_evaluate(tr_loss, model, trial, epoch, ignore_keys_for_eval)
                    self.control.should_evaluate = should_evaluate
                    self.control.should_save = should_save
                
                if self.control.should_training_stop:
                    break
                
                if DebugOption.TPU_METRICS_DEBUG in self.args.debug:
                    if is_torch_tpu_available():
                        # tpu-comment: Logging debug metrics for PyTorch/XLA (compile, execute times, ops, etc.)
                        xm.master_print(met.metrics_report())
                    else:
                        logger.warning(
                            "You enabled PyTorch/XLA debug metrics but you don't have a TPU "
                            "configured. Check your training configuration if this is unexpected."
                        )
            else:
                raise ValueError("aggregation_freq error.")

        # ------------------------- 训练结束 -------------------------
        if args.past_index and hasattr(self, "_past"):
            # Clean the state at the end of training
            delattr(self, "_past")

        logger.info("\n\nTraining completed. Do not forget to share your model on huggingface.co/models =)\n\n")
        if args.load_best_model_at_end and self.state.best_model_checkpoint is not None:
            # Wait for everyone to get here so we are sur the model has been saved by process 0.
            if is_torch_tpu_available():
                xm.rendezvous("load_best_model_at_end")
            elif args.local_rank != -1:
                dist.barrier()
            elif is_sagemaker_mp_enabled():
                smp.barrier()

            self._load_best_model()

        # add remaining tr_loss
        self._total_loss_scalar += tr_loss.item()
        train_loss = self._total_loss_scalar / self.state.global_step
        # 计算loss
        # train_loss = torch.mean(torch.stack(tr_losses)).item()
        # ------------------------- End of CeZO added -------------------------

        metrics = speed_metrics("train", start_time, num_samples=num_train_samples, num_steps=self.state.max_steps)
        self.store_flos()
        metrics["total_flos"] = self.state.total_flos
        metrics["train_loss"] = train_loss

        self.is_in_train = False

        self._memory_tracker.stop_and_update_metrics(metrics)

        self.log(metrics)

        run_dir = self._get_output_dir(trial)
        checkpoints_sorted = self._sorted_checkpoints(use_mtime=False, output_dir=run_dir)

        # Delete the last checkpoint when save_total_limit=1 if it's different from the best checkpoint.
        if self.state.best_model_checkpoint is not None and self.args.save_total_limit == 1:
            for checkpoint in checkpoints_sorted:
                if checkpoint != self.state.best_model_checkpoint:
                    logger.info(f"Deleting older checkpoint [{checkpoint}] due to args.save_total_limit")
                    shutil.rmtree(checkpoint)

        self.control = self.callback_handler.on_train_end(args, self.state, self.control)

        return TrainOutput(self.state.global_step, train_loss, metrics)

    ############## FedAvg ##############
    def train_for_single_model_batch(self, args, model, epoch, step, inputs, tr_loss, steps_in_epoch):
        # MeZO added: estimate gradient
        if args.trainer == "zo":
            tr_loss_step = self.zo_step(model, inputs)
        else:
            if (
                ((step + 1) % args.gradient_accumulation_steps != 0)
                and args.local_rank != -1
                and args._no_sync_in_gradient_accumulation
            ):
                # Avoid unnecessary DDP synchronization since there will be no backward pass on this example.
                with model.no_sync():
                    tr_loss_step = self.training_step(model, inputs)
            else:
                tr_loss_step = self.training_step(model, inputs)

        if (
            args.logging_nan_inf_filter
            and not is_torch_tpu_available()
            and (torch.isnan(tr_loss_step) or torch.isinf(tr_loss_step))
        ):
            # if loss is nan or inf simply add the average of previous logged losses
            tr_loss += tr_loss / (1 + self.state.global_step - self._globalstep_last_logged)
        else:
            tr_loss += tr_loss_step

        self.current_flos += float(self.floating_point_ops(inputs))

        # Optimizer step for deepspeed must be called on every step regardless of the value of gradient_accumulation_steps
        if self.deepspeed:
            self.deepspeed.step()


        if (step + 1) % args.gradient_accumulation_steps == 0 or (
            # last step in epoch but step is always smaller than gradient_accumulation_steps
            steps_in_epoch <= args.gradient_accumulation_steps
            and (step + 1) == steps_in_epoch
        ):
            # MeZO added: update model with the estimated gradient
            if args.trainer == "zo":
                self.zo_update(model)
            else:
                # Gradient clipping
                if args.max_grad_norm is not None and args.max_grad_norm > 0 and not self.deepspeed:
                    # deepspeed does its own clipping

                    if self.do_grad_scaling:
                        # Reduce gradients first for XLA
                        if is_torch_tpu_available():
                            gradients = xm._fetch_gradients(self.optimizer)
                            xm.all_reduce("sum", gradients, scale=1.0 / xm.xrt_world_size())
                        # AMP: gradients need unscaling
                        self.scaler.unscale_(self.optimizer)

                    if is_sagemaker_mp_enabled() and args.fp16:
                        self.optimizer.clip_master_grads(args.max_grad_norm)
                    elif hasattr(self.optimizer, "clip_grad_norm"):
                        # Some optimizers (like the sharded optimizer) have a specific way to do gradient clipping
                        self.optimizer.clip_grad_norm(args.max_grad_norm)
                    elif hasattr(model, "clip_grad_norm_"):
                        # Some models (like FullyShardedDDP) have a specific way to do gradient clipping
                        model.clip_grad_norm_(args.max_grad_norm)
                    else:
                        # Revert to normal clipping otherwise, handling Apex or full precision
                        nn.utils.clip_grad_norm_(
                            amp.master_params(self.optimizer) if self.use_apex else model.parameters(),
                            args.max_grad_norm,
                        )

                # Optimizer step
                optimizer_was_run = True
                if self.deepspeed:
                    pass  # called outside the loop
                elif is_torch_tpu_available():
                    if self.do_grad_scaling:
                        self.scaler.step(self.optimizer)
                        self.scaler.update()
                    else:
                        xm.optimizer_step(self.optimizer)
                elif self.do_grad_scaling:
                    scale_before = self.scaler.get_scale()
                    self.scaler.step(self.optimizer)
                    self.scaler.update()
                    scale_after = self.scaler.get_scale()
                    optimizer_was_run = scale_before <= scale_after
                else:
                    self.optimizer.step()

                if optimizer_was_run and not self.deepspeed:
                    self.lr_scheduler.step()
                model.zero_grad()

            self.state.global_step += 1
            self.state.epoch = epoch + (step + 1) / steps_in_epoch
            self.control = self.callback_handler.on_step_end(args, self.state, self.control)

            self._maybe_log_save_evaluate(tr_loss, model, trial, epoch, ignore_keys_for_eval)
        else:
            self.control = self.callback_handler.on_substep_end(args, self.state, self.control)

        return model, tr_loss

    def train_for_single_model_epoch(self, args, model, train_dataloader, epoch, epochs_trained,
                                    tr_loss, steps_trained_in_current_epoch, len_dataloader,
                                    resume_from_checkpoint, steps_trained_progress_bar):
        if isinstance(train_dataloader, DataLoader) and isinstance(train_dataloader.sampler, DistributedSampler):
            train_dataloader.sampler.set_epoch(epoch)
        elif hasattr(train_dataloader, "dataset") and isinstance(train_dataloader.dataset, IterableDatasetShard):
            train_dataloader.dataset.set_epoch(epoch)

        if is_torch_tpu_available():
            parallel_loader = pl.ParallelLoader(train_dataloader, [args.device]).per_device_loader(args.device)
            epoch_iterator = parallel_loader
        else:
            epoch_iterator = train_dataloader

        # Reset the past mems state at the beginning of each epoch if necessary.
        if args.past_index >= 0:
            self._past = None

        steps_in_epoch = (
            len(epoch_iterator)
            if len_dataloader is not None
            else args.max_steps * args.gradient_accumulation_steps
        )
        self.control = self.callback_handler.on_epoch_begin(args, self.state, self.control)

        if epoch == epochs_trained and resume_from_checkpoint is not None and steps_trained_in_current_epoch == 0:
            self._load_rng_state(resume_from_checkpoint)

        step = -1
        for step, inputs in enumerate(epoch_iterator):

            # Skip past any already trained steps if resuming training
            if steps_trained_in_current_epoch > 0:
                steps_trained_in_current_epoch -= 1
                if steps_trained_progress_bar is not None:
                    steps_trained_progress_bar.update(1)
                if steps_trained_in_current_epoch == 0:
                    self._load_rng_state(resume_from_checkpoint)
                continue
            elif steps_trained_progress_bar is not None:
                steps_trained_progress_bar.close()
                steps_trained_progress_bar = None

            if step % args.gradient_accumulation_steps == 0:
                self.control = self.callback_handler.on_step_begin(args, self.state, self.control)

            # MeZO added: estimate gradient
            if args.trainer == "zo":
                tr_loss_step = self.zo_step(model, inputs)
            else:
                if (
                    ((step + 1) % args.gradient_accumulation_steps != 0)
                    and args.local_rank != -1
                    and args._no_sync_in_gradient_accumulation
                ):
                    # Avoid unnecessary DDP synchronization since there will be no backward pass on this example.
                    with model.no_sync():
                        tr_loss_step = self.training_step(model, inputs)
                else:
                    tr_loss_step = self.training_step(model, inputs)

            if (
                args.logging_nan_inf_filter
                and not is_torch_tpu_available()
                and (torch.isnan(tr_loss_step) or torch.isinf(tr_loss_step))
            ):
                # if loss is nan or inf simply add the average of previous logged losses
                tr_loss += tr_loss / (1 + self.state.global_step - self._globalstep_last_logged)
            else:
                tr_loss += tr_loss_step

            self.current_flos += float(self.floating_point_ops(inputs))

            # Optimizer step for deepspeed must be called on every step regardless of the value of gradient_accumulation_steps
            if self.deepspeed:
                self.deepspeed.step()


            if (step + 1) % args.gradient_accumulation_steps == 0 or (
                # last step in epoch but step is always smaller than gradient_accumulation_steps
                steps_in_epoch <= args.gradient_accumulation_steps
                and (step + 1) == steps_in_epoch
            ):
                # MeZO added: update model with the estimated gradient
                if args.trainer == "zo":
                    self.zo_update(model)
                else:
                    # Gradient clipping
                    if args.max_grad_norm is not None and args.max_grad_norm > 0 and not self.deepspeed:
                        # deepspeed does its own clipping

                        if self.do_grad_scaling:
                            # Reduce gradients first for XLA
                            if is_torch_tpu_available():
                                gradients = xm._fetch_gradients(self.optimizer)
                                xm.all_reduce("sum", gradients, scale=1.0 / xm.xrt_world_size())
                            # AMP: gradients need unscaling
                            self.scaler.unscale_(self.optimizer)

                        if is_sagemaker_mp_enabled() and args.fp16:
                            self.optimizer.clip_master_grads(args.max_grad_norm)
                        elif hasattr(self.optimizer, "clip_grad_norm"):
                            # Some optimizers (like the sharded optimizer) have a specific way to do gradient clipping
                            self.optimizer.clip_grad_norm(args.max_grad_norm)
                        elif hasattr(model, "clip_grad_norm_"):
                            # Some models (like FullyShardedDDP) have a specific way to do gradient clipping
                            model.clip_grad_norm_(args.max_grad_norm)
                        else:
                            # Revert to normal clipping otherwise, handling Apex or full precision
                            nn.utils.clip_grad_norm_(
                                amp.master_params(self.optimizer) if self.use_apex else model.parameters(),
                                args.max_grad_norm,
                            )

                    # Optimizer step
                    optimizer_was_run = True
                    if self.deepspeed:
                        pass  # called outside the loop
                    elif is_torch_tpu_available():
                        if self.do_grad_scaling:
                            self.scaler.step(self.optimizer)
                            self.scaler.update()
                        else:
                            xm.optimizer_step(self.optimizer)
                    elif self.do_grad_scaling:
                        scale_before = self.scaler.get_scale()
                        self.scaler.step(self.optimizer)
                        self.scaler.update()
                        scale_after = self.scaler.get_scale()
                        optimizer_was_run = scale_before <= scale_after
                    else:
                        self.optimizer.step()

                    if optimizer_was_run and not self.deepspeed:
                        self.lr_scheduler.step()
                    model.zero_grad()

                self.state.global_step += 1
                self.state.epoch = epoch + (step + 1) / steps_in_epoch
                self.control = self.callback_handler.on_step_end(args, self.state, self.control)

                # self._maybe_log_save_evaluate(tr_loss, model, trial, epoch, ignore_keys_for_eval)
            else:
                self.control = self.callback_handler.on_substep_end(args, self.state, self.control)

            if self.control.should_epoch_stop or self.control.should_training_stop:
                break
        if step < 0:
            logger.warning(
                "There seems to be not a single sample in your epoch_iterator, stopping training at step"
                f" {self.state.global_step}! This is expected if you're using an IterableDataset and set"
                f" num_steps ({max_steps}) higher than the number of available samples."
            )
            self.control.should_training_stop = True
        return model, tr_loss

    def federated_adding(self, running_model, model_state_dicts, fed_num):
        # 将模型中相对original更新的部分加到updated上
        original = model_state_dicts["original"]
        updated = model_state_dicts["updated"]
        
        if self.is_parallel:
            with torch.no_grad():
                for name, param in running_model.module.named_parameters():
                    if param.requires_grad:
                        updated[name] = updated[name] + (param.data - original[name]) / fed_num
        else:
            with torch.no_grad():
                for name, param in running_model.named_parameters():
                    if param.requires_grad:
                        updated[name] = updated[name] + (param.data - original[name]) / fed_num

    # 3倍显存实现。既可用于FO，又可用于ZO
    def _inner_training_loop_Federated_3times(
        self, batch_size=None, args=None, resume_from_checkpoint=None, trial=None, ignore_keys_for_eval=None
    ):
        """
        We overload the original training loop to add linear probing and MeZO. Search key word "MeZO added"
        for those updates.
        """
        
        # print(f"self.args.fp16: {self.args.fp16}")
        # print(f"self.args.fp16_full_eval: {self.args.fp16_full_eval}")
        # print(f"self.do_grad_scaling: {self.do_grad_scaling}")
        # raise NotImplementedError()
        
        # 创建一个文件处理器，设置日志文件路径
        file_handler = py_logging.FileHandler(os.path.join(self.args.output_dir, self.args.log_file))
        file_handler.setLevel(py_logging.INFO)
        # 将处理器添加到 logger
        logger.addHandler(file_handler)
        
        self._train_batch_size = batch_size
        fed_num = self.args.fed_num
        # Data loader and number of training steps
        train_dataloader = self.get_train_dataloader()

        # MeZO added: Linear probing
        if self.args.linear_probing:

            def _get_token_prediction_layer(model):
                if model.config.model_type == "opt":
                    return model.lm_head
                else:
                    raise NotImplementedError(model.config.model_type)

            def _extract_features(model, *args, **kwargs):
                """some magic for getting features pre last layer"""
                features = {}
                def __hook(model_, input_, output_):
                    features["features"] = input_[0].detach()

                _get_token_prediction_layer(model).register_forward_hook(__hook)
                model.forward(*args, **kwargs)
                return features["features"]

            logger.info("Linear probing")
            logger.info("Starting to get features for training dataset")
            targets = []
            features = []
            with torch.inference_mode():
                for step, inputs in enumerate(tqdm(train_dataloader)):
                    for k, v in inputs.items():
                        if isinstance(v, torch.Tensor):
                            inputs[k] = v.to(self.model.device)
                        
                    feature = _extract_features(self.model, **inputs)
                    target = inputs["labels"]

                    # Shift the target (bc it's autoregressive LM) and add the corresponding part
                    assert not self.args.train_as_classification and self.args.only_train_option
                    feature, target = feature[:, :-1], target[:, 1:]
                    for _i, _len in enumerate(inputs["option_len"]):
                        features.append(feature[_i, -_len:])
                        targets.append(target[_i, -_len:])

            logger.info("Finished getting features for training dataset")

            features = torch.cat(features, dim=0).cpu().numpy()
            targets = torch.cat(targets, dim=0).cpu().numpy()
            # Whether to use bias
            if self.model.config.model_type in ["opt", "gpt2"]:
                use_bias = False
            else:
                raise NotImplementedError
            # Set early stopping
            tol = 0.01 if self.args.lp_early_stopping else 1e-4 # 1e-4 is scipy default
            max_iter = 1000 if self.args.lp_early_stopping else 5000

            logger.info("Fitting logistic regression...")
            reg = LogisticRegressionCV(max_iter=max_iter, fit_intercept=use_bias, multi_class="multinomial", random_state=0, tol=tol, n_jobs=-1).fit(features, targets)
            logger.info("Done")

            logger.info("Assigning weights to model")
            decoder = _get_token_prediction_layer(self.model)
            coef_torch = torch.tensor(reg.coef_, device=decoder.weight.device, dtype=decoder.weight.dtype)
            if use_bias:
                bias_torch = torch.tensor(reg.intercept_, device=decoder.weight.device, dtype=decoder.weight.dtype)
            if coef_torch.shape[0] == 1: # The regressor only detects two classes
                assert len(reg.classes_) == 2
                coef_torch = torch.cat([-coef_torch / 2, coef_torch / 2], dim=0)
                if use_bias:
                    bias_torch = torch.cat([-bias_torch / 2, bias_torch / 2], dim=0)

            for _i, token_id in enumerate(reg.classes_):
                decoder.weight.data[token_id] = coef_torch[_i]
                if use_bias:
                    decoder.bias.data[token_id] = bias_torch[_i]

            return None

        # Setting up training control variables:
        # number of training epochs: num_train_epochs
        # number of training steps per epoch: num_update_steps_per_epoch
        # total number of training steps to execute: max_steps
        total_train_batch_size = args.train_batch_size * args.gradient_accumulation_steps * args.world_size

        len_dataloader = None
        # 获取数据集大小，并计算训练epoch数
        if has_length(train_dataloader):
            len_dataloader = len(train_dataloader)
            num_update_steps_per_epoch = len_dataloader // args.gradient_accumulation_steps
            num_update_steps_per_epoch = max(num_update_steps_per_epoch, 1)
            num_examples = self.num_examples(train_dataloader)
            if args.max_steps > 0:
                max_steps = args.max_steps
                num_train_epochs = args.max_steps // num_update_steps_per_epoch + int(
                    args.max_steps % num_update_steps_per_epoch > 0
                )
                # May be slightly incorrect if the last batch in the training dataloader has a smaller size but it's
                # the best we can do.
                num_train_samples = args.max_steps * total_train_batch_size
            else:
                max_steps = math.ceil(args.num_train_epochs * num_update_steps_per_epoch)
                num_train_epochs = math.ceil(args.num_train_epochs)
                num_train_samples = self.num_examples(train_dataloader) * args.num_train_epochs
        elif args.max_steps > 0:  # Rely on max_steps when dataloader does not have a working size
            raise NotImplementedError(
                "The training dataloader does not have a length, please set max_steps to avoid an incorrect number of training steps."
            )
            max_steps = args.max_steps
            # Setting a very large number of epochs so we go as many times as necessary over the iterator.
            num_train_epochs = sys.maxsize
            num_update_steps_per_epoch = max_steps
            num_examples = total_train_batch_size * args.max_steps
            num_train_samples = args.max_steps * total_train_batch_size
        else:
            raise ValueError(
                "args.max_steps must be set to a positive value if dataloader does not have a length, was"
                f" {args.max_steps}"
            )

        if DebugOption.UNDERFLOW_OVERFLOW in self.args.debug:
            if self.args.n_gpu > 1:
                # nn.DataParallel(model) replicates the model, creating new variables and module
                # references registered here no longer work on other gpus, breaking the module
                raise ValueError(
                    "Currently --debug underflow_overflow is not supported under DP. Please use DDP"
                    " (torch.distributed.launch)."
                )
            else:
                debug_overflow = DebugUnderflowOverflow(self.model)  # noqa

        delay_optimizer_creation = (
            self.sharded_ddp is not None
            and self.sharded_ddp != ShardedDDPOption.SIMPLE
            or is_sagemaker_mp_enabled()
            or self.fsdp is not None
        )
        if args.deepspeed:
            deepspeed_engine, optimizer, lr_scheduler = deepspeed_init(
                self, num_training_steps=max_steps, resume_from_checkpoint=resume_from_checkpoint
            )
            self.model = deepspeed_engine.module
            self.model_wrapped = deepspeed_engine
            self.deepspeed = deepspeed_engine
            self.optimizer = optimizer
            self.lr_scheduler = lr_scheduler
        elif not delay_optimizer_creation:
            self.create_optimizer_and_scheduler(num_training_steps=max_steps)

        self.state = TrainerState()
        self.state.is_hyper_param_search = trial is not None

        # Activate gradient checkpointing if needed
        if args.gradient_checkpointing:
            self.model.gradient_checkpointing_enable()

        model = self._wrap_model(self.model_wrapped)

        if is_sagemaker_mp_enabled() and resume_from_checkpoint is not None:
            self._load_from_checkpoint(resume_from_checkpoint, model)

        # for the rest of this function `model` is the outside model, whether it was wrapped or not
        if model is not self.model:
            self.model_wrapped = model

        if delay_optimizer_creation:
            self.create_optimizer_and_scheduler(num_training_steps=max_steps)

        # Check if saved optimizer or scheduler states exist
        self._load_optimizer_and_scheduler(resume_from_checkpoint)

        # important: at this point:
        # self.model         is the Transformers Model
        # self.model_wrapped is DDP(Transformers Model), Deepspeed(Transformers Model), etc.

        # Train!
        logger.info("***** Running training *****")
        logger.info(f"  Num examples = {num_examples}")
        logger.info(f"  Num Epochs = {num_train_epochs}")
        logger.info(f"  Instantaneous batch size per device = {args.per_device_train_batch_size}")
        logger.info(f"  Total train batch size (w. parallel, distributed & accumulation) = {total_train_batch_size}")
        logger.info(f"  Gradient Accumulation steps = {args.gradient_accumulation_steps}")
        logger.info(f"  Total optimization steps = {max_steps}")
        logger.info(
            f"  Number of trainable parameters = {sum(p.numel() for p in model.parameters() if p.requires_grad)}"
        )

        self.state.epoch = 0
        start_time = time.time()
        epochs_trained = 0
        steps_trained_in_current_epoch = 0
        steps_trained_progress_bar = None

        # Check if continuing training from a checkpoint
        if resume_from_checkpoint is not None and os.path.isfile(
            os.path.join(resume_from_checkpoint, TRAINER_STATE_NAME)
        ):
            self.state = TrainerState.load_from_json(os.path.join(resume_from_checkpoint, TRAINER_STATE_NAME))
            epochs_trained = self.state.global_step // num_update_steps_per_epoch
            if not args.ignore_data_skip:
                steps_trained_in_current_epoch = self.state.global_step % (num_update_steps_per_epoch)
                steps_trained_in_current_epoch *= args.gradient_accumulation_steps
            else:
                steps_trained_in_current_epoch = 0

            logger.info("  Continuing training from checkpoint, will skip to saved global_step")
            logger.info(f"  Continuing training from epoch {epochs_trained}")
            logger.info(f"  Continuing training from global step {self.state.global_step}")
            if not args.ignore_data_skip:
                logger.info(
                    f"  Will skip the first {epochs_trained} epochs then the first {steps_trained_in_current_epoch} "
                    "batches in the first epoch. If this takes a lot of time, you can add the `--ignore_data_skip` "
                    "flag to your launch command, but you will resume the training on data already seen by your model."
                )
                if self.is_local_process_zero() and not args.disable_tqdm:
                    steps_trained_progress_bar = tqdm(total=steps_trained_in_current_epoch)
                    steps_trained_progress_bar.set_description("Skipping the first batches")

        # Update the references
        self.callback_handler.model = self.model
        self.callback_handler.optimizer = self.optimizer
        self.callback_handler.lr_scheduler = self.lr_scheduler
        self.callback_handler.train_dataloader = train_dataloader
        if self.hp_name is not None and self._trial is not None:
            # use self._trial because the SigOpt/Optuna hpo only call `_hp_search_setup(trial)` instead of passing trial
            # parameter to Train when using DDP.
            self.state.trial_name = self.hp_name(self._trial)
        if trial is not None:
            assignments = trial.assignments if self.hp_search_backend == HPSearchBackend.SIGOPT else trial
            self.state.trial_params = hp_params(assignments)
        else:
            self.state.trial_params = None
        # This should be the same if the state has been saved but in case the training arguments changed, it's safer
        # to set this after the load.
        self.state.max_steps = max_steps
        self.state.num_train_epochs = num_train_epochs
        self.state.is_local_process_zero = self.is_local_process_zero()
        self.state.is_world_process_zero = self.is_world_process_zero()

        # tr_loss is a tensor to avoid synchronization of TPUs through .item()
        tr_loss = torch.tensor(0.0).to(args.device)
        # _total_loss_scalar is updated everytime .item() has to be called on tr_loss and stores the sum of all losses
        self._total_loss_scalar = 0.0
        self._globalstep_last_logged = self.state.global_step
        model.zero_grad()

        self.control = self.callback_handler.on_train_begin(args, self.state, self.control)

        # Skip the first epochs_trained epochs to get the random state of the dataloader at the right point.
        if not args.ignore_data_skip:
            for epoch in range(epochs_trained):
                is_random_sampler = hasattr(train_dataloader, "sampler") and isinstance(
                    train_dataloader.sampler, RandomSampler
                )
                if is_torch_less_than_1_11 or not is_random_sampler:
                    # We just need to begin an iteration to create the randomization of the sampler.
                    # That was before PyTorch 1.11 however...
                    for _ in train_dataloader:
                        break
                else:
                    # Otherwise we need to call the whooooole sampler cause there is some random operation added
                    # AT THE VERY END!
                    _ = list(train_dataloader.sampler)

        # ------------------------- Federated added -------------------------
        # 为 Federated 创建对应的 dataloaders
        self.is_parallel = isinstance(model, torch.nn.DataParallel) or isinstance(model, torch.nn.parallel.DistributedDataParallel) # 检查模型是否被 DataParallel 包装
        train_dataloaders = self.split_dataloader(train_dataloader, fed_num) # 将原始DataLoader均分为fed_num份
        # 各个client的参数、优化器和学习率调度器
        model_state_dicts = {"original": copy.deepcopy(model.module.state_dict() if self.is_parallel else model.state_dict()),
                             "updated": copy.deepcopy(model.module.state_dict() if self.is_parallel else model.state_dict())}
        optimizer_state_dicts = []
        lr_scheduler_state_dicts = []
        for i in range(fed_num):
            optimizer_state_dicts.append(copy.deepcopy(self.optimizer.state_dict())) # 获取优化器参数
            lr_scheduler_state_dicts.append(copy.deepcopy(self.lr_scheduler.state_dict())) # 获取学习率参数
        
        # ------------------------- 开始训练 -------------------------
        for epoch in range(epochs_trained, num_train_epochs):
            tr_losses = []
            # training for federated learning
            if self.args.aggregation_freq == "batch": # 逐batch聚合
                if not self.is_batch_nums_equal(train_dataloaders):
                    raise ValueError("All train_dataloaders must have the same number of batches.")
                # 训练前的配置
                epoch_iterators = []
                steps_in_epochs = []
                for i in range(fed_num):
                    train_dataloader = train_dataloaders[i]
                    
                    if isinstance(train_dataloader, DataLoader) and isinstance(train_dataloader.sampler, DistributedSampler):
                        train_dataloader.sampler.set_epoch(epoch)
                    elif hasattr(train_dataloader, "dataset") and isinstance(train_dataloader.dataset, IterableDatasetShard):
                        train_dataloader.dataset.set_epoch(epoch)

                    if is_torch_tpu_available():
                        parallel_loader = pl.ParallelLoader(train_dataloader, [args.device]).per_device_loader(args.device)
                        epoch_iterator = parallel_loader
                    else:
                        epoch_iterator = train_dataloader
                    
                    epoch_iterators.append(epoch_iterator)
                    
                    steps_in_epoch = (
                        len(epoch_iterator)
                        if len_dataloader is not None
                        else args.max_steps * args.gradient_accumulation_steps
                    )
                    steps_in_epochs.append(steps_in_epoch)

                    # Reset the past mems state at the beginning of each epoch if necessary.
                    if args.past_index >= 0:
                        self._past = None

                    self.control = self.callback_handler.on_epoch_begin(args, self.state, self.control)
                if epoch == epochs_trained and resume_from_checkpoint is not None and steps_trained_in_current_epoch == 0:
                    self._load_rng_state(resume_from_checkpoint)
                # 遍历每个batch
                for step, epoch_inputs in enumerate(zip(*epoch_iterators)):
                    # Skip past any already trained steps if resuming training
                    if steps_trained_in_current_epoch > 0:
                        steps_trained_in_current_epoch -= 1
                        if steps_trained_progress_bar is not None:
                            steps_trained_progress_bar.update(1)
                        if steps_trained_in_current_epoch == 0:
                            self._load_rng_state(resume_from_checkpoint)
                        continue
                    elif steps_trained_progress_bar is not None:
                        steps_trained_progress_bar.close()
                        steps_trained_progress_bar = None

                    if step % args.gradient_accumulation_steps == 0:
                        self.control = self.callback_handler.on_step_begin(args, self.state, self.control)
                    
                    # 逐batches训练
                    for i in range(fed_num):
                        inputs = epoch_inputs[i]
                        epoch_iterator = epoch_iterators[i]
                        steps_in_epoch = steps_in_epochs[i]
                        # -------------- 初始化模型参数 --------------
                        if self.is_parallel: # 获取模型参数
                            model.module.load_state_dict(model_state_dicts["original"])
                        else:
                            model.load_state_dict(model_state_dicts["original"])
                        self.optimizer.load_state_dict(optimizer_state_dicts[i]) # 获取优化器参数
                        self.lr_scheduler.load_state_dict(lr_scheduler_state_dicts[i]) # 获取学习率参数
                        torch.cuda.empty_cache()
                        # -------------- 训练模型 --------------
                        model, tr_loss = self.train_for_single_model_batch(args, model, epoch, step, inputs, tr_loss, steps_in_epoch)
                        # -------------- 保存模型参数 --------------
                        self.federated_adding(model, model_state_dicts, fed_num)
                        optimizer_state_dicts[i] = copy.deepcopy(self.optimizer.state_dict()) # 保存优化器参数
                        lr_scheduler_state_dicts[i] = copy.deepcopy(self.lr_scheduler.state_dict()) # 保存学习率参数
                        torch.cuda.empty_cache()
                    # federated aggregation
                    model_state_dicts["original"] = copy.deepcopy(model_state_dicts["updated"])
                    torch.cuda.empty_cache()
                    
                tr_losses.append(tr_loss)
                
                # 后处理
                self.state.epoch = epoch + 1
                
                self.control = self.callback_handler.on_epoch_end(args, self.state, self.control)
                
                if (epoch + 1) % self.args.eval_epoch == 0:
                    should_evaluate = self.control.should_evaluate
                    should_save = self.control.should_save
                    self.control.should_evaluate = True
                    self.control.should_save = False
                    self._maybe_log_save_evaluate(tr_loss, model, trial, epoch, ignore_keys_for_eval)
                    self.control.should_evaluate = should_evaluate
                    self.control.should_save = should_save
                
                # 本model的本epoch是否训练结束
                if self.control.should_epoch_stop or self.control.should_training_stop:
                    break
                if step < 0:
                    logger.warning(
                        "There seems to be not a single sample in your epoch_iterator, stopping training at step"
                        f" {self.state.global_step}! This is expected if you're using an IterableDataset and set"
                        f" num_steps ({max_steps}) higher than the number of available samples."
                    )
                    self.control.should_training_stop = True
                if DebugOption.TPU_METRICS_DEBUG in self.args.debug:
                    if is_torch_tpu_available():
                        # tpu-comment: Logging debug metrics for PyTorch/XLA (compile, execute times, ops, etc.)
                        xm.master_print(met.metrics_report())
                    else:
                        logger.warning(
                            "You enabled PyTorch/XLA debug metrics but you don't have a TPU "
                            "configured. Check your training configuration if this is unexpected."
                        )
            elif re.search(r'(\d+)batch', self.args.aggregation_freq): # 形如 "[L]batch" 的聚合频率
                L = int(re.search(r'(\d+)batch', self.args.aggregation_freq).group(1))
                if not self.is_batch_nums_equal(train_dataloaders):
                    raise ValueError("All train_dataloaders must have the same number of batches.")
                # 训练前的配置
                epoch_iterators = []
                steps_in_epochs = []
                for i in range(fed_num):
                    train_dataloader = train_dataloaders[i]
                    
                    if isinstance(train_dataloader, DataLoader) and isinstance(train_dataloader.sampler, DistributedSampler):
                        train_dataloader.sampler.set_epoch(epoch)
                    elif hasattr(train_dataloader, "dataset") and isinstance(train_dataloader.dataset, IterableDatasetShard):
                        train_dataloader.dataset.set_epoch(epoch)

                    if is_torch_tpu_available():
                        parallel_loader = pl.ParallelLoader(train_dataloader, [args.device]).per_device_loader(args.device)
                        epoch_iterator = parallel_loader
                    else:
                        epoch_iterator = train_dataloader
                    
                    epoch_iterators.append(epoch_iterator)
                    
                    steps_in_epoch = (
                        len(epoch_iterator)
                        if len_dataloader is not None
                        else args.max_steps * args.gradient_accumulation_steps
                    )
                    steps_in_epochs.append(steps_in_epoch)

                # Reset the past mems state at the beginning of each epoch if necessary.
                if args.past_index >= 0:
                    self._past = None

                self.control = self.callback_handler.on_epoch_begin(args, self.state, self.control)
                if epoch == epochs_trained and resume_from_checkpoint is not None and steps_trained_in_current_epoch == 0:
                    self._load_rng_state(resume_from_checkpoint)

                # 初始化
                iterators = [iter(dl) for dl in epoch_iterators] # 每个 DataLoader 的迭代器
                step_counters = [0] * fed_num  # step 计数器列表
                exhausted_iterators = [False] * fed_num # 跟踪每个 DataLoader 是否已经遍历完
                # 遍历所有 dataLoader
                while not all(exhausted_iterators):
                    for i, it in enumerate(iterators):
                        if not exhausted_iterators[i]:
                            # 训练该 dataLoader 中的 L 个 batch
                            steps_in_epoch = steps_in_epochs[i]
                            # -------------- 初始化模型参数 --------------
                            if self.is_parallel: # 获取模型参数
                                model.module.load_state_dict(model_state_dicts["original"])
                            else:
                                model.load_state_dict(model_state_dicts["original"])
                            self.optimizer.load_state_dict(optimizer_state_dicts[i]) # 获取优化器参数
                            self.lr_scheduler.load_state_dict(lr_scheduler_state_dicts[i]) # 获取学习率参数
                            torch.cuda.empty_cache()
                            # -------------- 训练 L 个 batch --------------
                            try:
                                # 尝试获取该 DataLoader 的 L 个 batch
                                for _ in range(L):
                                    # 获取 batch inputs
                                    step = step_counters[i]
                                    step_counters[i] += 1  # 增加 step 计数器
                                    inputs = next(it)
                                    # Skip past any already trained steps if resuming training
                                    if steps_trained_in_current_epoch > 0:
                                        steps_trained_in_current_epoch -= 1
                                        if steps_trained_progress_bar is not None:
                                            steps_trained_progress_bar.update(1)
                                        if steps_trained_in_current_epoch == 0:
                                            self._load_rng_state(resume_from_checkpoint)
                                        continue
                                    elif steps_trained_progress_bar is not None:
                                        steps_trained_progress_bar.close()
                                        steps_trained_progress_bar = None

                                    if step % args.gradient_accumulation_steps == 0:
                                        self.control = self.callback_handler.on_step_begin(args, self.state, self.control)
                                    # -------------- 训练模型 --------------
                                    model, tr_loss = self.train_for_single_model_batch(args, model, epoch, step, inputs, tr_loss, steps_in_epoch)
                            except StopIteration:
                                # 如果 DataLoader 中的数据已经被完全遍历，则标记为已耗尽
                                exhausted_iterators[i] = True
                            except Exception as e:
                                # 其他异常就是真的异常了
                                raise e
                            # -------------- 保存模型参数 --------------
                            self.federated_adding(model, model_state_dicts, fed_num)
                            optimizer_state_dicts[i] = copy.deepcopy(self.optimizer.state_dict()) # 保存优化器参数
                            lr_scheduler_state_dicts[i] = copy.deepcopy(self.lr_scheduler.state_dict()) # 保存学习率参数
                            torch.cuda.empty_cache()
                    # 每个 dataloader 都跑完 L 个 batch 了，聚合分发新模型
                    model_state_dicts["original"] = copy.deepcopy(model_state_dicts["updated"])
                    torch.cuda.empty_cache()
                
                tr_losses.append(tr_loss)
                
                # 后处理
                self.state.epoch = epoch + 1
                
                self.control = self.callback_handler.on_epoch_end(args, self.state, self.control)
                
                if (epoch + 1) % self.args.eval_epoch == 0:
                    should_evaluate = self.control.should_evaluate
                    should_save = self.control.should_save
                    self.control.should_evaluate = True
                    self.control.should_save = False
                    self._maybe_log_save_evaluate(tr_loss, model, trial, epoch, ignore_keys_for_eval)
                    self.control.should_evaluate = should_evaluate
                    self.control.should_save = should_save
                
                # 本model的本epoch是否训练结束
                if self.control.should_epoch_stop or self.control.should_training_stop:
                    break
                if step < 0:
                    logger.warning(
                        "There seems to be not a single sample in your epoch_iterator, stopping training at step"
                        f" {self.state.global_step}! This is expected if you're using an IterableDataset and set"
                        f" num_steps ({max_steps}) higher than the number of available samples."
                    )
                    self.control.should_training_stop = True
                if DebugOption.TPU_METRICS_DEBUG in self.args.debug:
                    if is_torch_tpu_available():
                        # tpu-comment: Logging debug metrics for PyTorch/XLA (compile, execute times, ops, etc.)
                        xm.master_print(met.metrics_report())
                    else:
                        logger.warning(
                            "You enabled PyTorch/XLA debug metrics but you don't have a TPU "
                            "configured. Check your training configuration if this is unexpected."
                        )
            elif self.args.aggregation_freq == "epoch": # 逐epoch聚合
                for i in range(fed_num):
                    # -------------- 初始化模型参数 --------------
                    if self.is_parallel: # 获取模型参数
                        model.module.load_state_dict(model_state_dicts["original"])
                    else:
                        model.load_state_dict(model_state_dicts["original"])
                    self.optimizer.load_state_dict(optimizer_state_dicts[i]) # 获取优化器参数
                    self.lr_scheduler.load_state_dict(lr_scheduler_state_dicts[i]) # 获取学习率参数
                    torch.cuda.empty_cache()
                    # -------------- 训练模型 --------------
                    model, tr_loss = self.train_for_single_model_epoch(
                        args, model, train_dataloaders[i], epoch, epochs_trained, 
                        tr_loss, steps_trained_in_current_epoch, len_dataloader,
                        resume_from_checkpoint, steps_trained_progress_bar)
                    # -------------- 保存模型参数 --------------
                    self.federated_adding(model, model_state_dicts, fed_num)
                    optimizer_state_dicts[i] = copy.deepcopy(self.optimizer.state_dict()) # 保存优化器参数
                    lr_scheduler_state_dicts[i] = copy.deepcopy(self.lr_scheduler.state_dict()) # 保存学习率参数
                    torch.cuda.empty_cache()
                tr_losses.append(tr_loss)
                
                # federated aggregation
                model_state_dicts["original"] = copy.deepcopy(model_state_dicts["updated"])
                torch.cuda.empty_cache()
                
                # 后处理
                self.state.epoch = epoch + 1
                
                self.control = self.callback_handler.on_epoch_end(args, self.state, self.control)
                
                if (epoch + 1) % self.args.eval_epoch == 0:
                    should_evaluate = self.control.should_evaluate
                    should_save = self.control.should_save
                    self.control.should_evaluate = True
                    self.control.should_save = False
                    self._maybe_log_save_evaluate(tr_loss, model, trial, epoch, ignore_keys_for_eval)
                    self.control.should_evaluate = should_evaluate
                    self.control.should_save = should_save
                
                if self.control.should_training_stop:
                    break
                
                if DebugOption.TPU_METRICS_DEBUG in self.args.debug:
                    if is_torch_tpu_available():
                        # tpu-comment: Logging debug metrics for PyTorch/XLA (compile, execute times, ops, etc.)
                        xm.master_print(met.metrics_report())
                    else:
                        logger.warning(
                            "You enabled PyTorch/XLA debug metrics but you don't have a TPU "
                            "configured. Check your training configuration if this is unexpected."
                        )
            else:
                raise ValueError("aggregation_freq error.")

        # ------------------------- 训练结束 -------------------------
        if args.past_index and hasattr(self, "_past"):
            # Clean the state at the end of training
            delattr(self, "_past")

        logger.info("\n\nTraining completed. Do not forget to share your model on huggingface.co/models =)\n\n")
        if args.load_best_model_at_end and self.state.best_model_checkpoint is not None:
            # Wait for everyone to get here so we are sur the model has been saved by process 0.
            if is_torch_tpu_available():
                xm.rendezvous("load_best_model_at_end")
            elif args.local_rank != -1:
                dist.barrier()
            elif is_sagemaker_mp_enabled():
                smp.barrier()

            self._load_best_model()

        # add remaining tr_loss
        self._total_loss_scalar += tr_loss.item()
        train_loss = self._total_loss_scalar / self.state.global_step
        # 计算loss
        # train_loss = torch.mean(torch.stack(tr_losses)).item()
        # ------------------------- End of Federated added -------------------------

        metrics = speed_metrics("train", start_time, num_samples=num_train_samples, num_steps=self.state.max_steps)
        self.store_flos()
        metrics["total_flos"] = self.state.total_flos
        metrics["train_loss"] = train_loss

        self.is_in_train = False

        self._memory_tracker.stop_and_update_metrics(metrics)

        self.log(metrics)

        run_dir = self._get_output_dir(trial)
        checkpoints_sorted = self._sorted_checkpoints(use_mtime=False, output_dir=run_dir)

        # Delete the last checkpoint when save_total_limit=1 if it's different from the best checkpoint.
        if self.state.best_model_checkpoint is not None and self.args.save_total_limit == 1:
            for checkpoint in checkpoints_sorted:
                if checkpoint != self.state.best_model_checkpoint:
                    logger.info(f"Deleting older checkpoint [{checkpoint}] due to args.save_total_limit")
                    shutil.rmtree(checkpoint)

        self.control = self.callback_handler.on_train_end(args, self.state, self.control)

        return TrainOutput(self.state.global_step, train_loss, metrics)

    ############## Federated ##############
    def forward_for_single_model_batch(self, args, model, epoch, step, inputs, tr_loss, steps_in_epoch):
        # MeZO added: estimate gradient
        self.original_forward = self.model.forward
        
        if args.trainer == "zo":
            raise ValueError()
            tr_loss_step = self.zo_step(model, inputs)
        else:
            if (
                ((step + 1) % args.gradient_accumulation_steps != 0)
                and args.local_rank != -1
                and args._no_sync_in_gradient_accumulation
            ):
                # Avoid unnecessary DDP synchronization since there will be no backward pass on this example.
                with model.no_sync():
                    tr_loss_step = self.training_step(model, inputs)
            else:
                tr_loss_step = self.training_step(model, inputs)

        if (
            args.logging_nan_inf_filter
            and not is_torch_tpu_available()
            and (torch.isnan(tr_loss_step) or torch.isinf(tr_loss_step))
        ):
            # if loss is nan or inf simply add the average of previous logged losses
            tr_loss += tr_loss / (1 + self.state.global_step - self._globalstep_last_logged)
        else:
            tr_loss += tr_loss_step

        self.current_flos += float(self.floating_point_ops(inputs))

        # Optimizer step for deepspeed must be called on every step regardless of the value of gradient_accumulation_steps
        if self.deepspeed:
            self.deepspeed.step()

        if (step + 1) % args.gradient_accumulation_steps == 0 or (
            # last step in epoch but step is always smaller than gradient_accumulation_steps
            steps_in_epoch <= args.gradient_accumulation_steps
            and (step + 1) == steps_in_epoch
        ):
            self.state.global_step += 1
            self.state.epoch = epoch + (step + 1) / steps_in_epoch
            self.control = self.callback_handler.on_step_end(args, self.state, self.control)
            self.aggre_step += 1

            # self._maybe_log_save_evaluate(tr_loss, model, trial, epoch, ignore_keys_for_eval)
        else:
            self.control = self.callback_handler.on_substep_end(args, self.state, self.control)
        return model, tr_loss
    
    def forward_for_single_model_epoch(self, args, model, train_dataloader, epoch, epochs_trained,
                                    tr_loss, steps_trained_in_current_epoch, len_dataloader,
                                    resume_from_checkpoint, steps_trained_progress_bar):
        if isinstance(train_dataloader, DataLoader) and isinstance(train_dataloader.sampler, DistributedSampler):
            train_dataloader.sampler.set_epoch(epoch)
        elif hasattr(train_dataloader, "dataset") and isinstance(train_dataloader.dataset, IterableDatasetShard):
            train_dataloader.dataset.set_epoch(epoch)

        if is_torch_tpu_available():
            parallel_loader = pl.ParallelLoader(train_dataloader, [args.device]).per_device_loader(args.device)
            epoch_iterator = parallel_loader
        else:
            epoch_iterator = train_dataloader

        # Reset the past mems state at the beginning of each epoch if necessary.
        if args.past_index >= 0:
            self._past = None

        steps_in_epoch = (
            len(epoch_iterator)
            if len_dataloader is not None
            else args.max_steps * args.gradient_accumulation_steps
        )
        self.control = self.callback_handler.on_epoch_begin(args, self.state, self.control)

        if epoch == epochs_trained and resume_from_checkpoint is not None and steps_trained_in_current_epoch == 0:
            self._load_rng_state(resume_from_checkpoint)

        step = -1
        for step, inputs in enumerate(epoch_iterator):

            # Skip past any already trained steps if resuming training
            if steps_trained_in_current_epoch > 0:
                steps_trained_in_current_epoch -= 1
                if steps_trained_progress_bar is not None:
                    steps_trained_progress_bar.update(1)
                if steps_trained_in_current_epoch == 0:
                    self._load_rng_state(resume_from_checkpoint)
                continue
            elif steps_trained_progress_bar is not None:
                steps_trained_progress_bar.close()
                steps_trained_progress_bar = None

            if step % args.gradient_accumulation_steps == 0:
                self.control = self.callback_handler.on_step_begin(args, self.state, self.control)

            # MeZO added: estimate gradient
            if args.trainer == "zo":
                raise ValueError()
                tr_loss_step = self.zo_step(model, inputs)
            else:
                if (
                    ((step + 1) % args.gradient_accumulation_steps != 0)
                    and args.local_rank != -1
                    and args._no_sync_in_gradient_accumulation
                ):
                    # Avoid unnecessary DDP synchronization since there will be no backward pass on this example.
                    with model.no_sync():
                        tr_loss_step = self.training_step(model, inputs)
                else:
                    tr_loss_step = self.training_step(model, inputs)

            if (
                args.logging_nan_inf_filter
                and not is_torch_tpu_available()
                and (torch.isnan(tr_loss_step) or torch.isinf(tr_loss_step))
            ):
                # if loss is nan or inf simply add the average of previous logged losses
                tr_loss += tr_loss / (1 + self.state.global_step - self._globalstep_last_logged)
            else:
                tr_loss += tr_loss_step

            self.current_flos += float(self.floating_point_ops(inputs))

            # Optimizer step for deepspeed must be called on every step regardless of the value of gradient_accumulation_steps
            if self.deepspeed:
                self.deepspeed.step()

            if (step + 1) % args.gradient_accumulation_steps == 0 or (
                # last step in epoch but step is always smaller than gradient_accumulation_steps
                steps_in_epoch <= args.gradient_accumulation_steps
                and (step + 1) == steps_in_epoch
            ):
                self.state.global_step += 1
                self.state.epoch = epoch + (step + 1) / steps_in_epoch
                self.control = self.callback_handler.on_step_end(args, self.state, self.control)
                self.aggre_step += 1

                # self._maybe_log_save_evaluate(tr_loss, model, trial, epoch, ignore_keys_for_eval)
            else:
                self.control = self.callback_handler.on_substep_end(args, self.state, self.control)

            if self.control.should_epoch_stop or self.control.should_training_stop:
                break
        if step < 0:
            logger.warning(
                "There seems to be not a single sample in your epoch_iterator, stopping training at step"
                f" {self.state.global_step}! This is expected if you're using an IterableDataset and set"
                f" num_steps ({max_steps}) higher than the number of available samples."
            )
            self.control.should_training_stop = True
        return model, tr_loss
    
    def move_state_dict_to_device(self, state_dict, device):
        # 创建一个新的state_dict来存储移动后的Tensor
        state_dict_on_device = {}
        for key, param in state_dict.items():
            # 将每个Tensor移动到指定的设备上
            state_dict_on_device[key] = param.to(device)
        return state_dict_on_device
    
    def federated_averaging(self, model_state_dicts):
        # 假设所有模型具有相同的结构，我们可以从第一个模型的状态字典中获取键
        if not model_state_dicts:
            raise ValueError("model_state_dicts cannot be empty")
        # 获得 fed_num
        fed_num = len(model_state_dicts)
        # 初始化一个空的字典来存储全局平均参数
        global_state_dict = {}
        # 遍历第一个模型的参数
        for key in model_state_dicts[0].keys():
            # 初始化该参数的累加和
            sum_param = torch.zeros_like(model_state_dicts[0][key])
            # 遍历所有模型的参数
            for model_state_dict in model_state_dicts:
                # 确保所有模型都有这个键
                if key not in model_state_dict:
                    raise KeyError(f"Key {key} missing in some model state dicts")
                # 累加参数
                sum_param += model_state_dict[key]
            # 计算平均值
            avg_param = sum_param / fed_num
            # 存储到全局状态字典中
            global_state_dict[key] = avg_param
        # 返回全局平均后的模型状态字典
        return global_state_dict

    def federated_optim(self, args, model):
        if self.aggre_step == 0: # 无需聚合
            return model
        else: # 需训练的参数梯度缩小 self.aggre_step 倍
            for name, param in model.named_parameters():
                if param.requires_grad:
                    param.grad = param.grad / self.aggre_step
        self.aggre_step = 0 # 置零
            
        # Gradient clipping
        if args.max_grad_norm is not None and args.max_grad_norm > 0 and not self.deepspeed:
            # deepspeed does its own clipping

            if self.do_grad_scaling:
                # Reduce gradients first for XLA
                if is_torch_tpu_available():
                    gradients = xm._fetch_gradients(self.optimizer)
                    xm.all_reduce("sum", gradients, scale=1.0 / xm.xrt_world_size())
                # AMP: gradients need unscaling
                self.scaler.unscale_(self.optimizer)

            if is_sagemaker_mp_enabled() and args.fp16:
                self.optimizer.clip_master_grads(args.max_grad_norm)
            elif hasattr(self.optimizer, "clip_grad_norm"):
                # Some optimizers (like the sharded optimizer) have a specific way to do gradient clipping
                self.optimizer.clip_grad_norm(args.max_grad_norm)
            elif hasattr(model, "clip_grad_norm_"):
                # Some models (like FullyShardedDDP) have a specific way to do gradient clipping
                model.clip_grad_norm_(args.max_grad_norm)
            else:
                # Revert to normal clipping otherwise, handling Apex or full precision
                nn.utils.clip_grad_norm_(
                    amp.master_params(self.optimizer) if self.use_apex else model.parameters(),
                    args.max_grad_norm,
                )

        # Optimizer step
        optimizer_was_run = True
        if self.deepspeed:
            pass  # called outside the loop
        elif is_torch_tpu_available():
            if self.do_grad_scaling:
                self.scaler.step(self.optimizer)
                self.scaler.update()
            else:
                xm.optimizer_step(self.optimizer)
        elif self.do_grad_scaling:
            scale_before = self.scaler.get_scale()
            self.scaler.step(self.optimizer)
            self.scaler.update()
            scale_after = self.scaler.get_scale()
            optimizer_was_run = scale_before <= scale_after
        else:
            self.optimizer.step()

        if optimizer_was_run and not self.deepspeed:
            self.lr_scheduler.step()
        model.zero_grad()
        
        return model

    ############## FedSGD ##############
    # 1倍显存实现Federated Learning。仅可用于batch FO
    def _inner_training_loop_FedSGD(
        self, batch_size=None, args=None, resume_from_checkpoint=None, trial=None, ignore_keys_for_eval=None
    ):
        """
        We overload the original training loop to add linear probing and MeZO. Search key word "MeZO added"
        for those updates.
        """
        assert args.trainer == "regular", "This is only for FO"
        assert not self.deepspeed, "Not implement fot deepspeed"
        
        # 创建一个文件处理器，设置日志文件路径
        file_handler = py_logging.FileHandler(os.path.join(self.args.output_dir, self.args.log_file))
        file_handler.setLevel(py_logging.INFO)
        # 将处理器添加到 logger
        logger.addHandler(file_handler)
        
        self._train_batch_size = batch_size
        fed_num = self.args.fed_num
        # Data loader and number of training steps
        train_dataloader = self.get_train_dataloader()

        # MeZO added: Linear probing
        if self.args.linear_probing:

            def _get_token_prediction_layer(model):
                if model.config.model_type == "opt":
                    return model.lm_head
                else:
                    raise NotImplementedError(model.config.model_type)

            def _extract_features(model, *args, **kwargs):
                """some magic for getting features pre last layer"""
                features = {}
                def __hook(model_, input_, output_):
                    features["features"] = input_[0].detach()

                _get_token_prediction_layer(model).register_forward_hook(__hook)
                model.forward(*args, **kwargs)
                return features["features"]

            logger.info("Linear probing")
            logger.info("Starting to get features for training dataset")
            targets = []
            features = []
            with torch.inference_mode():
                for step, inputs in enumerate(tqdm(train_dataloader)):
                    for k, v in inputs.items():
                        if isinstance(v, torch.Tensor):
                            inputs[k] = v.to(self.model.device)
                        
                    feature = _extract_features(self.model, **inputs)
                    target = inputs["labels"]

                    # Shift the target (bc it's autoregressive LM) and add the corresponding part
                    assert not self.args.train_as_classification and self.args.only_train_option
                    feature, target = feature[:, :-1], target[:, 1:]
                    for _i, _len in enumerate(inputs["option_len"]):
                        features.append(feature[_i, -_len:])
                        targets.append(target[_i, -_len:])

            logger.info("Finished getting features for training dataset")

            features = torch.cat(features, dim=0).cpu().numpy()
            targets = torch.cat(targets, dim=0).cpu().numpy()
            # Whether to use bias
            if self.model.config.model_type in ["opt", "gpt2"]:
                use_bias = False
            else:
                raise NotImplementedError
            # Set early stopping
            tol = 0.01 if self.args.lp_early_stopping else 1e-4 # 1e-4 is scipy default
            max_iter = 1000 if self.args.lp_early_stopping else 5000

            logger.info("Fitting logistic regression...")
            reg = LogisticRegressionCV(max_iter=max_iter, fit_intercept=use_bias, multi_class="multinomial", random_state=0, tol=tol, n_jobs=-1).fit(features, targets)
            logger.info("Done")

            logger.info("Assigning weights to model")
            decoder = _get_token_prediction_layer(self.model)
            coef_torch = torch.tensor(reg.coef_, device=decoder.weight.device, dtype=decoder.weight.dtype)
            if use_bias:
                bias_torch = torch.tensor(reg.intercept_, device=decoder.weight.device, dtype=decoder.weight.dtype)
            if coef_torch.shape[0] == 1: # The regressor only detects two classes
                assert len(reg.classes_) == 2
                coef_torch = torch.cat([-coef_torch / 2, coef_torch / 2], dim=0)
                if use_bias:
                    bias_torch = torch.cat([-bias_torch / 2, bias_torch / 2], dim=0)

            for _i, token_id in enumerate(reg.classes_):
                decoder.weight.data[token_id] = coef_torch[_i]
                if use_bias:
                    decoder.bias.data[token_id] = bias_torch[_i]

            return None

        # Setting up training control variables:
        # number of training epochs: num_train_epochs
        # number of training steps per epoch: num_update_steps_per_epoch
        # total number of training steps to execute: max_steps
        total_train_batch_size = args.train_batch_size * args.gradient_accumulation_steps * args.world_size

        len_dataloader = None
        # 获取数据集大小，并计算训练epoch数
        if has_length(train_dataloader):
            len_dataloader = len(train_dataloader)
            num_update_steps_per_epoch = len_dataloader // args.gradient_accumulation_steps
            num_update_steps_per_epoch = max(num_update_steps_per_epoch, 1)
            num_examples = self.num_examples(train_dataloader)
            if args.max_steps > 0:
                max_steps = args.max_steps
                num_train_epochs = args.max_steps // num_update_steps_per_epoch + int(
                    args.max_steps % num_update_steps_per_epoch > 0
                )
                # May be slightly incorrect if the last batch in the training dataloader has a smaller size but it's
                # the best we can do.
                num_train_samples = args.max_steps * total_train_batch_size
            else:
                max_steps = math.ceil(args.num_train_epochs * num_update_steps_per_epoch)
                num_train_epochs = math.ceil(args.num_train_epochs)
                num_train_samples = self.num_examples(train_dataloader) * args.num_train_epochs
        elif args.max_steps > 0:  # Rely on max_steps when dataloader does not have a working size
            raise NotImplementedError(
                "The training dataloader does not have a length, please set max_steps to avoid an incorrect number of training steps."
            )
            max_steps = args.max_steps
            # Setting a very large number of epochs so we go as many times as necessary over the iterator.
            num_train_epochs = sys.maxsize
            num_update_steps_per_epoch = max_steps
            num_examples = total_train_batch_size * args.max_steps
            num_train_samples = args.max_steps * total_train_batch_size
        else:
            raise ValueError(
                "args.max_steps must be set to a positive value if dataloader does not have a length, was"
                f" {args.max_steps}"
            )

        if DebugOption.UNDERFLOW_OVERFLOW in self.args.debug:
            if self.args.n_gpu > 1:
                # nn.DataParallel(model) replicates the model, creating new variables and module
                # references registered here no longer work on other gpus, breaking the module
                raise ValueError(
                    "Currently --debug underflow_overflow is not supported under DP. Please use DDP"
                    " (torch.distributed.launch)."
                )
            else:
                debug_overflow = DebugUnderflowOverflow(self.model)  # noqa

        delay_optimizer_creation = (
            self.sharded_ddp is not None
            and self.sharded_ddp != ShardedDDPOption.SIMPLE
            or is_sagemaker_mp_enabled()
            or self.fsdp is not None
        )
        if args.deepspeed:
            deepspeed_engine, optimizer, lr_scheduler = deepspeed_init(
                self, num_training_steps=max_steps, resume_from_checkpoint=resume_from_checkpoint
            )
            self.model = deepspeed_engine.module
            self.model_wrapped = deepspeed_engine
            self.deepspeed = deepspeed_engine
            self.optimizer = optimizer
            self.lr_scheduler = lr_scheduler
        elif not delay_optimizer_creation:
            self.create_optimizer_and_scheduler(num_training_steps=max_steps)

        self.state = TrainerState()
        self.state.is_hyper_param_search = trial is not None

        # Activate gradient checkpointing if needed
        if args.gradient_checkpointing:
            self.model.gradient_checkpointing_enable()

        model = self._wrap_model(self.model_wrapped)

        if is_sagemaker_mp_enabled() and resume_from_checkpoint is not None:
            self._load_from_checkpoint(resume_from_checkpoint, model)

        # for the rest of this function `model` is the outside model, whether it was wrapped or not
        if model is not self.model:
            self.model_wrapped = model

        if delay_optimizer_creation:
            self.create_optimizer_and_scheduler(num_training_steps=max_steps)

        # Check if saved optimizer or scheduler states exist
        self._load_optimizer_and_scheduler(resume_from_checkpoint)

        # important: at this point:
        # self.model         is the Transformers Model
        # self.model_wrapped is DDP(Transformers Model), Deepspeed(Transformers Model), etc.

        # Train!
        logger.info("***** Running training *****")
        logger.info(f"  Num examples = {num_examples}")
        logger.info(f"  Num Epochs = {num_train_epochs}")
        logger.info(f"  Instantaneous batch size per device = {args.per_device_train_batch_size}")
        logger.info(f"  Total train batch size (w. parallel, distributed & accumulation) = {total_train_batch_size}")
        logger.info(f"  Gradient Accumulation steps = {args.gradient_accumulation_steps}")
        logger.info(f"  Total optimization steps = {max_steps}")
        logger.info(
            f"  Number of trainable parameters = {sum(p.numel() for p in model.parameters() if p.requires_grad)}"
        )

        self.state.epoch = 0
        start_time = time.time()
        epochs_trained = 0
        steps_trained_in_current_epoch = 0
        steps_trained_progress_bar = None

        # Check if continuing training from a checkpoint
        if resume_from_checkpoint is not None and os.path.isfile(
            os.path.join(resume_from_checkpoint, TRAINER_STATE_NAME)
        ):
            self.state = TrainerState.load_from_json(os.path.join(resume_from_checkpoint, TRAINER_STATE_NAME))
            epochs_trained = self.state.global_step // num_update_steps_per_epoch
            if not args.ignore_data_skip:
                steps_trained_in_current_epoch = self.state.global_step % (num_update_steps_per_epoch)
                steps_trained_in_current_epoch *= args.gradient_accumulation_steps
            else:
                steps_trained_in_current_epoch = 0

            logger.info("  Continuing training from checkpoint, will skip to saved global_step")
            logger.info(f"  Continuing training from epoch {epochs_trained}")
            logger.info(f"  Continuing training from global step {self.state.global_step}")
            if not args.ignore_data_skip:
                logger.info(
                    f"  Will skip the first {epochs_trained} epochs then the first {steps_trained_in_current_epoch} "
                    "batches in the first epoch. If this takes a lot of time, you can add the `--ignore_data_skip` "
                    "flag to your launch command, but you will resume the training on data already seen by your model."
                )
                if self.is_local_process_zero() and not args.disable_tqdm:
                    steps_trained_progress_bar = tqdm(total=steps_trained_in_current_epoch)
                    steps_trained_progress_bar.set_description("Skipping the first batches")

        # Update the references
        self.callback_handler.model = self.model
        self.callback_handler.optimizer = self.optimizer
        self.callback_handler.lr_scheduler = self.lr_scheduler
        self.callback_handler.train_dataloader = train_dataloader
        if self.hp_name is not None and self._trial is not None:
            # use self._trial because the SigOpt/Optuna hpo only call `_hp_search_setup(trial)` instead of passing trial
            # parameter to Train when using DDP.
            self.state.trial_name = self.hp_name(self._trial)
        if trial is not None:
            assignments = trial.assignments if self.hp_search_backend == HPSearchBackend.SIGOPT else trial
            self.state.trial_params = hp_params(assignments)
        else:
            self.state.trial_params = None
        # This should be the same if the state has been saved but in case the training arguments changed, it's safer
        # to set this after the load.
        self.state.max_steps = max_steps
        self.state.num_train_epochs = num_train_epochs
        self.state.is_local_process_zero = self.is_local_process_zero()
        self.state.is_world_process_zero = self.is_world_process_zero()

        # tr_loss is a tensor to avoid synchronization of TPUs through .item()
        tr_loss = torch.tensor(0.0).to(args.device)
        # _total_loss_scalar is updated everytime .item() has to be called on tr_loss and stores the sum of all losses
        self._total_loss_scalar = 0.0
        self._globalstep_last_logged = self.state.global_step
        model.zero_grad()

        self.control = self.callback_handler.on_train_begin(args, self.state, self.control)

        # Skip the first epochs_trained epochs to get the random state of the dataloader at the right point.
        if not args.ignore_data_skip:
            for epoch in range(epochs_trained):
                is_random_sampler = hasattr(train_dataloader, "sampler") and isinstance(
                    train_dataloader.sampler, RandomSampler
                )
                if is_torch_less_than_1_11 or not is_random_sampler:
                    # We just need to begin an iteration to create the randomization of the sampler.
                    # That was before PyTorch 1.11 however...
                    for _ in train_dataloader:
                        break
                else:
                    # Otherwise we need to call the whooooole sampler cause there is some random operation added
                    # AT THE VERY END!
                    _ = list(train_dataloader.sampler)

        # ------------------------- Federated added -------------------------
        # 为 Federated 创建对应的 dataloaders
        self.is_parallel = isinstance(model, torch.nn.DataParallel) or isinstance(model, torch.nn.parallel.DistributedDataParallel) # 检查模型是否被 DataParallel 包装
        train_dataloaders = self.split_dataloader(train_dataloader, fed_num) # 将原始DataLoader均分为fed_num份
        # What parameters to optimize 
        self.named_parameters_to_optim = []
        for name, param in model.named_parameters():
            if param.requires_grad:
                self.named_parameters_to_optim.append((name, param))
        self.aggre_step = 0 # 聚合了多少个step，用以等比缩小梯度
        # ------------------------- 开始训练 -------------------------
        for epoch in range(epochs_trained, num_train_epochs):
            tr_losses = []
            # training for federated learning
            if self.args.aggregation_freq == "batch": # 逐batch聚合
                if not self.is_batch_nums_equal(train_dataloaders):
                    raise ValueError("All train_dataloaders must have the same number of batches.")
                # 训练前的配置
                epoch_iterators = []
                for i in range(fed_num):
                    train_dataloader = train_dataloaders[i]
                    
                    if isinstance(train_dataloader, DataLoader) and isinstance(train_dataloader.sampler, DistributedSampler):
                        train_dataloader.sampler.set_epoch(epoch)
                    elif hasattr(train_dataloader, "dataset") and isinstance(train_dataloader.dataset, IterableDatasetShard):
                        train_dataloader.dataset.set_epoch(epoch)

                    if is_torch_tpu_available():
                        parallel_loader = pl.ParallelLoader(train_dataloader, [args.device]).per_device_loader(args.device)
                        epoch_iterator = parallel_loader
                    else:
                        epoch_iterator = train_dataloader
                    
                    epoch_iterators.append(epoch_iterator)

                    # Reset the past mems state at the beginning of each epoch if necessary.
                    if args.past_index >= 0:
                        self._past = None

                    self.control = self.callback_handler.on_epoch_begin(args, self.state, self.control)
                if epoch == epochs_trained and resume_from_checkpoint is not None and steps_trained_in_current_epoch == 0:
                    self._load_rng_state(resume_from_checkpoint)
                # 遍历每个batch
                for step, epoch_inputs in enumerate(zip(*epoch_iterators)):
                    # Skip past any already trained steps if resuming training
                    if steps_trained_in_current_epoch > 0:
                        steps_trained_in_current_epoch -= 1
                        if steps_trained_progress_bar is not None:
                            steps_trained_progress_bar.update(1)
                        if steps_trained_in_current_epoch == 0:
                            self._load_rng_state(resume_from_checkpoint)
                        continue
                    elif steps_trained_progress_bar is not None:
                        steps_trained_progress_bar.close()
                        steps_trained_progress_bar = None

                    if step % args.gradient_accumulation_steps == 0:
                        self.control = self.callback_handler.on_step_begin(args, self.state, self.control)
                    
                    # 逐batches训练
                    for i in range(fed_num):
                        inputs = epoch_inputs[i]
                        epoch_iterator = epoch_iterators[i]
                        steps_in_epoch = (
                            len(epoch_iterator)
                            if len_dataloader is not None
                            else args.max_steps * args.gradient_accumulation_steps
                        )
                        model, tr_loss = self.forward_for_single_model_batch(args, model, epoch, step, inputs, tr_loss, steps_in_epoch)
                    # aggregation
                    model = self.federated_optim(args, model)
                tr_losses.append(tr_loss)
                
                # 后处理
                self.state.epoch = epoch + 1
                
                self.control = self.callback_handler.on_epoch_end(args, self.state, self.control)
                
                if (epoch + 1) % self.args.eval_epoch == 0:
                    should_evaluate = self.control.should_evaluate
                    should_save = self.control.should_save
                    self.control.should_evaluate = True
                    self.control.should_save = False
                    self._maybe_log_save_evaluate(tr_loss, model, trial, epoch, ignore_keys_for_eval)
                    self.control.should_evaluate = should_evaluate
                    self.control.should_save = should_save
                
                # 本model的本epoch是否训练结束
                if self.control.should_epoch_stop or self.control.should_training_stop:
                    break
                if step < 0:
                    logger.warning(
                        "There seems to be not a single sample in your epoch_iterator, stopping training at step"
                        f" {self.state.global_step}! This is expected if you're using an IterableDataset and set"
                        f" num_steps ({max_steps}) higher than the number of available samples."
                    )
                    self.control.should_training_stop = True
                if DebugOption.TPU_METRICS_DEBUG in self.args.debug:
                    if is_torch_tpu_available():
                        # tpu-comment: Logging debug metrics for PyTorch/XLA (compile, execute times, ops, etc.)
                        xm.master_print(met.metrics_report())
                    else:
                        logger.warning(
                            "You enabled PyTorch/XLA debug metrics but you don't have a TPU "
                            "configured. Check your training configuration if this is unexpected."
                        )
            elif re.search(r'(\d+)batch', self.args.aggregation_freq): # 形如 "[L]batch" 的聚合频率
                raise NotImplementedError()
                L = int(re.search(r'(\d+)batch', self.args.aggregation_freq).group(1))
                if not self.is_batch_nums_equal(train_dataloaders):
                    raise ValueError("All train_dataloaders must have the same number of batches.")
                # 训练前的配置
                epoch_iterators = []
                for i in range(fed_num):
                    train_dataloader = train_dataloaders[i]
                    
                    if isinstance(train_dataloader, DataLoader) and isinstance(train_dataloader.sampler, DistributedSampler):
                        train_dataloader.sampler.set_epoch(epoch)
                    elif hasattr(train_dataloader, "dataset") and isinstance(train_dataloader.dataset, IterableDatasetShard):
                        train_dataloader.dataset.set_epoch(epoch)

                    if is_torch_tpu_available():
                        parallel_loader = pl.ParallelLoader(train_dataloader, [args.device]).per_device_loader(args.device)
                        epoch_iterator = parallel_loader
                    else:
                        epoch_iterator = train_dataloader
                    
                    epoch_iterators.append(epoch_iterator)

                # Reset the past mems state at the beginning of each epoch if necessary.
                if args.past_index >= 0:
                    self._past = None

                self.control = self.callback_handler.on_epoch_begin(args, self.state, self.control)
                if epoch == epochs_trained and resume_from_checkpoint is not None and steps_trained_in_current_epoch == 0:
                    self._load_rng_state(resume_from_checkpoint)
                # 遍历每个batch
                for step, epoch_inputs in enumerate(zip(*epoch_iterators)):
                    # Skip past any already trained steps if resuming training
                    if steps_trained_in_current_epoch > 0:
                        steps_trained_in_current_epoch -= 1
                        if steps_trained_progress_bar is not None:
                            steps_trained_progress_bar.update(1)
                        if steps_trained_in_current_epoch == 0:
                            self._load_rng_state(resume_from_checkpoint)
                        continue
                    elif steps_trained_progress_bar is not None:
                        steps_trained_progress_bar.close()
                        steps_trained_progress_bar = None

                    if step % args.gradient_accumulation_steps == 0:
                        self.control = self.callback_handler.on_step_begin(args, self.state, self.control)
                    
                    # 逐batches训练
                    for i in range(fed_num):
                        inputs = epoch_inputs[i]
                        epoch_iterator = epoch_iterators[i]
                        steps_in_epoch = (
                            len(epoch_iterator)
                            if len_dataloader is not None
                            else args.max_steps * args.gradient_accumulation_steps
                        )
                        model, tr_loss = self.forward_for_single_model_batch(args, model, epoch, step, inputs, tr_loss, steps_in_epoch)
                    # aggregation
                    if (step + 1) % L == 0:
                        model = self.federated_optim(args, model)
                # 遍历完所有的batches，为了避免还有未聚合的batches，再次聚合
                model = self.federated_optim(args, model)
                
                tr_losses.append(tr_loss)
                
                # 后处理
                self.state.epoch = epoch + 1
                
                self.control = self.callback_handler.on_epoch_end(args, self.state, self.control)
                
                if (epoch + 1) % self.args.eval_epoch == 0:
                    should_evaluate = self.control.should_evaluate
                    should_save = self.control.should_save
                    self.control.should_evaluate = True
                    self.control.should_save = False
                    self._maybe_log_save_evaluate(tr_loss, model, trial, epoch, ignore_keys_for_eval)
                    self.control.should_evaluate = should_evaluate
                    self.control.should_save = should_save
                
                # 本model的本epoch是否训练结束
                if self.control.should_epoch_stop or self.control.should_training_stop:
                    break
                if step < 0:
                    logger.warning(
                        "There seems to be not a single sample in your epoch_iterator, stopping training at step"
                        f" {self.state.global_step}! This is expected if you're using an IterableDataset and set"
                        f" num_steps ({max_steps}) higher than the number of available samples."
                    )
                    self.control.should_training_stop = True
                if DebugOption.TPU_METRICS_DEBUG in self.args.debug:
                    if is_torch_tpu_available():
                        # tpu-comment: Logging debug metrics for PyTorch/XLA (compile, execute times, ops, etc.)
                        xm.master_print(met.metrics_report())
                    else:
                        logger.warning(
                            "You enabled PyTorch/XLA debug metrics but you don't have a TPU "
                            "configured. Check your training configuration if this is unexpected."
                        )
            elif self.args.aggregation_freq == "epoch": # 逐epoch聚合
                raise NotImplementedError()
                for i in range(fed_num):
                    model, tr_loss = self.forward_for_single_model_epoch(
                        args, model, train_dataloaders[i], epoch, epochs_trained, 
                        tr_loss, steps_trained_in_current_epoch, len_dataloader,
                        resume_from_checkpoint, steps_trained_progress_bar)
                tr_losses.append(tr_loss)
                
                # 无显式的federated aggregation，而是累计多个梯度之后一次性后向传播，更新optimizer和lr_scheduler
                model = self.federated_optim(args, model)
                
                # 后处理
                self.state.epoch = epoch + 1
                
                self.control = self.callback_handler.on_epoch_end(args, self.state, self.control)
                
                if (epoch + 1) % self.args.eval_epoch == 0:
                    should_evaluate = self.control.should_evaluate
                    should_save = self.control.should_save
                    self.control.should_evaluate = True
                    self.control.should_save = False
                    self._maybe_log_save_evaluate(tr_loss, model, trial, epoch, ignore_keys_for_eval)
                    self.control.should_evaluate = should_evaluate
                    self.control.should_save = should_save
                
                if self.control.should_training_stop:
                    break
                
                if DebugOption.TPU_METRICS_DEBUG in self.args.debug:
                    if is_torch_tpu_available():
                        # tpu-comment: Logging debug metrics for PyTorch/XLA (compile, execute times, ops, etc.)
                        xm.master_print(met.metrics_report())
                    else:
                        logger.warning(
                            "You enabled PyTorch/XLA debug metrics but you don't have a TPU "
                            "configured. Check your training configuration if this is unexpected."
                        )
            else:
                raise ValueError("aggregation_freq error.")

        # ------------------------- 训练结束 -------------------------
        if args.past_index and hasattr(self, "_past"):
            # Clean the state at the end of training
            delattr(self, "_past")

        logger.info("\n\nTraining completed. Do not forget to share your model on huggingface.co/models =)\n\n")
        if args.load_best_model_at_end and self.state.best_model_checkpoint is not None:
            # Wait for everyone to get here so we are sur the model has been saved by process 0.
            if is_torch_tpu_available():
                xm.rendezvous("load_best_model_at_end")
            elif args.local_rank != -1:
                dist.barrier()
            elif is_sagemaker_mp_enabled():
                smp.barrier()

            self._load_best_model()

        # add remaining tr_loss
        self._total_loss_scalar += tr_loss.item()
        train_loss = self._total_loss_scalar / self.state.global_step
        # 计算loss
        # train_loss = torch.mean(torch.stack(tr_losses)).item()
        # ------------------------- End of Federated added -------------------------

        metrics = speed_metrics("train", start_time, num_samples=num_train_samples, num_steps=self.state.max_steps)
        self.store_flos()
        metrics["total_flos"] = self.state.total_flos
        metrics["train_loss"] = train_loss

        self.is_in_train = False

        self._memory_tracker.stop_and_update_metrics(metrics)

        self.log(metrics)

        run_dir = self._get_output_dir(trial)
        checkpoints_sorted = self._sorted_checkpoints(use_mtime=False, output_dir=run_dir)

        # Delete the last checkpoint when save_total_limit=1 if it's different from the best checkpoint.
        if self.state.best_model_checkpoint is not None and self.args.save_total_limit == 1:
            for checkpoint in checkpoints_sorted:
                if checkpoint != self.state.best_model_checkpoint:
                    logger.info(f"Deleting older checkpoint [{checkpoint}] due to args.save_total_limit")
                    shutil.rmtree(checkpoint)

        self.control = self.callback_handler.on_train_end(args, self.state, self.control)

        return TrainOutput(self.state.global_step, train_loss, metrics)

    ############## FedAvg ##############
    # K倍显存实现Federated Learning。既可用于FO，又可用于ZO
    def _inner_training_loop_FedAvg(
        self, batch_size=None, args=None, resume_from_checkpoint=None, trial=None, ignore_keys_for_eval=None
    ):
        """
        We overload the original training loop to add linear probing and MeZO. Search key word "MeZO added"
        for those updates.
        """
        
        # 创建一个文件处理器，设置日志文件路径
        file_handler = py_logging.FileHandler(os.path.join(self.args.output_dir, self.args.log_file))
        file_handler.setLevel(py_logging.INFO)
        # 将处理器添加到 logger
        logger.addHandler(file_handler)
        
        self._train_batch_size = batch_size
        fed_num = self.args.fed_num
        # Data loader and number of training steps
        train_dataloader = self.get_train_dataloader()

        # MeZO added: Linear probing
        if self.args.linear_probing:

            def _get_token_prediction_layer(model):
                if model.config.model_type == "opt":
                    return model.lm_head
                else:
                    raise NotImplementedError(model.config.model_type)

            def _extract_features(model, *args, **kwargs):
                """some magic for getting features pre last layer"""
                features = {}
                def __hook(model_, input_, output_):
                    features["features"] = input_[0].detach()

                _get_token_prediction_layer(model).register_forward_hook(__hook)
                model.forward(*args, **kwargs)
                return features["features"]

            logger.info("Linear probing")
            logger.info("Starting to get features for training dataset")
            targets = []
            features = []
            with torch.inference_mode():
                for step, inputs in enumerate(tqdm(train_dataloader)):
                    for k, v in inputs.items():
                        if isinstance(v, torch.Tensor):
                            inputs[k] = v.to(self.model.device)
                        
                    feature = _extract_features(self.model, **inputs)
                    target = inputs["labels"]

                    # Shift the target (bc it's autoregressive LM) and add the corresponding part
                    assert not self.args.train_as_classification and self.args.only_train_option
                    feature, target = feature[:, :-1], target[:, 1:]
                    for _i, _len in enumerate(inputs["option_len"]):
                        features.append(feature[_i, -_len:])
                        targets.append(target[_i, -_len:])

            logger.info("Finished getting features for training dataset")

            features = torch.cat(features, dim=0).cpu().numpy()
            targets = torch.cat(targets, dim=0).cpu().numpy()
            # Whether to use bias
            if self.model.config.model_type in ["opt", "gpt2"]:
                use_bias = False
            else:
                raise NotImplementedError
            # Set early stopping
            tol = 0.01 if self.args.lp_early_stopping else 1e-4 # 1e-4 is scipy default
            max_iter = 1000 if self.args.lp_early_stopping else 5000

            logger.info("Fitting logistic regression...")
            reg = LogisticRegressionCV(max_iter=max_iter, fit_intercept=use_bias, multi_class="multinomial", random_state=0, tol=tol, n_jobs=-1).fit(features, targets)
            logger.info("Done")

            logger.info("Assigning weights to model")
            decoder = _get_token_prediction_layer(self.model)
            coef_torch = torch.tensor(reg.coef_, device=decoder.weight.device, dtype=decoder.weight.dtype)
            if use_bias:
                bias_torch = torch.tensor(reg.intercept_, device=decoder.weight.device, dtype=decoder.weight.dtype)
            if coef_torch.shape[0] == 1: # The regressor only detects two classes
                assert len(reg.classes_) == 2
                coef_torch = torch.cat([-coef_torch / 2, coef_torch / 2], dim=0)
                if use_bias:
                    bias_torch = torch.cat([-bias_torch / 2, bias_torch / 2], dim=0)

            for _i, token_id in enumerate(reg.classes_):
                decoder.weight.data[token_id] = coef_torch[_i]
                if use_bias:
                    decoder.bias.data[token_id] = bias_torch[_i]

            return None

        # Setting up training control variables:
        # number of training epochs: num_train_epochs
        # number of training steps per epoch: num_update_steps_per_epoch
        # total number of training steps to execute: max_steps
        total_train_batch_size = args.train_batch_size * args.gradient_accumulation_steps * args.world_size

        len_dataloader = None
        # 获取数据集大小，并计算训练epoch数
        if has_length(train_dataloader):
            len_dataloader = len(train_dataloader)
            num_update_steps_per_epoch = len_dataloader // args.gradient_accumulation_steps
            num_update_steps_per_epoch = max(num_update_steps_per_epoch, 1)
            num_examples = self.num_examples(train_dataloader)
            if args.max_steps > 0:
                max_steps = args.max_steps
                num_train_epochs = args.max_steps // num_update_steps_per_epoch + int(
                    args.max_steps % num_update_steps_per_epoch > 0
                )
                # May be slightly incorrect if the last batch in the training dataloader has a smaller size but it's
                # the best we can do.
                num_train_samples = args.max_steps * total_train_batch_size
            else:
                max_steps = math.ceil(args.num_train_epochs * num_update_steps_per_epoch)
                num_train_epochs = math.ceil(args.num_train_epochs)
                num_train_samples = self.num_examples(train_dataloader) * args.num_train_epochs
        elif args.max_steps > 0:  # Rely on max_steps when dataloader does not have a working size
            raise NotImplementedError(
                "The training dataloader does not have a length, please set max_steps to avoid an incorrect number of training steps."
            )
            max_steps = args.max_steps
            # Setting a very large number of epochs so we go as many times as necessary over the iterator.
            num_train_epochs = sys.maxsize
            num_update_steps_per_epoch = max_steps
            num_examples = total_train_batch_size * args.max_steps
            num_train_samples = args.max_steps * total_train_batch_size
        else:
            raise ValueError(
                "args.max_steps must be set to a positive value if dataloader does not have a length, was"
                f" {args.max_steps}"
            )

        if DebugOption.UNDERFLOW_OVERFLOW in self.args.debug:
            if self.args.n_gpu > 1:
                # nn.DataParallel(model) replicates the model, creating new variables and module
                # references registered here no longer work on other gpus, breaking the module
                raise ValueError(
                    "Currently --debug underflow_overflow is not supported under DP. Please use DDP"
                    " (torch.distributed.launch)."
                )
            else:
                debug_overflow = DebugUnderflowOverflow(self.model)  # noqa

        delay_optimizer_creation = (
            self.sharded_ddp is not None
            and self.sharded_ddp != ShardedDDPOption.SIMPLE
            or is_sagemaker_mp_enabled()
            or self.fsdp is not None
        )
        if args.deepspeed:
            deepspeed_engine, optimizer, lr_scheduler = deepspeed_init(
                self, num_training_steps=max_steps, resume_from_checkpoint=resume_from_checkpoint
            )
            self.model = deepspeed_engine.module
            self.model_wrapped = deepspeed_engine
            self.deepspeed = deepspeed_engine
            self.optimizer = optimizer
            self.lr_scheduler = lr_scheduler
        elif not delay_optimizer_creation:
            self.create_optimizer_and_scheduler(num_training_steps=max_steps)

        self.state = TrainerState()
        self.state.is_hyper_param_search = trial is not None

        # Activate gradient checkpointing if needed
        if args.gradient_checkpointing:
            self.model.gradient_checkpointing_enable()

        model = self._wrap_model(self.model_wrapped)

        if is_sagemaker_mp_enabled() and resume_from_checkpoint is not None:
            self._load_from_checkpoint(resume_from_checkpoint, model)

        # for the rest of this function `model` is the outside model, whether it was wrapped or not
        if model is not self.model:
            self.model_wrapped = model

        if delay_optimizer_creation:
            self.create_optimizer_and_scheduler(num_training_steps=max_steps)

        # Check if saved optimizer or scheduler states exist
        self._load_optimizer_and_scheduler(resume_from_checkpoint)

        # important: at this point:
        # self.model         is the Transformers Model
        # self.model_wrapped is DDP(Transformers Model), Deepspeed(Transformers Model), etc.

        # Train!
        logger.info("***** Running training *****")
        logger.info(f"  Num examples = {num_examples}")
        logger.info(f"  Num Epochs = {num_train_epochs}")
        logger.info(f"  Instantaneous batch size per device = {args.per_device_train_batch_size}")
        logger.info(f"  Total train batch size (w. parallel, distributed & accumulation) = {total_train_batch_size}")
        logger.info(f"  Gradient Accumulation steps = {args.gradient_accumulation_steps}")
        logger.info(f"  Total optimization steps = {max_steps}")
        logger.info(
            f"  Number of trainable parameters = {sum(p.numel() for p in model.parameters() if p.requires_grad)}"
        )

        self.state.epoch = 0
        start_time = time.time()
        epochs_trained = 0
        steps_trained_in_current_epoch = 0
        steps_trained_progress_bar = None

        # Check if continuing training from a checkpoint
        if resume_from_checkpoint is not None and os.path.isfile(
            os.path.join(resume_from_checkpoint, TRAINER_STATE_NAME)
        ):
            self.state = TrainerState.load_from_json(os.path.join(resume_from_checkpoint, TRAINER_STATE_NAME))
            epochs_trained = self.state.global_step // num_update_steps_per_epoch
            if not args.ignore_data_skip:
                steps_trained_in_current_epoch = self.state.global_step % (num_update_steps_per_epoch)
                steps_trained_in_current_epoch *= args.gradient_accumulation_steps
            else:
                steps_trained_in_current_epoch = 0

            logger.info("  Continuing training from checkpoint, will skip to saved global_step")
            logger.info(f"  Continuing training from epoch {epochs_trained}")
            logger.info(f"  Continuing training from global step {self.state.global_step}")
            if not args.ignore_data_skip:
                logger.info(
                    f"  Will skip the first {epochs_trained} epochs then the first {steps_trained_in_current_epoch} "
                    "batches in the first epoch. If this takes a lot of time, you can add the `--ignore_data_skip` "
                    "flag to your launch command, but you will resume the training on data already seen by your model."
                )
                if self.is_local_process_zero() and not args.disable_tqdm:
                    steps_trained_progress_bar = tqdm(total=steps_trained_in_current_epoch)
                    steps_trained_progress_bar.set_description("Skipping the first batches")

        # Update the references
        self.callback_handler.model = self.model
        self.callback_handler.optimizer = self.optimizer
        self.callback_handler.lr_scheduler = self.lr_scheduler
        self.callback_handler.train_dataloader = train_dataloader
        if self.hp_name is not None and self._trial is not None:
            # use self._trial because the SigOpt/Optuna hpo only call `_hp_search_setup(trial)` instead of passing trial
            # parameter to Train when using DDP.
            self.state.trial_name = self.hp_name(self._trial)
        if trial is not None:
            assignments = trial.assignments if self.hp_search_backend == HPSearchBackend.SIGOPT else trial
            self.state.trial_params = hp_params(assignments)
        else:
            self.state.trial_params = None
        # This should be the same if the state has been saved but in case the training arguments changed, it's safer
        # to set this after the load.
        self.state.max_steps = max_steps
        self.state.num_train_epochs = num_train_epochs
        self.state.is_local_process_zero = self.is_local_process_zero()
        self.state.is_world_process_zero = self.is_world_process_zero()

        # tr_loss is a tensor to avoid synchronization of TPUs through .item()
        tr_loss = torch.tensor(0.0).to(args.device)
        # _total_loss_scalar is updated everytime .item() has to be called on tr_loss and stores the sum of all losses
        self._total_loss_scalar = 0.0
        self._globalstep_last_logged = self.state.global_step
        model.zero_grad()

        self.control = self.callback_handler.on_train_begin(args, self.state, self.control)

        # Skip the first epochs_trained epochs to get the random state of the dataloader at the right point.
        if not args.ignore_data_skip:
            for epoch in range(epochs_trained):
                is_random_sampler = hasattr(train_dataloader, "sampler") and isinstance(
                    train_dataloader.sampler, RandomSampler
                )
                if is_torch_less_than_1_11 or not is_random_sampler:
                    # We just need to begin an iteration to create the randomization of the sampler.
                    # That was before PyTorch 1.11 however...
                    for _ in train_dataloader:
                        break
                else:
                    # Otherwise we need to call the whooooole sampler cause there is some random operation added
                    # AT THE VERY END!
                    _ = list(train_dataloader.sampler)

        # ------------------------- Federated added -------------------------
        # 为 Federated 创建对应的 dataloaders
        self.is_parallel = isinstance(model, torch.nn.DataParallel) or isinstance(model, torch.nn.parallel.DistributedDataParallel) # 检查模型是否被 DataParallel 包装
        train_dataloaders = self.split_dataloader(train_dataloader, fed_num) # 将原始DataLoader均分为fed_num份
        # What parameters to optimize 
        self.named_parameters_to_optim = []
        for name, param in model.named_parameters():
            if param.requires_grad:
                self.named_parameters_to_optim.append((name, param))
        # state_dicts for all models
        # model_state_dicts = []
        # optimizer_state_dicts = []
        # lr_scheduler_state_dicts = []
        # for i in range(fed_num):
        #     model_state_dicts.append(copy.deepcopy(model.module.state_dict() if self.is_parallel else model.state_dict())) # 获取模型参数
        #     optimizer_state_dicts.append(copy.deepcopy(self.optimizer.state_dict())) # 获取优化器参数
        #     lr_scheduler_state_dicts.append(copy.deepcopy(self.lr_scheduler.state_dict())) # 获取学习率参数
        models = []
        optimizers = []
        lr_schedulers = []
        for i in range(fed_num):
            model_state_dicts.append(copy.deepcopy(model.module.state_dict() if self.is_parallel else model.state_dict())) # 获取模型参数
            optimizer_state_dicts.append(copy.deepcopy(self.optimizer.state_dict())) # 获取优化器参数
            lr_scheduler_state_dicts.append(copy.deepcopy(self.lr_scheduler.state_dict())) # 获取学习率参数
        # ------------------------- 开始训练 -------------------------
        for epoch in range(epochs_trained, num_train_epochs):
            tr_losses = []
            # training for federated learning
            if self.args.aggregation_freq == "batch": # 逐batch聚合
                raise NotImplementedError()
                if not self.is_batch_nums_equal(train_dataloaders):
                    raise ValueError("All train_dataloaders must have the same number of batches.")
                # 训练前的配置
                epoch_iterators = []
                for i in range(fed_num):
                    train_dataloader = train_dataloaders[i]
                    
                    if isinstance(train_dataloader, DataLoader) and isinstance(train_dataloader.sampler, DistributedSampler):
                        train_dataloader.sampler.set_epoch(epoch)
                    elif hasattr(train_dataloader, "dataset") and isinstance(train_dataloader.dataset, IterableDatasetShard):
                        train_dataloader.dataset.set_epoch(epoch)

                    if is_torch_tpu_available():
                        parallel_loader = pl.ParallelLoader(train_dataloader, [args.device]).per_device_loader(args.device)
                        epoch_iterator = parallel_loader
                    else:
                        epoch_iterator = train_dataloader
                    
                    epoch_iterators.append(epoch_iterator)

                    # Reset the past mems state at the beginning of each epoch if necessary.
                    if args.past_index >= 0:
                        self._past = None

                    self.control = self.callback_handler.on_epoch_begin(args, self.state, self.control)
                if epoch == epochs_trained and resume_from_checkpoint is not None and steps_trained_in_current_epoch == 0:
                    self._load_rng_state(resume_from_checkpoint)
                # 遍历每个batch
                for step, epoch_inputs in enumerate(zip(*epoch_iterators)):
                    # Skip past any already trained steps if resuming training
                    if steps_trained_in_current_epoch > 0:
                        steps_trained_in_current_epoch -= 1
                        if steps_trained_progress_bar is not None:
                            steps_trained_progress_bar.update(1)
                        if steps_trained_in_current_epoch == 0:
                            self._load_rng_state(resume_from_checkpoint)
                        continue
                    elif steps_trained_progress_bar is not None:
                        steps_trained_progress_bar.close()
                        steps_trained_progress_bar = None

                    if step % args.gradient_accumulation_steps == 0:
                        self.control = self.callback_handler.on_step_begin(args, self.state, self.control)
                    
                    # 逐batches训练
                    for i in range(fed_num):
                        inputs = epoch_inputs[i]
                        epoch_iterator = epoch_iterators[i]
                        steps_in_epoch = (
                            len(epoch_iterator)
                            if len_dataloader is not None
                            else args.max_steps * args.gradient_accumulation_steps
                        )
                        # -------------- 初始化模型参数 --------------
                        if self.is_parallel: # 获取模型参数
                            model.module.load_state_dict(model_state_dicts[i])
                        else:
                            model.load_state_dict(model_state_dicts[i])
                        self.optimizer.load_state_dict(optimizer_state_dicts[i]) # 获取优化器参数
                        self.lr_scheduler.load_state_dict(lr_scheduler_state_dicts[i]) # 获取学习率参数
                        # -------------- 训练模型 --------------
                        model, tr_loss = self.train_for_single_model_batch(args, model, epoch, step, inputs, tr_loss, steps_in_epoch)
                        # -------------- 保存模型参数 --------------
                        model_state_dicts[i] = copy.deepcopy(model.state_dict())
                        torch.cuda.empty_cache()
                    # aggregation
                    global_state_dict = self.federated_averaging(model_state_dicts)
                    del model_state_dicts
                    torch.cuda.empty_cache()
                    model_state_dicts = [copy.deepcopy(global_state_dict) for _ in range(fed_num)]
                    del global_state_dict
                    torch.cuda.empty_cache()
                tr_losses.append(tr_loss)
                
                # 后处理
                self.state.epoch = epoch + 1
                
                self.control = self.callback_handler.on_epoch_end(args, self.state, self.control)
                
                if (epoch + 1) % self.args.eval_epoch == 0:
                    should_evaluate = self.control.should_evaluate
                    should_save = self.control.should_save
                    self.control.should_evaluate = True
                    self.control.should_save = False
                    self._maybe_log_save_evaluate(tr_loss, model, trial, epoch, ignore_keys_for_eval)
                    self.control.should_evaluate = should_evaluate
                    self.control.should_save = should_save
                
                # 本model的本epoch是否训练结束
                if self.control.should_epoch_stop or self.control.should_training_stop:
                    break
                if step < 0:
                    logger.warning(
                        "There seems to be not a single sample in your epoch_iterator, stopping training at step"
                        f" {self.state.global_step}! This is expected if you're using an IterableDataset and set"
                        f" num_steps ({max_steps}) higher than the number of available samples."
                    )
                    self.control.should_training_stop = True
                if DebugOption.TPU_METRICS_DEBUG in self.args.debug:
                    if is_torch_tpu_available():
                        # tpu-comment: Logging debug metrics for PyTorch/XLA (compile, execute times, ops, etc.)
                        xm.master_print(met.metrics_report())
                    else:
                        logger.warning(
                            "You enabled PyTorch/XLA debug metrics but you don't have a TPU "
                            "configured. Check your training configuration if this is unexpected."
                        )
            elif re.search(r'(\d+)batch', self.args.aggregation_freq): # TODO: 形如 "[L]batch" 的聚合频率
                L = int(re.search(r'(\d+)batch', self.args.aggregation_freq).group(1))
                if not self.is_batch_nums_equal(train_dataloaders):
                    raise ValueError("All train_dataloaders must have the same number of batches.")
                # 训练前的配置
                epoch_iterators = []
                for i in range(fed_num):
                    train_dataloader = train_dataloaders[i]
                    
                    if isinstance(train_dataloader, DataLoader) and isinstance(train_dataloader.sampler, DistributedSampler):
                        train_dataloader.sampler.set_epoch(epoch)
                    elif hasattr(train_dataloader, "dataset") and isinstance(train_dataloader.dataset, IterableDatasetShard):
                        train_dataloader.dataset.set_epoch(epoch)

                    if is_torch_tpu_available():
                        parallel_loader = pl.ParallelLoader(train_dataloader, [args.device]).per_device_loader(args.device)
                        epoch_iterator = parallel_loader
                    else:
                        epoch_iterator = train_dataloader
                    
                    epoch_iterators.append(epoch_iterator)

                # Reset the past mems state at the beginning of each epoch if necessary.
                if args.past_index >= 0:
                    self._past = None

                self.control = self.callback_handler.on_epoch_begin(args, self.state, self.control)
                if epoch == epochs_trained and resume_from_checkpoint is not None and steps_trained_in_current_epoch == 0:
                    self._load_rng_state(resume_from_checkpoint)
                # 遍历每个batch
                aggre_ok_flag = True # 是否还存在step没被聚合
                for step, epoch_inputs in enumerate(zip(*epoch_iterators)):
                    # Skip past any already trained steps if resuming training
                    if steps_trained_in_current_epoch > 0:
                        steps_trained_in_current_epoch -= 1
                        if steps_trained_progress_bar is not None:
                            steps_trained_progress_bar.update(1)
                        if steps_trained_in_current_epoch == 0:
                            self._load_rng_state(resume_from_checkpoint)
                        continue
                    elif steps_trained_progress_bar is not None:
                        steps_trained_progress_bar.close()
                        steps_trained_progress_bar = None

                    if step % args.gradient_accumulation_steps == 0:
                        self.control = self.callback_handler.on_step_begin(args, self.state, self.control)
                    
                    # 逐batches训练
                    for i in range(fed_num):
                        inputs = epoch_inputs[i]
                        epoch_iterator = epoch_iterators[i]
                        steps_in_epoch = (
                            len(epoch_iterator)
                            if len_dataloader is not None
                            else args.max_steps * args.gradient_accumulation_steps
                        )
                        # -------------- 初始化模型参数 --------------
                        if self.is_parallel: # 获取模型参数
                            model.module.load_state_dict(model_state_dicts[i])
                        else:
                            model.load_state_dict(model_state_dicts[i])
                        self.optimizer.load_state_dict(optimizer_state_dicts[i]) # 获取优化器参数
                        self.lr_scheduler.load_state_dict(lr_scheduler_state_dicts[i]) # 获取学习率参数
                        # -------------- 训练模型 --------------
                        model, tr_loss = self.train_for_single_model_batch(args, model, epoch, step, inputs, tr_loss, steps_in_epoch)
                        # -------------- 保存模型参数 --------------
                        model_state_dicts[i] = copy.deepcopy(model.state_dict())
                        torch.cuda.empty_cache()
                    # aggregation
                    if (step + 1) % L == 0:
                        global_state_dict = self.federated_averaging(model_state_dicts)
                        del model_state_dicts
                        torch.cuda.empty_cache()
                        model_state_dicts = [copy.deepcopy(global_state_dict) for _ in range(fed_num)]
                        del global_state_dict
                        torch.cuda.empty_cache()
                        aggre_ok_flag = True
                    else:
                        aggre_ok_flag = False
                # 遍历完所有的batches，为了避免还有未聚合的batches，再次聚合
                if not aggre_ok_flag:
                    global_state_dict = self.federated_averaging(model_state_dicts)
                    del model_state_dicts
                    torch.cuda.empty_cache()
                    model_state_dicts = [copy.deepcopy(global_state_dict) for _ in range(fed_num)]
                    del global_state_dict
                    torch.cuda.empty_cache()
                
                tr_losses.append(tr_loss)
                
                # 后处理
                self.state.epoch = epoch + 1
                
                self.control = self.callback_handler.on_epoch_end(args, self.state, self.control)
                
                if (epoch + 1) % self.args.eval_epoch == 0:
                    should_evaluate = self.control.should_evaluate
                    should_save = self.control.should_save
                    self.control.should_evaluate = True
                    self.control.should_save = False
                    self._maybe_log_save_evaluate(tr_loss, model, trial, epoch, ignore_keys_for_eval)
                    self.control.should_evaluate = should_evaluate
                    self.control.should_save = should_save
                
                # 本model的本epoch是否训练结束
                if self.control.should_epoch_stop or self.control.should_training_stop:
                    break
                if step < 0:
                    logger.warning(
                        "There seems to be not a single sample in your epoch_iterator, stopping training at step"
                        f" {self.state.global_step}! This is expected if you're using an IterableDataset and set"
                        f" num_steps ({max_steps}) higher than the number of available samples."
                    )
                    self.control.should_training_stop = True
                if DebugOption.TPU_METRICS_DEBUG in self.args.debug:
                    if is_torch_tpu_available():
                        # tpu-comment: Logging debug metrics for PyTorch/XLA (compile, execute times, ops, etc.)
                        xm.master_print(met.metrics_report())
                    else:
                        logger.warning(
                            "You enabled PyTorch/XLA debug metrics but you don't have a TPU "
                            "configured. Check your training configuration if this is unexpected."
                        )
            elif self.args.aggregation_freq == "epoch": # TODO: 逐epoch聚合
                for i in range(fed_num):
                    # -------------- 初始化模型参数 --------------
                    if self.is_parallel: # 获取模型参数
                        model.module.load_state_dict(model_state_dicts[i])
                    else:
                        model.load_state_dict(model_state_dicts[i])
                    self.optimizer.load_state_dict(optimizer_state_dicts[i]) # 获取优化器参数
                    self.lr_scheduler.load_state_dict(lr_scheduler_state_dicts[i]) # 获取学习率参数
                    # -------------- 训练模型 --------------
                    model, tr_loss = self.train_for_single_model_epoch(
                        args, model, train_dataloaders[i], epoch, epochs_trained, 
                        tr_loss, steps_trained_in_current_epoch, len_dataloader,
                        resume_from_checkpoint, steps_trained_progress_bar)
                    # -------------- 保存模型参数 --------------
                    model_state_dicts[i] = copy.deepcopy(model.state_dict())
                    torch.cuda.empty_cache()
                tr_losses.append(tr_loss)
                
                # federated aggregation
                global_state_dict = self.federated_averaging(model_state_dicts)
                del model_state_dicts
                torch.cuda.empty_cache()
                model_state_dicts = [copy.deepcopy(global_state_dict) for _ in range(fed_num)]
                del global_state_dict
                torch.cuda.empty_cache()
                
                # 后处理
                self.state.epoch = epoch + 1
                
                self.control = self.callback_handler.on_epoch_end(args, self.state, self.control)
                
                if (epoch + 1) % self.args.eval_epoch == 0:
                    should_evaluate = self.control.should_evaluate
                    should_save = self.control.should_save
                    self.control.should_evaluate = True
                    self.control.should_save = False
                    self._maybe_log_save_evaluate(tr_loss, model, trial, epoch, ignore_keys_for_eval)
                    self.control.should_evaluate = should_evaluate
                    self.control.should_save = should_save
                
                if self.control.should_training_stop:
                    break
                
                if DebugOption.TPU_METRICS_DEBUG in self.args.debug:
                    if is_torch_tpu_available():
                        # tpu-comment: Logging debug metrics for PyTorch/XLA (compile, execute times, ops, etc.)
                        xm.master_print(met.metrics_report())
                    else:
                        logger.warning(
                            "You enabled PyTorch/XLA debug metrics but you don't have a TPU "
                            "configured. Check your training configuration if this is unexpected."
                        )
            else:
                raise ValueError("aggregation_freq error.")

        # ------------------------- 训练结束 -------------------------
        if args.past_index and hasattr(self, "_past"):
            # Clean the state at the end of training
            delattr(self, "_past")

        logger.info("\n\nTraining completed. Do not forget to share your model on huggingface.co/models =)\n\n")
        if args.load_best_model_at_end and self.state.best_model_checkpoint is not None:
            # Wait for everyone to get here so we are sur the model has been saved by process 0.
            if is_torch_tpu_available():
                xm.rendezvous("load_best_model_at_end")
            elif args.local_rank != -1:
                dist.barrier()
            elif is_sagemaker_mp_enabled():
                smp.barrier()

            self._load_best_model()

        # add remaining tr_loss
        self._total_loss_scalar += tr_loss.item()
        train_loss = self._total_loss_scalar / self.state.global_step
        # 计算loss
        # train_loss = torch.mean(torch.stack(tr_losses)).item()
        # ------------------------- End of Federated added -------------------------

        metrics = speed_metrics("train", start_time, num_samples=num_train_samples, num_steps=self.state.max_steps)
        self.store_flos()
        metrics["total_flos"] = self.state.total_flos
        metrics["train_loss"] = train_loss

        self.is_in_train = False

        self._memory_tracker.stop_and_update_metrics(metrics)

        self.log(metrics)

        run_dir = self._get_output_dir(trial)
        checkpoints_sorted = self._sorted_checkpoints(use_mtime=False, output_dir=run_dir)

        # Delete the last checkpoint when save_total_limit=1 if it's different from the best checkpoint.
        if self.state.best_model_checkpoint is not None and self.args.save_total_limit == 1:
            for checkpoint in checkpoints_sorted:
                if checkpoint != self.state.best_model_checkpoint:
                    logger.info(f"Deleting older checkpoint [{checkpoint}] due to args.save_total_limit")
                    shutil.rmtree(checkpoint)

        self.control = self.callback_handler.on_train_end(args, self.state, self.control)

        return TrainOutput(self.state.global_step, train_loss, metrics)

    ############## MeZO ##############
    def _inner_training_loop_MeZO(
        self, batch_size=None, args=None, resume_from_checkpoint=None, trial=None, ignore_keys_for_eval=None
    ):
        """
        We overload the original training loop to add linear probing and MeZO. Search key word "MeZO added"
        for those updates.
        """
        
        # 创建一个文件处理器，设置日志文件路径
        file_handler = py_logging.FileHandler(os.path.join(self.args.output_dir, self.args.log_file))
        file_handler.setLevel(py_logging.INFO)
        # 将处理器添加到 logger
        logger.addHandler(file_handler)
        
        self._train_batch_size = batch_size
        # Data loader and number of training steps
        train_dataloader = self.get_train_dataloader()

        # MeZO added: Linear probing
        if self.args.linear_probing:

            def _get_token_prediction_layer(model):
                if model.config.model_type == "opt":
                    return model.lm_head
                else:
                    raise NotImplementedError(model.config.model_type)

            def _extract_features(model, *args, **kwargs):
                """some magic for getting features pre last layer"""
                features = {}
                def __hook(model_, input_, output_):
                    features["features"] = input_[0].detach()

                _get_token_prediction_layer(model).register_forward_hook(__hook)
                model.forward(*args, **kwargs)
                return features["features"]

            logger.info("Linear probing")
            logger.info("Starting to get features for training dataset")
            targets = []
            features = []
            with torch.inference_mode():
                for step, inputs in enumerate(tqdm(train_dataloader)):
                    for k, v in inputs.items():
                        if isinstance(v, torch.Tensor):
                            inputs[k] = v.to(self.model.device)
                        
                    feature = _extract_features(self.model, **inputs)
                    target = inputs["labels"]

                    # Shift the target (bc it's autoregressive LM) and add the corresponding part
                    assert not self.args.train_as_classification and self.args.only_train_option
                    feature, target = feature[:, :-1], target[:, 1:]
                    for _i, _len in enumerate(inputs["option_len"]):
                        features.append(feature[_i, -_len:])
                        targets.append(target[_i, -_len:])

            logger.info("Finished getting features for training dataset")

            features = torch.cat(features, dim=0).cpu().numpy()
            targets = torch.cat(targets, dim=0).cpu().numpy()
            # Whether to use bias
            if self.model.config.model_type in ["opt", "gpt2"]:
                use_bias = False
            else:
                raise NotImplementedError
            # Set early stopping
            tol = 0.01 if self.args.lp_early_stopping else 1e-4 # 1e-4 is scipy default
            max_iter = 1000 if self.args.lp_early_stopping else 5000

            logger.info("Fitting logistic regression...")
            reg = LogisticRegressionCV(max_iter=max_iter, fit_intercept=use_bias, multi_class="multinomial", random_state=0, tol=tol, n_jobs=-1).fit(features, targets)
            logger.info("Done")

            logger.info("Assigning weights to model")
            decoder = _get_token_prediction_layer(self.model)
            coef_torch = torch.tensor(reg.coef_, device=decoder.weight.device, dtype=decoder.weight.dtype)
            if use_bias:
                bias_torch = torch.tensor(reg.intercept_, device=decoder.weight.device, dtype=decoder.weight.dtype)
            if coef_torch.shape[0] == 1: # The regressor only detects two classes
                assert len(reg.classes_) == 2
                coef_torch = torch.cat([-coef_torch / 2, coef_torch / 2], dim=0)
                if use_bias:
                    bias_torch = torch.cat([-bias_torch / 2, bias_torch / 2], dim=0)

            for _i, token_id in enumerate(reg.classes_):
                decoder.weight.data[token_id] = coef_torch[_i]
                if use_bias:
                    decoder.bias.data[token_id] = bias_torch[_i]

            return None

        # Setting up training control variables:
        # number of training epochs: num_train_epochs
        # number of training steps per epoch: num_update_steps_per_epoch
        # total number of training steps to execute: max_steps
        total_train_batch_size = args.train_batch_size * args.gradient_accumulation_steps * args.world_size

        len_dataloader = None
        if has_length(train_dataloader):
            len_dataloader = len(train_dataloader)
            num_update_steps_per_epoch = len_dataloader // args.gradient_accumulation_steps
            num_update_steps_per_epoch = max(num_update_steps_per_epoch, 1)
            num_examples = self.num_examples(train_dataloader)
            if args.max_steps > 0:
                max_steps = args.max_steps
                num_train_epochs = args.max_steps // num_update_steps_per_epoch + int(
                    args.max_steps % num_update_steps_per_epoch > 0
                )
                # May be slightly incorrect if the last batch in the training dataloader has a smaller size but it's
                # the best we can do.
                num_train_samples = args.max_steps * total_train_batch_size
            else:
                max_steps = math.ceil(args.num_train_epochs * num_update_steps_per_epoch)
                num_train_epochs = math.ceil(args.num_train_epochs)
                num_train_samples = self.num_examples(train_dataloader) * args.num_train_epochs
        elif args.max_steps > 0:  # Rely on max_steps when dataloader does not have a working size
            max_steps = args.max_steps
            # Setting a very large number of epochs so we go as many times as necessary over the iterator.
            num_train_epochs = sys.maxsize
            num_update_steps_per_epoch = max_steps
            num_examples = total_train_batch_size * args.max_steps
            num_train_samples = args.max_steps * total_train_batch_size
        else:
            raise ValueError(
                "args.max_steps must be set to a positive value if dataloader does not have a length, was"
                f" {args.max_steps}"
            )

        if DebugOption.UNDERFLOW_OVERFLOW in self.args.debug:
            if self.args.n_gpu > 1:
                # nn.DataParallel(model) replicates the model, creating new variables and module
                # references registered here no longer work on other gpus, breaking the module
                raise ValueError(
                    "Currently --debug underflow_overflow is not supported under DP. Please use DDP"
                    " (torch.distributed.launch)."
                )
            else:
                debug_overflow = DebugUnderflowOverflow(self.model)  # noqa

        delay_optimizer_creation = (
            self.sharded_ddp is not None
            and self.sharded_ddp != ShardedDDPOption.SIMPLE
            or is_sagemaker_mp_enabled()
            or self.fsdp is not None
        )
        if args.deepspeed:
            deepspeed_engine, optimizer, lr_scheduler = deepspeed_init(
                self, num_training_steps=max_steps, resume_from_checkpoint=resume_from_checkpoint
            )
            self.model = deepspeed_engine.module
            self.model_wrapped = deepspeed_engine
            self.deepspeed = deepspeed_engine
            self.optimizer = optimizer
            self.lr_scheduler = lr_scheduler
        elif not delay_optimizer_creation:
            self.create_optimizer_and_scheduler(num_training_steps=max_steps)

        self.state = TrainerState()
        self.state.is_hyper_param_search = trial is not None

        # Activate gradient checkpointing if needed
        if args.gradient_checkpointing:
            self.model.gradient_checkpointing_enable()

        model = self._wrap_model(self.model_wrapped)

        if is_sagemaker_mp_enabled() and resume_from_checkpoint is not None:
            self._load_from_checkpoint(resume_from_checkpoint, model)

        # for the rest of this function `model` is the outside model, whether it was wrapped or not
        if model is not self.model:
            self.model_wrapped = model

        if delay_optimizer_creation:
            self.create_optimizer_and_scheduler(num_training_steps=max_steps)

        # Check if saved optimizer or scheduler states exist
        self._load_optimizer_and_scheduler(resume_from_checkpoint)

        # important: at this point:
        # self.model         is the Transformers Model
        # self.model_wrapped is DDP(Transformers Model), Deepspeed(Transformers Model), etc.

        # Train!
        logger.info("***** Running training *****")
        logger.info(f"  Num examples = {num_examples}")
        logger.info(f"  Num Epochs = {num_train_epochs}")
        logger.info(f"  Instantaneous batch size per device = {args.per_device_train_batch_size}")
        logger.info(f"  Total train batch size (w. parallel, distributed & accumulation) = {total_train_batch_size}")
        logger.info(f"  Gradient Accumulation steps = {args.gradient_accumulation_steps}")
        logger.info(f"  Total optimization steps = {max_steps}")
        logger.info(
            f"  Number of trainable parameters = {sum(p.numel() for p in model.parameters() if p.requires_grad)}"
        )

        self.state.epoch = 0
        start_time = time.time()
        epochs_trained = 0
        steps_trained_in_current_epoch = 0
        steps_trained_progress_bar = None

        # Check if continuing training from a checkpoint
        if resume_from_checkpoint is not None and os.path.isfile(
            os.path.join(resume_from_checkpoint, TRAINER_STATE_NAME)
        ):
            self.state = TrainerState.load_from_json(os.path.join(resume_from_checkpoint, TRAINER_STATE_NAME))
            epochs_trained = self.state.global_step // num_update_steps_per_epoch
            if not args.ignore_data_skip:
                steps_trained_in_current_epoch = self.state.global_step % (num_update_steps_per_epoch)
                steps_trained_in_current_epoch *= args.gradient_accumulation_steps
            else:
                steps_trained_in_current_epoch = 0

            logger.info("  Continuing training from checkpoint, will skip to saved global_step")
            logger.info(f"  Continuing training from epoch {epochs_trained}")
            logger.info(f"  Continuing training from global step {self.state.global_step}")
            if not args.ignore_data_skip:
                logger.info(
                    f"  Will skip the first {epochs_trained} epochs then the first {steps_trained_in_current_epoch} "
                    "batches in the first epoch. If this takes a lot of time, you can add the `--ignore_data_skip` "
                    "flag to your launch command, but you will resume the training on data already seen by your model."
                )
                if self.is_local_process_zero() and not args.disable_tqdm:
                    steps_trained_progress_bar = tqdm(total=steps_trained_in_current_epoch)
                    steps_trained_progress_bar.set_description("Skipping the first batches")

        # Update the references
        self.callback_handler.model = self.model
        self.callback_handler.optimizer = self.optimizer
        self.callback_handler.lr_scheduler = self.lr_scheduler
        self.callback_handler.train_dataloader = train_dataloader
        if self.hp_name is not None and self._trial is not None:
            # use self._trial because the SigOpt/Optuna hpo only call `_hp_search_setup(trial)` instead of passing trial
            # parameter to Train when using DDP.
            self.state.trial_name = self.hp_name(self._trial)
        if trial is not None:
            assignments = trial.assignments if self.hp_search_backend == HPSearchBackend.SIGOPT else trial
            self.state.trial_params = hp_params(assignments)
        else:
            self.state.trial_params = None
        # This should be the same if the state has been saved but in case the training arguments changed, it's safer
        # to set this after the load.
        self.state.max_steps = max_steps
        self.state.num_train_epochs = num_train_epochs
        self.state.is_local_process_zero = self.is_local_process_zero()
        self.state.is_world_process_zero = self.is_world_process_zero()

        # tr_loss is a tensor to avoid synchronization of TPUs through .item()
        tr_loss = torch.tensor(0.0).to(args.device)
        # _total_loss_scalar is updated everytime .item() has to be called on tr_loss and stores the sum of all losses
        self._total_loss_scalar = 0.0
        self._globalstep_last_logged = self.state.global_step
        model.zero_grad()

        self.control = self.callback_handler.on_train_begin(args, self.state, self.control)

        # Skip the first epochs_trained epochs to get the random state of the dataloader at the right point.
        if not args.ignore_data_skip:
            for epoch in range(epochs_trained):
                is_random_sampler = hasattr(train_dataloader, "sampler") and isinstance(
                    train_dataloader.sampler, RandomSampler
                )
                if is_torch_less_than_1_11 or not is_random_sampler:
                    # We just need to begin an iteration to create the randomization of the sampler.
                    # That was before PyTorch 1.11 however...
                    for _ in train_dataloader:
                        break
                else:
                    # Otherwise we need to call the whooooole sampler cause there is some random operation added
                    # AT THE VERY END!
                    _ = list(train_dataloader.sampler)

        for epoch in range(epochs_trained, num_train_epochs):
            if isinstance(train_dataloader, DataLoader) and isinstance(train_dataloader.sampler, DistributedSampler):
                train_dataloader.sampler.set_epoch(epoch)
            elif hasattr(train_dataloader, "dataset") and isinstance(train_dataloader.dataset, IterableDatasetShard):
                train_dataloader.dataset.set_epoch(epoch)

            if is_torch_tpu_available():
                parallel_loader = pl.ParallelLoader(train_dataloader, [args.device]).per_device_loader(args.device)
                epoch_iterator = parallel_loader
            else:
                epoch_iterator = train_dataloader

            # Reset the past mems state at the beginning of each epoch if necessary.
            if args.past_index >= 0:
                self._past = None

            steps_in_epoch = (
                len(epoch_iterator)
                if len_dataloader is not None
                else args.max_steps * args.gradient_accumulation_steps
            )
            self.control = self.callback_handler.on_epoch_begin(args, self.state, self.control)

            if epoch == epochs_trained and resume_from_checkpoint is not None and steps_trained_in_current_epoch == 0:
                self._load_rng_state(resume_from_checkpoint)

            step = -1
            for step, inputs in enumerate(epoch_iterator):

                # Skip past any already trained steps if resuming training
                if steps_trained_in_current_epoch > 0:
                    steps_trained_in_current_epoch -= 1
                    if steps_trained_progress_bar is not None:
                        steps_trained_progress_bar.update(1)
                    if steps_trained_in_current_epoch == 0:
                        self._load_rng_state(resume_from_checkpoint)
                    continue
                elif steps_trained_progress_bar is not None:
                    steps_trained_progress_bar.close()
                    steps_trained_progress_bar = None

                if step % args.gradient_accumulation_steps == 0:
                    self.control = self.callback_handler.on_step_begin(args, self.state, self.control)

                # MeZO added: estimate gradient
                if args.trainer == "zo":
                    tr_loss_step = self.zo_step(model, inputs)
                else:
                    if (
                        ((step + 1) % args.gradient_accumulation_steps != 0)
                        and args.local_rank != -1
                        and args._no_sync_in_gradient_accumulation
                    ):
                        # Avoid unnecessary DDP synchronization since there will be no backward pass on this example.
                        with model.no_sync():
                            tr_loss_step = self.training_step(model, inputs)
                    else:
                        tr_loss_step = self.training_step(model, inputs)

                if (
                    args.logging_nan_inf_filter
                    and not is_torch_tpu_available()
                    and (torch.isnan(tr_loss_step) or torch.isinf(tr_loss_step))
                ):
                    # if loss is nan or inf simply add the average of previous logged losses
                    tr_loss += tr_loss / (1 + self.state.global_step - self._globalstep_last_logged)
                else:
                    tr_loss += tr_loss_step

                self.current_flos += float(self.floating_point_ops(inputs))

                # Optimizer step for deepspeed must be called on every step regardless of the value of gradient_accumulation_steps
                if self.deepspeed:
                    self.deepspeed.step()


                if (step + 1) % args.gradient_accumulation_steps == 0 or (
                    # last step in epoch but step is always smaller than gradient_accumulation_steps
                    steps_in_epoch <= args.gradient_accumulation_steps
                    and (step + 1) == steps_in_epoch
                ):
                    # MeZO added: update model with the estimated gradient
                    if args.trainer == "zo":
                        self.zo_update(model)
                    else:
                        # Gradient clipping
                        if args.max_grad_norm is not None and args.max_grad_norm > 0 and not self.deepspeed:
                            # deepspeed does its own clipping

                            if self.do_grad_scaling:
                                # Reduce gradients first for XLA
                                if is_torch_tpu_available():
                                    gradients = xm._fetch_gradients(self.optimizer)
                                    xm.all_reduce("sum", gradients, scale=1.0 / xm.xrt_world_size())
                                # AMP: gradients need unscaling
                                self.scaler.unscale_(self.optimizer)

                            if is_sagemaker_mp_enabled() and args.fp16:
                                self.optimizer.clip_master_grads(args.max_grad_norm)
                            elif hasattr(self.optimizer, "clip_grad_norm"):
                                # Some optimizers (like the sharded optimizer) have a specific way to do gradient clipping
                                self.optimizer.clip_grad_norm(args.max_grad_norm)
                            elif hasattr(model, "clip_grad_norm_"):
                                # Some models (like FullyShardedDDP) have a specific way to do gradient clipping
                                model.clip_grad_norm_(args.max_grad_norm)
                            else:
                                # Revert to normal clipping otherwise, handling Apex or full precision
                                nn.utils.clip_grad_norm_(
                                    amp.master_params(self.optimizer) if self.use_apex else model.parameters(),
                                    args.max_grad_norm,
                                )

                        # Optimizer step
                        optimizer_was_run = True
                        if self.deepspeed:
                            pass  # called outside the loop
                        elif is_torch_tpu_available():
                            if self.do_grad_scaling:
                                self.scaler.step(self.optimizer)
                                self.scaler.update()
                            else:
                                xm.optimizer_step(self.optimizer)
                        elif self.do_grad_scaling:
                            scale_before = self.scaler.get_scale()
                            self.scaler.step(self.optimizer)
                            self.scaler.update()
                            scale_after = self.scaler.get_scale()
                            optimizer_was_run = scale_before <= scale_after
                        else:
                            self.optimizer.step()

                        if optimizer_was_run and not self.deepspeed:
                            self.lr_scheduler.step()
                        model.zero_grad()

                    self.state.global_step += 1
                    self.state.epoch = epoch + (step + 1) / steps_in_epoch
                    self.control = self.callback_handler.on_step_end(args, self.state, self.control)

                    self._maybe_log_save_evaluate(tr_loss, model, trial, epoch, ignore_keys_for_eval)
                else:
                    self.control = self.callback_handler.on_substep_end(args, self.state, self.control)

                if self.control.should_epoch_stop or self.control.should_training_stop:
                    break
            if step < 0:
                logger.warning(
                    "There seems to be not a single sample in your epoch_iterator, stopping training at step"
                    f" {self.state.global_step}! This is expected if you're using an IterableDataset and set"
                    f" num_steps ({max_steps}) higher than the number of available samples."
                )
                self.control.should_training_stop = True

            self.control = self.callback_handler.on_epoch_end(args, self.state, self.control)
            self._maybe_log_save_evaluate(tr_loss, model, trial, epoch, ignore_keys_for_eval)

            if DebugOption.TPU_METRICS_DEBUG in self.args.debug:
                if is_torch_tpu_available():
                    # tpu-comment: Logging debug metrics for PyTorch/XLA (compile, execute times, ops, etc.)
                    xm.master_print(met.metrics_report())
                else:
                    logger.warning(
                        "You enabled PyTorch/XLA debug metrics but you don't have a TPU "
                        "configured. Check your training configuration if this is unexpected."
                    )
            if self.control.should_training_stop:
                break

        if args.past_index and hasattr(self, "_past"):
            # Clean the state at the end of training
            delattr(self, "_past")

        logger.info("\n\nTraining completed. Do not forget to share your model on huggingface.co/models =)\n\n")
        if args.load_best_model_at_end and self.state.best_model_checkpoint is not None:
            # Wait for everyone to get here so we are sur the model has been saved by process 0.
            if is_torch_tpu_available():
                xm.rendezvous("load_best_model_at_end")
            elif args.local_rank != -1:
                dist.barrier()
            elif is_sagemaker_mp_enabled():
                smp.barrier()

            self._load_best_model()

        # add remaining tr_loss
        self._total_loss_scalar += tr_loss.item()
        train_loss = self._total_loss_scalar / self.state.global_step

        metrics = speed_metrics("train", start_time, num_samples=num_train_samples, num_steps=self.state.max_steps)
        self.store_flos()
        metrics["total_flos"] = self.state.total_flos
        metrics["train_loss"] = train_loss

        self.is_in_train = False

        self._memory_tracker.stop_and_update_metrics(metrics)

        self.log(metrics)

        run_dir = self._get_output_dir(trial)
        checkpoints_sorted = self._sorted_checkpoints(use_mtime=False, output_dir=run_dir)

        # Delete the last checkpoint when save_total_limit=1 if it's different from the best checkpoint.
        if self.state.best_model_checkpoint is not None and self.args.save_total_limit == 1:
            for checkpoint in checkpoints_sorted:
                if checkpoint != self.state.best_model_checkpoint:
                    logger.info(f"Deleting older checkpoint [{checkpoint}] due to args.save_total_limit")
                    shutil.rmtree(checkpoint)

        self.control = self.callback_handler.on_train_end(args, self.state, self.control)

        return TrainOutput(self.state.global_step, train_loss, metrics)

    ############## federated ##############
    def create_optimizer_single(self, model):
        """
        Setup the optimizer.

        We provide a reasonable default that works well. If you want to use something else, you can pass a tuple in the
        Trainer's init through `optimizers`, or subclass and override this method in a subclass.
        """
        # opt_model = self.model_wrapped if is_sagemaker_mp_enabled() else self.model
        opt_model = model

        decay_parameters = get_parameter_names(opt_model, ALL_LAYERNORM_LAYERS)
        decay_parameters = [name for name in decay_parameters if "bias" not in name]
        optimizer_grouped_parameters = [
            {
                "params": [
                    p for n, p in opt_model.named_parameters() if (n in decay_parameters and p.requires_grad)
                ],
                "weight_decay": self.args.weight_decay,
            },
            {
                "params": [
                    p for n, p in opt_model.named_parameters() if (n not in decay_parameters and p.requires_grad)
                ],
                "weight_decay": 0.0,
            },
        ]

        optimizer_cls, optimizer_kwargs = Trainer.get_optimizer_cls_and_kwargs(self.args)

        if self.sharded_ddp == ShardedDDPOption.SIMPLE:
            optimizer = OSS(
                params=optimizer_grouped_parameters,
                optim=optimizer_cls,
                **optimizer_kwargs,
            )
        else:
            optimizer = optimizer_cls(optimizer_grouped_parameters, **optimizer_kwargs)
            if optimizer_cls.__name__ == "Adam8bit":
                import bitsandbytes

                manager = bitsandbytes.optim.GlobalOptimManager.get_instance()

                skipped = 0
                for module in opt_model.modules():
                    if isinstance(module, nn.Embedding):
                        skipped += sum({p.data_ptr(): p.numel() for p in module.parameters()}.values())
                        print(f"skipped {module}: {skipped/2**20}M params")
                        manager.register_module_override(module, "weight", {"optim_bits": 32})
                        logger.debug(f"bitsandbytes: will optimize {module} in fp32")
                print(f"skipped: {skipped/2**20}M params")

        if is_sagemaker_mp_enabled():
            optimizer = smp.DistributedOptimizer(optimizer)

        return optimizer

    def create_scheduler_single(self, num_training_steps: int, optimizer: torch.optim.Optimizer):
        """
        Setup the scheduler. The optimizer of the trainer must have been set up either before this method is called or
        passed as an argument.

        Args:
            num_training_steps (int): The number of training steps to do.
        """
        lr_scheduler = get_scheduler(
            self.args.lr_scheduler_type,
            # optimizer=optimizer if optimizer is None else optimizer,
            optimizer=optimizer,
            num_warmup_steps=self.args.get_warmup_steps(num_training_steps),
            num_training_steps=num_training_steps,
        )
        return lr_scheduler

    def create_optimizer_and_scheduler_single(self, num_training_steps: int, model):
        """
        Setup the optimizer and the learning rate scheduler.

        We provide a reasonable default that works well. If you want to use something else, you can pass a tuple in the
        Trainer's init through `optimizers`, or subclass and override this method (or `create_optimizer` and/or
        `create_scheduler`) in a subclass.
        """
        optimizer = self.create_optimizer_single(model)
        if IS_SAGEMAKER_MP_POST_1_10 and smp.state.cfg.fp16:
            # If smp >= 1.10 and fp16 is enabled, we unwrap the optimizer
            optimizer = optimizer.optimizer
        else:
            optimizer = optimizer
        scheduler = self.create_scheduler_single(num_training_steps=num_training_steps, optimizer=optimizer)
        
        return optimizer, scheduler

    def _load_optimizer_and_scheduler_single(self, checkpoint, model_wrapped, optimizer, lr_scheduler):
        """If optimizer and scheduler states exist, load them."""
        if checkpoint is None:
            return

        if self.deepspeed:
            # deepspeed loads optimizer/lr_scheduler together with the model in deepspeed_init
            return

        checkpoint_file_exists = (
            glob.glob(os.path.join(checkpoint, OPTIMIZER_NAME) + "_*")
            if is_sagemaker_mp_enabled()
            else os.path.isfile(os.path.join(checkpoint, OPTIMIZER_NAME))
        )
        if checkpoint_file_exists and os.path.isfile(os.path.join(checkpoint, SCHEDULER_NAME)):
            # Load in optimizer and scheduler states
            if is_torch_tpu_available():
                # On TPU we have to take some extra precautions to properly load the states on the right device.
                optimizer_state = torch.load(os.path.join(checkpoint, OPTIMIZER_NAME), map_location="cpu")
                with warnings.catch_warnings(record=True) as caught_warnings:
                    lr_scheduler_state = torch.load(os.path.join(checkpoint, SCHEDULER_NAME), map_location="cpu")
                reissue_pt_warnings(caught_warnings)

                xm.send_cpu_data_to_device(optimizer_state, self.args.device)
                xm.send_cpu_data_to_device(lr_scheduler_state, self.args.device)

                optimizer.load_state_dict(optimizer_state)
                lr_scheduler.load_state_dict(lr_scheduler_state)
            else:
                if is_sagemaker_mp_enabled():
                    if os.path.isfile(os.path.join(checkpoint, "user_content.pt")):
                        # Optimizer checkpoint was saved with smp >= 1.10
                        def opt_load_hook(mod, opt):
                            opt.load_state_dict(smp.load(os.path.join(checkpoint, OPTIMIZER_NAME), partial=True))

                    else:
                        # Optimizer checkpoint was saved with smp < 1.10
                        def opt_load_hook(mod, opt):
                            if IS_SAGEMAKER_MP_POST_1_10:
                                opt.load_state_dict(
                                    smp.load(os.path.join(checkpoint, OPTIMIZER_NAME), partial=True, back_compat=True)
                                )
                            else:
                                opt.load_state_dict(smp.load(os.path.join(checkpoint, OPTIMIZER_NAME), partial=True))

                    model_wrapped.register_post_step_hook(opt_load_hook)
                else:
                    # We use the CPU when training on one GPU to avoid OOM for GPU RAM when training big models.
                    # In distributed training however, we load directly on each GPU and risk the GPU OOM as it's more
                    # likely to get OOM on CPU (since we load num_gpu times the optimizer state
                    map_location = self.args.device if self.args.world_size > 1 else "cpu"
                    optimizer.load_state_dict(
                        torch.load(os.path.join(checkpoint, OPTIMIZER_NAME), map_location=map_location)
                    )
                with warnings.catch_warnings(record=True) as caught_warnings:
                    lr_scheduler.load_state_dict(torch.load(os.path.join(checkpoint, SCHEDULER_NAME)))
                reissue_pt_warnings(caught_warnings)
                if self.do_grad_scaling and os.path.isfile(os.path.join(checkpoint, SCALER_NAME)):
                    self.scaler.load_state_dict(torch.load(os.path.join(checkpoint, SCALER_NAME)))

    ############## zo ##############
    def zo_perturb_parameters(self, random_seed=None, scaling_factor=1):
        """
        Perturb the parameters with random vector z.
        Input: 
        - random_seed: random seed for MeZO in-place perturbation (if it's None, we will use self.zo_random_seed)
        - scaling_factor: theta = theta + scaling_factor * z * eps
        """

        # Set the random seed to ensure that we sample the same z for perturbation/update
        torch.manual_seed(random_seed if random_seed is not None else self.zo_random_seed)
        
        for name, param in self.named_parameters_to_optim:
            z = torch.normal(mean=0, std=1, size=param.data.size(), device=param.data.device, dtype=param.data.dtype)
            param.data = param.data + scaling_factor * z * self.args.zo_eps


    def zo_forward(self, model, inputs):
        """
        Get (no gradient) loss from the model. Dropout is turned off too.
        """
        model.eval()
        if self.args.non_diff:
            # Non-differentiable objective (may require autoregressive generation)
            return self.zo_forward_nondiff(model, inputs)

        with torch.inference_mode():
            inputs = self._prepare_inputs(inputs)
            with self.compute_loss_context_manager():
                loss = self.compute_loss(model, inputs)
            if self.args.n_gpu > 1:
                # Warning: this is copied from the original Huggingface Trainer. Untested.
                loss = loss.mean()  # mean() to average on multi-gpu parallel training
        return loss.detach()


    def zo_forward_nondiff(self, model, inputs):
        """
        Get (no gradient) non-diffiable loss from the model.
        """
        model.eval()
        assert self.args.task_name == "SQuAD", "Non differentiable objective only supports SQuAD for now."

        with torch.inference_mode():
            inputs = self._prepare_inputs(inputs)
            args = self.args
            outputs = self.model.generate(
                inputs["input_ids"], do_sample=args.sampling, temperature=args.temperature, 
                num_beams=args.num_beams, top_p=args.top_p, top_k=args.top_k, max_new_tokens=min(args.max_new_tokens, args.max_length - inputs["input_ids"].size(1)), 
                num_return_sequences=1, eos_token_id=[self.tokenizer.encode(args.eos_token, add_special_tokens=False)[-1], self.tokenizer.eos_token_id],
            )
            output_text = []
            for i in range(len(outputs)):
                output_text.append(self.tokenizer.decode(outputs[i][inputs["input_ids"].size(1):], skip_special_tokens=True).strip())
            f1s = [f1(output_text[i], inputs['gold'][i]) for i in range(len(output_text))]
        
        return -torch.tensor(np.mean(f1s), dtype=torch.float32)


    def zo_step(self, model, inputs):
        """
        Estimate gradient by MeZO. Return the loss from f(theta + z)
        """
        args = self.args

        # What parameters to optimize 
        self.named_parameters_to_optim = []
        for name, param in model.named_parameters():
            if param.requires_grad:
                self.named_parameters_to_optim.append((name, param))

        # Sample the random seed for sampling z
        self.zo_random_seed = np.random.randint(1000000000)

        # First function evaluation
        self.zo_perturb_parameters(scaling_factor=1)
        loss1 = self.zo_forward(model, inputs)

        # Second function evaluation
        self.zo_perturb_parameters(scaling_factor=-2)
        loss2 = self.zo_forward(model, inputs)

        self.projected_grad = ((loss1 - loss2) / (2 * self.args.zo_eps)).item()

        # No gradient accumulation support
        assert self.args.gradient_accumulation_steps == 1

        # Reset model back to its parameters at start of step
        self.zo_perturb_parameters(scaling_factor=1)
        
        return loss1


    def zo_update(self, model):
        """
        Update the parameters with the estimated gradients.
        """
        args = self.args

        # Reset the random seed for sampling zs
        torch.manual_seed(self.zo_random_seed)     

        for name, param in self.named_parameters_to_optim:
            # Resample z
            z = torch.normal(mean=0, std=1, size=param.data.size(), device=param.data.device, dtype=param.data.dtype)
            if "bias" not in name and "layer_norm" not in name and "layernorm" not in name:
                param.data = param.data - self._get_learning_rate() * (self.projected_grad * z + args.weight_decay * param.data)
            else:
                param.data = param.data - self._get_learning_rate() * (self.projected_grad * z)

        self.lr_scheduler.step()

    ############## zo - CeZO ##############
    def zo_perturb_parameters_CeZO(self, model, random_seed, scaling_factor=1):
        """
        Perturb the parameters with random vector z.
        Input: 
        - random_seed: random seed for MeZO in-place perturbation
        - scaling_factor: theta = theta + scaling_factor * z * eps
        """

        # Set the random seed to ensure that we sample the same z for perturbation/update
        torch.manual_seed(random_seed)
        
        for name, param in model.named_parameters():
            if param.requires_grad:
                z = torch.normal(mean=0, std=1, size=param.data.size(), device=param.data.device, dtype=param.data.dtype)
                param.data = param.data + scaling_factor * z * self.args.zo_eps

    def zo_step_CeZO(self, model, inputs, random_seed):
        """
        Estimate gradient by MeZO. Return the loss from f(theta + z)
        """
        # First function evaluation
        self.zo_perturb_parameters_CeZO(model, random_seed=random_seed, scaling_factor=1)
        loss1 = self.zo_forward(model, inputs)

        # Second function evaluation
        self.zo_perturb_parameters_CeZO(model, random_seed=random_seed, scaling_factor=-2)
        loss2 = self.zo_forward(model, inputs)

        projected_grad = ((loss1 - loss2) / (2 * self.args.zo_eps)).item()
        projected_grad = 1 if projected_grad > 0 else -1 # onebit

        # No gradient accumulation support
        assert self.args.gradient_accumulation_steps == 1

        # Reset model back to its parameters at start of step
        self.zo_perturb_parameters_CeZO(model, random_seed=random_seed, scaling_factor=1)
        
        return loss1, projected_grad

    def zo_update_CeZO(self, model, random_seed, projected_grad, fed_num=None):
        """
        Update the parameters with the estimated gradients.
        """
        if fed_num is None:
            fed_num = 1
        
        # Reset the random seed for sampling zs
        torch.manual_seed(random_seed)

        for name, param in model.named_parameters():
            if param.requires_grad:
                # Resample z
                z = torch.normal(mean=0, std=1, size=param.data.size(), device=param.data.device, dtype=param.data.dtype)
                if "bias" not in name and "layer_norm" not in name and "layernorm" not in name:
                    param.data = param.data - (self.args.learning_rate * (projected_grad * z + self.args.weight_decay * param.data)) / fed_num
                else:
                    param.data = param.data - (self.args.learning_rate * (projected_grad * z)) / fed_num

    ############## Misc overload functions ##############


    def _set_signature_columns_if_needed(self):
        """
        We overload this function for non-differentiable objective training to pass "gold" -- the gold text for the task
        """
        if self._signature_columns is None:
            # Inspect model forward signature to keep only the arguments it accepts.
            signature = inspect.signature(self.model.forward)
            self._signature_columns = list(signature.parameters.keys())
            # Labels may be named label or label_ids, the default data collator handles that.
            self._signature_columns += list(set(["label", "label_ids"] + self.label_names))
            self._signature_columns += ["gold"]

    
    def save_model(self, output_dir: Optional[str] = None, _internal_call: bool = False):
        """
        We overload this function to fix an FSDP saving bug (before fix, it will likely cause OOM) 
        """

        if output_dir is None:
            output_dir = self.args.output_dir

        if is_torch_tpu_available():
            self._save_tpu(output_dir)
        elif is_sagemaker_mp_enabled():
            # Calling the state_dict needs to be done on the wrapped model and on all processes.
            os.makedirs(output_dir, exist_ok=True)
            state_dict = self.model_wrapped.state_dict()
            if self.args.should_save:
                self._save(output_dir, state_dict=state_dict)
            if IS_SAGEMAKER_MP_POST_1_10:
                # 'user_content.pt' indicates model state_dict saved with smp >= 1.10
                Path(os.path.join(output_dir, "user_content.pt")).touch()
        elif (
            ShardedDDPOption.ZERO_DP_2 in self.args.sharded_ddp
            or ShardedDDPOption.ZERO_DP_3 in self.args.sharded_ddp
            or self.fsdp is not None
        ):
            from torch.distributed.fsdp import FullyShardedDataParallel as FSDP, StateDictType, FullStateDictConfig
            full_state_dict_config = FullStateDictConfig(offload_to_cpu=True, rank0_only=True)

            # Fix the FSDP loading bug
            with FSDP.state_dict_type(self.model, StateDictType.FULL_STATE_DICT, full_state_dict_config):
                state_dict = self.model.state_dict()
            # state_dict = self.model.state_dict()

            if self.args.should_save:
                self._save(output_dir, state_dict=state_dict)
        elif self.deepspeed:
            # this takes care of everything as long as we aren't under zero3
            if self.args.should_save:
                self._save(output_dir)

            if is_deepspeed_zero3_enabled():
                # It's too complicated to try to override different places where the weights dump gets
                # saved, so since under zero3 the file is bogus, simply delete it. The user should
                # either user deepspeed checkpoint to resume or to recover full weights use
                # zero_to_fp32.py stored in the checkpoint.
                if self.args.should_save:
                    file = os.path.join(output_dir, WEIGHTS_NAME)
                    if os.path.isfile(file):
                        # logger.info(f"deepspeed zero3: removing {file}, see zero_to_fp32.py to recover weights")
                        os.remove(file)

                # now save the real model if stage3_gather_16bit_weights_on_model_save=True
                # if false it will not be saved.
                # This must be called on all ranks
                if not self.deepspeed.save_16bit_model(output_dir, WEIGHTS_NAME):
                    logger.warning(
                        "deepspeed.save_16bit_model didn't save the model, since"
                        " stage3_gather_16bit_weights_on_model_save=false. Saving the full checkpoint instead, use"
                        " zero_to_fp32.py to recover weights"
                    )
                    self.deepspeed.save_checkpoint(output_dir)

        elif self.args.should_save:
            self._save(output_dir)

        # Push to the Hub when `save_model` is called by the user.
        if self.args.push_to_hub and not _internal_call:
            self.push_to_hub(commit_message="Model save")
