import torch
import wandb
import pdb


class Logger(object):
    def __init__(self, runs, info=None, use_wandb=False):
        self.info = info
        self.use_wandb = use_wandb
        self.results = [[] for _ in range(runs)]

    def add_result(self, run, result):
        assert len(result) == 3
        assert run >= 0 and run < len(self.results)
        self.results[run].append(result)

    def print_statistics(self, run=None):
        if run is not None:
            result = 100 * torch.tensor(self.results[run])
            argmax = result[:, 1].argmax().item()
            print(f'Run {run + 1:02d}:')
            print(f'Highest Train: {result[:, 0].max():.2f}')
            print(f'Highest Valid: {result[:, 1].max():.2f}')
            print(f'  Final Train: {result[argmax, 0]:.2f}')
            print(f'   Final Test: {result[argmax, 2]:.2f}')
            if self.use_wandb:
                log_dict = {"Highest Train" : result[:, 0].max(),
                        "Highest Valid" : result[:, 1].max(),
                        "Final Train" : result[argmax, 0],
                        "Final Test" : result[argmax, 2]}
                wandb.log(log_dict)
        else:
            best_results = []
            for r in self.results:
                r = 100 * torch.tensor(r)
                train1 = r[:, 0].max().item()
                valid = r[:, 1].max().item()
                train2 = r[r[:, 1].argmax(), 0].item()
                test = r[r[:, 1].argmax(), 2].item()
                best_results.append((train1, valid, train2, test))

            best_result = torch.tensor(best_results)

            print(f'All runs:')
            r = best_result[:, 0]
            print(f'Highest Train: {r.mean():.2f} ± {r.std():.2f}')
            r = best_result[:, 1]
            print(f'Highest Valid: {r.mean():.2f} ± {r.std():.2f}')
            r = best_result[:, 2]
            print(f'  Final Train: {r.mean():.2f} ± {r.std():.2f}')
            r = best_result[:, 3]
            print(f'   Final Test: {r.mean():.2f} ± {r.std():.2f}')
            
            if self.use_wandb:
                log_dict = {"Highest Train" : best_result[:,0].mean().item(),
                            "Highest Valid" : best_result[:,1].mean().item(),
                            "Final Train" : best_result[:,2].mean().item(),
                            "Final Test" : best_result[:,3].mean().item()}
                wandb.log(log_dict)
                wandb.run.summary["Test_Mean"] =  best_result[:,3].mean().item()
                wandb.run.summary["Test_Std"] =  best_result[:,3].std().item()