#!/usr/bin/env python
# coding=utf-8
# Copyright 2020 The HuggingFace Inc. team. All rights reserved.
#
# Licensed under the Apache License, Version 2.0 (the "License");
# you may not use this file except in compliance with the License.
# You may obtain a copy of the License at
#
#     http://www.apache.org/licenses/LICENSE-2.0
#
# Unless required by applicable law or agreed to in writing, software
# distributed under the License is distributed on an "AS IS" BASIS,
# WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
# See the License for the specific language governing permissions and
# limitations under the License.
""" Finetuning the library models for sequence classification on GLUE."""
# You can also adapt this script on your own text classification task. Pointers for this are left as comments.

import logging
import os
import random
import sys
import torch
from dataclasses import dataclass, field
from typing import Optional
from collections import defaultdict

import numpy as np
import pickle
from datasets import load_dataset, load_metric, Dataset, DatasetDict

import transformers
from transformers import (
    AutoConfig,
    AutoModelForSequenceClassification,
    AutoModelForCausalLM,
    AutoTokenizer,
    DataCollatorWithPadding,
    EvalPrediction,
    HfArgumentParser,
    PretrainedConfig,
    Trainer,
    TrainingArguments,
    default_data_collator,
    set_seed,
)
from transformers.trainer_utils import get_last_checkpoint, is_main_process
from transformers.utils import check_min_version
from torch import pca_lowrank
from transformers.trainer_pt_utils import get_module
from tqdm import tqdm
import matplotlib.pyplot as plt
from sklearn.decomposition import IncrementalPCA
from sklearn.metrics.pairwise import cosine_similarity
import loralib

from gpt2_data_utils import FT_Dataset


# Will error if the minimal version of Transformers is not installed. Remove at your own risks.
check_min_version("4.4.0")

task_to_keys = {
    "cola": ("sentence", None),
    "mnli": ("premise", "hypothesis"),
    "mrpc": ("sentence1", "sentence2"),
    "qnli": ("question", "sentence"),
    "qqp": ("question1", "question2"),
    "rte": ("sentence1", "sentence2"),
    "sst2": ("sentence", None),
    "stsb": ("sentence1", "sentence2"),
    "wnli": ("sentence1", "sentence2"),
}

logger = logging.getLogger(__name__)


@dataclass
class DataTrainingArguments:
    """
    Arguments pertaining to what data we are going to input our model for training and eval.

    Using `HfArgumentParser` we can turn this class
    into argparse arguments to be able to specify them on
    the command line.
    """

    task_name: Optional[str] = field(
        default=None,
        metadata={"help": "The name of the glue task to train on: " + ", ".join(task_to_keys.keys())},
    )
    dataset_name: Optional[str] = field(
        default=None,
        metadata={"help": "The name of the dataset to train on: " + ", ".join(task_to_keys.keys())},
    )
    dataset_config_name: Optional[str] = field(
        default=None, metadata={"help": "The configuration name of the dataset to use (via the datasets library)."}
    )
    label_column: Optional[str] = field(
        default=None, metadata={"help": "Dataset column name for labels"}
    )
    use_all_columns: bool = field(
        default=False, metadata={"help": "Use all dataset columns"}
    )
    max_seq_length: Optional[int] = field(
        default=None,
        metadata={
            "help": "The maximum total input sequence length after tokenization. Sequences longer "
                    "than this will be truncated, sequences shorter will be padded."
        },
    )
    pad_to_max_length: bool = field(
        default=True,
        metadata={
            "help": "Whether to pad all samples to `max_seq_length`. "
                    "If False, will pad the samples dynamically when batching to the maximum length in the batch."
        },
    )
    max_train_samples: Optional[int] = field(
        default=None,
        metadata={
            "help": "For debugging purposes or quicker training, truncate the number of training examples to this "
                    "value if set."
        },
    )
    max_val_samples: Optional[int] = field(
        default=None,
        metadata={
            "help": "For debugging purposes or quicker training, truncate the number of validation examples to this "
                    "value if set."
        },
    )
    max_test_samples: Optional[int] = field(
        default=None,
        metadata={
            "help": "For debugging purposes or quicker training, truncate the number of test examples to this "
                    "value if set."
        },
    )
    train_file: Optional[str] = field(
        default=None, metadata={"help": "A csv or a json file containing the training data."}
    )
    validation_file: Optional[str] = field(
        default=None, metadata={"help": "A csv or a json file containing the validation data."}
    )
    test_file: Optional[str] = field(default=None, metadata={"help": "A csv or a json file containing the test data."})
    data_cache_dir: Optional[str] = field(default=None, metadata={"help": "Directory path to save cache files."})

    def __post_init__(self):
        if self.task_name is not None:
            self.task_name = self.task_name.lower()
            if self.task_name not in task_to_keys.keys():
                raise ValueError("Unknown task, you should pick one in " + ",".join(task_to_keys.keys()))
        elif self.dataset_name is not None:
            pass
        elif self.train_file is None or self.validation_file is None:
            raise ValueError("Need either a GLUE task or a training/validation file.")
        else:
            train_extension = self.train_file.split(".")[-1]
            assert train_extension in ["csv", "json"], "`train_file` should be a csv or a json file."
            validation_extension = self.validation_file.split(".")[-1]
            assert (
                    validation_extension == train_extension
            ), "`validation_file` should have the same extension (csv or json) as `train_file`."


@dataclass
class ModelArguments:
    """
    Arguments pertaining to which model/config/tokenizer we are going to fine-tune from.
    """

    model_name_or_path: str = field(
        metadata={"help": "Path to pretrained model or model identifier from huggingface.co/models"}
    )
    cache_dir: Optional[str] = field(
        default=None,
        metadata={"help": "Where do you want to store the pretrained models downloaded from huggingface.co"},
    )
    pca_on_acts: Optional[bool] = field(
        default=False,
        metadata={"help": "choose a good init for lora matrices"}
    )
    sparse: Optional[bool] = field(
        default=False,
        metadata={"help": "perform sparse pca"}
    )
    lda_on_acts: Optional[bool] = field(
        default=False,
        metadata={"help": "choose a good init for lora matrices"}
    )
    init_importances: Optional[bool] = field(
        default=False,
        metadata={"help": "use importance scores for initializing LoRA subspace"}
    )
    init_importances_with_grads: Optional[bool] = field(
        default=False,
        metadata={"help": "use importance scores for initializing LoRA subspace"}
    )
    smooth_grads: Optional[bool] = field(
        default=False,
        metadata={"help": "include smoothing of importance weights and uncertainty estimate"}
    )
    scale_by_singular_vals: Optional[bool] = field(
        default=False,
        metadata={"help": "scale eigenvectors after PCA by singular values"}
    )
    early_stop: Optional[bool] = field(
        default=False,
        metadata={"help": "Use early stopping in case PCA components have converged."}
    )
    early_stop_sim_thresh: Optional[float] = field(
        default=0.99,
        metadata={"help": "Threshold for cosine similarity to determine convergence of a single component."}
    )
    early_stop_p: Optional[float] = field(
        default=0.9,
        metadata={"help": "Percentage of components that have converged (via early_stop_sim_thresh) to early stop."}
    )
    n_components: Optional[int] = field(
        default=16,
        metadata={"help": "Number of principle components."}
    )



def main():
    # See all possible arguments in src/transformers/training_args.py
    # or by passing the --help flag to this script.
    # We now keep distinct sets of args, for a cleaner separation of concerns.
    parser = HfArgumentParser((ModelArguments, DataTrainingArguments, TrainingArguments))
    if len(sys.argv) == 2 and sys.argv[1].endswith(".json"):
        # If we pass only one argument to the script and it's the path to a json file,
        # let's parse it to get our arguments.
        model_args, data_args, training_args = parser.parse_json_file(json_file=os.path.abspath(sys.argv[1]))
    else:
        model_args, data_args, training_args = parser.parse_args_into_dataclasses()

    # torch.use_deterministic_algorithms(training_args.use_deterministic_algorithms)
    logger.info("use_deterministic_algorithms: " + str(torch.are_deterministic_algorithms_enabled()))

    # Setup logging
    logging.basicConfig(
        format="%(asctime)s - %(levelname)s - %(name)s -   %(message)s",
        datefmt="%m/%d/%Y %H:%M:%S",
        handlers=[logging.StreamHandler(sys.stdout)],
    )
    logger.setLevel(logging.INFO if is_main_process(training_args.local_rank) else logging.WARN)

    # Log on each process the small summary:
    logger.warning(
        f"Process rank: {training_args.local_rank}, device: {training_args.device}, n_gpu: {training_args.n_gpu}"
        + f"distributed training: {bool(training_args.local_rank != -1)}, 16-bits training: {training_args.fp16}"
    )
    # Set the verbosity to info of the Transformers logger (on main process only):
    if is_main_process(training_args.local_rank):
        transformers.utils.logging.set_verbosity_info()
        transformers.utils.logging.enable_default_handler()
        transformers.utils.logging.enable_explicit_format()
    logger.info(f"Training/evaluation parameters {training_args}")

    # Set seed before initializing model.
    set_seed(training_args.seed)

    # Get the datasets: you can either provide your own CSV/JSON training and evaluation files (see below)
    # or specify a GLUE benchmark task (the dataset will be downloaded automatically from the datasets Hub).
    #
    # For CSV/JSON files, this script will use as labels the column called 'label' and as pair of sentences the
    # sentences in columns called 'sentence1' and 'sentence2' if such column exists or the first two columns not named
    # label if at least two columns are provided.
    #
    # If the CSVs/JSONs contain only one non-label column, the script does single sentence classification on this
    # single column. You can easily tweak this behavior (see below)
    #
    # In distributed training, the load_dataset function guarantee that only one local process can concurrently
    # download the dataset.
    if data_args.task_name is not None:
        # Downloading and loading a dataset from the hub.
        datasets = load_dataset("glue", data_args.task_name, cache_dir=data_args.data_cache_dir)
        data_ref = data_args.task_name
    elif data_args.dataset_name == "e2e_nlg":
        data_ref = "e2e_nlg"
    elif data_args.dataset_name is not None:
        datasets = load_dataset(data_args.dataset_name, data_args.dataset_config_name)
        data_ref = data_args.dataset_name
    else:
        # Loading a dataset from your local files.
        # CSV/JSON training and evaluation files are needed.
        data_files = {"train": data_args.train_file, "validation": data_args.validation_file}

        # Get the test dataset: you can provide your own CSV/JSON test file (see below)
        # when you use `do_predict` without specifying a GLUE benchmark task.
        if training_args.do_predict:
            if data_args.test_file is not None:
                train_extension = data_args.train_file.split(".")[-1]
                test_extension = data_args.test_file.split(".")[-1]
                assert (
                        test_extension == train_extension
                ), "`test_file` should have the same extension (csv or json) as `train_file`."
                data_files["test"] = data_args.test_file
            else:
                raise ValueError("Need either a GLUE task, a dataset or a test file for `do_predict`.")

        for key in data_files.keys():
            logger.info(f"load a local file for {key}: {data_files[key]}")

        if data_args.train_file.endswith(".csv"):
            # Loading a dataset from local csv files
            datasets = load_dataset("csv", data_files=data_files, cache_dir=data_args.data_cache_dir)
        else:
            # Loading a dataset from local json files
            datasets = load_dataset("json", data_files=data_files, cache_dir=data_args.data_cache_dir)
        data_ref = ""
    # See more about loading any type of standard or custom dataset at
    # https://huggingface.co/docs/datasets/loading_datasets.html.

    # Labels
    num_labels = None
    if data_args.task_name is not None:
        is_regression = data_args.task_name == "stsb"
        if not is_regression:
            label_list = datasets["train"].features["label"].names
            num_labels = len(label_list)
        else:
            num_labels = 1
        model_cls = AutoModelForSequenceClassification
    elif data_args.dataset_name is not None:
        if data_args.dataset_name == "e2e_nlg":
            model_cls = AutoModelForCausalLM
        else:
            if data_args.use_all_columns:
                merge_columns = lambda example: {'text': ' '.join(example.values())}
                datasets = datasets.map(
                    merge_columns,
                    load_from_cache_file=True
                )
            column_names = datasets["train"].column_names
            text_column_name = "text" if "text" in column_names else column_names[0]
            if data_args.label_column is None:
                logger.info(f"label column name not set, assuming language modeling task")
                model_cls = AutoModelForCausalLM
            else:
                model_cls = AutoModelForSequenceClassification
    else:
        # Trying to have good defaults here, don't hesitate to tweak to your needs.
        is_regression = datasets["train"].features["label"].dtype in ["float32", "float64"]
        if is_regression:
            num_labels = 1
        else:
            # A useful fast method:
            # https://huggingface.co/docs/datasets/package_reference/main_classes.html#datasets.Dataset.unique
            label_list = datasets["train"].unique("label")
            label_list.sort()  # Let's sort it for determinism
            num_labels = len(label_list)
        if data_args.label_column is None:
            logger.info(f"label column name not set, assuming language modeling task")
            model_cls = AutoModelForCausalLM
        else:
            model_cls = AutoModelForSequenceClassification

    # Load pretrained model and tokenizer
    #
    # In distributed training, the .from_pretrained methods guarantee that only one local process can concurrently
    # download model & vocab.
    if model_cls == AutoModelForSequenceClassification:
        config = AutoConfig.from_pretrained(
            model_args.model_name_or_path,
            num_labels=num_labels,
            finetuning_task=data_args.task_name,
            cache_dir=model_args.cache_dir,
            revision="main"
        )
    else:
        config = AutoConfig.from_pretrained(
            model_args.model_name_or_path,
            cache_dir=model_args.cache_dir,
            revision="main"
        )
    tokenizer = AutoTokenizer.from_pretrained(
        model_args.model_name_or_path,
        cache_dir=model_args.cache_dir,
        use_fast=True,
        revision="main"
    )
    model = model_cls.from_pretrained(
        model_args.model_name_or_path,
        from_tf=bool(".ckpt" in model_args.model_name_or_path),
        config=config,
        cache_dir=model_args.cache_dir,
        revision="main"
    )

    names_to_exclude = None
    if 'roberta' in model_args.model_name_or_path:
        names_to_exclude = ['classifier.dense', 'classifier.out_proj']
    if 'deberta' in model_args.model_name_or_path:
        names_to_exclude = ['pooler.dense', 'classifier']
    model = loralib.LoRALayer.convert(module=model, kind='LoRA', lora_rank=model_args.n_components,
                                      lora_alpha=1, names_to_exclude=names_to_exclude)

    if model_args.pca_on_acts or model_args.lda_on_acts:
        # collect layer inputs during forward pass with hooks

        activations = {}

        def get_activation(name):
            def hook(model, input, output):
                try:
                    activations[name] = input.detach().cpu().numpy()
                except AttributeError:
                    if len(input) > 0 and input[0].shape[0] == training_args.per_device_train_batch_size:
                        activations[name] = input[0].detach().cpu().numpy()
            return hook

        for name, module in model.named_modules():
            if name != '' and (isinstance(module, loralib.Linear)):
                module.register_forward_hook(get_activation(name))


    if model_args.init_importances:
        trainable_params = ['query', 'key', 'value', 'attention.output', 'intermediate', 'output.dense']
        for name, param in model.named_parameters():
            param.requires_grad = False
            if name.startswith('deberta') or name.startswith('roberta'):
                for trainable_param in trainable_params:
                    if not model_args.pca_on_acts and not model_args.lda_on_acts:
                        if trainable_param in name and 'weight' in name and not 'LayerNorm' in name and not "Embedding" in name:
                            param.requires_grad = True
                            break
                    else:
                        if trainable_param in name:
                            param.requires_grad = True
                            break

    if data_args.max_seq_length is None:
        max_seq_length = tokenizer.model_max_length
        if max_seq_length > 1024:
            logger.warn(
                f"The tokenizer picked seems to have a very large `model_max_length` ({tokenizer.model_max_length}). "
                "Picking 1024 instead. You can change that default value by passing --max_seq_length xxx."
            )
        max_seq_length = 1024
    else:
        if data_args.max_seq_length > tokenizer.model_max_length:
            logger.warn(
                f"The max_seq_length passed ({data_args.max_seq_length}) is larger than the maximum length for the model"
                f"({tokenizer.model_max_length}). Using max_seq_length={tokenizer.model_max_length}."
            )
        max_seq_length = min(data_args.max_seq_length, tokenizer.model_max_length)

    # Preprocessing the datasets
    if model_cls == AutoModelForCausalLM and data_args.task_name is None:

        data_collator = default_data_collator

        compute_metrics = None

        if data_args.dataset_name == "e2e_nlg":

            train_data = FT_Dataset("../NLG/data/e2e/train.jsonl", 1, max_seq_length, joint_lm=True)    
            valid_data = FT_Dataset("../NLG/data/e2e/valid.jsonl", 1, max_seq_length, joint_lm=True)

            td = {"id": [], "input_ids": [], "labels": [], "attention_mask": []}
            for i in range(len(train_data)):
                example = train_data[i]
                td["id"].append(i)
                td["input_ids"].append(example["input"])
                td["labels"].append(example["input"])
                m = example["mask"]
                m[int(m.sum().item())] = 1.
                td["attention_mask"].append(m)

            vd = {"id": [], "input_ids": [], "labels": [], "attention_mask": []}
            for i in range(len(valid_data)):
                example = valid_data[i]
                vd["id"].append(i)
                vd["input_ids"].append(example["input"])
                vd["labels"].append(example["input"])
                m = example["mask"]
                m[int(m.sum().item())] = 1. # index of last non zero token is 0 but should be 1
                vd["attention_mask"].append(m)

            datasets = DatasetDict({
                "train": Dataset.from_dict(td),
                "val": Dataset.from_dict(vd)
            })

        else:

            # Preprocessing the datasets.
            # First we tokenize all the texts

            def tokenize_function(examples):
                return tokenizer(examples[text_column_name])

            tokenized_datasets = datasets.map(
                tokenize_function,
                batched=True,
                remove_columns=column_names,
                load_from_cache_file=True
            )

            # Main data processing function that will concatenate all texts from our dataset and generate chunks of max_seq_length.
            def group_texts(examples):
                # Concatenate all texts.
                concatenated_examples = {k: sum(examples[k], []) for k in examples.keys()}
                total_length = len(concatenated_examples[list(examples.keys())[0]])
                # We drop the small remainder, we could add padding if the model supported it instead of this drop, you can
                # customize this part to your needs.
                total_length = (total_length // max_seq_length) * max_seq_length
                # Split by chunks of max_len.
                result = {
                    k: [t[i : i + max_seq_length] for i in range(0, total_length, max_seq_length)]
                    for k, t in concatenated_examples.items()
                }
                result["labels"] = result["input_ids"].copy()
                return result

            # Note that with `batched=True`, this map processes 1,000 texts together, so group_texts throws away a remainder
            # for each of those groups of 1,000 texts. You can adjust that batch_size here but a higher value might be slower
            # to preprocess.
            #
            # To speed up this part, we use multiprocessing. See the documentation of the map method for more information:
            # https://huggingface.co/docs/datasets/package_reference/main_classes.html#datasets.Dataset.map

            datasets = tokenized_datasets.map(group_texts, batched=True, load_from_cache_file=True)

    elif model_cls == AutoModelForSequenceClassification and data_args.task_name is None:

        raise NotImplementedError

    else:

        if data_args.task_name is not None:
            sentence1_key, sentence2_key = task_to_keys[data_args.task_name]
        else:
            # Again, we try to have some nice defaults but don't hesitate to tweak to your use case.
            non_label_column_names = [name for name in datasets["train"].column_names if name != "label"]
            if "sentence1" in non_label_column_names and "sentence2" in non_label_column_names:
                sentence1_key, sentence2_key = "sentence1", "sentence2"
            else:
                if len(non_label_column_names) >= 2:
                    sentence1_key, sentence2_key = non_label_column_names[:2]
                else:
                    sentence1_key, sentence2_key = non_label_column_names[0], None

        # Padding strategy
        if data_args.pad_to_max_length:
            padding = "max_length"
        else:
            # We will pad later, dynamically at batch creation, to the max sequence length in each batch
            padding = False

        # Some models have set the order of the labels to use, so let's make sure we do use it.
        label_to_id = None
        if num_labels is not None and (
                model.config.label2id != PretrainedConfig(num_labels=num_labels).label2id
                and data_args.task_name is not None
                and not is_regression
        ):
            # Some have all caps in their config, some don't.
            label_name_to_id = {k.lower(): v for k, v in model.config.label2id.items()}
            if list(sorted(label_name_to_id.keys())) == list(sorted(label_list)):
                label_to_id = {i: int(label_name_to_id[label_list[i]]) for i in range(num_labels)}
            else:
                logger.warn(
                    "Your model seems to have been trained with labels, but they don't match the dataset: ",
                    f"model labels: {list(sorted(label_name_to_id.keys()))}, dataset labels: {list(sorted(label_list))}."
                    "\nIgnoring the model labels as a result.",
                )
        elif data_args.task_name is None and not is_regression:
            label_to_id = {v: i for i, v in enumerate(label_list)}

        def preprocess_function(examples):
            # Tokenize the texts
            args = (
                (examples[sentence1_key],) if sentence2_key is None else (examples[sentence1_key], examples[sentence2_key])
            )
            result = tokenizer(*args, padding=padding, max_length=max_seq_length, truncation=True)

            # Map labels to IDs (not necessary for GLUE tasks)
            if label_to_id is not None and "label" in examples:
                result["label"] = [(label_to_id[l] if l != -1 else -1) for l in examples["label"]]
            return result

        datasets = datasets.map(preprocess_function, batched=True, load_from_cache_file=True)

        # Get the metric function
        if data_args.task_name is not None:
            metric = load_metric("glue", data_args.task_name)

        # You can define your custom compute_metrics function. It takes an `EvalPrediction` object (a namedtuple with a
        # predictions and label_ids field) and has to return a dictionary string to float.
        def compute_metrics(p: EvalPrediction):
            preds = p.predictions[0] if isinstance(p.predictions, tuple) else p.predictions
            preds = np.squeeze(preds) if is_regression else np.argmax(preds, axis=1)
            if data_args.task_name is not None:
                result = metric.compute(predictions=preds, references=p.label_ids)
                if len(result) > 1:
                    result["combined_score"] = np.mean(list(result.values())).item()
                return result
            elif is_regression:
                return {"mse": ((preds - p.label_ids) ** 2).mean().item()}
            else:
                return {"accuracy": (preds == p.label_ids).astype(np.float32).mean().item()}

        # Data collator will default to DataCollatorWithPadding, so we change it if we already did the padding.
        if data_args.pad_to_max_length:
            data_collator = default_data_collator
        elif training_args.fp16:
            data_collator = DataCollatorWithPadding(tokenizer, pad_to_multiple_of=8)
        else:
            data_collator = None

    if "train" not in datasets:
        raise ValueError("--do_train requires a train dataset")
    train_dataset = datasets["train"]
    if data_args.max_train_samples is not None:
        train_dataset = train_dataset.select(range(data_args.max_train_samples))

    # Log a few random samples from the training set:
    rand_samples = []
    if training_args.do_train:
        for index in random.sample(range(len(train_dataset)), 3):
            rand_samples.append(train_dataset[index])
            logger.info(f"Sample {index} of the training set: {train_dataset[index]}.")

    old_train_epochs = training_args.num_train_epochs
    print("Computing initialisation for LoRA finetuning...")
    if model_args.init_importances_with_grads or model_args.pca_on_acts or model_args.lda_on_acts:
        print("Running one epoch through training data...")
        training_args.num_train_epochs = 1
        old_train_epochs -= 1
        # Initialize our Trainer
        trainer = Trainer(
            model=model,
            args=training_args,
            train_dataset=train_dataset,
            eval_dataset=None,
            compute_metrics=compute_metrics,
            tokenizer=tokenizer,
            data_collator=data_collator,
        )

        # Accumulate gradients over entire finetuning dataset
        train_dataloader = trainer.get_train_dataloader()
        tr_loss = 0.
        grads = {}
        steps = 0
        hidden_state_dict = {}
        previous_components_dict = defaultdict(list)
        has_converged = defaultdict(lambda: (False, 0))
        
        model.eval()
        n_samples = trainer.get_train_dataloader().batch_size
        for step, inputs in tqdm(enumerate(train_dataloader)):
            prepared_inputs = trainer._prepare_inputs(inputs)
            if model_args.pca_on_acts or model_args.lda_on_acts:
                prepared_inputs['output_hidden_states'] = True
                if model_args.early_stop and step > 1:
                    layer_converged = [v[0] for v in has_converged.values()]
                    converged_per_layer = [v[1] for v in has_converged.values()]
                    print(f"\t{sum(layer_converged)}/{len(layer_converged)} layers have converged."
                          f"Average components converged: {np.mean(converged_per_layer).round(3)}")
                    # early-exit if has converged 
                    if all(layer_converged):
                        print("Early exit. All PCA components have converged.")
                        break

                # perform forward pass!
                with torch.no_grad():
                    _ = model(**prepared_inputs)
                    indices = torch.nonzero(prepared_inputs['attention_mask']).cpu()
                    labels = prepared_inputs['labels'].cpu()
                    labels = labels[indices[:, 0]]

                    for name, input in activations.items():

                        if has_converged[name][0]:
                            continue

                        states = input[indices[:, 0], indices[:, 1], :]

                        if model_args.pca_on_acts:
                            try:
                                hidden_state_dict[name].partial_fit(states)
                            except KeyError:
                                if model_args.sparse:
                                    hidden_state_dict[name] = loralib.IncrementalSparsePCA(n_components=model_args.n_components).partial_fit(states)
                                else:
                                    hidden_state_dict[name] = IncrementalPCA(n_components=model_args.n_components).partial_fit(states)
                        else:
                            try:
                                hidden_state_dict[name].partial_fit(states, y=labels)
                            except KeyError:
                                hidden_state_dict[name] = loralib.IncrementalLDA(
                                    unique_classes=np.arange(config.num_labels),
                                    n_components=model_args.n_components
                                ).partial_fit(states, y=labels)

                        # check for convergence
                        components = hidden_state_dict[name].components_
                        if name in previous_components_dict:
                            if len(components.shape) == 1:
                                components = components.reshape(1, -1)
                            # consider as converged if enough components have converged via cossim
                            sim = cosine_similarity(components, previous_components_dict[name]).diagonal()
                            converged_components = (sim >= model_args.early_stop_sim_thresh).sum()
                            has_converged[name] = (converged_components >= model_args.early_stop_p * len(sim), 
                                                    converged_components)
                        previous_components_dict[name] = components.copy()
                    
                    activations = {}
            else:
                tr_loss += trainer.compute_loss(model, prepared_inputs)
                tr_loss.backward()
                steps += 1
                for n, p in model.named_parameters():
                    if p.requires_grad:
                        if n not in grads:
                            grads[n] = torch.abs(p.grad.detach().cpu())
                        else:
                            grads[n] += torch.abs(p.grad.detach().cpu())
                model.zero_grad()

    model.cpu()

    if not model_args.pca_on_acts and not model_args.lda_on_acts:
        if model_args.init_importances_with_grads:
            print("Importance scores are pre-trained weights + grads!")
        else:
            print("Importance scores are pre-trained weights")

        importance_scores = {}
        for n, p in model.named_parameters():
            if p.requires_grad:
                if model_args.init_importances_with_grads:
                    if model_args.smooth_grads:
                        imps = p.data.detach().cpu() * torch.stack(grads[n])
                        # weights = 0.85 ** torch.arange(len(imps)) - 0.85 ** torch.arange(1, len(imps)+1)
                        weights = 0.5 * torch.ones(len(imps))
                        weights = weights.flip(0).view(-1, 1, 1)
                        smoothed = imps * weights
                        deltas = torch.abs(imps - smoothed)
                        uncertainties = deltas * weights
                        coeffs = uncertainties * smoothed
                        importance_scores[n] = coeffs.sum(0)
                    else:
                        # weigh pre-trained weights by sensitivity to fine-tuning task
                        importance_scores[n] = p.data.cpu() * (grads[n] / steps)
                else:
                    # use raw weights
                    importance_scores[n] = p.data.cpu()

        print("Compute SVD on importance scores!!!")
        for name in importance_scores.keys():

            _, s, v = torch.linalg.svd(importance_scores[name], full_matrices=False)
            module = get_module(name, model, parent_idx=1)

            print(f"Initialize {name} with SVD on importance scores!!!")
            if model_args.scale_by_singular_vals:
                module.lora_A.data = (v[:model_args.n_components, :] * s).reshape(module.lora_A.shape).to(module.lora_A.device)
            else:
                module.lora_A.data = v[:model_args.n_components, :].reshape(module.lora_A.shape).to(module.lora_A.device)
    else:
        metric_dict = defaultdict(dict)
        for name, module in model.named_modules():
                
            if name in hidden_state_dict:

                u = hidden_state_dict[name].components_

                if not model_args.sparse:

                    if isinstance(hidden_state_dict[name], loralib.IncrementalLDA):
                        s = hidden_state_dict[name].eigenvalues_
                    else:
                        s = hidden_state_dict[name].singular_values_

                    if u.shape[0] > 1:
                        s /= np.max(s)
                        if s.shape[0] == u.shape[0]:
                            s = s.reshape(-1, 1)
                        else:
                            s = s.reshape(1, -1)
                    else:
                        s = 1

                    if model_args.pca_on_acts:
                        var_exp = hidden_state_dict[name].explained_variance_[:model_args.n_components]
                        sum_normed = var_exp / var_exp.sum()
                        max_normed = var_exp / var_exp.max()
                        var_exp_ratio = hidden_state_dict[name].explained_variance_ratio_[:model_args.n_components]
                        exp_var_dict = {
                            'raw': var_exp,
                            'sum': sum_normed,
                            'max': max_normed,
                            'ratio': var_exp_ratio
                        }
                        metric_dict[name] = exp_var_dict
                        if len(var_exp) < model_args.n_components:
                            nticks = config.num_labels
                        else:
                            nticks = model_args.n_components
                        total_variance = np.sum(var_exp)
                        cum_var_exp = np.cumsum(var_exp)
                        plt.bar(range(nticks), var_exp, alpha=0.5, align="center")
                        plt.step(range(nticks), cum_var_exp, where='mid')
                        plt.ylabel("Explained Variance")
                        plt.xlabel("Principal Components")
                        plt.title("Total Variance Explained: %.2f" % total_variance)
                        os.makedirs(f"plots/{model_args.model_name_or_path}/pca", exist_ok=True)
                        plt.tight_layout()
                        plt.gcf().savefig(f"plots/{model_args.model_name_or_path}/pca/{name}_{data_args.task_name}.svg")
                        plt.close()
                    else:
                        eigenvals = hidden_state_dict[name].eigenvalues_
                        disc_ratio = eigenvals / eigenvals.sum()
                        disc_raw = eigenvals
                        disc_dict = {
                            'raw': disc_raw,
                            'ratio': disc_ratio
                        }
                        metric_dict[name] = disc_dict
                        cum_discr = np.cumsum(disc_ratio)

                        if len(disc_ratio) < model_args.n_components:
                            nticks = config.num_labels
                        else:
                            nticks = model_args.n_components
                        plt.bar(range(nticks), disc_ratio, alpha=0.5, align="center")
                        plt.step(range(nticks), cum_discr, where='mid')
                        plt.ylabel("Discrimination ratio")
                        plt.xlabel("Components")
                        plt.title("Total Discrimination ratio: %.2f" % sum(disc_ratio))
                        os.makedirs(f"plots/{model_args.model_name_or_path}/lda", exist_ok=True)
                        plt.tight_layout()
                        plt.gcf().savefig(f"plots/{model_args.model_name_or_path}/lda/{name}_{data_args.task_name}.svg")
                        plt.close()


                    if model_args.scale_by_singular_vals:
                        u *= s

                    module.lora_A.data = torch.from_numpy(u).float().reshape(module.lora_A.shape).to(module.lora_A.device)
                        
    state_dict = model.state_dict()

    suffix = ""
    if model_args.init_importances:
        suffix += "_weights"

    if model_args.init_importances_with_grads:
        suffix += '_with_grads'
        if model_args.smooth_grads:
            suffix += '_smoothed'
    elif model_args.pca_on_acts:
        suffix += '_pca'
    elif model_args.lda_on_acts:
        suffix += '_lda'

    if model_args.sparse:
        suffix += '_pca_sparse'

    if model_args.scale_by_singular_vals:
        suffix += '_scaled'

    if suffix == "_weights":
        torch.save(state_dict, os.path.join(training_args.output_dir, f"{model_args.model_name_or_path.split('/')[-1]}_r_{model_args.n_components}{suffix}.bin"))
    else:
        torch.save(state_dict, os.path.join(training_args.output_dir, f"{model_args.model_name_or_path.split('/')[-1]}_{data_ref}_r_{model_args.n_components}{suffix}.bin"))

    if model_args.pca_on_acts or model_args.lda_on_acts:
        data_name = data_args.task_name if data_args.task_name is not None else data_args.dataset_name
        filename = f"{model_args.model_name_or_path.split('/')[-1]}_{data_name}_r_{model_args.n_components}{suffix}.pkl"
        pickle.dump(metric_dict, open(os.path.join(training_args.output_dir, filename), "wb"))


if __name__ == "__main__":
    main()