from flax import nnx
from numpy.random import default_rng

from offline import base
from offline.modules.base import TrainState, get_optimizer
from offline.bc.arguments import Arguments, build_argument_parser
from offline.bc.modules import BCPolicy
from offline.bc.train import TrainerState, train_fn
from offline.types import OfflineDataWithInfos
from offline.utils.data import DataLoader
from offline.utils.dataset import prepare_sa_dataset
from offline.utils.logger import Logger
from offline.utils.nnx import default_nnx_rngs


def init_fn(args: Arguments, data: OfflineDataWithInfos) -> TrainerState:
    policy = BCPolicy(
        action_dim=data.data.actions.shape[1],
        hidden_features=args.hidden_features,
        layer_norm=args.layer_norm,
        num_layers=args.num_layers,
        observation_dim=data.data.observations.shape[1],
        rngs=default_nnx_rngs(args.seed),
        unsquash=args.unsquash,
    )
    rng = default_rng(args.seed)
    sa_loader = DataLoader(
        batch_size=args.batch_size,
        dataset=prepare_sa_dataset(data.data),
        drop_last=True,
        rng=rng,
    )
    train_state = TrainState(
        model=policy,
        optimizer=get_optimizer(policy, learning_rate=args.learning_rate),
    )
    graphdef, graphstate = nnx.split(train_state)

    return TrainerState(
        eval_state=None,
        graphdef=graphdef,
        graphstate=graphstate,
        sa_iter=sa_loader.repeat_forever(),
    )


def load_fn(
    step: int | None,
    action_dim: int,
    hidden_features: int,
    layer_norm: bool,
    num_layers: int,
    logger: Logger,
    observation_dim: int,
    unsquash: bool,
    **kwargs,
):
    del kwargs

    def model_fn():
        return BCPolicy(
            action_dim=action_dim,
            hidden_features=hidden_features,
            layer_norm=layer_norm,
            num_layers=num_layers,
            observation_dim=observation_dim,
            rngs=default_nnx_rngs(0),
            unsquash=unsquash,
        )

    policy = base.default_load_fn(logger=logger, model_fn=model_fn, step=step)
    return policy, None


if __name__ == "__main__":
    base.run(
        arguments_class=Arguments,
        init_fn=init_fn,
        train_fn=train_fn,
        **vars(build_argument_parser().parse_args()),
    )
