from flax import nnx
import jax.random
from numpy.random import Generator, default_rng

from offline import base
from offline.bppo.arguments import Arguments, build_argument_parser
from offline.bppo.modules import BPPOPolicy
from offline.bppo.train import TrainerState, sarsa_fn, train_fn
from offline.bppo.utils import prepare_bppo_dataset
from offline.modules.actor.base import GaussianActor
from offline.modules.base import TrainState, get_optimizer
from offline.modules.critic import QCritic
from offline.lbp.__main__ import (
    init_behavior_actor_fn,
    init_behavior_vcritic_fn,
)
from offline.types import OfflineData, OfflineDataWithInfos
from offline.utils.data import DataLoader
from offline.utils.dataset import prepare_sarsa_dataset
from offline.utils.logger import Logger
from offline.utils.nnx import default_nnx_rngs


def init_behavior_qcritic_fn(
    batch_size: int,
    data: OfflineData,
    gamma: float,
    hidden_features: int,
    layer_norm: bool,
    learning_rate: float,
    logger: Logger,
    num_layers: int,
    numpy_rng: Generator,
    rngs: nnx.Rngs,
    sarsa_steps: int,
    tau: float,
    update_every: int,
) -> QCritic:
    qcritic = QCritic(
        action_dim=data.actions.shape[1],
        hidden_features=hidden_features,
        layer_norm=layer_norm,
        num_layers=num_layers,
        observation_dim=data.observations.shape[1],
        rngs=rngs,
    )
    qcritic = sarsa_fn(
        gamma=gamma,
        logger=logger,
        optimizer=get_optimizer(model=qcritic, learning_rate=learning_rate),
        qcritic=qcritic,
        sarsa_iter=DataLoader(
            prepare_sarsa_dataset(data=data),
            batch_size=batch_size,
            drop_last=True,
            rng=numpy_rng,
        ).repeat_forever(),
        steps=sarsa_steps,
        tau=tau,
        update_every=update_every,
    )
    qcritic.eval()
    return qcritic


def init_behavior_fn(
    args: Arguments, data: OfflineData, numpy_rng: Generator, rngs: nnx.Rngs
):
    actor = init_behavior_actor_fn(
        batch_size=args.batch_size,
        bc_steps=args.bc_steps,
        data=data,
        hidden_features=args.hidden_features,
        layer_norm=args.layer_norm,
        learning_rate=args.learning_rate,
        logger=args.logger,
        num_layers=args.num_layers,
        numpy_rng=numpy_rng,
        rngs=rngs,
    )
    if args.save:
        args.logger.save_model("behavior_actor", model=actor)
    qcritic = init_behavior_qcritic_fn(
        batch_size=args.batch_size,
        data=data,
        gamma=args.gamma,
        hidden_features=args.hidden_features,
        layer_norm=args.layer_norm,
        learning_rate=args.learning_rate,
        logger=args.logger,
        num_layers=args.num_layers,
        numpy_rng=numpy_rng,
        rngs=rngs,
        sarsa_steps=args.sarsa_steps,
        tau=args.tau,
        update_every=args.update_every,
    )
    if args.save:
        args.logger.save_model("behavior_qcritic", model=qcritic)
    vcritic = init_behavior_vcritic_fn(
        batch_size=args.batch_size,
        data=data,
        gamma=args.gamma,
        hidden_features=args.hidden_features,
        layer_norm=args.layer_norm,
        learning_rate=args.learning_rate,
        logger=args.logger,
        num_layers=args.num_layers,
        numpy_rng=numpy_rng,
        rngs=rngs,
        v_learning_steps=args.v_learning_steps,
    )
    if args.save:
        args.logger.save_model("behavior_vcritic", model=vcritic)
    return actor, qcritic, vcritic


def init_fn(args: Arguments, data: OfflineDataWithInfos) -> TrainerState:
    if not args.unsquash:
        raise NotImplementedError()

    numpy_rng = default_rng(args.seed)
    rngs = default_nnx_rngs(args.seed)
    actor, qcritic, vcritic = init_behavior_fn(
        args=args, data=data.data, numpy_rng=numpy_rng, rngs=rngs
    )
    bppo_dataset = prepare_bppo_dataset(
        actor=actor, observations=data.data.observations, vcritic=vcritic
    )
    bppo_loader = DataLoader(
        bppo_dataset, batch_size=args.batch_size, drop_last=True, rng=numpy_rng
    )
    policy = BPPOPolicy(actor=actor, critic=qcritic)
    train_state = TrainState(
        model=policy,
        optimizer=get_optimizer(
            policy.actor,
            every_k_schedule=1 if args.stochastic_update else len(bppo_loader),
            learning_rate=args.learning_rate,
            max_gradient_norm=args.max_gradient_norm,
        ),
    )
    graphdef, graphstate = nnx.split(train_state)

    return TrainerState(
        bppo_loader=bppo_loader,
        eval_state=None,
        graphdef=graphdef,
        graphstate=graphstate,
        train_key=jax.random.key(args.seed),
    )


def load_fn(
    step: int | None,
    action_dim: int,
    hidden_features: int,
    layer_norm: bool,
    num_layers: int,
    logger: Logger,
    observation_dim: int,
    **kwargs,
):
    del kwargs

    def actor_model_fn():
        return GaussianActor(
            action_dim=action_dim,
            hidden_features=hidden_features,
            layer_norm=layer_norm,
            num_layers=num_layers,
            observation_dim=observation_dim,
            rngs=default_nnx_rngs(0),
        )

    if step is None:
        actor = logger.restore_model("actor", model_fn=actor_model_fn)
    else:
        actor = logger.restore_model(
            "checkpoints", f"actor_{step}", model_fn=actor_model_fn
        )
    critic = QCritic(
        action_dim=action_dim,
        hidden_features=hidden_features,
        layer_norm=layer_norm,
        num_layers=num_layers,
        observation_dim=observation_dim,
        rngs=default_nnx_rngs(0),
    )
    policy = BPPOPolicy(actor=actor, critic=critic)
    del policy.critic
    return policy, None


if __name__ == "__main__":
    base.run(
        arguments_class=Arguments,
        init_fn=init_fn,
        train_fn=train_fn,
        **vars(build_argument_parser().parse_args()),
    )
