from dataclasses import dataclass

from flax import nnx
import jax.random
from numpy.random import default_rng
from scipy.stats import chi2

from offline import base
from offline.lbp import arguments as lbp
from offline.lbp.__main__ import load_fn
from offline.lbp.modules import (
    ActorCriticFilter,
    ActorFilter,
    LBPPolicy,
    LBPTrainState,
)
from offline.lbp.train import TrainerState, train_fn
from offline.lbp.utils import prepare_noise_qlearning_dataset
from offline.modules.actor.base import GaussianActor
from offline.modules.base import TargetModel, get_optimizer
from offline.modules.critic import VCritic
from offline.types import OfflineData, OfflineDataWithInfos
from offline.utils.data import DataLoader
from offline.utils.logger import Logger
from offline.utils.nnx import default_nnx_rngs


__all__ = ["load_fn"]


@dataclass(frozen=True)
class Arguments(lbp.Arguments):
    parent_logger: Logger


def build_argument_parser(**kwargs):
    parser = lbp.build_argument_parser(
        extra=True,
        fix_keys=(
            "env",
            "hidden_features",
            "layer_norm",
            "normalize_observations",
            "num_layers",
        ),
        **kwargs,
    )
    return parser


def init_behavior_fn(args: Arguments, data: OfflineData):
    def actor_model_fn():
        return GaussianActor(
            action_dim=data.actions.shape[1],
            hidden_features=args.hidden_features,
            layer_norm=args.layer_norm,
            num_layers=args.num_layers,
            observation_dim=data.observations.shape[1],
            rngs=default_nnx_rngs(0),
        )

    actor = args.parent_logger.restore_model(
        "behavior_actor", model_fn=actor_model_fn
    )

    def vcritic_model_fn():
        return VCritic(
            hidden_features=args.hidden_features,
            layer_norm=args.layer_norm,
            num_layers=args.num_layers,
            observation_dim=data.observations.shape[1],
            rngs=default_nnx_rngs(0),
        )

    vcritic = args.parent_logger.restore_model(
        "behavior_vcritic", model_fn=vcritic_model_fn
    )
    args.parent_logger.wait()
    return actor, vcritic


def init_fn(args: Arguments, data: OfflineDataWithInfos) -> TrainerState:
    if not args.unsquash:
        raise NotImplementedError()

    numpy_rng = default_rng(args.seed)
    behavior_actor, behavior_vcritic = init_behavior_fn(
        args=args, data=data.data
    )
    noise_dataset, q_learning_dataset = prepare_noise_qlearning_dataset(
        actor=behavior_actor,
        critic=behavior_vcritic,
        data=data.data,
        sparse=args.sparse,
    )
    policy = LBPPolicy(
        action_dim=data.data.actions.shape[1],
        behavior_actor=behavior_actor,
        deterministic=args.deterministic,
        ensemble_size=args.ensemble_size,
        hidden_features=args.hidden_features,
        layer_norm=args.layer_norm,
        num_layers=args.num_layers,
        observation_dim=data.data.observations.shape[1],
        rngs=default_nnx_rngs(args.seed),
    )
    train_state = LBPTrainState(
        actor_optimizer=get_optimizer(
            policy, learning_rate=args.learning_rate, wrt=ActorFilter
        ),
        critic_optimizer=get_optimizer(
            policy.critic, learning_rate=args.learning_rate
        ),
        policy=policy,
        target_policy=TargetModel(policy, poi=ActorCriticFilter),
    )
    graphdef, graphstate = nnx.split(train_state)
    train_actor_key, train_critic_key, train_regularizer_key = jax.random.split(
        jax.random.key(args.seed), 3
    )

    return TrainerState(
        eval_state=None,
        graphdef=graphdef,
        graphstate=graphstate,
        min_target=float(data.data.rewards.min()) / (1 - args.gamma),
        noise_iter=DataLoader(
            noise_dataset,
            batch_size=args.batch_size,
            drop_last=True,
            rng=numpy_rng,
        ).repeat_forever(),
        ood_threshold=float(
            chi2.isf(args.ood_probability, data.data.actions.shape[1])
        ),
        q_learning_iter=DataLoader(
            q_learning_dataset,
            batch_size=args.batch_size,
            drop_last=True,
            rng=numpy_rng,
        ).repeat_forever(),
        train_actor_key=train_actor_key,
        train_critic_key=train_critic_key,
        train_regularizer_key=train_regularizer_key,
    )


def postprocess_args(**kwargs):
    kwargs["parent_logger"] = kwargs["logger"].parent
    return kwargs


if __name__ == "__main__":
    base.run(
        arguments_class=Arguments,
        init_fn=init_fn,
        train_fn=train_fn,
        **postprocess_args(**vars(build_argument_parser().parse_args())),
    )
