from flax import nnx
import jax.random
from numpy.random import Generator, default_rng
from optax import cosine_decay_schedule
from scipy.stats import chi2

from offline import base
from offline.modules.actor.base import GaussianActor
from offline.modules.base import TargetModel, get_optimizer
from offline.modules.critic import VCritic
from offline.lbp.arguments import Arguments, build_argument_parser
from offline.lbp.modules import (
    ActorCriticFilter,
    ActorFilter,
    LBPPolicy,
    LBPTrainState,
)
from offline.lbp.train import (
    TrainerState,
    behavior_cloning_fn,
    pretrain_critic_fn,
    train_fn,
    v_learning_fn,
)
from offline.lbp.utils import (
    get_max_min_reward,
    normalize_rewards,
    prepare_actor_regularizer_qlearning_dataset,
    prepare_v_learning_dataset,
)
from offline.types import OfflineData, OfflineDataWithInfos
from offline.utils.data import DataLoader
from offline.utils.dataset import prepare_sa_dataset
from offline.utils.logger import Logger
from offline.utils.nnx import default_nnx_rngs


def init_behavior_actor_fn(
    batch_size: int,
    bc_steps: int,
    data: OfflineData,
    hidden_features: int,
    layer_norm: bool,
    learning_rate: float,
    logger: Logger,
    num_layers: int,
    numpy_rng: Generator,
    rngs: nnx.Rngs,
) -> GaussianActor:
    actor = GaussianActor(
        action_dim=data.actions.shape[1],
        hidden_features=hidden_features,
        layer_norm=layer_norm,
        num_layers=num_layers,
        observation_dim=data.observations.shape[1],
        rngs=rngs,
    )
    actor = behavior_cloning_fn(
        actor=actor,
        logger=logger,
        optimizer=get_optimizer(model=actor, learning_rate=learning_rate),
        sa_iter=DataLoader(
            prepare_sa_dataset(data=data),
            batch_size=batch_size,
            drop_last=True,
            rng=numpy_rng,
        ).repeat_forever(),
        steps=bc_steps,
    )
    actor.eval()
    return actor


def init_behavior_vcritic_fn(
    batch_size: int,
    data: OfflineData,
    gamma: float,
    hidden_features: int,
    layer_norm: bool,
    learning_rate: float,
    logger: Logger,
    num_layers: int,
    numpy_rng: Generator,
    rngs: nnx.Rngs,
    v_learning_steps: int,
) -> VCritic:
    vcritic = VCritic(
        hidden_features=hidden_features,
        layer_norm=layer_norm,
        num_layers=num_layers,
        observation_dim=data.observations.shape[1],
        rngs=rngs,
    )
    vcritic = v_learning_fn(
        logger=logger,
        optimizer=get_optimizer(model=vcritic, learning_rate=learning_rate),
        steps=v_learning_steps,
        v_learning_iter=DataLoader(
            prepare_v_learning_dataset(data=data, gamma=gamma),
            batch_size=batch_size,
            drop_last=True,
            rng=numpy_rng,
        ).repeat_forever(),
        vcritic=vcritic,
    )
    vcritic.eval()
    return vcritic


def init_behavior_fn(
    args: Arguments, data: OfflineData, numpy_rng: Generator, rngs: nnx.Rngs
):
    actor = init_behavior_actor_fn(
        batch_size=args.batch_size,
        bc_steps=args.bc_steps,
        data=data,
        hidden_features=args.hidden_features,
        layer_norm=args.layer_norm,
        learning_rate=args.learning_rate,
        logger=args.logger,
        num_layers=args.num_layers,
        numpy_rng=numpy_rng,
        rngs=rngs,
    )
    if args.save:
        args.logger.save_model("behavior_actor", model=actor)
    vcritic = init_behavior_vcritic_fn(
        batch_size=args.batch_size,
        data=data,
        gamma=args.gamma,
        hidden_features=args.hidden_features,
        layer_norm=args.layer_norm,
        learning_rate=args.learning_rate,
        logger=args.logger,
        num_layers=args.num_layers,
        numpy_rng=numpy_rng,
        rngs=rngs,
        v_learning_steps=args.v_learning_steps,
    )
    if args.save:
        args.logger.save_model("behavior_vcritic", model=vcritic)
    return actor, vcritic


def init_fn(args: Arguments, data: OfflineDataWithInfos) -> TrainerState:
    if not args.unsquash:
        raise NotImplementedError()

    data_without_info = data.data
    max_reward, min_reward = get_max_min_reward(args.dataset)
    if max_reward is None:
        max_reward = float(data_without_info.rewards.max())
    if min_reward is None:
        min_reward = float(data_without_info.rewards.min())

    if args.normalize_rewards:
        coefficient = args.reward_multiplier * normalize_rewards(
            data=data.data, gamma=args.gamma
        )
        data_without_info = data_without_info._replace(
            rewards=coefficient * data_without_info.rewards
        )
        max_reward = coefficient * max_reward
        min_reward = coefficient * min_reward

    numpy_rng = default_rng(args.seed)
    rngs = default_nnx_rngs(args.seed)
    behavior_actor, behavior_vcritic = init_behavior_fn(
        args=args, data=data_without_info, numpy_rng=numpy_rng, rngs=rngs
    )
    actor_dataset, regularizer_dataset, q_learning_dataset = (
        prepare_actor_regularizer_qlearning_dataset(
            actor=behavior_actor,
            critic=behavior_vcritic,
            data=data_without_info,
            offset=0 if args.sparse else (max_reward - min_reward),
        )
    )
    q_learning_loader = DataLoader(
        q_learning_dataset,
        batch_size=args.batch_size,
        drop_last=True,
        rng=numpy_rng,
    )
    policy = LBPPolicy(
        action_dim=data_without_info.actions.shape[1],
        behavior_actor=behavior_actor,
        ensemble_size=args.ensemble_size,
        hidden_features=args.hidden_features,
        layer_norm=args.layer_norm,
        num_layers=args.num_layers,
        observation_dim=data_without_info.observations.shape[1],
        rngs=rngs,
        zero_mean=args.zero_mean,
    )
    if args.constant_schedule:
        schedule = args.learning_rate
    else:
        schedule = cosine_decay_schedule(
            args.learning_rate, args.total_steps // args.update_every
        )
    if args.pretrain_steps > 0:
        policy.critic = pretrain_critic_fn(
            gamma=args.gamma,
            logger=args.logger,
            optimizer=get_optimizer(
                policy.critic,
                learning_rate=args.learning_rate,
                max_gradient_norm=args.max_gradient_norm,
            ),
            qcritic=policy.critic,
            q_learning_iter=q_learning_loader.repeat_forever(),
            steps=args.pretrain_steps,
            tau=args.tau,
            update_every=args.update_every,
        )
    train_state = LBPTrainState(
        actor_optimizer=get_optimizer(
            policy,
            learning_rate=schedule,
            max_gradient_norm=args.max_gradient_norm,
            wrt=ActorFilter,
        ),
        critic_optimizer=get_optimizer(
            policy.critic,
            learning_rate=args.learning_rate,
            max_gradient_norm=args.max_gradient_norm,
        ),
        policy=policy,
        target_policy=TargetModel(policy, poi=ActorCriticFilter),
    )
    graphdef, graphstate = nnx.split(train_state)
    train_actor_key, train_critic_key, train_regularizer_key = jax.random.split(
        jax.random.key(args.seed), 3
    )

    return TrainerState(
        actor_iter=DataLoader(
            actor_dataset,
            batch_size=args.batch_size,
            drop_last=True,
            rng=numpy_rng,
        ).repeat_forever(),
        eval_state=None,
        graphdef=graphdef,
        graphstate=graphstate,
        min_target=min_reward / (1 - args.gamma),
        ood_threshold=float(
            chi2.isf(args.ood_probability, data_without_info.actions.shape[1])
        ),
        q_learning_iter=DataLoader(
            q_learning_dataset,
            batch_size=args.batch_size,
            drop_last=True,
            rng=numpy_rng,
        ).repeat_forever(),
        regularizer_iter=DataLoader(
            regularizer_dataset,
            batch_size=args.batch_size,
            drop_last=True,
            rng=numpy_rng,
        ).repeat_forever(),
        train_actor_key=train_actor_key,
        train_critic_key=train_critic_key,
        train_regularizer_key=train_regularizer_key,
    )


def load_fn(
    step: int | None,
    action_dim: int,
    ensemble_size: int,
    hidden_features: int,
    layer_norm: bool,
    num_layers: int,
    logger: Logger,
    observation_dim: int,
    zero_mean: bool,
    **kwargs,
):
    del kwargs

    def behavior_actor_model_fn():
        return GaussianActor(
            action_dim=action_dim,
            hidden_features=hidden_features,
            layer_norm=layer_norm,
            num_layers=num_layers,
            observation_dim=observation_dim,
            rngs=default_nnx_rngs(0),
        )

    behavior_actor = logger.restore_model(
        "behavior_actor", model_fn=behavior_actor_model_fn
    )

    def model_fn():
        return LBPPolicy(
            action_dim=action_dim,
            behavior_actor=behavior_actor,
            ensemble_size=ensemble_size,
            hidden_features=hidden_features,
            layer_norm=layer_norm,
            num_layers=num_layers,
            observation_dim=observation_dim,
            rngs=default_nnx_rngs(0),
            zero_mean=zero_mean,
        )

    policy = base.default_load_fn(
        logger=logger, model_fn=model_fn, poi=LBPPolicy.POI, step=step
    )
    del policy.critic
    return policy, None


if __name__ == "__main__":
    base.run(
        arguments_class=Arguments,
        init_fn=init_fn,
        skip_reward_normalization=True,
        train_fn=train_fn,
        **vars(build_argument_parser().parse_args()),
    )
