from dataclasses import dataclass

from flax import nnx
import jax.random
from numpy.random import default_rng
from optax import cosine_decay_schedule

from offline import base
from offline.lbp.utils import get_max_min_reward
from offline.svr import arguments as svr
from offline.svr.__main__ import load_fn
from offline.svr.modules import ActorFilter, SVRPolicy, SVRTrainState
from offline.svr.train import TrainerState, train_fn
from offline.svr.utils import prepare_svr_dataset
from offline.modules.actor.base import DeterministicActor
from offline.modules.base import TargetModel, get_optimizer
from offline.types import OfflineData, OfflineDataWithInfos
from offline.utils.data import ArrayDataLoader, DataLoader
from offline.utils.dataset import prepare_q_learning_dataset
from offline.utils.logger import Logger
from offline.utils.nnx import default_nnx_rngs


__all__ = ["load_fn"]


@dataclass(frozen=True)
class Arguments(svr.Arguments):
    parent_logger: Logger


def build_argument_parser(**kwargs):
    parser = svr.build_argument_parser(
        extra=True,
        fix_keys=(
            "dataset",
            "hidden_features",
            "layer_norm",
            "normalize_observations",
            "normalize_rewards",
            "num_layers",
            "reward_multiplier",
        ),
        **kwargs,
    )
    return parser


def init_behavior_fn(args: Arguments, data: OfflineData):
    def actor_model_fn():
        return DeterministicActor(
            action_dim=data.actions.shape[1],
            hidden_features=args.hidden_features,
            layer_norm=args.layer_norm,
            num_layers=args.num_layers,
            observation_dim=data.observations.shape[1],
            rngs=default_nnx_rngs(0),
            squash=True,
        )

    actor = args.parent_logger.restore_model(
        "behavior_actor", model_fn=actor_model_fn
    )
    args.parent_logger.wait()
    return actor


def init_fn(args: Arguments, data: OfflineDataWithInfos) -> TrainerState:
    if args.unsquash:
        raise NotImplementedError()

    data_without_info = data.data

    _, min_reward = get_max_min_reward(args.dataset)
    if min_reward is None:
        min_reward = float(data_without_info.rewards.min())

    numpy_rng = default_rng(args.seed)
    rngs = default_nnx_rngs(args.seed)
    behavior_actor = init_behavior_fn(args=args, data=data_without_info)
    svr_dataset = prepare_svr_dataset(
        actor=behavior_actor, data=data_without_info, sample_std=args.sample_std
    )
    policy = SVRPolicy(
        action_dim=data_without_info.actions.shape[1],
        ensemble_size=args.ensemble_size,
        hidden_features=args.hidden_features,
        layer_norm=args.layer_norm,
        num_layers=args.num_layers,
        observation_dim=data_without_info.observations.shape[1],
        rngs=rngs,
    )
    if args.constant_schedule:
        schedule = args.learning_rate
    else:
        schedule = cosine_decay_schedule(
            args.learning_rate, args.total_steps // args.update_every
        )
    train_state = SVRTrainState(
        actor_optimizer=get_optimizer(
            policy,
            learning_rate=schedule,
            max_gradient_norm=args.max_gradient_norm,
            wrt=ActorFilter,
        ),
        critic_optimizer=get_optimizer(
            policy.critic,
            learning_rate=args.learning_rate,
            max_gradient_norm=args.max_gradient_norm,
        ),
        policy=policy,
        target_policy=TargetModel(policy),
    )
    graphdef, graphstate = nnx.split(train_state)
    train_key = jax.random.key(args.seed)

    return TrainerState(
        eval_state=None,
        graphdef=graphdef,
        graphstate=graphstate,
        svr_iter=DataLoader(
            svr_dataset,
            batch_size=args.batch_size,
            drop_last=True,
            rng=numpy_rng,
        ).repeat_forever(),
        targets_regularizer=min_reward / (1 - args.gamma),
        train_key=train_key,
    )

def postprocess_args(**kwargs):
    kwargs["parent_logger"] = kwargs["logger"].parent
    return kwargs


if __name__ == "__main__":
    base.run(
        arguments_class=Arguments,
        init_fn=init_fn,
        skip_reward_normalization=True,
        train_fn=train_fn,
        **postprocess_args(**vars(build_argument_parser().parse_args())),
    )
