import torch.nn as nn
import torch.nn.functional as F
import torch


''' ConvNet '''
class ConvNet(nn.Module):
    def __init__(self, channel, num_classes, net_width, net_depth, net_act, net_norm, net_pooling, im_size = (32,32)):
        super(ConvNet, self).__init__()

        self.features, shape_feat = self._make_layers(channel, net_width, net_depth, net_norm, net_act, net_pooling, im_size)
        num_feat = shape_feat[0]*shape_feat[1]*shape_feat[2]
        self.classifier = nn.Linear(num_feat, num_classes)

    def forward(self, x):
        # print("MODEL DATA ON: ", x.get_device(), "MODEL PARAMS ON: ", self.classifier.weight.data.get_device())
        out = self.features(x)
        out = out.view(out.size(0), -1)
        out = self.classifier(out)
        return out

    def _get_activation(self, net_act):
        if net_act == 'sigmoid':
            return nn.Sigmoid()
        elif net_act == 'relu':
            return nn.ReLU(inplace=True)
        elif net_act == 'leakyrelu':
            return nn.LeakyReLU(negative_slope=0.01)
        else:
            exit('unknown activation function: %s'%net_act)

    def _get_pooling(self, net_pooling):
        if net_pooling == 'maxpooling':
            return nn.MaxPool2d(kernel_size=2, stride=2)
        elif net_pooling == 'avgpooling':
            return nn.AvgPool2d(kernel_size=2, stride=2)
        elif net_pooling == 'none':
            return None
        else:
            exit('unknown net_pooling: %s'%net_pooling)

    def _get_normlayer(self, net_norm, shape_feat):
        # shape_feat = (c*h*w)
        if net_norm == 'batchnorm':
            return nn.BatchNorm2d(shape_feat[0], affine=True)
        elif net_norm == 'layernorm':
            return nn.LayerNorm(shape_feat, elementwise_affine=True)
        elif net_norm == 'instancenorm':
            return nn.GroupNorm(shape_feat[0], shape_feat[0], affine=True)
        elif net_norm == 'groupnorm':
            return nn.GroupNorm(4, shape_feat[0], affine=True)
        elif net_norm == 'none':
            return None
        else:
            exit('unknown net_norm: %s'%net_norm)

    def _make_layers(self, channel, net_width, net_depth, net_norm, net_act, net_pooling, im_size):
        layers = []
        in_channels = channel
        shape_feat = [in_channels, im_size[0], im_size[1]]
        for d in range(net_depth):
            layers += [nn.Conv2d(in_channels, net_width, kernel_size=3, padding=3 if channel == 1 and d == 0 else 1)]
            shape_feat[0] = net_width
            if net_norm != 'none':
                layers += [self._get_normlayer(net_norm, shape_feat)]
            layers += [self._get_activation(net_act)]
            in_channels = net_width
            if net_pooling != 'none':
                layers += [self._get_pooling(net_pooling)]
                shape_feat[1] //= 2
                shape_feat[2] //= 2


        return nn.Sequential(*layers), shape_feat
    
    def _get_layers(self):
        return self.features, self.classifier
    
    def _set_layers(self, features, classifier):
        self.features, self.classifier = features, classifier
    
    

    

''' MLP '''
class MLP(nn.Module):
    def __init__(self, channel, num_classes, net_width, net_act):
        super(MLP, self).__init__()
        self.features, shape_feat = self._make_layers(channel, net_width, net_act)
        self.classifier = nn.Linear(shape_feat, num_classes)
        
    def forward(self, x):
        out = x.view(x.size(0), -1)
        out = self.features(out)
        out = self.classifier(out)
        return out

    def _get_activation(self, net_act):
        if net_act == 'sigmoid':
            return nn.Sigmoid()
        elif net_act == 'relu':
            return nn.ReLU(inplace=True)
        elif net_act == 'leakyrelu':
            return nn.LeakyReLU(negative_slope=0.01)
        else:
            exit('unknown activation function: %s'%net_act)

    def _make_layers(self, channel, net_width, net_act):
        layers = []
        layers += [nn.Linear(28*28*1 if channel==1 else 32*32*3, net_width[0])]
        layers += [self._get_activation(net_act)]
        for d in range(len(net_width)-1):
            layers += [nn.Linear(net_width[d], net_width[d+1])]
            layers += [self._get_activation(net_act)]

        shape_feat=net_width[-1]

        return nn.Sequential(*layers), shape_feat
    
    def _get_layers(self):
        return self.features, self.classifier
    
    def _set_layers(self, features, classifier):
        self.features, self.classifier = features, classifier
        
        
''' Linear '''
class Linear(nn.Module):
    def __init__(self, channel, num_classes, im_size):
        super(Linear, self).__init__()
        in_size = channel*im_size[0]*im_size[1]
        self.classifier = nn.Linear(in_size, num_classes)
        
    def forward(self, x):
        out = x.view(x.size(0), -1)
        out = self.classifier(out)
        return out

    
    def _get_layers(self):
        return 0, self.classifier
    
    def _set_layers(self, _, classifier):
        self.classifier = classifier