import os
import pickle
from collections import OrderedDict

from dassl.data.datasets import DATASET_REGISTRY, Datum, DatasetBase
from dassl.utils import listdir_nohidden, mkdir_if_missing

from .oxford_pets import OxfordPets


@DATASET_REGISTRY.register()
class ImageNet(DatasetBase):

    dataset_dir = "imagenet"

    def __init__(self, cfg):
        root = os.path.abspath(os.path.expanduser(cfg.DATASET.ROOT)).replace('/data/user/', '/data8/user/')
        self.dataset_dir = os.path.join(root, self.dataset_dir)
        self.image_dir = os.path.join(self.dataset_dir, "images")
        clsname = cfg.DATASET.CLASSNAMES.split('/')[-1].split('.txt')[0]
        # clsname = 'increasing_classes_19'
        self.preprocessed = os.path.join(self.dataset_dir, clsname, "preprocessed.pkl")
        self.split_fewshot_dir = os.path.join(self.dataset_dir, clsname, "split_fewshot")
        mkdir_if_missing(self.split_fewshot_dir)
        self.use_first_parent = False #cfg.DATASET.TREE
        if self.use_first_parent:
            import numpy as np
            # self.mapping = np.load('/data8/user/ProTeCt/prepro/first_parents_to_leaves_mapping.npy', allow_pickle=True).tolist()
            # self.mapping = np.load('/data8/user/ProTeCt/prepro/second_parents_to_leaves_mapping.npy', allow_pickle=True).tolist()
            # self.mapping = np.load('/data8/user/ProTeCt/prepro/third_parents_to_leaves_mapping.npy', allow_pickle=True).tolist()
            self.mapping = np.load('/data8/user/ProTeCt/prepro/fifth_parents_to_leaves_mapping.npy', allow_pickle=True).tolist()
            
            unique_parents = set(self.mapping.values())
            self.label_mapping = {}
            for idx, name in enumerate(unique_parents):
                self.label_mapping[name] = idx
        if os.path.exists(self.preprocessed):
            with open(self.preprocessed, "rb") as f:
                preprocessed = pickle.load(f)
                train = preprocessed["train"]
                test = preprocessed["test"]
        else:
            text_file = os.path.join(self.dataset_dir, 'increasing_classes', cfg.DATASET.CLASSNAMES)
            classnames = self.read_classnames(text_file)
            train = self.read_data(classnames, "train")
            # Follow standard practice to perform evaluation on the val set
            # Also used as the val set (so evaluate the last-step model)
            test = self.read_data(classnames, "val")

            preprocessed = {"train": train, "test": test}
            with open(self.preprocessed, "wb") as f:
                pickle.dump(preprocessed, f, protocol=pickle.HIGHEST_PROTOCOL)
        
        num_shots = cfg.DATASET.NUM_SHOTS
        if num_shots >= 1:
            seed = cfg.SEED
            preprocessed = os.path.join(self.split_fewshot_dir, f"shot_{num_shots}-seed_{seed}.pkl")
            
            if os.path.exists(preprocessed):
                print(f"Loading preprocessed few-shot data from {preprocessed}")
                with open(preprocessed, "rb") as file:
                    data = pickle.load(file)
                    train = data["train"]
            else:
                train = self.generate_fewshot_dataset(train, num_shots=num_shots)
                data = {"train": train}
                print(f"Saving preprocessed few-shot data to {preprocessed}")
                with open(preprocessed, "wb") as file:
                    pickle.dump(data, file, protocol=pickle.HIGHEST_PROTOCOL)

        subsample = cfg.DATASET.SUBSAMPLE_CLASSES
        train, test = OxfordPets.subsample_classes(train, test, subsample=subsample)

        super().__init__(train_x=train, val=test, test=test)

    @staticmethod
    def read_classnames(text_file):
        """Return a dictionary containing
        key-value pairs of <folder name>: <class name>.
        """
        classnames = OrderedDict()
        with open(text_file, "r") as f:
            lines = f.readlines()
            for line in lines:
                line = line.strip().split(" ")
                folder = line[0]
                classname = " ".join(line[1:])
                classnames[folder] = classname
        return classnames

    def read_data(self, classnames, split_dir):
        split_dir = os.path.join(self.image_dir, split_dir)
        folders = sorted(f.name for f in os.scandir(split_dir) if f.is_dir())
        items = []

        for label, folder in enumerate(folders):
            imnames = listdir_nohidden(os.path.join(split_dir, folder))
            if folder not in classnames:
                continue
            classname = classnames[folder]
            if self.use_first_parent:                
                # print("Leaf", classname)
                clss = classname.split(' ')
                clssn = '_'.join(clss)
                clssn = clssn.replace('/', 'or')
                if clssn in ['newt', 'slot_machine', 'duck', 'scarf', 'gown', 'mountain']:
                    clssn += '_1'
                # print(clssn)
                classname = self.mapping[clssn]
                # print("Parent", classname)
                label = self.label_mapping[classname]

            for imname in imnames:
                impath = os.path.join(split_dir, folder, imname).replace('/data/user/', '/data8/user/')
                item = Datum(impath=impath, label=label, classname=classname)
                items.append(item)

        return items
