#    Copyright 2023 Haotian Liu
#
#    Licensed under the Apache License, Version 2.0 (the "License");
#    you may not use this file except in compliance with the License.
#    You may obtain a copy of the License at
#
#        http://www.apache.org/licenses/LICENSE-2.0
#
#    Unless required by applicable law or agreed to in writing, software
#    distributed under the License is distributed on an "AS IS" BASIS,
#    WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
#    See the License for the specific language governing permissions and
#    limitations under the License.


import os
import warnings
import shutil

from transformers import AutoTokenizer, AutoModelForCausalLM, AutoConfig, BitsAndBytesConfig
import torch
from llava.model import *
from llava.constants import *
from llava.train.train_utils import rank0_print

import logging
logging.getLogger('transformers').setLevel(logging.ERROR)
warnings.filterwarnings("ignore", category=UserWarning)

def load_retriever_model(model_path, model_base, model_name, doc_model_init, compute_dtype=None, device_map="auto", attn_implementation="flash_attention_2", customized_config=None, **kwargs):
    kwargs = {"device_map": device_map, **kwargs}

    if compute_dtype is not None:
        kwargs["torch_dtype"] = compute_dtype
    else:
        kwargs["torch_dtype"] = torch.float16

    if customized_config is not None:
        kwargs["config"] = customized_config

    # Load LLaVA model
    rank0_print("Loading Retrieval model...")
    # For retrieval model
    from llava.model.language_model.llava_qwen_dual_ret import LlavaQwenConfig

    lora_cfg_pretrained = LlavaQwenConfig.from_pretrained(model_path)

    if model_base is None:  # Directly loaded through model_path
        model = LlavaQwenForCausalLMWithDualRet.from_pretrained(
            model_path,
            config=lora_cfg_pretrained,
            attn_implementation=attn_implementation,
            **kwargs)

        tokenizer = AutoTokenizer.from_pretrained(model_path,
                                                  model_max_length=model.config.tokenizer_model_max_length,
                                                  padding_side="right",
                                                  use_fast=False)
    else:
        model = LlavaQwenForCausalLMWithDualRet.from_pretrained(
            model_base,
            config=lora_cfg_pretrained,
            attn_implementation=attn_implementation,
            **kwargs)
        # model_base is shared to the doc_model of the model.
        if doc_model_init:
            model.copy_model()

        tokenizer = AutoTokenizer.from_pretrained(model_base,
                                                  model_max_length=model.config.tokenizer_model_max_length,
                                                  padding_side="right",
                                                  use_fast=False)

        rank0_print("Loading additional model weights...")
        if os.path.exists(os.path.join(model_path, "non_lora_trainables.bin")):
            non_lora_trainables = torch.load(os.path.join(model_path, "non_lora_trainables.bin"), map_location="cpu")
        else:
            # this is probably from HF Hub
            from huggingface_hub import hf_hub_download

            def load_from_hf(repo_id, filename, subfolder=None):
                cache_file = hf_hub_download(repo_id=repo_id, filename=filename, subfolder=subfolder)
                return torch.load(cache_file, map_location="cpu")

            non_lora_trainables = load_from_hf(model_path, "non_lora_trainables.bin")
        non_lora_trainables = {(k[11:] if k.startswith("base_model.") else k): v for k, v in
                               non_lora_trainables.items()}
        if any(k.startswith("model.model.") for k in non_lora_trainables):
            non_lora_trainables = {(k[6:] if k.startswith("model.") else k): v for k, v in non_lora_trainables.items()}
        model.load_state_dict(non_lora_trainables, strict=False)

        from peft import PeftModel

        rank0_print("Loading LoRA weights...")
        model = PeftModel.from_pretrained(model, model_path)
        rank0_print("Merging LoRA weights...")
        model = model.merge_and_unload()
        rank0_print("Model is loaded...")

        rank0_print(f"Model Class: {model.__class__.__name__}")


    mm_use_im_start_end = getattr(model.config, "mm_use_im_start_end", False)
    mm_use_im_patch_token = getattr(model.config, "mm_use_im_patch_token", True)
    if mm_use_im_patch_token:
        tokenizer.add_tokens([DEFAULT_IMAGE_PATCH_TOKEN], special_tokens=True)
    if mm_use_im_start_end:
        tokenizer.add_tokens([DEFAULT_IM_START_TOKEN, DEFAULT_IM_END_TOKEN], special_tokens=True)

    vision_tower = model.get_vision_tower()
    if not vision_tower.is_loaded:
        vision_tower.load_model(device_map=device_map)
    if device_map != "auto":
        vision_tower.to(device="cuda", dtype=torch.float16)
    image_processor = vision_tower.image_processor

    assert model.config.tokenizer_model_max_length == 2048
    context_len = model.config.tokenizer_model_max_length

    return tokenizer, model, image_processor, context_len



def load_reranker_model(model_path, model_base, model_name, compute_dtype=None, device_map="auto", attn_implementation="flash_attention_2", customized_config=None, **kwargs):
    kwargs = {"device_map": device_map, **kwargs}

    if compute_dtype is not None:
        kwargs["torch_dtype"] = compute_dtype
    else:
        kwargs["torch_dtype"] = torch.float16

    if customized_config is not None:
        kwargs["config"] = customized_config

    # Load LLaVA model
    rank0_print("Loading Retrieval model...")
    # For retrieval model
    from llava.model.language_model.llava_qwen_dual_ret import LlavaQwenConfig

    lora_cfg_pretrained = LlavaQwenConfig.from_pretrained(model_path)

    if model_base is None:  # Directly loaded through model_path
        model = LlavaQwenForCausalLMWithDualReRank.from_pretrained(
            model_path,
            config=lora_cfg_pretrained,
            attn_implementation=attn_implementation,
            **kwargs)

        tokenizer = AutoTokenizer.from_pretrained(model_path,
                                                  model_max_length=model.config.tokenizer_model_max_length,
                                                  padding_side="right",
                                                  use_fast=False)
    else:
        model = LlavaQwenForCausalLMWithDualReRank.from_pretrained(
            model_base,
            config=lora_cfg_pretrained,
            attn_implementation=attn_implementation,
            **kwargs)

        tokenizer = AutoTokenizer.from_pretrained(model_base,
                                                  model_max_length=model.config.tokenizer_model_max_length,
                                                  padding_side="right",
                                                  use_fast=False)

        rank0_print("Loading additional model weights...")
        if os.path.exists(os.path.join(model_path, "non_lora_trainables.bin")):
            non_lora_trainables = torch.load(os.path.join(model_path, "non_lora_trainables.bin"), map_location="cpu")
        else:
            # this is probably from HF Hub
            from huggingface_hub import hf_hub_download

            def load_from_hf(repo_id, filename, subfolder=None):
                cache_file = hf_hub_download(repo_id=repo_id, filename=filename, subfolder=subfolder)
                return torch.load(cache_file, map_location="cpu")

            non_lora_trainables = load_from_hf(model_path, "non_lora_trainables.bin")
        non_lora_trainables = {(k[11:] if k.startswith("base_model.") else k): v for k, v in
                               non_lora_trainables.items()}
        if any(k.startswith("model.model.") for k in non_lora_trainables):
            non_lora_trainables = {(k[6:] if k.startswith("model.") else k): v for k, v in non_lora_trainables.items()}
        model.load_state_dict(non_lora_trainables, strict=False)

        from peft import PeftModel

        rank0_print("Loading LoRA weights...")
        model = PeftModel.from_pretrained(model, model_path)
        rank0_print("Merging LoRA weights...")
        model = model.merge_and_unload()
        rank0_print("Model is loaded...")

        rank0_print(f"Model Class: {model.__class__.__name__}")


    mm_use_im_start_end = getattr(model.config, "mm_use_im_start_end", False)
    mm_use_im_patch_token = getattr(model.config, "mm_use_im_patch_token", True)
    if mm_use_im_patch_token:
        tokenizer.add_tokens([DEFAULT_IMAGE_PATCH_TOKEN], special_tokens=True)
    if mm_use_im_start_end:
        tokenizer.add_tokens([DEFAULT_IM_START_TOKEN, DEFAULT_IM_END_TOKEN], special_tokens=True)

    vision_tower = model.get_vision_tower()
    if not vision_tower.is_loaded:
        vision_tower.load_model(device_map=device_map)
    if device_map != "auto":
        vision_tower.to(device="cuda", dtype=torch.float16)
    image_processor = vision_tower.image_processor

    assert model.config.tokenizer_model_max_length == 3072
    context_len = model.config.tokenizer_model_max_length

    return tokenizer, model, image_processor, context_len


def load_reranker_interleaved_model(model_path, model_base, model_name, compute_dtype=None, device_map="auto", attn_implementation="flash_attention_2", customized_config=None, **kwargs):
    kwargs = {"device_map": device_map, **kwargs}

    if compute_dtype is not None:
        kwargs["torch_dtype"] = compute_dtype
    else:
        kwargs["torch_dtype"] = torch.float16

    if customized_config is not None:
        kwargs["config"] = customized_config

    # Load LLaVA model
    rank0_print("Loading Retrieval model...")
    # For retrieval model
    from llava.model.language_model.llava_qwen_dual_ret import LlavaQwenConfig

    lora_cfg_pretrained = LlavaQwenConfig.from_pretrained(model_path)

    if model_base is None:  # Directly loaded through model_path
        model = LlavaQwenForCausalLMWithDualReRankInterleaved.from_pretrained(
            model_path,
            config=lora_cfg_pretrained,
            attn_implementation=attn_implementation,
            **kwargs)

        tokenizer = AutoTokenizer.from_pretrained(model_path,
                                                  model_max_length=model.config.tokenizer_model_max_length,
                                                  padding_side="right",
                                                  use_fast=False)
    else:
        model = LlavaQwenForCausalLMWithDualReRankInterleaved.from_pretrained(
            model_base,
            config=lora_cfg_pretrained,
            attn_implementation=attn_implementation,
            **kwargs)

        tokenizer = AutoTokenizer.from_pretrained(model_base,
                                                  model_max_length=model.config.tokenizer_model_max_length,
                                                  padding_side="right",
                                                  use_fast=False)

        rank0_print("Loading additional model weights...")
        if os.path.exists(os.path.join(model_path, "non_lora_trainables.bin")):
            non_lora_trainables = torch.load(os.path.join(model_path, "non_lora_trainables.bin"), map_location="cpu")
        else:
            # this is probably from HF Hub
            from huggingface_hub import hf_hub_download

            def load_from_hf(repo_id, filename, subfolder=None):
                cache_file = hf_hub_download(repo_id=repo_id, filename=filename, subfolder=subfolder)
                return torch.load(cache_file, map_location="cpu")

            non_lora_trainables = load_from_hf(model_path, "non_lora_trainables.bin")
        non_lora_trainables = {(k[11:] if k.startswith("base_model.") else k): v for k, v in
                               non_lora_trainables.items()}
        if any(k.startswith("model.model.") for k in non_lora_trainables):
            non_lora_trainables = {(k[6:] if k.startswith("model.") else k): v for k, v in non_lora_trainables.items()}
        model.load_state_dict(non_lora_trainables, strict=False)

        from peft import PeftModel

        rank0_print("Loading LoRA weights...")
        model = PeftModel.from_pretrained(model, model_path)
        rank0_print("Merging LoRA weights...")
        model = model.merge_and_unload()
        rank0_print("Model is loaded...")

        rank0_print(f"Model Class: {model.__class__.__name__}")


    mm_use_im_start_end = getattr(model.config, "mm_use_im_start_end", False)
    mm_use_im_patch_token = getattr(model.config, "mm_use_im_patch_token", True)
    if mm_use_im_patch_token:
        tokenizer.add_tokens([DEFAULT_IMAGE_PATCH_TOKEN], special_tokens=True)
    if mm_use_im_start_end:
        tokenizer.add_tokens([DEFAULT_IM_START_TOKEN, DEFAULT_IM_END_TOKEN], special_tokens=True)

    vision_tower = model.get_vision_tower()
    if not vision_tower.is_loaded:
        vision_tower.load_model(device_map=device_map)
    if device_map != "auto":
        vision_tower.to(device="cuda", dtype=torch.float16)
    image_processor = vision_tower.image_processor

    assert model.config.tokenizer_model_max_length == 32768
    context_len = model.config.tokenizer_model_max_length

    return tokenizer, model, image_processor, context_len